/*

    Linux WLAN 

    The contents of this file are subject to the Mozilla Public
    License Version 1.0 (the "License"); you may not use this file
    except in compliance with the License. You may obtain a copy of
    the License at http://www.mozilla.org/MPL/

    Software distributed under the License is distributed on an "AS
    IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
    implied. See the License for the specific language governing
    rights and limitations under the License.

    The initial developer of the original code is Mark S. Mathews
    <mark@absoval.com>.  Portions created by Mark S. Mathews
    are Copyright (C) 1998 AbsoluteValue Software, Inc.  All Rights Reserved.
    
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/* Ugly hack for LinuxPPC R4, don't have time to figure it out right now */
#if defined(_ARCH_PPC)
#undef __GLIBC__
#endif


#include <sys/socket.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <wlan/version.h>
#include <wlan/wlan_compat.h>
#include <wlan/am930mib.h>
#include <wlan/wlan_ioctl.h>
#include "wlanctl.h"
#include "prmib.h"

int		getmib( int fd, char *interface, UINT32 mibcode );
void	pr_netlist(wlan_netlist_t *list);
int		process_bsscreate( int fd, int argc, char **argv );
int		process_bssjoin( int fd, int argc, char **argv );
int		process_getmib( int fd, int argc, char **argv );
int		process_netlist( int fd, int argc, char **argv );
int		process_scan( int fd, int argc, char **argv );
int		process_privacy( int fd, int argc, char **argv );
int		process_ethconv( int fd, int argc, char **argv );
int		process_authen( int fd, int argc, char **argv );
int		process_assoc( int fd, int argc, char **argv );
void	strtoUINT48( UINT8 *a, char *s );
void	strtoUINT40( UINT8 *a, char *s );
void	usage(void);

char appname[MAX_APPNAME_LEN + 1];

int main(int argc, char *argv[])
{
	int result = 0;
	int fd;
	wlan_req_t	req;

	strcpy( appname, APPNAME );

	if ( argc >=  2 )
	{
		
		if ( strcasecmp(argv[1], "version") == 0 )
		{
			printf("%s version " WLAN_RELEASE "\n", appname);
			return result;
		}

		/* get a socket */
		fd = socket(AF_INET, SOCK_STREAM, 0);
		if ( fd != -1 )
		{
			/* Test that there is a card */
			strcpy( req.name, argv[1]);
			req.result = 0;
			req.data = NULL;
			req.len = 0;
			result = ioctl( fd, WLAN_TEST, &req);

			if (result < 0)
			{
				fprintf(stderr, "%s: ioctl failed testing interface \'%s\'\n",
						appname, argv[1]);
				fprintf(stderr, "interface \'%s\' is probably an invalid wireless network interface name\n", argv[1] );
				fprintf(stderr, "OR a wireless card is not inserted in the specified interface\n");
			}
			else
			{
				if ( req.result == 0xf0f0 )
				{
					/* now, what does the user want? */
					if ( argc > 2 )
					{
						/* ok, there's more arguments */
						if ( strcasecmp(argv[2], SCAN) == 0 )
						{
						   result = process_scan(fd, argc, argv);
						}
						else if ( strcasecmp(argv[2], NETLIST) == 0 )
						{
						   result = process_netlist(fd, argc, argv);
						}
						else if ( strcasecmp(argv[2], BSSCREATE) == 0 )
						{
						   result = process_bsscreate(fd, argc, argv);
						}
						else if ( strcasecmp(argv[2], BSSJOIN) == 0 )
						{
						   result = process_bssjoin(fd, argc, argv);
						}
						else if ( strcasecmp(argv[2], GETMIB) == 0 )
						{
						   result = process_getmib(fd, argc, argv);
						}
						else if ( strcasecmp(argv[2], PRIVACY) == 0 )
						{
						   result = process_privacy(fd, argc, argv);
						}
						else if ( strcasecmp(argv[2], ETHCONV) == 0 )
						{
						   result = process_ethconv(fd, argc, argv);
						}
						else if ( strcasecmp(argv[2], AUTHEN) == 0 )
						{
						   result = process_authen(fd, argc, argv);
						}
						else if ( strcasecmp(argv[2], ASSOC) == 0 )
						{
						   result = process_assoc(fd, argc, argv);
						}
						else if ( strcasecmp(argv[2], "getmibitem") == 0 )
						{
							fprintf(stderr, "%s: cmd currently unsupported\n", 
									appname);
						}
						else if ( strcasecmp(argv[2], "setmibitem") == 0 )
						{
							fprintf(stderr, "%s: cmd currently unsupported\n", 
									appname);
						}
						else
						{
							fprintf(stderr, "%s: %s is an invalid command\n",
									appname, argv[2]);
							usage();
							result = 1;
						}
					}
					else
					{
						/* ok, we just want a general status msg */
						/* maybe some stats, joined?, ESSID, BSSID etc. */
						printf("some general stats\n");
					}

				}
				else
				{
					fprintf(stderr,"%s: ioctl successful but driver not responding!\n", appname);
					result = 1;
				}
			}
		}
		else
		{
			fprintf(stderr, "%s: No appropriate sockets!\n", appname);
			result = 1;
		}
	}
	else
	{
		usage();
		result = 1;
	}

	return result;
}


/*----------------------------------------------------------------
*	getmib
*
*	Called by process_getmib when the user has specified the 'getmib'
*	command on the command line.
*
*	This function does the actual read the contents of one or more Managment
*	Information Block (MIB) structures from the specified wireless
*	network device and display the values.  Issuing just the getmib
*   command to wlanctl displays ALL MIB's.  Otherwise, a particular
*   MIB is specified following the getmib command.
*   
*   Valid MIB values are:
*		0x00 for local mib
*		0x02 for address mib
*		0x03 for mac (media access control) mib
*		0x04 for statistics mib
*		0x05 for managment mib
*		0x07 for physical mib
*	Arguments:
*		fd 		- socket file descriptor
*		argc  	- the argument count passed to main 
*		argv	- the argument list passed to main 
*
*	returns: 0 if successful, not 0 otherwise
*
----------------------------------------------------------------*/
int  getmib( int fd, char *interface, UINT32 mibcode )
{
	wlan_req_t 		req;
	wlan_getmib_t	mib;
	int 			result;

	strcpy( req.name, interface);
	req.len = sizeof(wlan_getmib_t);
	req.data = &mib;
	req.result = 0;
	mib.mibcode = mibcode;

	result = 0;

	switch (mibcode)
	{
		case SUMIB_LOCAL:
			if ( (result = ioctl( fd, WLAN_GETMIB, &req)) == 0 )
			{
				prlocalmib( &mib.data.local ); 
			}
			else
			{
				fprintf(stderr, "%s: ioctl failed for mibcode 0x%02lx.\n", appname, mibcode);
			}
			break;
		case SUMIB_ADDR:
			if ( (result = ioctl( fd, WLAN_GETMIB, &req)) == 0 )
			{
				praddrmib( &mib.data.addr ); 
			}
			else
			{
				fprintf(stderr, "%s: ioctl failed for mibcode 0x%02lx.\n", appname, mibcode);
			}
			break;
		case SUMIB_MAC:
			if ( (result = ioctl( fd, WLAN_GETMIB, &req)) == 0 )
			{
				prmacmib( &mib.data.mac ); 
			}
			else
			{
				fprintf(stderr, "%s: ioctl failed for mibcode 0x%02lx.\n", appname, mibcode);
			}
			break;
		case SUMIB_STAT:
			if ( (result = ioctl( fd, WLAN_GETMIB, &req)) == 0 )
			{
				prstatsmib( &mib.data.stats ); 
			}
			else
			{
				fprintf(stderr, "%s: ioctl failed for mibcode 0x%02lx.\n", appname, mibcode);
			}
			break;
		case SUMIB_MGMT:
			if ( (result = ioctl( fd, WLAN_GETMIB, &req)) == 0 )
			{
				prmgmtmib( &mib.data.mgmt ); 
			}
			else
			{
				fprintf(stderr, "%s: ioctl failed for mibcode 0x%02lx.\n", appname, mibcode);
			}
			break;
		case SUMIB_PHY:
			if ( (result = ioctl( fd, WLAN_GETMIB, &req)) == 0 )
			{
				prphymib( &mib.data.phy ); 
			}
			else
			{
				fprintf(stderr, "%s: ioctl failed for mibcode 0x%02lx.\n", appname, mibcode);
			}
			break;
		default:
			fprintf(stderr, "%s: invalid mibcode of 0x%02lx.\n", appname, mibcode);
			result = 1;
			break;
	}

	return result;
}


/*----------------------------------------------------------------
*	usage
*
*	This function displays the proper syntax of the wlanctl utility.
*
*	Arguments:
*		no arguments
*
*	returns: function doesn't return anything
*
----------------------------------------------------------------*/
void usage(void)
{
	printf("\n%s : control utility for am79C930 based DS wlan cards\n", appname);
	printf("   usage: %s interface|version [cmd [cmdarg ...]]\n\n", appname);
	printf("          where \"interface\" is the name of a wireless\n");
	printf("          network interface.  Running 'ifconfig' will list\n");
	printf("          all network interfaces.\n\n");
	printf("          Refer to wlanctl man page for more details.\n\n");
}


/*----------------------------------------------------------------
*	strtoUINT40
*
*	This function converts a character string that represents an
*	a 40 bit quantity.
*
*	Arguments:
*		a	- 40 bit buffer
*		s	- character string representing a 40 key quantity
*
*	returns: function doesn't return anything
*
----------------------------------------------------------------*/
void strtoUINT40( UINT8 *a, char *s )
{
	char	*p;
	int		i;
	UINT	val;

	for ( i = 0; i < 4; i++)
	{
		p = strchr( s, ':');
		if ( p == NULL )
		{
			fprintf(stderr, "%s: UINT48 format error!\n", appname);
			exit(1);
		}
		else
		{
			*p = '\0';
			sscanf( s, "%x", &val);
			a[i] = (UINT8)val;
			s = p+1;
		}
	}
	sscanf( s, "%x", &val);
	a[i] = (UINT8)val;
}


/*----------------------------------------------------------------
*	strtoUINT48
*
*	This function converts a character string that represents an
*	IEEE 802 address into an unsigned 48 bit number.
*
*	Arguments:
*		a	- IEEE 802 48 bit address
*		s	- character string representing a 48 bit IEEE 802 address
*
*	returns: function doesn't return anything
*
----------------------------------------------------------------*/
void strtoUINT48( UINT8 *a, char *s )
{
	char	*p;
	int		i;
	UINT	val;

	for ( i = 0; i < 5; i++)
	{
		p = strchr( s, ':');
		if ( p == NULL )
		{
			fprintf(stderr, "%s: UINT48 format error!\n", appname);
			exit(1);
		}
		else
		{
			*p = '\0';
			sscanf( s, "%x", &val);
			a[i] = (UINT8)val;
			s = p+1;
		}
	}
	sscanf( s, "%x", &val);
	a[i] = (UINT8)val;
}


/*----------------------------------------------------------------
*	pr_netlist
*
*	Called when the user has requested the 'netlist' command at the
*	command line.
*
*	This function will traverse the list of BSS's in the network
*	list and print the channel, the bssid, the beacon interval,
*	the capabilities information, and the ssid for each BSS.
*
*	Arguments:
*		list	- network list containing all current BSS's
*
*
*	returns: function doesn't return anything
*
----------------------------------------------------------------*/
void pr_netlist(wlan_netlist_t *list)
{
	int 			i;
	char 			buf[80];
	wlan_ie_ssid_t *ie;

	for ( i = 0; i < list->nitems; i++)
	{
/*		printf("ch  bssid              bcn   cap_info  ssid\n");
		printf("------------------------------------------------------\n"); */
		printf("%02d  ", list->netlist[i].channel);
		mac2str( list->netlist[i].bssid, buf);
		printf("%s  ", buf);
		printf("%04d  ", list->netlist[i].bcn_int);
		printf("0x%04x  ", list->netlist[i].cap_info);
		ie = (wlan_ie_ssid_t*)list->netlist[i].ssid;
		memcpy(buf, ie->ssid, ie->len);
		buf[ie->len] = '\0';
		printf("%s\n", buf);
	}
}


/*----------------------------------------------------------------
*	process_scan
*
*	Called when the user has requested the 'scan' command at the
*	command line.
*
*	This function will scan the range of channels beginning and
*	ending with	"start" channel and "end" channel repsectively
*	for the specified basic service set id (BSSID).  The user
*	must specify the starting and ending channels to scan, how
*	long to scan on each channel, what BSS to scan for, the BSS
*   type and the scan type.
*   
*	When complete, issuing the "netlist" command will display the
*	list of BSS's found as a result of the scan.
*
*	Arguments:
*		fd 		- socket file descriptor
*		argc  	- the argument count passed to main 
*		argv	- the argument list passed to main 
*
*	The arguments for the wlanctl scan command as they should appear
*	on the command line in the following order are:
*		name - name of wireless network interface 
*       cmd - wlanctl command being called, in this case, it is 'scan'
*		startch - first channel to start scan
*		endch - last channel to scan
*		timech - time, in microseconds, to scan each channel
*		bssid - a colon separated IEEE 802 address indicating which
*				bss to scan for.  An address of ff:ff:ff:ff:ff:ff
*				scans for ALL bss's.
*		bsstype - specifies the type of bss to scan for.  Valid values
*				are "ind", "infr" or both, where "ind" is an independent
*				bss, namely funtioning in ad-hoc mode, and "infr" is
*				an infrastructure bss, namely we're scanning for access points.
*				Note that both "ind" and "infr" bss's can be operable
*				simultaneously; hence, the scan for "both".
*		scantype - specifies the type of scan to perform: active or passive
*				active - sends out probe requests expecting a probe response
*				passive - listens for beacons without sending probe requests
*		ssid - "Service Set Identifier": a string (with a maximum
*			length of 15 characters) that identifies the new BSS.
*			This is not to be confused with the BSSID, namely the
*			IEEE 802 48 bit address.
*
*	returns: 0 if successful, not 0 otherwise
*
----------------------------------------------------------------*/
int process_scan( int fd, int argc, char **argv )
{
	int			result;

	result = 0;

	if ( argc < 10 )
	{
		fprintf(stderr, "%s: not enough arguments for scan command; ",
				appname);
		fprintf(stderr, "see wlanctl man page.\n");
		usage();
	}
	else
	{
		wlan_req_t	req;
		wlan_scan_t cmd;

		strcpy( req.name, argv[1]);
		req.result = 0;
		req.data = &cmd;
		req.len = sizeof(cmd);
		cmd.startch = atoi( argv[3] );
		cmd.endch = atoi( argv[4] );
		cmd.timech = atoi( argv[5] );
		strtoUINT48( cmd.bssid, argv[6] );

		if (strcmp(argv[7], "both") == 0 )
		{
			cmd.bsstype = BSSTYPE_IND | BSSTYPE_INFRA;
		}
		else if ( strcmp(argv[7], "independent") == 0 )
		{
			cmd.bsstype = BSSTYPE_IND;
		}
		else
		{
			cmd.bsstype = BSSTYPE_INFRA;
		}

		if ( strcmp( argv[8], "active") == 0 )
		{
			cmd.scantype = 0;
		}
		else
		{
			cmd.scantype = 1;
		}

		cmd.ssid[0] = WLAN_EID_SSID;
		if ( (cmd.ssid[1] = strlen(argv[9])) <= 32 )
		{
			memcpy( &(cmd.ssid[2]), argv[9], cmd.ssid[1]);
		}
		else
		{
			fprintf(stderr, "warning: ssid longer than 32 chars, using wildcard.");
			cmd.ssid[1] = 0;
			cmd.ssid[2] = 0;
		}

		result = ioctl( fd, WLAN_SCAN, &req);

		switch( req.result)
		{
			case 0: 
				process_netlist( fd, argc, argv );
				break;
			case 1: 
				fprintf(stderr, "%s: command discarded, driver is already scanning\n", appname);
				break;
			case 2: 
				fprintf(stderr, "%s: h/w scan command failed!\n", appname);
				break;
			default:
				fprintf(stderr, "%s: unknown error.\n", appname);
				break;
		}
	}
	
    return result;
}


/*----------------------------------------------------------------
*	process_getmib
*
*	Called when the user has requested the 'getmib' command at the
*	command line.
*
*	This function calls getmib to read the contents of one or more
*   Managment Information Block (MIB) structures from the specified
*   wireless network device and display the values.  Issuing just the
*   getmib command to wlanctl displays ALL MIB's.  Otherwise, a
*   particular MIB(s) is specified following the getmib command.
*   
*   Valid MIB values are:
*		0x00 for local mib
*		0x02 for address mib
*		0x03 for mac (media access control) mib
*		0x04 for statistics mib
*		0x05 for managment mib
*		0x07 for physical mib
*	Arguments:
*		fd 		- socket file descriptor
*		argc  	- the argument count passed to main 
*		argv	- the argument list passed to main 
*
*	returns: 0 if successful, not 0 otherwise
*
----------------------------------------------------------------*/
int process_getmib( int fd, int argc, char **argv )
{
	int			result;

	result = 0;

	/* ok, which mib's? */
	if ( argc >= 4 )
	{
		int		i;
		UINT32	mibcode;

		for ( i = 3; i < argc; i++ )
		{
		   mibcode = atoi(argv[i]);
		   result = getmib( fd, argv[1], mibcode );
		}
	}
	else
	{
		/* OK, print them all */
		result = getmib( fd, argv[1], SUMIB_LOCAL);
		result = getmib( fd, argv[1], SUMIB_ADDR);
		result = getmib( fd, argv[1], SUMIB_MAC);
		result = getmib( fd, argv[1], SUMIB_STAT);
		result = getmib( fd, argv[1], SUMIB_MGMT);
		result = getmib( fd, argv[1], SUMIB_PHY);
	}

	return result;
}


/*----------------------------------------------------------------
*	process_netlist
*
*	Called when the user has requested the 'netlist' command at the
*	command line.
*
*	This function prints all the BSS's in the current list of BSS's
*	generated from the last scan.
*   
*	Arguments:
*		fd 		- socket file descriptor
*		argc  	- the argument count passed to main 
*		argv	- the argument list passed to main 
*
*	returns: 0 if successful, not 0 otherwise
*
----------------------------------------------------------------*/
int process_netlist( int fd, int argc, char **argv )
{
	int					result;
	wlan_req_t			req;
	wlan_netlist_len_t	cmd1;
	wlan_netlist_t		*cmd2;

	result = 0;
	strcpy( req.name, argv[1]);
	req.result = 0;
	req.data = &cmd1;
	req.len = sizeof(cmd1);
	result = ioctl( fd, WLAN_NETLIST_LEN, &req);

	if ( result || req.result )
	{
		fprintf(stderr,"%s: ioctl failed in netlist_len command.\n", appname);
	}
	else
	{
		cmd2 = malloc( sizeof(wlan_netlist_t) + (cmd1.nitems * sizeof(netitem_t)));
		cmd2->nitems = cmd1.nitems;
		req.result = 0;
		req.data = cmd2;
		req.len = sizeof(wlan_netlist_t) + (cmd1.nitems * sizeof(netitem_t));
		result = ioctl( fd, WLAN_NETLIST, &req);

		if ( result || req.result )
		{
			fprintf(stderr,"%s: ioctl failed in netlist command\n", appname);
		}
		else
		{
			pr_netlist( cmd2 );
		}
	}

	return result;
}


/*----------------------------------------------------------------
*	process_bsscreate
*
*	Called when the user has requested the 'bsscreate' command at the
*	command line.
*
*	This function creates a BSS based on the arguments passed to the
*   command.  The arguments are in the order that follows:
*		channel - the channel number to establish the bss on.
*		beacon interval - time (in microseconds) between the
*			transmission of beacon frames in the new BSS.
*		atim window - "Announcement Traffic Indication Message"
*			The atim window is the period of time following a target
*			beacon transmission time (TBTT) that power saving MAC
*			entities should remain awake listening for atim frames.
*			Units for the atim window are 1024 microseconds.
*		ssid - "Service Set Identifier": a string (with a maximum
*			length of 15 characters) that identifies the new BSS.
*			This is not to be confused with the BSSID, namely the
*			IEEE 802 48 bit address.
*   
*	Arguments:
*		fd 		- socket file descriptor
*		argc  	- the argument count passed to main 
*		argv	- the argument list passed to main 
*
*	returns: 0 if successful, not 0 otherwise
*
----------------------------------------------------------------*/
int process_bsscreate( int fd, int argc, char **argv )
{
	int					result;

	result = 0;

	if (argc < 7)
	{
		fprintf(stderr,"%s: not enough arguments for bsscreate command; ",
				appname);
		fprintf(stderr, "see wlanctl man page.\n");
		usage();
	}
	else
	{
		wlan_req_t	req;
		wlan_bsscreate_t cmd;

		strcpy( req.name, argv[1]);
		req.result = 0;
		req.data = &cmd;
		req.len = sizeof(cmd);
		cmd.channel = atoi(argv[3]);
		cmd.beacon_int = atoi(argv[4]);
		cmd.atim_win = atoi(argv[5]);
		cmd.ssid[0] = WLAN_EID_SSID;
		cmd.ssid[1] = strlen(argv[6]);
		strcpy( &(cmd.ssid[2]), argv[6]);
		result = ioctl( fd, WLAN_BSSCREATE, &req);

		if ( result )
		{
			fprintf(stderr,"%s: ioctl failed for bsscreate command.\n",
					appname);
		}
	}

	return result;
}


/*----------------------------------------------------------------
*	process_bssjoin
*
*	Called when the user has requested the 'bssjoin' command at the
*	command line.
*
*	This function forces the wireless network device to join
*	(syncronize) with the named BSS.  The named BSS provided as an
*	argument to the bssjoin command must be a valid BSS in the
*	network list.
*   
*	Arguments:
*		fd 		- socket file descriptor
*		argc  	- the argument count passed to main 
*		argv	- the argument list passed to main 
*
*	returns: 0 if successful, not 0 otherwise
*
----------------------------------------------------------------*/
int process_bssjoin( int fd, int argc, char **argv )
{
	int					result;
	wlan_req_t			req;
	wlan_bssjoin_t		cmd;

	result = 0;

	if ( argc < 4 )
	{
		fprintf(stderr, 
			"%s: not enough arguments for the bssjoin command; ", appname);
		fprintf(stderr, "see wlanctl man page.\n");
		usage();
		result = 1;
	}
	else
	{
		strcpy( req.name, argv[1]);
		req.result = 0;
		req.data = &cmd;
		req.len = sizeof(cmd);
		strtoUINT48( cmd.bssid, argv[3] );

		result = ioctl( fd, WLAN_BSSJOIN, &req);

		if ( req.result )
		{
			fprintf(stderr,
			"%s: Either the bssid was invalid or the sync command failed.\n",
				appname);
			result = 1;
		}
	}

	return result;
}


/*----------------------------------------------------------------
*	process_privacy
*
*	Called when the user has requested the 'privacy' command at the
*	command line.
*
*	This function collects the privacy settings and sends them down
*	to the driver.
*   
*	Arguments:
*		fd 		- socket file descriptor
*		argc  	- the argument count passed to main 
*		argv	- the argument list passed to main 
*
*	returns: 0 if successful, not 0 otherwise
*
----------------------------------------------------------------*/
int process_privacy( int fd, int argc, char **argv )
{
	int					result;
	wlan_req_t			req;
	wlan_privacy_t		cmd;
	int					i;


	result = 0;

	if ( argc < 4 )
	{
		fprintf(stderr, 
			"%s: not enough arguments for the bssjoin command; ", appname);
		fprintf(stderr, "see wlanctl man page.\n");
		usage();
		result = 1;
	}
	else
	{
		strcpy( req.name, argv[1]);
		req.result = 0;
		req.data = &cmd;
		req.len = sizeof(cmd);

		cmd.defkey = atoi(argv[3]);
		cmd.exclude_unencrypted = (strcmp(argv[4], "exclude_unenecrypted")==0);
		for ( i = 0; i < 4; i++)
		{
			strtoUINT40( cmd.keys[i], argv[i + 5] );
		}

		result = ioctl( fd, WLAN_PRIVACY, &req);

		if ( req.result )
		{
			fprintf(stderr,
			"%s: There's a bad arg or the device isn't present.\n",
				appname);
			result = 1;
		}
	}

	return result;
}

/*----------------------------------------------------------------
*	process_ethconv
*
*	Called when the user wants to change the default ethernet 
*	conversion.
*
*	Arguments:
*		fd 		- socket file descriptor
*		argc  	- the argument count passed to main 
*		argv	- the argument list passed to main 
*
*	returns: 0 if successful, not 0 otherwise
*
----------------------------------------------------------------*/
int process_ethconv( int fd, int argc, char **argv )
{
	int					result;
	wlan_req_t			req;
	wlan_ethconv_t		cmd;


	result = 0;

	if ( argc < 4 )
	{
		fprintf(stderr, 
			"%s: not enough arguments for the bssjoin command; ", appname);
		fprintf(stderr, "see wlanctl man page.\n");
		usage();
		result = 1;
	}
	else
	{
		strcpy( req.name, argv[1]);
		req.result = 0;
		req.data = &cmd;
		req.len = sizeof(cmd);

		cmd.ethconv_type = WLAN_ETHCONV_ENCAP;
		if ( strcmp(argv[3], "encapsulation")==0 )
		{
			cmd.ethconv_type = WLAN_ETHCONV_ENCAP;
		}
		if ( strcmp(argv[3], "rfc1042")==0 )
		{
			cmd.ethconv_type = WLAN_ETHCONV_RFC1042;
		}
		if ( strcmp(argv[3], "802.1h")==0 )
		{
			cmd.ethconv_type = WLAN_ETHCONV_8021h;
		}

		result = ioctl( fd, WLAN_ETHCONV, &req);

		if ( req.result )
		{
			fprintf(stderr,
			"%s: There's a bad arg or the device isn't present.\n",
				appname);
			result = 1;
		}
	}

	return result;
}


/*----------------------------------------------------------------
*	process_authen
*
*	Called when the user wants to establish the authentication
*	relationship with the AP at the joined BSSID.
*
*	Arguments:
*		fd 		- socket file descriptor
*		argc  	- the argument count passed to main 
*		argv	- the argument list passed to main 
*
*	returns: 0 if successful, not 0 otherwise
*
----------------------------------------------------------------*/
int process_authen( int fd, int argc, char **argv )
{
	int					result;
	wlan_req_t			req;

	result = 0;

	strcpy( req.name, argv[1]);
	req.result = 0;
	req.data = NULL;
	req.len = 0;

	result = ioctl( fd, WLAN_AUTHENTICATE, &req);

	/* TODO: error handling needs alot of work */
	if ( req.result )
	{
		fprintf(stderr,
		"%s: There's a bad arg or the device isn't present.\n",
			appname);
		result = 1;
	}
	return result;
}

/*----------------------------------------------------------------
*	process_assoc
*
*	Called when the user wants to establish the association
*	relationship with the AP at the joined ESSID.
*
*	Arguments:
*		fd 		- socket file descriptor
*		argc  	- the argument count passed to main 
*		argv	- the argument list passed to main 
*
*	returns: 0 if successful, not 0 otherwise
*
----------------------------------------------------------------*/
int process_assoc( int fd, int argc, char **argv )
{
	int					result;
	wlan_req_t			req;

	result = 0;

	strcpy( req.name, argv[1]);
	req.result = 0;
	req.data = NULL;
	req.len = 0;

	result = ioctl( fd, WLAN_ASSOCIATE, &req);

	/* TODO: error handling needs alot of work */
	if ( req.result )
	{
		fprintf(stderr,
		"%s: There's a bad arg or the device isn't present.\n",
			appname);
		result = 1;
	}
	return result;
}


