/* prframe.c: Routines for decoding and printing 802.11 frames
*	----------------------------------------------------------------
*  
*	The contents of this file are subject to the Mozilla Public
*	License Version 1.0 (the "License"); you may not use this file
*	except in compliance with the License. You may obtain a copy of
*	the License at http://www.mozilla.org/MPL/
*
*	Software distributed under the License is distributed on an "AS
*	IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*	implied. See the License for the specific language governing
*	rights and limitations under the License.
*
*	The initial developer of the original code is Jo-Ellen F. Mathews
*	<joellen@absoval.com>.  Portions were created by Jo-Ellen F.
*	Mathews and Mark S. Mathews and are Copyright (C) 1998
*	AbsoluteValue Software, Inc.  All Rights Reserved.
*
----------------------------------------------------------------*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/* Ugly hack for LinuxPPC R4, don't have time to figure it out right now */
#if defined(_ARCH_PPC)
#undef __GLIBC__
#endif

#include <sys/socket.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/param.h>
#include <unistd.h>
#include <wlan/version.h>
#include <wlan/wlan_compat.h>
#include <wlan/am930mib.h>
#include <wlan/wlan_ioctl.h>
#include "wlandump.h"

/*----------------------------------------------------------------
*	Global Variables
----------------------------------------------------------------*/
wlan_p80211_frame_t		MACframes[] =	
{
	{ "mgmt", "assocreq", print_mgmt_assocreq },
	{ "mgmt", "assocresp", print_mgmt_assocresp },
	{ "mgmt", "atim", print_mgmt_ibssatim },
	{ "mgmt", "authen", print_mgmt_authen },
	{ "mgmt", "beacon", print_mgmt_beacon },
	{ "mgmt", "deauthen", print_mgmt_deauthen },
	{ "mgmt", "disassoc", print_mgmt_disassoc },
	{ "mgmt", "probereq", print_mgmt_probereq },
	{ "mgmt", "proberesp", print_mgmt_proberesp },
	{ "mgmt", "reassocreq", print_mgmt_reassocreq },
	{ "mgmt", "reassocresp", print_mgmt_reassocresp },
	{ "data", "data only", print_data_frame },
	{ "data", "data+cfack", print_data_frame },
	{ "data", "data+cfpoll", print_data_frame },
	{ "data", "data+cfack+cfpoll", print_data_frame },
	{ "data", "null", print_data_frame },
	{ "data", "cfack", print_data_frame },
	{ "data", "cfpoll", print_data_frame },
	{ "data", "cfack+cfpoll", print_data_frame },
	{ "ctl", "pspoll", print_ctl_pspoll },
	{ "ctl", "rts", print_ctl_rts },
	{ "ctl", "cts", print_ctl_cts },
	{ "ctl", "ack", print_ctl_ack },
	{ "ctl", "cfend", print_ctl_cfend },
	{ "ctl", "cfendcfack", print_ctl_cfendcfack }
};

/*----------------------------------------------------------------
*	print_80211_frame
*
*	This function receives an 802.11 frame, decodes the frame and
*   prints it.
*
*	Arguments:
*		frame_buffer	- a wireless network frame
*		len				- length of the frame
*
*	returns: 0 if successful, not 0 otherwise
*
----------------------------------------------------------------*/
int	print_80211_frame( wlan_sniffer_t *snif, UINT8 *frame_buffer, UINT16 len )
{
	int				result;
	int				index;
	unsigned int	type;
	unsigned int	stype;
	p80211_hdr_t	*phdr;
	int				i;
	UINT8			*pbuf;

	result = 0;
	index = -1;
	phdr = (p80211_hdr_t *)frame_buffer;

    type =  WLAN_GET_FC_FTYPE(phdr->a3.fc);
    stype =  WLAN_GET_FC_FSTYPE(phdr->a3.fc);

	if ( WLAN_GET_FC_ISWEP(phdr->a3.fc) && !opt_nodecrypt )
	{
		wlandump_wep_decrypt( frame_buffer, len);
	}

	switch ( type )
	{
		case WLAN_FTYPE_MGMT:
			if ( opt_noshowmgmt )
			{
				return result;
			}
			switch ( stype )
			{
				case WLAN_FSTYPE_ASSOCREQ:
            		index = MGMT_TYPE_ASSOCREQ;
					break;
				case WLAN_FSTYPE_ASSOCRESP:
            		index = MGMT_TYPE_ASSOCRESP;
					break;
				case WLAN_FSTYPE_REASSOCREQ:
            		index = MGMT_TYPE_REASSOCREQ;
					break;
				case WLAN_FSTYPE_REASSOCRESP:
            		index = MGMT_TYPE_REASSOCRESP;
					break;
				case WLAN_FSTYPE_PROBEREQ:
            		index = MGMT_TYPE_PROBEREQ;
					break;
				case WLAN_FSTYPE_PROBERESP:
            		index = MGMT_TYPE_PROBERESP;
					break;
				case WLAN_FSTYPE_BEACON:
            		index = MGMT_TYPE_BEACON;
					break;
				case WLAN_FSTYPE_ATIM:
            		index = MGMT_TYPE_ATIM;
					break;
				case WLAN_FSTYPE_DISASSOC:
            		index = MGMT_TYPE_DISASSOC;
					break;
				case WLAN_FSTYPE_AUTHEN:
            		index = MGMT_TYPE_AUTHEN;
					break;
				case WLAN_FSTYPE_DEAUTHEN:
            		index = MGMT_TYPE_DEAUTHEN;
					break;
				default:
            		printf("???");
					break;
			}
			break;
		case WLAN_FTYPE_CTL:
			if ( opt_noshowctl )
			{
				return result;
			}
			switch ( stype )
			{
				case WLAN_FSTYPE_PSPOLL:
					index = CTL_TYPE_PSPOLL;
					break;
				case WLAN_FSTYPE_RTS:
					index = CTL_TYPE_RTS;
					break;
				case WLAN_FSTYPE_CTS:
					index = CTL_TYPE_CTS;
					break;
				case WLAN_FSTYPE_ACK:
					index = CTL_TYPE_ACK;
					break;
				case WLAN_FSTYPE_CFEND:
					index = CTL_TYPE_CFEND;
					break;
				case WLAN_FSTYPE_CFENDCFACK:
					index = CTL_TYPE_CFENDCFACK;
					break;
				default:
            		printf("???");
					break;
			}
			break;
		case WLAN_FTYPE_DATA:
			if ( opt_noshowdata )
			{
				return result;
			}
			switch ( stype )
			{
				case WLAN_FSTYPE_DATAONLY:
            		index=DATA_TYPE_DATAONLY;
					break;
				case WLAN_FSTYPE_DATA_CFACK:
            		index=DATA_TYPE_DATA_CFACK;
					break;
				case WLAN_FSTYPE_DATA_CFPOLL:
            		index=DATA_TYPE_DATA_CFPOLL;
					break;
				case WLAN_FSTYPE_DATA_CFACK_CFPOLL:
            		index=DATA_TYPE_DATA_CFACK_CFPOLL;
					break;
				case WLAN_FSTYPE_NULL:
            		index=DATA_TYPE_NULL;
					break;
				case WLAN_FSTYPE_CFACK:
            		index=DATA_TYPE_CFACK;
					break;
				case WLAN_FSTYPE_CFPOLL:
            		index=DATA_TYPE_CFPOLL;
					break;
				case WLAN_FSTYPE_CFACK_CFPOLL:
            		index=DATA_TYPE_CFACK_CFPOLL;
					break;
				default:
            		printf("???");
					break;
			}
			break;
		default:
            printf("???/???");
           	printf("\n\n");
			break;
	}

	if ( (index != -1) && (result == 0) )
	{
		/* Print the sniffer stats */
		if ( !(index == MGMT_TYPE_BEACON && opt_nobeacon) )
		{
			if ( snif->istx )
			{
				printf("TX\n");
			}
			else
			{
				printf("RX rssi=%d ch=%d rate=%d\n", 
					snif->rssi, snif->ch, snif->rate);
			}

			/* if raw display is selected, print out the raw frame */
			if ( opt_showraw )
			{
				printf("    ");
				pbuf = frame_buffer;
				for ( i = 0; i < len; i++, pbuf++ )
				{
					printf("%02x ", *pbuf );
					if ( (i % 16) == 15 )
					{
						printf("\n    ");
					}
				}
				if ( ((i-1) % 16) != 15 ) printf("\n");
			}
	
			/* print specific frame type */
			(*(MACframes[index].pfunc))(frame_buffer, index, len);
	
			/* jump to FCS and print it */
			if ( !(index == MGMT_TYPE_BEACON && opt_nobeacon) )
			{
				frame_buffer = frame_buffer + (len - 4);
				printf("FCS=0x%02x%02x%02x%02x\n", *frame_buffer, *(frame_buffer+1),
					*(frame_buffer+2), *(frame_buffer+3) );
	
				printf( "----------------------------"
						"----------------------------\n");
			}
		}
	}
	else
	{
		int i,j;
		fprintf(stderr, "%s: Can not determine frame type.  Hex dump:\n", appname);
		for ( i=0; i < len; i++)
		{
			for ( j = 0; j < 16 && i + j < len; j++)
			{
				fprintf(stderr,"%02x ",  *(frame_buffer + i + j));
			}
			i += j - 1;
			fprintf(stderr,"\n");
		}
	}

	return result;
}


/*----------------------------------------------------------------
*	print_80211_frame_ctl
*
*	This function decodes and prints the frame control portion
*	of an 802.11 frame header
*
*	Arguments:
*		phdr - pointer to header of a wireless network frame
*		index	- represents the index into the global array of 	
*		        frame types 	
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_80211_frame_ctl( p80211_hdr_t *phdr, int index )
{
   	printf("%s/%s", MACframes[index].type, MACframes[index].stype);

    if( WLAN_GET_FC_TODS(phdr->a3.fc) )
    	printf("|toDS");
	
    if( WLAN_GET_FC_FROMDS(phdr->a3.fc) )
    	printf("|frDS");
	
    if( WLAN_GET_FC_MOREFRAG(phdr->a3.fc) )
    	printf("|morFrag");

    if( WLAN_GET_FC_RETRY(phdr->a3.fc) )
    	printf("|retry");

    if( WLAN_GET_FC_PWRMGT(phdr->a3.fc) )
    	printf("|pwrMgt");

    if( WLAN_GET_FC_MOREDATA(phdr->a3.fc) )
    	printf("|morData");

    if( WLAN_GET_FC_ISWEP(phdr->a3.fc) )
    	printf("|wep");

    if( WLAN_GET_FC_ORDER(phdr->a3.fc) )
    	printf("|ord");
}


/*----------------------------------------------------------------
*	print_80211_ctl_header
*
*	This function decodes and prints the frame header for 802.11
*	control frames.
*
*	Arguments:
*		phdr - pointer to header of a wireless network frame
*		index	- represents the index into the global array of 	
*		        frame types 	
*		len - length of frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_80211_ctl_header( p80211_hdr_t *phdr, int index, UINT16 len )
{

	print_80211_frame_ctl( phdr, index );

	switch ( index )
	{
		case CTL_TYPE_PSPOLL:
   			printf("\nBSSID=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a1[0],
				phdr->a3.a1[1], phdr->a3.a1[2], phdr->a3.a1[3],
				phdr->a3.a1[4], phdr->a3.a1[5]);
   			printf("TA=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a2[0],
				phdr->a3.a2[1], phdr->a3.a2[2], phdr->a3.a2[3], phdr->a3.a2[4],
				phdr->a3.a2[5]);
   			printf("\naid=0x%04x|", phdr->a3.dur);
			break;
		case CTL_TYPE_RTS:
   			printf("\nRA=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a1[0],
				phdr->a3.a1[1], phdr->a3.a1[2], phdr->a3.a1[3],
				phdr->a3.a1[4], phdr->a3.a1[5]);
   			printf("TA=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a2[0],
				phdr->a3.a2[1], phdr->a3.a2[2], phdr->a3.a2[3], phdr->a3.a2[4],
				phdr->a3.a2[5]);
   			printf("\ndur=0x%04x|", phdr->a3.dur);
			break;
		case CTL_TYPE_CTS:
		case CTL_TYPE_ACK:
   			printf("\nRA=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a1[0],
				phdr->a3.a1[1], phdr->a3.a1[2], phdr->a3.a1[3],
				phdr->a3.a1[4], phdr->a3.a1[5]);
   			printf("\ndur=0x%04x|", phdr->a3.dur);
			break;
		case CTL_TYPE_CFEND:
		case CTL_TYPE_CFENDCFACK:
   			printf("\nRA=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a1[0],
				phdr->a3.a1[1], phdr->a3.a1[2], phdr->a3.a1[3],
				phdr->a3.a1[4], phdr->a3.a1[5]);
   			printf("BSSID=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a2[0],
				phdr->a3.a2[1], phdr->a3.a2[2], phdr->a3.a2[3], phdr->a3.a2[4],
				phdr->a3.a2[5]);
   			printf("\ndur=0x%04x|", phdr->a3.dur);
			break;
	}
	printf("len=%u|", len );
}


/*----------------------------------------------------------------
*	print_80211_data_header
*
*	This function decodes and prints the frame header for 802.11
*	data frames.
*
*	Arguments:
*		phdr - pointer to header of a wireless network frame
*		index	- represents the index into the global array of 	
*		        frame types 	
*		len - length of frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_80211_data_header( p80211_hdr_t *phdr, int index, UINT16 len )
{
	int		toDS;
	int		fromDS;

	print_80211_frame_ctl( phdr, index );

	toDS = WLAN_GET_FC_TODS(phdr->a3.fc);
	fromDS = WLAN_GET_FC_FROMDS(phdr->a3.fc);

	if ( !toDS && !fromDS ) /* toDS == 0, fromDS == 0 */
	{
   		printf("\nDA=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a1[0],
			phdr->a3.a1[1], phdr->a3.a1[2], phdr->a3.a1[3],
			phdr->a3.a1[4], phdr->a3.a1[5]);
   		printf("SA=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a2[0],
			phdr->a3.a2[1], phdr->a3.a2[2], phdr->a3.a2[3], phdr->a3.a2[4],
			phdr->a3.a2[5]);
   		printf("BSSID=%02x:%02x:%02x:%02x:%02x:%02x", phdr->a3.a3[0],
			phdr->a3.a3[1], phdr->a3.a3[2], phdr->a3.a3[3], phdr->a3.a3[4],
			phdr->a3.a3[5]);
	}

	if ( !toDS && fromDS ) /* toDS == 0, fromDS == 1 */
	{
   		printf("\nDA=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a1[0],
			phdr->a3.a1[1], phdr->a3.a1[2], phdr->a3.a1[3],
			phdr->a3.a1[4], phdr->a3.a1[5]);
   		printf("BSSID=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a2[0],
			phdr->a3.a2[1], phdr->a3.a2[2], phdr->a3.a2[3], phdr->a3.a2[4],
			phdr->a3.a2[5]);
   		printf("SA=%02x:%02x:%02x:%02x:%02x:%02x", phdr->a3.a3[0],
			phdr->a3.a3[1], phdr->a3.a3[2], phdr->a3.a3[3], phdr->a3.a3[4],
			phdr->a3.a3[5]);
	}

	if ( toDS && !fromDS ) /* toDS == 1, fromDS == 0 */
	{
   		printf("\nBSSID%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a1[0],
			phdr->a3.a1[1], phdr->a3.a1[2], phdr->a3.a1[3],
			phdr->a3.a1[4], phdr->a3.a1[5]);
   		printf("SA=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a2[0],
			phdr->a3.a2[1], phdr->a3.a2[2], phdr->a3.a2[3], phdr->a3.a2[4],
			phdr->a3.a2[5]);
   		printf("DA=%02x:%02x:%02x:%02x:%02x:%02x", phdr->a3.a3[0],
			phdr->a3.a3[1], phdr->a3.a3[2], phdr->a3.a3[3], phdr->a3.a3[4],
			phdr->a3.a3[5]);
	}

	if ( toDS && fromDS ) /* toDS == 1, fromDS == 1 */
	{
   		printf("\nRA%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a1[0],
			phdr->a3.a1[1], phdr->a3.a1[2], phdr->a3.a1[3],
			phdr->a3.a1[4], phdr->a3.a1[5]);
   		printf("TA=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a2[0],
			phdr->a3.a2[1], phdr->a3.a2[2], phdr->a3.a2[3], phdr->a3.a2[4],
			phdr->a3.a2[5]);
   		printf("DA=%02x:%02x:%02x:%02x:%02x:%02x", phdr->a3.a3[0],
			phdr->a3.a3[1], phdr->a3.a3[2], phdr->a3.a3[3], phdr->a3.a3[4],
			phdr->a3.a3[5]);
   		printf(" SA=%02x:%02x:%02x:%02x:%02x:%02x", phdr->a4.a4[0],
			phdr->a4.a4[1], phdr->a4.a4[2], phdr->a4.a4[3],
			phdr->a4.a4[4], phdr->a4.a4[5]);
	}

   	printf("\ndur=0x%04x|", phdr->a3.dur);
   	printf("seq fragnum=%u,seq seqnum=%u\n", WLAN_GET_SEQ_FRGNUM(phdr->a3.seq),
   		WLAN_GET_SEQ_SEQNUM(phdr->a3.seq));

    if( WLAN_GET_FC_ISWEP(phdr->a3.fc) )
	{
   		UINT8	*ptr;

		ptr = ((UINT8 *)phdr) + WLAN_HDR_A3_LEN;

   		printf("wep.iv=%02x%02x%02x, wep.keyid=%02d, ",
			ptr[0], ptr[1], ptr[2], (ptr[3] >> 6));
	}

	printf("frame len=%u\n", len );
}


/*----------------------------------------------------------------
*	print_80211_mgmt_header
*
*	This function decodes and prints the frame header for 802.11
*	management frames.
*
*	Arguments:
*		phdr - pointer to header of a wireless network frame
*		index	- represents the index into the global array of 	
*		        frame types 	
*		len - length of frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_80211_mgmt_header( p80211_hdr_t *phdr, int index, UINT16 len )
{
	print_80211_frame_ctl( phdr, index );

    if( WLAN_GET_FC_ISWEP(phdr->a3.fc) )
	{
   		UINT8	*ptr;

		ptr = ((UINT8 *)phdr) + ((UINT8)(sizeof(p80211_hdr_t)));

   		printf("\nWEP Init Vector=%02x%02x%02x, KeyID=%02d\n",
			ptr[0], ptr[1], ptr[2], (ptr[3] >> 6));
	}

   	printf("\nDA=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a1[0],
		phdr->a3.a1[1], phdr->a3.a1[2], phdr->a3.a1[3],
		phdr->a3.a1[4], phdr->a3.a1[5]);
   	printf("SA=%02x:%02x:%02x:%02x:%02x:%02x ", phdr->a3.a2[0],
		phdr->a3.a2[1], phdr->a3.a2[2], phdr->a3.a2[3], phdr->a3.a2[4],
		phdr->a3.a2[5]);
   	printf("BSSID=%02x:%02x:%02x:%02x:%02x:%02x", phdr->a3.a3[0],
		phdr->a3.a3[1], phdr->a3.a3[2], phdr->a3.a3[3], phdr->a3.a3[4],
		phdr->a3.a3[5]);

   	printf("\ndur=0x%04x|", phdr->a3.dur);
   	printf("seq fragnum=%u,seq seqnum=%u\n", WLAN_GET_SEQ_FRGNUM(phdr->a3.seq),
   		WLAN_GET_SEQ_SEQNUM(phdr->a3.seq));

	printf("len=%u", len );
}


/*----------------------------------------------------------------
*	print_ctl_ack
*
*	This function decodes and prints control frames of subtype
*	acknowledgement.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_ctl_ack( UINT8 *pbuf, int index, UINT16 len )
{
	/* print 802.11 frame header */

	print_80211_ctl_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */
}


/*----------------------------------------------------------------
*	print_ctl_cfend
*
*	This function decodes and prints control frames of subtype
*	contention-free end.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_ctl_cfend( UINT8 *pbuf, int index, UINT16 len )
{
	/* print 802.11 frame header */

	print_80211_ctl_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */
}


/*----------------------------------------------------------------
*	print_ctl_cfendcfack
*
*	This function decodes and prints control frames of subtype
*	contention-free end acknowledge.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_ctl_cfendcfack( UINT8 *pbuf, int index, UINT16 len )
{
	/* print 802.11 frame header */

	print_80211_ctl_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */
}


/*----------------------------------------------------------------
*	print_ctl_cts
*
*	This function decodes and prints control frames of subtype
*	clear to send.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_ctl_cts( UINT8 *pbuf, int index, UINT16 len )
{
	/* print 802.11 frame header */

	print_80211_ctl_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */
}


/*----------------------------------------------------------------
*	print_ctl_pspoll
*
*	This function decodes and prints control frames of subtype
*	power-save poll.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_ctl_pspoll( UINT8 *pbuf, int index, UINT16 len )
{
	/* print 802.11 frame header */

	print_80211_ctl_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */
}


/*----------------------------------------------------------------
*	print_ctl_rts
*
*	This function decodes and prints control frames of subtype
*	request to send.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_ctl_rts( UINT8 *pbuf, int index, UINT16 len )
{
	/* print 802.11 frame header */

	print_80211_ctl_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */
}



/*----------------------------------------------------------------
*	print_data_frame
*
*	This function decodes and prints all data type frames.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_data_frame( UINT8 *pbuf, int index, UINT16 len )
{
	int				i;
	p80211_hdr_t	*phdr;

	/* print 802.11 frame header */

	print_80211_data_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for the data frame */

	phdr = (p80211_hdr_t *)pbuf;

    if( WLAN_GET_FC_TODS(phdr->a3.fc) && WLAN_GET_FC_FROMDS(phdr->a3.fc) )
    {
    	pbuf = pbuf + WLAN_HDR_A4_LEN;
    	len = len - WLAN_HDR_A4_LEN;
    }
    else
    {
    	pbuf = pbuf + WLAN_HDR_A3_LEN;
    	len = len - WLAN_HDR_A3_LEN;
    }

    /* printing of data frame body doesn't include the 4 trailing bytes
       for the FCS */
    len = len - 4;

	printf("    ");
	for ( i = 0; i < len && i < opt_showdatalen; i++, pbuf++ )
	{
		printf("%02x ", *pbuf );
		if ( (i % 16) == 15 )
		{
			printf("\n    ");
		}
	}
}

/*----------------------------------------------------------------
*	print_mgmt_assocreq
*
*	This function decodes and prints management frames of subtype
*	association request.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_mgmt_assocreq( UINT8 *pbuf, int index, UINT16 len )
{
	wlan_fr_assocreq_t				frame;
 	wlan_p80211_frame_fields_t		fixed_fields[TOTAL_MGMT_FIXED_FIELDS];
 	wlan_p80211_frame_fields_t		info_elements[TOTAL_MGMT_INFO_ELEMENTS];

	/* print 802.11 frame header */

	print_80211_mgmt_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */

	memset( (void *)fixed_fields, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_FIXED_FIELDS));
	memset( (void *)info_elements, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_INFO_ELEMENTS));
	memset( (void *)&frame, 0, sizeof(wlan_fr_assocreq_t) );
	frame.buf = pbuf;
	/* decode functions assume the 4 trailing bytes for the FCS aren't
	   included in the length */
	frame.len = len - 4;

	wlan_mgmt_decode_assocreq( &frame );

	fixed_fields[MGMT_FF_CAP_INFO].exists = TRUE;
	fixed_fields[MGMT_FF_CAP_INFO].info = (void *)frame.cap_info;

	fixed_fields[MGMT_FF_LISTEN_INT].exists = TRUE;
	fixed_fields[MGMT_FF_LISTEN_INT].info = (void *)frame.listen_int;

	info_elements[MGMT_IE_SSID].exists = TRUE;
	info_elements[MGMT_IE_SSID].info = (void *)frame.ssid;

	info_elements[MGMT_IE_SUPP_RATES].exists = TRUE;
	info_elements[MGMT_IE_SUPP_RATES].info = (void *)frame.supp_rates;

	print_fixed_fields( fixed_fields, len );
	print_info_elements( info_elements, len );
}


/*----------------------------------------------------------------
*	print_mgmt_assocresp
*
*	This function decodes and prints management frames of subtype
*	association response.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_mgmt_assocresp( UINT8 *pbuf, int index, UINT16 len )
{
	wlan_fr_assocresp_t				frame;
 	wlan_p80211_frame_fields_t		fixed_fields[TOTAL_MGMT_FIXED_FIELDS];
 	wlan_p80211_frame_fields_t		info_elements[TOTAL_MGMT_INFO_ELEMENTS];

	/* print 802.11 frame header */

	print_80211_mgmt_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */

	memset( (void *)fixed_fields, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_FIXED_FIELDS));
	memset( (void *)info_elements, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_INFO_ELEMENTS));
	memset( (void *)&frame, 0, sizeof(wlan_fr_assocresp_t) );
	frame.buf = pbuf;
	/* decode functions assume the 4 trailing bytes for the FCS aren't
	   included in the length */
	frame.len = len - 4;

	wlan_mgmt_decode_assocresp( &frame );

	fixed_fields[MGMT_FF_CAP_INFO].exists = TRUE;
	fixed_fields[MGMT_FF_CAP_INFO].info = (void *)frame.cap_info;

	fixed_fields[MGMT_FF_STATUS].exists = TRUE;
	fixed_fields[MGMT_FF_STATUS].info = (void *)frame.status;

	fixed_fields[MGMT_FF_AID].exists = TRUE;
	fixed_fields[MGMT_FF_AID].info = (void *)frame.aid;

	info_elements[MGMT_IE_SUPP_RATES].exists = TRUE;
	info_elements[MGMT_IE_SUPP_RATES].info = (void *)frame.supp_rates;

	print_fixed_fields( fixed_fields, len );
	print_info_elements( info_elements, len );
}


/*----------------------------------------------------------------
*	print_mgmt_authen
*
*	This function decodes and prints management frames of subtype
*	authentication.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_mgmt_authen( UINT8 *pbuf, int index, UINT16 len )
{
	wlan_fr_authen_t				frame;
 	wlan_p80211_frame_fields_t		fixed_fields[TOTAL_MGMT_FIXED_FIELDS];
 	wlan_p80211_frame_fields_t		info_elements[TOTAL_MGMT_INFO_ELEMENTS];

	/* print 802.11 frame header */

	print_80211_mgmt_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */

	memset( (void *)fixed_fields, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_FIXED_FIELDS));
	memset( (void *)info_elements, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_INFO_ELEMENTS));
	memset( (void *)&frame, 0, sizeof(wlan_fr_authen_t) );
	frame.buf = pbuf;
	/* decode functions assume the 4 trailing bytes for the FCS aren't
	   included in the length */
	frame.len = len - 4;

	wlan_mgmt_decode_authen( &frame );

	fixed_fields[MGMT_FF_AUTH_ALG].exists = TRUE;
	fixed_fields[MGMT_FF_AUTH_ALG].info = (void *)frame.auth_alg;

	fixed_fields[MGMT_FF_AUTH_SEQ].exists = TRUE;
	fixed_fields[MGMT_FF_AUTH_SEQ].info = (void *)frame.auth_seq;

	fixed_fields[MGMT_FF_STATUS].exists = TRUE;
	fixed_fields[MGMT_FF_STATUS].info = (void *)frame.status;

	info_elements[MGMT_IE_CHALLENGE].exists = TRUE;
	info_elements[MGMT_IE_CHALLENGE].info = (void *)frame.challenge;

	print_fixed_fields( fixed_fields, len );
	print_info_elements( info_elements, len );
}


/*----------------------------------------------------------------
*	print_mgmt_beacon
*
*	This function decodes and prints management frames of subtype
*	beacon.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_mgmt_beacon( UINT8 *pbuf, int index, UINT16 len )
{
	wlan_fr_beacon_t				frame;
 	wlan_p80211_frame_fields_t		fixed_fields[TOTAL_MGMT_FIXED_FIELDS];
 	wlan_p80211_frame_fields_t		info_elements[TOTAL_MGMT_INFO_ELEMENTS];

	if ( !opt_nobeacon )
	{
		/* print 802.11 frame header */
	
		print_80211_mgmt_header( (p80211_hdr_t *)pbuf, index, len );
	
		/* print management frame body for this management frame subtype */
	
		memset( (void *)fixed_fields, 0,
			(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_FIXED_FIELDS));
		memset( (void *)info_elements, 0,
			(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_INFO_ELEMENTS));
		memset( (void *)&frame, 0, sizeof(wlan_fr_beacon_t) );
		frame.buf = pbuf;
		/* decode functions assume the 4 trailing bytes for the FCS aren't
		   included in the length */
		frame.len = len - 4;
	
		wlan_mgmt_decode_beacon( &frame );
	
		fixed_fields[MGMT_FF_TS].exists = TRUE;
		fixed_fields[MGMT_FF_TS].info = (void *)frame.ts;
	
		fixed_fields[MGMT_FF_BCN_INT].exists = TRUE;
		fixed_fields[MGMT_FF_BCN_INT].info = (void *)frame.bcn_int;
	
		fixed_fields[MGMT_FF_CAP_INFO].exists = TRUE;
		fixed_fields[MGMT_FF_CAP_INFO].info = (void *)frame.cap_info;
	
		info_elements[MGMT_IE_SSID].exists = TRUE;
		info_elements[MGMT_IE_SSID].info = (void *)frame.ssid;
	
		info_elements[MGMT_IE_SUPP_RATES].exists = TRUE;
		info_elements[MGMT_IE_SUPP_RATES].info = (void *)frame.supp_rates;
	
		info_elements[MGMT_IE_FH_PARMS].exists = TRUE;
		info_elements[MGMT_IE_FH_PARMS].info = (void *)frame.fh_parms;
	
		info_elements[MGMT_IE_DS_PARMS].exists = TRUE;
		info_elements[MGMT_IE_DS_PARMS].info = (void *)frame.ds_parms;
	
		info_elements[MGMT_IE_CF_PARMS].exists = TRUE;
		info_elements[MGMT_IE_CF_PARMS].info = (void *)frame.cf_parms;
	
		info_elements[MGMT_IE_IBSS_PARMS].exists = TRUE;
		info_elements[MGMT_IE_IBSS_PARMS].info = (void *)frame.ibss_parms;
	
		info_elements[MGMT_IE_TIM].exists = TRUE;
		info_elements[MGMT_IE_TIM].info = (void *)frame.tim;
	
		print_fixed_fields( fixed_fields, len );
		print_info_elements( info_elements, len );
	}
}


/*----------------------------------------------------------------
*	print_mgmt_deauthen
*
*	This function decodes and prints management frames of subtype
*	deauthentication.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_mgmt_deauthen( UINT8 *pbuf, int index, UINT16 len )
{
	wlan_fr_deauthen_t				frame;
 	wlan_p80211_frame_fields_t		fixed_fields[TOTAL_MGMT_FIXED_FIELDS];

	/* print 802.11 frame header */

	print_80211_mgmt_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */

	memset( (void *)fixed_fields, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_FIXED_FIELDS));
	memset( (void *)&frame, 0, sizeof(wlan_fr_deauthen_t) );
	frame.buf = pbuf;
	/* decode functions assume the 4 trailing bytes for the FCS aren't
	   included in the length */
	frame.len = len - 4;

	wlan_mgmt_decode_deauthen( &frame );

	fixed_fields[MGMT_FF_REASON].exists = TRUE;
	fixed_fields[MGMT_FF_REASON].info = (void *)frame.reason;

	print_fixed_fields( fixed_fields, len );
}


/*----------------------------------------------------------------
*	print_mgmt_disassoc
*
*	This function decodes and prints management frames of subtype
*	disassociation.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_mgmt_disassoc( UINT8 *pbuf, int index, UINT16 len )
{
	wlan_fr_disassoc_t				frame;
 	wlan_p80211_frame_fields_t		fixed_fields[TOTAL_MGMT_FIXED_FIELDS];

	/* print 802.11 frame header */

	print_80211_mgmt_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */

	memset( (void *)fixed_fields, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_FIXED_FIELDS));
	memset( (void *)&frame, 0, sizeof(wlan_fr_disassoc_t) );
	frame.buf = pbuf;
	/* decode functions assume the 4 trailing bytes for the FCS aren't
	   included in the length */
	frame.len = len - 4;

	wlan_mgmt_decode_disassoc( &frame );

	fixed_fields[MGMT_FF_REASON].exists = TRUE;
	fixed_fields[MGMT_FF_REASON].info = (void *)frame.reason;

	print_fixed_fields( fixed_fields, len );
}


/*----------------------------------------------------------------
*	print_mgmt_ibssatim
*
*	This function decodes and prints management frames of subtype
*	ibss atim.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_mgmt_ibssatim( UINT8 *pbuf, int index, UINT16 len )
{
	/* print 802.11 frame header */

	print_80211_mgmt_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */

	return;
}


/*----------------------------------------------------------------
*	print_mgmt_probereq
*
*	This function decodes and prints management frames of subtype
*	probe request.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_mgmt_probereq( UINT8 *pbuf, int index, UINT16 len )
{
	wlan_fr_probereq_t				frame;
 	wlan_p80211_frame_fields_t		info_elements[TOTAL_MGMT_INFO_ELEMENTS];

	/* print 802.11 frame header */

	print_80211_mgmt_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */

	memset( (void *)info_elements, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_INFO_ELEMENTS));
	memset( (void *)&frame, 0, sizeof(wlan_fr_probereq_t) );
	frame.buf = pbuf;
	/* decode functions assume the 4 trailing bytes for the FCS aren't
	   included in the length */
	frame.len = len - 4;

	wlan_mgmt_decode_probereq( &frame );

	info_elements[MGMT_IE_SSID].exists = TRUE;
	info_elements[MGMT_IE_SSID].info = (void *)frame.ssid;

	info_elements[MGMT_IE_SUPP_RATES].exists = TRUE;
	info_elements[MGMT_IE_SUPP_RATES].info = (void *)frame.supp_rates;

	print_info_elements( info_elements, len );
}


/*----------------------------------------------------------------
*	print_mgmt_proberesp
*
*	This function decodes and prints management frames of subtype
*	probe response.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_mgmt_proberesp( UINT8 *pbuf, int index, UINT16 len )
{
	wlan_fr_proberesp_t				frame;
 	wlan_p80211_frame_fields_t		fixed_fields[TOTAL_MGMT_FIXED_FIELDS];
 	wlan_p80211_frame_fields_t		info_elements[TOTAL_MGMT_INFO_ELEMENTS];

	/* print 802.11 frame header */

	print_80211_mgmt_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */

	memset( (void *)fixed_fields, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_FIXED_FIELDS));
	memset( (void *)info_elements, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_INFO_ELEMENTS));
	memset( (void *)&frame, 0, sizeof(wlan_fr_proberesp_t) );
	frame.buf = pbuf;
	/* decode functions assume the 4 trailing bytes for the FCS aren't
	   included in the length */
	frame.len = len - 4;

	wlan_mgmt_decode_proberesp( &frame );

	fixed_fields[MGMT_FF_TS].exists = TRUE;
	fixed_fields[MGMT_FF_TS].info = (void *)frame.ts;

	fixed_fields[MGMT_FF_BCN_INT].exists = TRUE;
	fixed_fields[MGMT_FF_BCN_INT].info = (void *)frame.bcn_int;

	fixed_fields[MGMT_FF_CAP_INFO].exists = TRUE;
	fixed_fields[MGMT_FF_CAP_INFO].info = (void *)frame.cap_info;

	info_elements[MGMT_IE_SSID].exists = TRUE;
	info_elements[MGMT_IE_SSID].info = (void *)frame.ssid;

	info_elements[MGMT_IE_SUPP_RATES].exists = TRUE;
	info_elements[MGMT_IE_SUPP_RATES].info = (void *)frame.supp_rates;

	info_elements[MGMT_IE_FH_PARMS].exists = TRUE;
	info_elements[MGMT_IE_FH_PARMS].info = (void *)frame.fh_parms;

	info_elements[MGMT_IE_DS_PARMS].exists = TRUE;
	info_elements[MGMT_IE_DS_PARMS].info = (void *)frame.ds_parms;

	info_elements[MGMT_IE_CF_PARMS].exists = TRUE;
	info_elements[MGMT_IE_CF_PARMS].info = (void *)frame.cf_parms;

	info_elements[MGMT_IE_IBSS_PARMS].exists = TRUE;
	info_elements[MGMT_IE_IBSS_PARMS].info = (void *)frame.ibss_parms;

	print_fixed_fields( fixed_fields, len );
	print_info_elements( info_elements, len );
}


/*----------------------------------------------------------------
*	print_mgmt_reassocreq
*
*	This function decodes and prints management frames of subtype
*	reassociation request.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_mgmt_reassocreq( UINT8 *pbuf, int index, UINT16 len )
{
	wlan_fr_reassocreq_t			frame;
 	wlan_p80211_frame_fields_t		fixed_fields[TOTAL_MGMT_FIXED_FIELDS];
 	wlan_p80211_frame_fields_t		info_elements[TOTAL_MGMT_INFO_ELEMENTS];

	/* print 802.11 frame header */

	print_80211_mgmt_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */

	memset( (void *)fixed_fields, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_FIXED_FIELDS));
	memset( (void *)info_elements, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_INFO_ELEMENTS));
	memset( (void *)&frame, 0, sizeof(wlan_fr_reassocreq_t) );
	frame.buf = pbuf;
	/* decode functions assume the 4 trailing bytes for the FCS aren't
	   included in the length */
	frame.len = len - 4;

	wlan_mgmt_decode_reassocreq( &frame );

	fixed_fields[MGMT_FF_CAP_INFO].exists = TRUE;
	fixed_fields[MGMT_FF_CAP_INFO].info = (void *)frame.cap_info;

	fixed_fields[MGMT_FF_LISTEN_INT].exists = TRUE;
	fixed_fields[MGMT_FF_LISTEN_INT].info = (void *)frame.listen_int;

	fixed_fields[MGMT_FF_CURR_AP].exists = TRUE;
	fixed_fields[MGMT_FF_CURR_AP].info = (void *)frame.curr_ap;

	info_elements[MGMT_IE_SSID].exists = TRUE;
	info_elements[MGMT_IE_SSID].info = (void *)frame.ssid;

	info_elements[MGMT_IE_SUPP_RATES].exists = TRUE;
	info_elements[MGMT_IE_SUPP_RATES].info = (void *)frame.supp_rates;

	print_fixed_fields( fixed_fields, len );
	print_info_elements( info_elements, len );
}


/*----------------------------------------------------------------
*	print_mgmt_reassocresp
*
*	This function decodes and prints management frames of subtype
*	reassociation response.
*
*	Arguments:
*		pbuf - pointer to the frame buffer
*		index	- index into the global array of frame types for a
*		        particular frame type and subtype
*		len	- the number of bytes in the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_mgmt_reassocresp( UINT8 *pbuf, int index, UINT16 len )
{
	wlan_fr_reassocresp_t			frame;
 	wlan_p80211_frame_fields_t		fixed_fields[TOTAL_MGMT_FIXED_FIELDS];
 	wlan_p80211_frame_fields_t		info_elements[TOTAL_MGMT_INFO_ELEMENTS];

	/* print 802.11 frame header */

	print_80211_mgmt_header( (p80211_hdr_t *)pbuf, index, len );

	/* print management frame body for this management frame subtype */

	memset( (void *)fixed_fields, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_FIXED_FIELDS));
	memset( (void *)info_elements, 0,
		(sizeof(wlan_p80211_frame_fields_t) * TOTAL_MGMT_INFO_ELEMENTS));
	memset( (void *)&frame, 0, sizeof(wlan_fr_reassocresp_t) );
	frame.buf = pbuf;
	/* decode functions assume the 4 trailing bytes for the FCS aren't
	   included in the length */
	frame.len = len - 4;

	wlan_mgmt_decode_reassocresp( &frame );

	fixed_fields[MGMT_FF_CAP_INFO].exists = TRUE;
	fixed_fields[MGMT_FF_CAP_INFO].info = (void *)frame.cap_info;

	fixed_fields[MGMT_FF_STATUS].exists = TRUE;
	fixed_fields[MGMT_FF_STATUS].info = (void *)frame.status;

	fixed_fields[MGMT_FF_AID].exists = TRUE;
	fixed_fields[MGMT_FF_AID].info = (void *)frame.aid;

	info_elements[MGMT_IE_SUPP_RATES].exists = TRUE;
	info_elements[MGMT_IE_SUPP_RATES].info = (void *)frame.supp_rates;

	print_fixed_fields( fixed_fields, len );
	print_info_elements( info_elements, len );
}


/*----------------------------------------------------------------
*	print_fixed_fields
*
*	This function looks in the array of fixed fields and prints the
*	information for only those fixed fields that exist.	
*
*	Arguments:
*		ff - array of all the fixed field types and associated info
*		len - length of the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_fixed_fields( wlan_p80211_frame_fields_t  *ff, UINT16 len )
{
	int		i;
	UINT8	*byte;
	UINT16	info;

	for ( i = 0; i < TOTAL_MGMT_FIXED_FIELDS; i++ )
	{
		switch( i )
		{
			case	MGMT_FF_AUTH_ALG:
					if ( ff[MGMT_FF_AUTH_ALG].exists )
					{
						printf("|authAlg=%u",
						*((UINT16 *)ff[MGMT_FF_AUTH_ALG].info));
					}
					break;
			case	MGMT_FF_AUTH_SEQ:
					if ( ff[MGMT_FF_AUTH_SEQ].exists )
					{
						printf("|authSeq=%u",
						*((UINT16 *)ff[MGMT_FF_AUTH_SEQ].info));
					}
					break;
			case	MGMT_FF_BCN_INT:
					if ( ff[MGMT_FF_BCN_INT].exists )
					{
						printf("|bcn_int=%u",
						*((UINT16 *)ff[MGMT_FF_BCN_INT].info));
					}
					break;
			case	MGMT_FF_CAP_INFO:
					if ( ff[MGMT_FF_CAP_INFO].exists )
					{
						info = *((UINT16 *)ff[MGMT_FF_CAP_INFO].info);
						printf("|capInfo=0x%02x", info);
						if ( WLAN_GET_MGMT_CAP_INFO_ESS( info ) )
							printf(",capInfo_ess");
						if ( WLAN_GET_MGMT_CAP_INFO_IBSS( info ) )
							printf(",capInfo_ibss");
						if ( WLAN_GET_MGMT_CAP_INFO_CFPOLLABLE( info ) )
							printf(",capInfo_cfpollable");
						if ( WLAN_GET_MGMT_CAP_INFO_CFPOLLREQ( info ) )
							printf(",capInfo_cfpollreq");
						if ( WLAN_GET_MGMT_CAP_INFO_PRIVACY( info ) )
							printf(",capInfo_privacy");
					}
					break;
			case	MGMT_FF_CURR_AP:
					if ( ff[MGMT_FF_CURR_AP].exists )
					{
						byte = (UINT8 *)ff[MGMT_FF_CURR_AP].info;
						printf("|currAPadd=%02x:%02x:%02x:%02x:%02x:%02x",
						*byte, *(byte + 1), *(byte + 2),
						*(byte + 3), *(byte + 4), *(byte + 5) );
					}
					break;
			case	MGMT_FF_LISTEN_INT:
					if ( ff[MGMT_FF_LISTEN_INT].exists )
					{
						printf("|listenInt=%u",
						*((UINT16 *)ff[MGMT_FF_LISTEN_INT].info));
					}
					break;
			case	MGMT_FF_REASON:
					if ( ff[MGMT_FF_REASON].exists )
					{
						printf("|reason=%u",
						*((UINT16 *)ff[MGMT_FF_REASON].info));
					}
					break;
			case	MGMT_FF_AID:
					if ( ff[MGMT_FF_AID].exists )
					{
						printf("|aid=%u",
						*((UINT16 *)ff[MGMT_FF_AID].info));
					}
					break;
			case	MGMT_FF_STATUS:
					if ( ff[MGMT_FF_STATUS].exists )
					{
						printf("|status=%u",
						*((UINT16 *)ff[MGMT_FF_STATUS].info));
					}
					break;
			case	MGMT_FF_TS:
					if ( ff[MGMT_FF_TS].exists )
					{
						printf("|ts=0x");
						byte = (UINT8 *)ff[MGMT_FF_TS].info;
						printf("%02x%02x%02x%02x%02x%02x%02x%02x", *byte,
							*(byte+1), *(byte+2), *(byte+3), *(byte+4),
							*(byte+5), *(byte+6), *(byte+7));
					}
					break;
		}
	}
	printf("\n");
}


/*----------------------------------------------------------------
*	print_info_elements
*
*	This function looks in the array of information elements and prints the
*	information for only those information elements that exist.	
*
*	Arguments:
*		ie - array of all the information element types and associated info
*		len - length of the frame
*
*	returns: nothing
*
----------------------------------------------------------------*/
void print_info_elements( wlan_p80211_frame_fields_t  *ie, UINT16 len )
{
	int		index;
	int		i;
	UINT16	n;

	for ( index = 0; index < TOTAL_MGMT_INFO_ELEMENTS; index++ )
	{
		switch ( index )
		{
			case MGMT_IE_SSID:
				if ( ie[MGMT_IE_SSID].exists )
				{
					wlan_ie_ssid_t	*info;

					info = (wlan_ie_ssid_t *)(ie[MGMT_IE_SSID].info);
					if ( info != NULL )
					{
						char	ssid_str[ WLAN_SSID_MAXLEN + 2 ];

						n = info->len;
						printf("ssid:elemID=%u,len=%u,", info->eid, n );
						memcpy( (void *)ssid_str, (void *)info->ssid, n );
						ssid_str[n] = '\0';
						printf("%s\n", ssid_str );
					}
				}
				break;
			case MGMT_IE_SUPP_RATES:
				if ( ie[MGMT_IE_SUPP_RATES].exists )
				{
					wlan_ie_supp_rates_t	*info;

					info = (wlan_ie_supp_rates_t *)(ie[MGMT_IE_SUPP_RATES].info);
					if (info != NULL)
					{
						n = info->len;
						printf("suppRates:elemID=%u,len=%u,0x", info->eid, n);
						for ( i = 0; i < n; i++ )
						{
							printf("%02x", info->rates[i]);
						}
						printf("\n");
					}
				}
				break;
			case MGMT_IE_FH_PARMS:
				if ( ie[MGMT_IE_FH_PARMS].exists )
				{
					wlan_ie_fh_parms_t	*info;

					info = (wlan_ie_fh_parms_t *)(ie[MGMT_IE_FH_PARMS].info);
					if (info != NULL)
					{
						printf("fhParms:elemID=%u,len=%u,dwell=%u",
							info->eid, info->len, info->dwell);
						printf("hopset=%u,hoppattern=%u,hopindex=%u\n",
							info->hopset, info->hoppattern, info->hopindex);
					}
				}
				break;
			case MGMT_IE_DS_PARMS:
				if ( ie[MGMT_IE_DS_PARMS].exists )
				{
					wlan_ie_ds_parms_t	*info;

					info = (wlan_ie_ds_parms_t *)(ie[MGMT_IE_DS_PARMS].info);
					if (info != NULL)
					{
						printf("dsParms:elemID=%u,len=%u,currChannel=%u\n",
							info->eid, info->len, info->curr_ch);
					}
				}
				break;
			case MGMT_IE_CF_PARMS:
				if ( ie[MGMT_IE_CF_PARMS].exists )
				{
					wlan_ie_cf_parms_t	*info;

					info = (wlan_ie_cf_parms_t *)(ie[MGMT_IE_CF_PARMS].info);
					if (info != NULL)
					{
						printf("cfParms:elemID=%u,len=%u,cfpCnt=%u",
							info->eid, info->len, info->cfp_cnt);
						printf("cfpPeriod=%u,cfpMaxdur=%u,cfpDurRemain=%u\n",
							info->cfp_period, info->cfp_maxdur,
							info->cfp_durremaining );
					}
				}
				break;
			case MGMT_IE_TIM:
				if ( ie[MGMT_IE_TIM].exists )
				{
					wlan_ie_tim_t	*info;

					info = (wlan_ie_tim_t *)(ie[MGMT_IE_TIM].info);
					if (info != NULL)
					{
						n = info->len;
						printf("tim:elemID=%u,len=%u,dtimCnt=%u,",
							info->eid, n, info->dtim_cnt);
						printf("dtimPer=%u,bitmapCtl=%u,virtBitmap=0x",
							info->dtim_period, info->bitmap_ctl);

						n = n - 3; 

						for ( i = 0; i < n; i++ )
						{
							printf("%02x", info->virt_bm[i]);
						}
						printf("\n");
					}
				}
				break;
			case MGMT_IE_IBSS_PARMS:
				if ( ie[MGMT_IE_IBSS_PARMS].exists )
				{
					wlan_ie_ibss_parms_t	*info;

					info=(wlan_ie_ibss_parms_t *)(ie[MGMT_IE_IBSS_PARMS].info);
					if (info != NULL)
					{
						printf("ibssParms:elemID=%u,len=%u,atimWin=%u\n",
							info->eid, info->len, info->atim_win);
					}
				}
				break;
			case MGMT_IE_CHALLENGE:
				if ( ie[MGMT_IE_CHALLENGE].exists )
				{
					wlan_ie_challenge_t	*info;

					info = (wlan_ie_challenge_t *)(ie[MGMT_IE_CHALLENGE].info);
					if (info != NULL)
					{
						n = info->len;
						printf("ChallengeText:elemID=%u,len=%u,challenge=0x\n",
							info->eid, n );
						for ( i = 0; i < n; i++ )
						{
							printf("%02x", info->challenge[i]);
						}
					}
				}
				break;
		}
	}
}


