/* wlandump.c: Entry point to frame 'snif' and dump program
*	----------------------------------------------------------------
*
*	The contents of this file are subject to the Mozilla Public
*	License Version 1.0 (the "License"); you may not use this file
*	except in compliance with the License. You may obtain a copy of
*	the License at http://www.mozilla.org/MPL/
*
*	Software distributed under the License is distributed on an "AS
*	IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*	implied. See the License for the specific language governing
*	rights and limitations under the License.
*
*	The initial developer of the original code is Jo-Ellen F. Mathews
*	<joellen@absoval.com>.  Portions were created by Jo-Ellen F.
*	Mathews and Mark S. Mathews and are Copyright (C) 1998
*	AbsoluteValue Software, Inc.  All Rights Reserved.
*
----------------------------------------------------------------*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/* Ugly hack for LinuxPPC R4, don't have time to figure it out right now */
#if defined(_ARCH_PPC)
#undef __GLIBC__
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/param.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>

#include <asm/types.h>
#include <linux/netlink.h>

#include <wlan/version.h>
#include <wlan/wlan_compat.h>
#include <wlan/am930mib.h>
#include <wlan/wlan_ioctl.h>
#include "wlandump.h"


#define APPNAME		"wlandump"


int		read_frames_from_file( char *filename );
void	usage(void);
void 	handlesig(int signo);
int		get_ioctl_socket(char *ifname);
void	sniff_interface(int ioctlfd);
void	strtoUINT40( UINT8 *a, char *s );

/*----------------------------------------------------------------
*	Global Variables
----------------------------------------------------------------*/
char				appname[APPNAME_MAX + 1];
char				*fullname;
wlan_sniffercmd_t	cmd;
char				ifname[16];
int					ioctlfd;
int					nlfd;

/* option flags */
int	opt_nobeacon = 0;		/* -b => don't show beacons */
int	opt_showrx = 1;			/* -r => don't show rx'd frames (on by default) */
int	opt_showtx = 0;			/* -t => show tx'd frames */
int	opt_noshowctl = 0;		/* -c => don't show ctl frames */
int	opt_noshowmgmt = 0;		/* -m => don't show mgmt frames */
int	opt_noshowdata = 0;		/* -d => don't show data frames */
int	opt_showdatalen = 2312;	/* -s <len> => show, at most, this many data bytes */
int	opt_showraw = 0;		/* -x => show the whole frame as raw bytes */
int	opt_nodecrypt = 0;		/* -w => don't decrypt data frame contents */
int	opt_nodefrag = 0;		/* -u => don't unfragment */
int	opt_showfrags = 0;		/* -f => show fragments as well as defragged frm */
int opt_wepkeyset = 0;		/* -k <keyindex(0-3)>-xx:xx:xx:xx:xx  */
UINT8	wep_key[WLAN_WEP_NKEYS][WLAN_WEP_KEYLEN];

char				opts[] = "brtcmdxwufs:k:";

/*----------------------------------------------------------------
*	Main
----------------------------------------------------------------*/
int main(int argc, char *argv[])
{
	int 				result = 0;
    int					optch;

	strcpy( appname, APPNAME );
	fullname = argv[0];
	
	/* if no args, print the usage msg */
	if ( argc < 2 )
	{
		usage();
		return -1;
	}

	/* make stdout line buffered */
	setlinebuf(stdout);

	/* initialize the command */
	cmd.cmd = SNIFFERCMD_REG;
	cmd.pid = getpid();
	cmd.flags = SNIFFLAG_SNIFRX;
	cmd.result = 0;

	/* collect the args */
    while ( ((optch = getopt(argc, argv, opts)) != EOF) && (result == 0) )
	{
		switch (optch)
		{
			case 'b':
				opt_nobeacon = 1;
				break;
			case 'r':
				opt_showrx = 0;
				cmd.flags &= ~SNIFFLAG_SNIFRX;
				break;
			case 't':
				opt_showtx = 0;
				cmd.flags |= SNIFFLAG_SNIFTX;
				break;
			case 'c':
				opt_noshowctl = 1;
				break;
			case 'm':
				opt_noshowmgmt = 1;
				break;
			case 'd':
				opt_noshowdata = 1;
				break;
			case 's':
				opt_showdatalen = atoi(optarg);
				break;
			case 'x':
				opt_showraw = 1;
				break;
			case 'w':
				opt_nodecrypt = 1;
				break;
			case 'u':
				opt_nodefrag = 1;
				break;
			case 'f':
				opt_showfrags = 1;
				break;
			case 'k':
				opt_wepkeyset = 1;
				strtoUINT40( wep_key[optarg[0] - '0'], optarg + 2);
				break;
			case 'v':
				printf("%s utility version %s\n", appname, WLAN_RELEASE);
				return 0;
				break;
			default:
				usage();
				return -1;
				break;
		}
	}

	/* if we made it this far, the options must be pretty much OK */
	if ( optind >= argc )
	{
		/* we're missing the interface argument */
		fprintf(stderr, "%s: missing argument - interface\n", fullname);
		usage();
		return -1;
	}
	else
	{
		/* Make the crc table for wep to use */
		wlandump_crc32_mktable();

		/* save the interface name */
		strcpy( ifname, argv[optind] );
		if ( (ioctlfd = get_ioctl_socket(ifname)) != -1)
		{
			sniff_interface(ioctlfd);
		}
		else
		{
			return -1;
		}
	}

	/* if we reach here, something went wrong in sniff_interface, return error */
	return -1;
}


/*----------------------------------------------------------------
*	get_ioctl_socket
*
*	Opens a socket and issues the WLAN_TEST ioctl to make sure
*	the named device is, in fact, a wireless interface.
*
*	Arguments:
*		ifname  	- e.g. "eth0"
*
*	returns: 0 if successful, not 0 otherwise
*
----------------------------------------------------------------*/
int get_ioctl_socket(char *ifname)
{
	wlan_req_t	req;
	int			result = 0;

	/* get a socket */
	ioctlfd = socket(AF_INET, SOCK_STREAM, 0);
	if ( ioctlfd != -1 )
	{
		/* Test that there is a card */
		strcpy( req.name, ifname);
		req.result = 0;
		req.data = NULL;
		req.len = 0;
		result = ioctl( ioctlfd, WLAN_TEST, &req);

		if (result < 0)
		{
			fprintf(stderr, "%s: ioctl WLAN_TEST failed. \'%s\'\n", appname, ifname);
			fprintf(stderr, "interface \'%s\' is probably an ", ifname);
			fprintf(stderr, "invalid wireless network interface name.\n");
		}
		else
		{
			if ( req.result == 0xf0f0 )
			{
				result = ioctlfd;
			}
		}
	}
	else
	{
		perror(appname);
		result = ioctlfd;
	}
	return result;
}
		
/*----------------------------------------------------------------
*	sniff_interface
*
*	Sends the SNIFFERCMD:SNIFFERCMD_REG message to the driver,
*	hooks the signals, opens the netlink socket and loops forever
*	reading and printing frames.
*
*	Arguments:
*		ioctlfd	- the socket file descriptor for ioctl commands
*
*	returns: should never return.  Program terminates via signal
*				handler.
*
----------------------------------------------------------------*/
void sniff_interface(int ioctlfd)
{
	/* wlan_sniffercmd_t	cmd;  uses global */
	/* char					ifname[];  uses global */
	int					result;
	int					i;
	wlan_req_t			req;
	UINT8				snifbuf[WLAN_A3FR_MAXLEN + sizeof(wlan_sniffer_t)];
	UINT8				*frame;
	wlan_sniffer_t		*snif;
	struct sockaddr_nl	nlskaddr;
	int					recvlen;




	/* send the CMD_REG ioctl */
	strcpy( req.name, ifname);
	req.result = 0;
	req.data = &cmd;
	req.len = sizeof(cmd);

	/* cmd values were setup from argv in main */
	result = ioctl( ioctlfd, WLAN_SNIFFERCMD, &req);

	if ( result != 0 )
	{
		if ( result == -ENOSYS )
		{
			fprintf(stderr, "%s: error - the driver wasn't compiled for sniffing.\n", fullname);
		}
		else
		{
			perror(fullname);
		}
	}
	else
	{
		if ( req.result == 0 )
		{
			/* set up the signal handler */
			for ( i = 1; i < _NSIG; i++)
			{
				signal( i, handlesig);
			}

			/* open the netlink socket */
			if ( (nlfd = socket( PF_NETLINK, SOCK_RAW, NETLINK_WLAN )) == -1 )
			{
				perror(fullname);
				return;
			}
			else
			{
				memset ( &nlskaddr, 0 , sizeof( nlskaddr ));
				nlskaddr.nl_family = (sa_family_t)PF_NETLINK;
				nlskaddr.nl_pid = (__u32)getpid();
				nlskaddr.nl_groups = 0;

				if ((bind( nlfd, (struct sockaddr*)&nlskaddr, sizeof(nlskaddr))) == -1 )
				{
					perror(fullname);
					return;
				}
				else
				{
					/* loop forever */
					while(1)
					{
						recvlen = recv( nlfd, snifbuf, sizeof(snifbuf), 0 );

						if ( recvlen > -1 )
						{
							snif = (wlan_sniffer_t*)snifbuf;
							frame = snifbuf + sizeof(wlan_sniffer_t);
							print_80211_frame( snif, frame, snif->len );
						}
						else
						{
							perror(fullname);
						}
					}
				}
			}
		}
		else
		{
			/* something is _still_ hosed from the ioctl */
		}
	}
	return;
}


/*----------------------------------------------------------------
*	handlesig
*
*	Cleans up after the sniffer.
*
*	Arguments:
*		signo	- signal number
*
*	returns: function doesn't return anything
*
----------------------------------------------------------------*/
void handlesig(int signo)
{
	wlan_req_t			req;

	/* send the CMD_UNREG ioctl */
	strcpy( req.name, ifname);
	req.result = 0;
	req.data = &cmd;
	req.len = sizeof(cmd);
	cmd.cmd = SNIFFERCMD_UNREG;

	/* we don't really care about returns at this point */
	ioctl( ioctlfd, WLAN_SNIFFERCMD, &req);

	close(ioctlfd);
	close(nlfd);
	
	exit(0);
}


/*----------------------------------------------------------------
*	strtoUINT40
*
*	This function converts a character string that represents an
*	a 40 bit quantity.
*
*	Arguments:
*		a	- 40 bit buffer
*		s	- character string representing a 40 key quantity
*
*	returns: function doesn't return anything
*
----------------------------------------------------------------*/
void strtoUINT40( UINT8 *a, char *s )
{
	char	*p;
	int		i;
	UINT	val;

	for ( i = 0; i < 4; i++)
	{
		p = strchr( s, ':');
		if ( p == NULL )
		{
			fprintf(stderr, "%s: UINT48 format error!\n", appname);
			exit(1);
		}
		else
		{
			*p = '\0';
			sscanf( s, "%x", &val);
			a[i] = (UINT8)val;
			s = p+1;
		}
	}
	sscanf( s, "%x", &val);
	a[i] = (UINT8)val;
}



/*----------------------------------------------------------------
*	usage
*
*	This function displays the proper syntax of the frdumper utility.
*
*	Arguments:
*		no arguments
*
*	returns: function doesn't return anything
*
----------------------------------------------------------------*/
void usage(void)
{
	printf("\n%s : 802.11 frame dump utility\n", appname);
	printf("    usage: %s [option ...] interface\n\n", appname);
	printf("          where valid options are:\n");
	printf("          -b        don't display beacon frames\n");
	printf("          -r        don't show received frames (shown by default)\n");
	printf("          -t        show transmitted frames\n");
	printf("          -c        don't show ctl frames\n");
	printf("          -m        don't show mgmt frames\n");
	printf("          -d        don't show data frames\n");
	printf("          -s <len>  show, at most, this many data field bytes\n");
	printf("          -x        show the whole frame as raw bytes\n");
	printf("          -w        don't decrypt data frame contents\n");
	printf("          -u        don't unfragment\n");
	printf("          -f        show fragments as well as defragged frame\n");
	printf("          -k <<keyindex(0-3)>-xx:xx:xx:xx:xx>\n");
	printf("                    set wep key[keyindex], don't foget to set all 4 keys\n");
	printf("          -v  display the program version and exit.\n\n");
	printf("  Note: this program doesn't do a whole lot of cmdline argument validation\n");
	printf("        so be careful when setting them.\n\n");
}
/* end of file frdumper.c */
