/* prism2mgmt.c: Management request handler functions.
* --------------------------------------------------------------------
*
* Linux WLAN 
*
*   The contents of this file are subject to the Mozilla Public
*   License Version 1.1 (the "License"); you may not use this file
*   except in compliance with the License. You may obtain a copy of
*   the License at http://www.mozilla.org/MPL/
*
*   Software distributed under the License is distributed on an "AS
*   IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*   implied. See the License for the specific language governing
*   rights and limitations under the License.
*
*   The initial developer of the original code is Mark S. Mathews
*   <mark@absoval.com>.  Portions created by Mark S. Mathews are 
*   Copyright (C) 1999 AbsoluteValue Software, Inc.  All Rights Reserved.
*
*   Alternatively, the contents of this file may be used under the
*   terms of the GNU Public License version 2 (the "GPL"), in which
*   case the provisions of the GPL are applicable instead of the
*   above.  If you wish to allow the use of your version of this file
*   only under the terms of the GPL and not to allow others to use
*   your version of this file under the MPL, indicate your decision
*   by deleting the provisions above and replace them with the notice
*   and other provisions required by the GPL.  If you do not delete
*   the provisions above, a recipient may use your version of this
*   file under either the MPL or the GPL.
*
* --------------------------------------------------------------------
*
* The initial author may be reached as mark@absoval.com, or 
* C/O AbsoluteValue Software Inc., P.O. Box 941149, 
* Maitland, FL, 32794-1149
*
* --------------------------------------------------------------------
*
* Portions of the development of this software were funded by 
* Intersil Corporation as part of PRISM(R) chipset product development.
*
* --------------------------------------------------------------------
*
* The functions in this file handle management requests sent from
* user mode.
*
* Most of these functions have two separate blocks of code that are
* compile-time conditional on WLAN_STA and WLAN_AP.  This is used
* to separate out the STA and AP responses to these management primitives.
* It's a choice (good, bad, indifferent?) to have the code in the same 
* place so it's clear that the same primitive  is implemented in both 
* cases but has different behavior.
*
* --------------------------------------------------------------------
*/

/*================================================================*/
/* System Includes */

#define __NO_VERSION__

#include <linux/config.h>
#define WLAN_DBVAR	prism2_debug
#include <wlan/wlan_compat.h>

#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/malloc.h>
#include <linux/netdevice.h>
#include <asm/io.h>
#include <asm/delay.h>
#include <asm/byteorder.h>

#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>
#include <pcmcia/cisreg.h>
#include <pcmcia/driver_ops.h>

/*================================================================*/
/* Project Includes */

#include <wlan/version.h>
#include <wlan/p80211types.h>
#include <wlan/p80211hdr.h>
#include <wlan/p80211mgmt.h>
#include <wlan/p80211conv.h>
#include <wlan/p80211msg.h>
#include <wlan/p80211netdev.h>
#include <wlan/p80211metadef.h>
#include <wlan/p80211metastruct.h>
#include <prism2/hfa384x.h>
#include <prism2/prism2mgmt.h>

/*================================================================*/
/* Local Constants */


/*================================================================*/
/* Local Macros */

/* Converts 802.11 format rate specifications to prism2 */
#define p80211rate_to_p2bit(n)	((((n)&~BIT7) == 2) ? BIT0 : \
				 (((n)&~BIT7) == 4) ? BIT1 : \
				 (((n)&~BIT7) == 11) ? BIT2 : \
				 (((n)&~BIT7) == 22) ? BIT3 : 0)

/*================================================================*/
/* Local Types */


/*================================================================*/
/* Local Static Definitions */


/*================================================================*/
/* Local Function Declarations */


/*================================================================*/
/* Function Definitions */


/*----------------------------------------------------------------
* prism2mgmt_powermgmt
*
* Set the power management state of this station's MAC.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
*	interrupt
----------------------------------------------------------------*/
int prism2mgmt_powermgmt(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int 			result = 0;
	p80211msg_dot11req_powermgmt_t	*msg = msgp;
	DBFENTER;

#if defined(WLAN_STA)
	/* Powermgmt is currently unsupported for STA */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	/* Powermgmt is never supported for AP */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2mgmt_scan
*
* Initiate a scan for BSSs.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
*	interrupt
----------------------------------------------------------------*/
int prism2mgmt_scan(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int 			result = 0;
	p80211msg_dot11req_scan_t	*msg = msgp;
	DBFENTER;

#if defined(WLAN_STA)
	/* Scan is currently unsupported for STA (handled by f/w) */
	/* TODO: there is f/w support for host managed scan, need */
	/*       to implement */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	/* Scan is currently unsupported for AP */
	/* TODO: Find out if host managed scan is supported in ap f/w */
	/*       we might want to use it to allow some "auto config"  */
	/*       mode where the AP scans for others and then selects  */
	/*       its channel and settings based on what it finds.     */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2mgmt_scan_results
*
* Retrieve the BSS description for one of the BSSs identified in
* a scan.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
*	interrupt
----------------------------------------------------------------*/
int prism2mgmt_scan_results(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int 			result = 0;
	p80211msg_dot11req_scan_results_t	*msg = msgp;
	DBFENTER;

#if defined(WLAN_STA)
	/* Same situation as scan */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;

#elif defined(WLAN_AP)
	/* Same situation as scan */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2mgmt_join
*
* Join a BSS whose BSS description was previously obtained with
* a scan.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
*	interrupt
----------------------------------------------------------------*/
int prism2mgmt_join(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int 			result = 0;
	p80211msg_dot11req_join_t	*msg = msgp;
	DBFENTER;

#if defined(WLAN_STA)
	/* TODO: Implement after scan */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	/* Never supported by APs */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2mgmt_authenticate
*
* Station should be begin an authentication exchange.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
*	interrupt
----------------------------------------------------------------*/
int prism2mgmt_authenticate(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int 			result = 0;
	p80211msg_dot11req_authenticate_t	*msg = msgp;
	DBFENTER;

#if defined(WLAN_STA)
	/* TODO: Decide how we're going to handle this one w/ Prism2 */
	/*       It could be entertaining since Prism2 doesn't have  */
	/*       an explicit way to control this */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	/* Never supported by APs */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2mgmt_deauthenticate
*
* Send a deauthenticate notification.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
*	interrupt
----------------------------------------------------------------*/
int prism2mgmt_deauthenticate(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int 			result = 0;
	p80211msg_dot11req_deauthenticate_t	*msg = msgp;
	DBFENTER;

#if defined(WLAN_STA)
	/* TODO: Decide how we're going to handle this one w/ Prism2 */
	/*       It could be entertaining since Prism2 doesn't have  */
	/*       an explicit way to control this */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	/* TODO: Decide how we're going to handle this one w/ Prism2 */
	/*       It could be entertaining since Prism2 doesn't have  */
	/*       an explicit way to control this */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2mgmt_associate
*
* Associate with an ESS.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
*	interrupt
----------------------------------------------------------------*/
int prism2mgmt_associate(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int 			result = 0;
	UINT16			reg;
	UINT16			port_type;
	p80211msg_dot11req_associate_t	*msg = msgp;
	DBFENTER;

#if defined(WLAN_STA)
	/* Set the TxRates */
	reg = 0x000f;
	hfa384x_drvr_setconfig16(hw, HFA384x_RID_TXRATECNTL, &reg);

	/* Set the PortType */
	port_type = 1; /* ess port */
	hfa384x_drvr_setconfig16(hw, HFA384x_RID_CNFPORTTYPE, &port_type);

	/* Enable the interrupts */
	reg = HFA384x_INTEN_INFDROP_SET(1) |
		HFA384x_INTEN_INFO_SET(1) |
		HFA384x_INTEN_ALLOC_SET(1) |
		HFA384x_INTEN_TXEXC_SET(1) |
		HFA384x_INTEN_TX_SET(1) |
		HFA384x_INTEN_RX_SET(1);
	outw( 0xffff, HFA384x_EVSTAT(priv->hw->iobase));
	outw( reg, HFA384x_INTEN(hw->iobase));
		
	/* Enable the Port */
	hfa384x_cmd_enable(hw, 0);
	
	/* Set the resultcode */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_success;

#elif defined(WLAN_AP)
	/* Never supported on AP */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2mgmt_reassociate
*
* Renew association because of a BSS change.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
*	interrupt
----------------------------------------------------------------*/
int prism2mgmt_reassociate(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int 			result = 0;
	p80211msg_dot11req_reassociate_t	*msg = msgp;
	DBFENTER;

#if defined(WLAN_STA)
	/* TODO: Not supported yet...not sure how we're going to do it */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	/* Never supported on AP */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2mgmt_disassociate
*
* Send a disassociation notification.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
*	interrupt
----------------------------------------------------------------*/
int prism2mgmt_disassociate(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int 			result = 0;
	p80211msg_dot11req_disassociate_t	*msg = msgp;
	DBFENTER;

#if defined(WLAN_STA)
	/* TODO: Not supported yet...not sure how we're going to do it */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	/* TODO: Not supported yet...not sure how we're going to do it */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2mgmt_start
*
* Start a BSS.  Any station can do this for IBSS, only AP for ESS.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
*	interrupt
----------------------------------------------------------------*/
int prism2mgmt_start(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int 			result = 0;
	p80211msg_dot11req_start_t	*msg = msgp;
	p80211pstrd_t		*pstr;	
	UINT8			bytebuf[80];
	hfa384x_bytestr_t	*p2bytestr = (hfa384x_bytestr_t*)bytebuf;
	hfa384x_PCFInfo_data_t	*pcfinfo = (hfa384x_PCFInfo_data_t*)bytebuf;
	UINT16			word;
	DBFENTER;

#if defined(WLAN_STA)

	/* Ad-Hoc not quite supported on Prism2 */
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	result = 0;

#elif defined(WLAN_AP)
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;

	/* Validate the command, if BSStype=infra is the tertiary loaded? */
	if ( (msg->bsstype.data == P80211ENUM_bsstype_independent && 
	      priv->cap_sup_sta.id != 4) ||
	     (msg->bsstype.data == P80211ENUM_bsstype_infrastructure && 
	      priv->cap_sup_sta.id != 5) ) {
		msg->resultcode.data = P80211ENUM_resultcode_invalid_parameters;
		return 0;
	}

	/* Set the REQUIRED config items */
	/* SSID */
	pstr = (p80211pstrd_t*)&(msg->ssid.data);
	prism2mgmt_pstr2bytestr(p2bytestr, pstr);
	result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFOWNSSID,
				bytebuf, HFA384x_RID_CNFOWNSSID_LEN);
	if ( result ) {
		WLAN_LOG_DEBUG0(1, "Unable to set SSID\n");
		goto failed;
	}

	/* bsstype */
	if ( msg->bsstype.data == P80211ENUM_bsstype_independent ) {
		word = HFA384x_PORTTYPE_IBSS;
		result = hfa384x_drvr_setconfig16(hw, HFA384x_RID_CNFPORTTYPE, &word);
		if ( result ) {
			WLAN_LOG_DEBUG1(1, "Unable to set port type=%d.\n", word);
			goto failed;
		}
	}

	/* beacon period */
	word = msg->beaconperiod.data;
	result = hfa384x_drvr_setconfig16(hw, HFA384x_RID_CNFAPBCNINT, &word);
	if ( result ) {
		WLAN_LOG_DEBUG1(1, "Unable to set beacon period=%d.\n", word);
		goto failed;
	}

	/* dschannel */
	word = msg->dschannel.data;
	result = hfa384x_drvr_setconfig16(hw, HFA384x_RID_CNFOWNCHANNEL, &word);
	if ( result ) {
		WLAN_LOG_DEBUG1(1, "Unable to set channel=%d.\n", word);
		goto failed;
	}
	/* Basic rates */
	word = p80211rate_to_p2bit(msg->basicrate1.data);
	if ( msg->basicrate2.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->basicrate2.data);
	}
	if ( msg->basicrate3.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->basicrate3.data);
	}
	if ( msg->basicrate4.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->basicrate4.data);
	}
	if ( msg->basicrate5.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->basicrate5.data);
	}
	if ( msg->basicrate6.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->basicrate6.data);
	}
	if ( msg->basicrate7.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->basicrate7.data);
	}
	if ( msg->basicrate8.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->basicrate8.data);
	}
	result = hfa384x_drvr_setconfig16(hw, HFA384x_RID_CNFBASICRATES, &word);
	if ( result ) {
		WLAN_LOG_DEBUG1(1, "Unable to set basicrates=%d.\n", word);
		goto failed;
	}

	/* Operational rates (supprates and txratecontrol) */
	word = msg->operationalrate1.data;
	if ( msg->operationalrate2.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->operationalrate2.data);
	}
	if ( msg->operationalrate3.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->operationalrate3.data);
	}
	if ( msg->operationalrate4.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->operationalrate4.data);
	}
	if ( msg->operationalrate5.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->operationalrate5.data);
	}
	if ( msg->operationalrate6.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->operationalrate6.data);
	}
	if ( msg->operationalrate7.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->operationalrate7.data);
	}
	if ( msg->operationalrate8.status == P80211ENUM_msgitem_status_data_ok ) {
		word |= p80211rate_to_p2bit(msg->operationalrate8.data);
	}
	if ( msg->bsstype.data == P80211ENUM_bsstype_infrastructure ) {
		result = hfa384x_drvr_setconfig16(hw, HFA384x_RID_CNFSUPPRATES, &word);
		if ( result ) {
			WLAN_LOG_DEBUG1(1, "Unable to set supprates=%d.\n", word);
			goto failed;
		}
		result = hfa384x_drvr_setconfig16(hw, HFA384x_RID_TXRATECNTL0, &word);
		if ( result ) {
			WLAN_LOG_DEBUG1(1, "Unable to set txrates=%d.\n", word);
			goto failed;
		}
	} else { /* assume ibss */
		result = hfa384x_drvr_setconfig16(hw, HFA384x_RID_TXRATECNTL, &word);
		if ( result ) {
			WLAN_LOG_DEBUG1(1, "Unable to set txrates=%d.\n", word);
			goto failed;
		}
	}

	/* ibssatimwindow */
	if (msg->bsstype.data != P80211ENUM_bsstype_independent ) {
		WLAN_LOG_INFO0("Setting atimwindow for non-ibss is pointless.\n");
	} else {
		word = msg->ibssatimwindow.data;
		result = hfa384x_drvr_setconfig16(hw, HFA384x_RID_CNFOWNATIMWIN, &word);
		if ( result ) {
			WLAN_LOG_DEBUG1(1, "Unable to set atimwindow=%d.\n", word);
			goto failed;
		}
	}

	/* The settings below this if statement only apply to infrastructure */
	/*  if we're not infrastructure, skip over them */
	if (msg->bsstype.data != P80211ENUM_bsstype_infrastructure ) {
		goto ibss_skip;
	}

	/* DTIM period */
	word = msg->dtimperiod.data;
	result = hfa384x_drvr_setconfig16(hw, HFA384x_RID_CNFOWNDTIMPER, &word);
	if ( result ) {
		WLAN_LOG_DEBUG1(1, "Unable to set dtim period=%d.\n", word);
		goto failed;
	}

	/* probedelay */
	WLAN_LOG_INFO0("No setting for probedelay in prism2, skipped.\n");

	/* cfpollable, cfpollreq, cfpperiod, cfpmaxduration */
	if (msg->cfpollable.data == P80211ENUM_truth_true && 
	    msg->cfpollreq.data == P80211ENUM_truth_true ) {
		WLAN_LOG_ERROR0("cfpollable=cfpollreq=true is illegal.\n");
		result = -1;
		goto failed;
	}

	/* read the PCFInfo and update */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_CNFAPPCFINFO, 
					pcfinfo, HFA384x_RID_CNFAPPCFINFO_LEN);
	if ( result ) {
		WLAN_LOG_DEBUG0(1, "Unable to read pcfinfo, not supported so skip it.\n");
		goto ibss_skip;
	}
	if ((msg->cfpollable.data == P80211ENUM_truth_false && 
	     msg->cfpollreq.data == P80211ENUM_truth_false) ) {
	    	pcfinfo->MediumOccupancyLimit = 0;
		pcfinfo->CFPPeriod = 0;
		pcfinfo->CFPMaxDuration = 0;
		pcfinfo->CFPFlags &= host2hfa384x_16((UINT16)~BIT0);
		
		if ( msg->cfpperiod.data == P80211ENUM_msgitem_status_data_ok ||
		     msg->cfpmaxduration.data == P80211ENUM_msgitem_status_data_ok ) {
			WLAN_LOG_WARNING0(
				"Setting cfpperiod or cfpmaxduration when "
				"cfpollable and cfreq are false is pointless.\n");
		}
	}
	if ((msg->cfpollable.data == P80211ENUM_truth_true ||
	     msg->cfpollreq.data == P80211ENUM_truth_true) ) {
		if ( msg->cfpollable.data == P80211ENUM_truth_true) {
			pcfinfo->CFPFlags |= host2hfa384x_16((UINT16)BIT0);
		}

		if ( msg->cfpperiod.status == P80211ENUM_msgitem_status_data_ok) {
			pcfinfo->CFPPeriod = msg->cfpperiod.data;
			pcfinfo->CFPPeriod = host2hfa384x_16(pcfinfo->CFPPeriod);
		}

		if ( msg->cfpmaxduration.status == P80211ENUM_msgitem_status_data_ok) {
			pcfinfo->CFPMaxDuration = msg->cfpmaxduration.data;
			pcfinfo->CFPMaxDuration = host2hfa384x_16(pcfinfo->CFPMaxDuration); 
			pcfinfo->MediumOccupancyLimit = pcfinfo->CFPMaxDuration;
		}
	}
	result = hfa384x_drvr_setconfig(hw, HFA384x_RID_CNFAPPCFINFO, 
					pcfinfo, HFA384x_RID_CNFAPPCFINFO_LEN);
	if ( result ) {
		WLAN_LOG_DEBUG0(1, "Unable to write pcfinfo, not supported so skip it.\n");
		/* goto failed; */
	}


ibss_skip:

	/* Enable the interrupts */
	word = HFA384x_INTEN_INFDROP_SET(1) |
		HFA384x_INTEN_INFO_SET(1) |
		HFA384x_INTEN_ALLOC_SET(1) |
		HFA384x_INTEN_TXEXC_SET(1) |
		HFA384x_INTEN_TX_SET(1) |
		HFA384x_INTEN_RX_SET(1);
	outw( 0xffff, HFA384x_EVSTAT(priv->hw->iobase));
	outw( word, HFA384x_INTEN(hw->iobase));

	/* Set the macmode so the frame setup code knows what to do */
	if ( msg->bsstype.data == P80211ENUM_bsstype_infrastructure ) {
		wlandev->macmode = WLAN_MACMODE_ESS_AP;
		word=2304;  /* lets extend the data length a bit */
		hfa384x_drvr_setconfig16(hw, HFA384x_RID_CNFMAXDATALEN, &word);
	}

	/* Set the BSSID to the same as our MAC */
	memcpy( wlandev->bssid, wlandev->netdev->dev_addr, WLAN_BSSID_LEN);

	/* Enable the Port */
	result = hfa384x_cmd_enable(hw, 0);
	if ( result ) {
		WLAN_LOG_DEBUG1(1, "Enable macport failed, result=%d.\n", result);
		goto failed;
	}
	
	msg->resultcode.data = P80211ENUM_resultcode_success;
	goto done;
failed:
	WLAN_LOG_DEBUG1(1, "Failed to set a config option, result=%d\n", result);
	msg->resultcode.data = P80211ENUM_resultcode_invalid_parameters;
done:
	result = 0;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2mgmt_readpda
*
* Collect the PDA data and put it in the message.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
----------------------------------------------------------------*/
int prism2mgmt_readpda(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	p80211msg_p2req_readpda_t	*msg = msgp;
	DBFENTER;

	/* This driver really shouldn't be active if we weren't able */
	/*  to read a PDA from a card.  Therefore, we assume the pda */
	/*  in priv->pda is good. */
	memcpy( msg->pda.data, priv->pda, HFA384x_PDA_LEN_MAX);
	msg->pda.status = P80211ENUM_msgitem_status_data_ok;

	msg->resultcode.data = P80211ENUM_resultcode_success;
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;

	DBFEXIT;
	return 0;
}


/*----------------------------------------------------------------
* prism2mgmt_auxport_state
*
* Enables/Disables the card's auxiliary port.  Should be called 
* before and after a sequence of auxport_read()/auxport_write() 
* calls.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
----------------------------------------------------------------*/
int prism2mgmt_auxport_state(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	p80211msg_p2req_auxport_state_t	*msg = msgp;
	DBFENTER;

	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	if (msg->enable.data == P80211ENUM_truth_true) {
		if ( hfa384x_cmd_aux_enable(hw) ) {
			msg->resultcode.data = P80211ENUM_resultcode_implementation_failure;
		} else {
			msg->resultcode.data = P80211ENUM_resultcode_success;
		}
	} else {
		hfa384x_cmd_aux_disable(hw);
		msg->resultcode.data = P80211ENUM_resultcode_success;
	}

	DBFEXIT;
	return 0;
}


/*----------------------------------------------------------------
* prism2mgmt_auxport_read
*
* Copies data from the card using the auxport.  The auxport must
* have previously been enabled.  Note: this is not the way to 
* do downloads, see the [ram|flash]dl functions.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
----------------------------------------------------------------*/
int prism2mgmt_auxport_read(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	p80211msg_p2req_auxport_read_t	*msg = msgp;
	UINT32			addr;
	UINT32			len;
	UINT8*			buf;
	UINT32			maxlen = sizeof(msg->data.data);
	DBFENTER;

	if ( hw->auxen ) {
		addr = msg->addr.data;
		len = msg->len.data;
		buf = msg->data.data;
		if ( len <= maxlen ) {  /* max read/write size */
			hfa384x_copy_from_aux(hw, addr, buf, len);
		} else {
			WLAN_LOG_DEBUG0(1,"Attempt to read > maxlen from auxport.\n");
			msg->resultcode.data = P80211ENUM_resultcode_refused;
		}

	} else {
		msg->resultcode.data = P80211ENUM_resultcode_refused;
	}
	msg->data.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;

	DBFEXIT;
	return 0;
}


/*----------------------------------------------------------------
* prism2mgmt_auxport_write
*
* Copies data to the card using the auxport.  The auxport must
* have previously been enabled.  Note: this is not the way to 
* do downloads, see the [ram|flash]dl functions.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
----------------------------------------------------------------*/
int prism2mgmt_auxport_write(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	p80211msg_p2req_auxport_write_t	*msg = msgp;
	UINT32			addr;
	UINT32			len;
	UINT8*			buf;
	UINT32			maxlen = sizeof(msg->data.data);
	DBFENTER;

	if ( hw->auxen ) {
		addr = msg->addr.data;
		len = msg->len.data;
		buf = msg->data.data;
		if ( len <= maxlen ) {  /* max read/write size */
			hfa384x_copy_to_aux(hw, addr, buf, len);
		} else {
			WLAN_LOG_DEBUG0(1,"Attempt to write > maxlen from auxport.\n");
			msg->resultcode.data = P80211ENUM_resultcode_refused;
		}

	} else {
		msg->resultcode.data = P80211ENUM_resultcode_refused;
	}
	msg->data.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;

	DBFEXIT;
	return 0;
}


/*----------------------------------------------------------------
* prism2mgmt_ramdl_state
*
* Establishes the beginning/end of a card RAM download session.
*
* It is expected that the ramdl_write() function will be called 
* one or more times between the 'enable' and 'disable' calls to
* this function.
*
* Note: This function should not be called when a mac comm port 
*       is active.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
----------------------------------------------------------------*/
int prism2mgmt_ramdl_state(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	p80211msg_p2req_ramdl_state_t	*msg = msgp;
	DBFENTER;

	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	if  ( msg->enable.data == P80211ENUM_truth_true ) {
		if ( hfa384x_drvr_ramdl_enable(hw, msg->exeaddr.data) ) {
			msg->resultcode.data = P80211ENUM_resultcode_implementation_failure;
		} else {
			msg->resultcode.data = P80211ENUM_resultcode_success;
		}
	} else {
		hfa384x_drvr_ramdl_disable(hw);
		msg->resultcode.data = P80211ENUM_resultcode_success;

		/*TODO: Reset everything....the MAC just restarted */
		udelay(1000);
		prism2sta_initmac(wlandev);
	}

	DBFEXIT;
	return 0;
}


/*----------------------------------------------------------------
* prism2mgmt_ramdl_write
*
* Writes a buffer to the card RAM using the download state.  This
* is for writing code to card RAM.  To just read or write raw data
* use the aux functions.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
----------------------------------------------------------------*/
int prism2mgmt_ramdl_write(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	p80211msg_p2req_ramdl_write_t	*msg = msgp;
	UINT32			addr;
	UINT32			len;
	UINT8			*buf;
	DBFENTER;

	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	/* first validate the length */
	if  ( msg->len.data > sizeof(msg->data.data) ) {
		msg->resultcode.status = P80211ENUM_resultcode_invalid_parameters;
		return 0;
	}
	/* call the hfa384x function to do the write */
	addr = msg->addr.data;
	len = msg->len.data;
	buf = msg->data.data;
	if ( hfa384x_drvr_ramdl_write(hw, addr, buf, len) ) {
		msg->resultcode.data = P80211ENUM_resultcode_refused;
		
	}
	msg->resultcode.data = P80211ENUM_resultcode_success;

	DBFEXIT;
	return 0;
}


/*----------------------------------------------------------------
* prism2mgmt_flashdl_state
*
* Establishes the beginning/end of a card Flash download session.
*
* It is expected that the flashdl_write() function will be called 
* one or more times between the 'enable' and 'disable' calls to
* this function.
*
* Note: This function should not be called when a mac comm port 
*       is active.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
----------------------------------------------------------------*/
int prism2mgmt_flashdl_state(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	p80211msg_p2req_flashdl_state_t	*msg = msgp;
	DBFENTER;

	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	if  ( msg->enable.data == P80211ENUM_truth_true ) {
		if ( hfa384x_drvr_flashdl_enable(hw) ) {
			msg->resultcode.data = P80211ENUM_resultcode_implementation_failure;
		} else {
			msg->resultcode.data = P80211ENUM_resultcode_success;
		}
	} else {
		hfa384x_drvr_flashdl_disable(hw);
		msg->resultcode.data = P80211ENUM_resultcode_success;

		/*TODO: Reset everything....the MAC just restarted */
		udelay(1000);
		prism2sta_initmac(wlandev);
	}

	DBFEXIT;
	return 0;
}


/*----------------------------------------------------------------
* prism2mgmt_flashdl_write
*
* 
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
----------------------------------------------------------------*/
int prism2mgmt_flashdl_write(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	p80211msg_p2req_flashdl_write_t	*msg = msgp;
	UINT32			addr;
	UINT32			len;
	UINT8			*buf;
	DBFENTER;

	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	/* first validate the length */
	if  ( msg->len.data > sizeof(msg->data.data) ) {
		msg->resultcode.status = P80211ENUM_resultcode_invalid_parameters;
		return 0;
	}
	/* call the hfa384x function to do the write */
	addr = msg->addr.data;
	len = msg->len.data;
	buf = msg->data.data;
	if ( hfa384x_drvr_flashdl_write(hw, addr, buf, len) ) {
		msg->resultcode.data = P80211ENUM_resultcode_refused;
		
	}
	msg->resultcode.data = P80211ENUM_resultcode_success;

	DBFEXIT;
	return 0;
}


/*----------------------------------------------------------------
* prism2mgmt_mm_state
*
* Enables the usage of the QoS parameter to the transmit 
* command.  Also sets (for now) a variable that contains a static
* QoS value to use for all transmitted frames.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
----------------------------------------------------------------*/
int prism2mgmt_mm_state(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	p80211msg_p2req_mm_state_t	*msg = msgp;
	DBFENTER;

	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	if  ( msg->enable.data == P80211ENUM_truth_true) {
		priv->qos_enable = 1;
		priv->qos_staticlevel = msg->level.data;
		msg->resultcode.status = P80211ENUM_resultcode_success;
	} else {
		priv->qos_enable = 0;
		priv->qos_staticlevel = 0;
		msg->resultcode.status = P80211ENUM_resultcode_success;
	}

	DBFEXIT;
	return 0;
}


/*----------------------------------------------------------------
* prism2mgmt_wlansniff
*
* Start or stop sniffing.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
*	interrupt
----------------------------------------------------------------*/
int prism2mgmt_wlansniff(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int 			result = 0;
	p80211msg_lnxreq_wlansniff_t	*msg = msgp;
	UINT16			word;

	DBFENTER;

	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;

	switch (msg->enable.data)
	{
	case P80211ENUM_truth_false:
		/* Confirm that we're in monitor mode */
		if ( !priv->sniffing ) {
			msg->resultcode.data = P80211ENUM_resultcode_invalid_parameters;
			result = 0;
		}
		/* Disable monitor mode */
		result = hfa384x_cmd_monitor(hw, HFA384x_MONITOR_DISABLE);
		if ( result ) {
			WLAN_LOG_DEBUG1(1,
				"failed to disable monitor mode, result=%d\n", 
				result);
			goto failed;
		}
		/* Disable port 0 */
		result = hfa384x_cmd_disable(hw, 0);
		if ( result ) {
			WLAN_LOG_DEBUG1(1,
			"failed to disable port 0 after sniffing, result=%d\n", 
			result);
			goto failed;
		}
		/* Clear the driver state */
		priv->sniffing = 0;

		/* Restore the wepflags */
		result = hfa384x_drvr_setconfig16(hw, 
				HFA384x_RID_CNFWEPFLAGS, 
				&(priv->presniff_wepflags));
		if ( result ) {
			WLAN_LOG_DEBUG2(1,
			"failed to restore wepflags=0x%04x, result=%d\n", 
			word,
			result);
			goto failed;
		}
		/* Set the port to its prior type and enable (if necessary) */
		if (priv->presniff_port_type != 0 ) {
			word = priv->presniff_port_type;
			result = hfa384x_drvr_setconfig16(hw, 
				HFA384x_RID_CNFPORTTYPE, &word);
			if ( result ) {
				WLAN_LOG_DEBUG1(1,
				"failed to restore porttype, result=%d\n", 
				result);
				goto failed;
			}
			/* Enable the port */
			result = hfa384x_cmd_enable(hw, 0);
			if ( result ) {
				WLAN_LOG_DEBUG1(1,
				"failed to enable port to presniff setting, result=%d\n", 
				result);
				goto failed;
			}
		}

		msg->resultcode.data = P80211ENUM_resultcode_success;
		result = 0;
		goto exit;
		break;
	case P80211ENUM_truth_true:
		/* Disable the port (if enabled), only check Port 0 */
		if ( hw->port_enabled[0] ) {
			/* Save macport 0 state */
			result = hfa384x_drvr_getconfig16(hw,
					HFA384x_RID_CNFPORTTYPE,
					&(priv->presniff_port_type));
			if ( result ) {
				WLAN_LOG_DEBUG1(1,"failed to read porttype, result=%d\n", result);
				goto failed;
			}
			/* Save the wepflags state */
			result = hfa384x_drvr_getconfig16(hw,
					HFA384x_RID_CNFWEPFLAGS,
					&(priv->presniff_wepflags));
			if ( result ) {
				WLAN_LOG_DEBUG1(1,"failed to read wepflags, result=%d\n", result);
				goto failed;
			}
			result = hfa384x_cmd_disable(hw, 0);
			if ( result ) {
				WLAN_LOG_DEBUG1(1,
				"failed to clear port 0 for sniffing, result=%d\n", 
				result);
				goto failed;
			}
		}
		else {
			priv->presniff_port_type = 0;
		}
		/* Set the channel we wish to sniff */
		word = msg->channel.data;
		result = hfa384x_drvr_setconfig16(hw, 
				HFA384x_RID_CNFOWNCHANNEL, &word);
		if ( result ) {
			WLAN_LOG_DEBUG2(1,
			"failed to set channel %d, result=%d\n", 
			word,
			result);
			goto failed;
		}
		/* Set the port type to pIbss */
		word = HFA384x_PORTTYPE_IBSS;
		result = hfa384x_drvr_setconfig16(hw, 
				HFA384x_RID_CNFPORTTYPE, &word);
		if ( result ) {
			WLAN_LOG_DEBUG2(1,
			"failed to set porttype %d, result=%d\n", 
			word,
			result);
			goto failed;
		}
		/* Set the wepflags for no decryption */
		word =	HFA384x_WEPFLAGS_DISABLE_TXCRYPT | 
			HFA384x_WEPFLAGS_DISABLE_RXCRYPT;
		result = hfa384x_drvr_setconfig16(hw, 
				HFA384x_RID_CNFWEPFLAGS, &word);
		if ( result ) {
			WLAN_LOG_DEBUG2(1,
			"failed to set wepflags=0x%04x, result=%d\n", 
			word,
			result);
			goto failed;
		}
		/* Enable the port */
		result = hfa384x_cmd_enable(hw, 0);
		if ( result ) {
			WLAN_LOG_DEBUG1(1,
			"failed to enable port for sniffing, result=%d\n", 
			result);
			goto failed;
		}
		/* Enable monitor mode */
		result = hfa384x_cmd_monitor(hw, HFA384x_MONITOR_ENABLE);
		if ( result ) {
			WLAN_LOG_DEBUG1(1,
			"failed to enable monitor mode, result=%d\n", 
			result);
			goto failed;
		}
		/* Set the driver state */
		priv->sniffing = 1;

		msg->resultcode.data = P80211ENUM_resultcode_success;
		result = 0;
		goto exit;
		break;
	default:
		msg->resultcode.data = P80211ENUM_resultcode_invalid_parameters;
		result = 0;
		goto exit;
		break;
	}

failed:
	msg->resultcode.data = P80211ENUM_resultcode_refused;
	result = 0;
exit:	
	DBFEXIT;
	return result;
}

