/* prism2sta.c: Implements the station functionality for prism2
* --------------------------------------------------------------------
*
* Linux WLAN 
*
*   The contents of this file are subject to the Mozilla Public
*   License Version 1.1 (the "License"); you may not use this file
*   except in compliance with the License. You may obtain a copy of
*   the License at http://www.mozilla.org/MPL/
*
*   Software distributed under the License is distributed on an "AS
*   IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*   implied. See the License for the specific language governing
*   rights and limitations under the License.
*
*   The initial developer of the original code is Mark S. Mathews
*   <mark@absoval.com>.  Portions created by Mark S. Mathews are 
*   Copyright (C) 1999 AbsoluteValue Software, Inc.  All Rights Reserved.
*
*   Alternatively, the contents of this file may be used under the
*   terms of the GNU Public License version 2 (the "GPL"), in which
*   case the provisions of the GPL are applicable instead of the
*   above.  If you wish to allow the use of your version of this file
*   only under the terms of the GPL and not to allow others to use
*   your version of this file under the MPL, indicate your decision
*   by deleting the provisions above and replace them with the notice
*   and other provisions required by the GPL.  If you do not delete
*   the provisions above, a recipient may use your version of this
*   file under either the MPL or the GPL.
*
* --------------------------------------------------------------------
*
* The initial author may be reached as mark@absoval.com, or 
* C/O AbsoluteValue Software Inc., P.O. Box 941149, 
* Maitland, FL, 32794-1149
*
* --------------------------------------------------------------------
*
* Portions of the development of this software were funded by 
* Intersil Corporation as part of PRISM(R) chipset product development.
*
* --------------------------------------------------------------------
*
* This file implements the module and linux pcmcia routines for the
* prism2 driver.
*
* --------------------------------------------------------------------
*/

/*================================================================*/
/* System Includes */

#include <linux/config.h>
#define WLAN_DBVAR	prism2_debug
#include <wlan/wlan_compat.h>

#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/malloc.h>
#include <linux/netdevice.h>
#include <asm/io.h>
#include <asm/delay.h>
#include <asm/byteorder.h>

#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>
#include <pcmcia/cisreg.h>
#include <pcmcia/driver_ops.h>

/*================================================================*/
/* Project Includes */

#include <wlan/version.h>
#include <wlan/p80211types.h>
#include <wlan/p80211hdr.h>
#include <wlan/p80211mgmt.h>
#include <wlan/p80211conv.h>
#include <wlan/p80211msg.h>
#include <wlan/p80211netdev.h>
#include <wlan/p80211req.h>
#include <wlan/p80211metadef.h>
#include <wlan/p80211metastruct.h>
#include <prism2/hfa384x.h>
#include <prism2/prism2mgmt.h>

/*================================================================*/
/* Local Constants */

#define PRISM2STA_MAGIC		(0x4a2d)
#define	INFOFRM_LEN_MAX		sizeof(hfa384x_ScanResults_t)

/*================================================================*/
/* Local Macros */

#define PRISM2_INFOBUF_MAX	(sizeof(hfa384x_HandoverAddr_t))
#define PRISM2_TXBUF_MAX	(sizeof(hfa384x_tx_frame_t) + \
				WLAN_DATA_MAXLEN - \
				WLAN_WEP_IV_LEN - \
				WLAN_WEP_ICV_LEN)

/*================================================================*/
/* Local Types */

/*================================================================*/
/* Local Static Definitions */

#if (WLAN_HOSTIF == WLAN_PCMCIA)
#define DRIVER_SUFFIX	"_cs"
/* #else , TODO: Fix .depend generation for multiple driver build */
/* #error "HOSTIF unsupported or undefined!" */
#endif

#if 0
#if defined(WLAN_STA)
static char		*version = "prism2sta" DRIVER_SUFFIX ".o: " WLAN_RELEASE;
static dev_info_t	dev_info = "prism2sta" DRIVER_SUFFIX;
#elif defined(WLAN_AP)
static char		*version = "prism2ap" DRIVER_SUFFIX ".o: " WLAN_RELEASE;
static dev_info_t	dev_info = "prism2ap" DRIVER_SUFFIX;
/* #else , TODO: Fix .depend generation for multiple driver build */
/* #error "Driver type (AP or STA) not defined!" */
#endif
#endif

static char		*version = "prism2" DRIVER_SUFFIX ".o: " WLAN_RELEASE;
static dev_info_t	dev_info = "prism2" DRIVER_SUFFIX;

static dev_link_t	*dev_list = NULL;	/* head of instance list */

/*----------------------------------------------------------------*/
/* --Module Parameters */

int		prism2_debug=0;			/* Debug output level, */
static u_int	irq_mask = 0xdeb8;		/* Interrupt mask */
static int	irq_list[4] = { -1 };		/* Interrupt list (alternative) */
static u_int	prism2_irq_evread_max=10;	/* Maximum number of ev_reads (loops)
						   in irq handler */

MODULE_PARM( prism2_debug, "i");	
MODULE_PARM( irq_mask, "i");
MODULE_PARM( irq_list, "1-4i");
MODULE_PARM( prism2_irq_evread_max, "i");


/*================================================================*/
/* Local Function Declarations */

int		init_module(void);
void		cleanup_module(void);
dev_link_t	*prism2sta_attach(void);
static void	prism2sta_detach(dev_link_t *link);
static void	prism2sta_config(dev_link_t *link);
static void	prism2sta_release(UINT32 arg);
static int 	prism2sta_event (event_t event, int priority, event_callback_args_t *args);

static int	prism2sta_open(wlandevice_t *wlandev);
static int	prism2sta_close(wlandevice_t *wlandev);
static void	prism2sta_reset(wlandevice_t *wlandev );
static int	prism2sta_txframe(wlandevice_t *wlandev, wlan_pb_t *pb);
static int	prism2sta_mlmerequest(wlandevice_t *wlandev, p80211msg_t *msg);

static void 	prism2sta_interrupt IRQ(int irq, void *dev_id, struct pt_regs *regs);
static void	prism2sta_int_infdrop(wlandevice_t *wlandev);
static void	prism2sta_int_info(wlandevice_t *wlandev);
static void	prism2sta_int_txexc(wlandevice_t *wlandev);
static void	prism2sta_int_tx(wlandevice_t *wlandev);
static void	prism2sta_int_rx(wlandevice_t *wlandev);
static void	prism2sta_int_rxmonitor( wlandevice_t *wlandev, UINT16 rxfid, hfa384x_rx_frame_t *rxdesc);
static void	prism2sta_int_alloc(wlandevice_t *wlandev);

static void	prism2sta_inf_handover(wlandevice_t *wlandev, void *buf);
static void	prism2sta_inf_tallies(wlandevice_t *wlandev, void *buf);
static void	prism2sta_inf_scanresults(wlandevice_t *wlandev, void *buf);
static void	prism2sta_inf_linkstatus(wlandevice_t *wlandev, void *buf);
static void	prism2sta_inf_assocstatus(wlandevice_t *wlandev, void *buf);

/*================================================================*/
/* Function Definitions */

/*----------------------------------------------------------------
* dmpmem
*
* Debug utility function to dump memory to the kernel debug log.
*
* Arguments:
*	buf	ptr data we want dumped
*	len	length of data
*
* Returns: 
*	nothing
* Side effects:
*
* Call context:
*	process thread
*	interrupt
----------------------------------------------------------------*/
inline void dmpmem(void *buf, int n)
{
	int c;
	for ( c= 0; c < n; c++) {
		if ( (c % 16) == 0 ) printk(KERN_DEBUG"dmp[%d]: ", c);
		printk("%02x ", ((UINT8*)buf)[c]);
		if ( (c % 16) == 15 ) printk("\n");
	}
	if ( (c % 16) != 0 ) printk("\n");
}

/*----------------------------------------------------------------
* cs_error
*
* Utility function to print card services error messages.
*
* Arguments:
*	handle	client handle identifying this CS client
*	func	CS function number that generated the error
*	ret	CS function return code
*
* Returns: 
*	nothing
* Side effects:
*
* Call context:
*	process thread
*	interrupt
----------------------------------------------------------------*/
static void cs_error(client_handle_t handle, int func, int ret)
{
#if CS_RELEASE_CODE < 0x2911
	CardServices(ReportError, dev_info, (void *)func, (void *)ret);
#else
	error_info_t err = { func, ret };
	CardServices(ReportError, handle, &err);
#endif
}


/*----------------------------------------------------------------
* prism2sta_open
*
* WLAN device open method.  Called from p80211netdev when kernel 
* device open (start) method is called in response to the 
* SIOCSIIFFLAGS ioctl changing the flags bit IFF_UP 
* from clear to set.
*
* Arguments:
*	wlandev		wlan device structure
*
* Returns: 
*	0	success
*	>0	f/w reported error
*	<0	driver reported error
*
* Side effects:
*
* Call context:
*	process thread
----------------------------------------------------------------*/
int prism2sta_open(wlandevice_t *wlandev)
{
	int			result = 0;
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	int			i;
	DBFENTER;

	/* Make sure at least 1 port is enabled */
	for ( i = 0; i < HFA384x_NUMPORTS_MAX; i++) {
		if ( priv->hw->port_enabled[i] != 0 ) break;
	}
	if ( i >= HFA384x_NUMPORTS_MAX ) {
		result = -ENOMEDIUM;
	}

	/* We don't currently have to do anything else.  */
	/* The setup of the MAC should have been done previously */
	/* via the mlme commands. */
	/* Higher layers know we're ready from dev->start==1 and */
	/* dev->tbusy==0.  Our rx path knows to pass up received */
	/* frames because of dev->flags&IFF_UP is true. */
	
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2sta_close
*
* WLAN device close method.  Called from p80211netdev when kernel 
* device close method is called in response to the 
* SIOCSIIFFLAGS ioctl changing the flags bit IFF_UP 
* from set to clear.
*
* Arguments:
*	wlandev		wlan device structure
*
* Returns: 
*	0	success
*	>0	f/w reported error
*	<0	driver reported error
*
* Side effects:
*
* Call context:
*	process thread
----------------------------------------------------------------*/
int prism2sta_close(wlandevice_t *wlandev)
{
	DBFENTER;

	/* We don't currently have to do anything else.  */
	/* Higher layers know we're not ready from dev->start==0 and */
	/* dev->tbusy==1.  Our rx path knows to pass up received */
	/* frames because of dev->flags&IFF_UP is false. */
	/* We leave the port enabled in case someone needs to receive */
	/* Info frames or send Notify frames.  All rx frames after this */
	/* will be dropped. */

	DBFEXIT;
	return 0;
}


/*----------------------------------------------------------------
* prism2sta_reset
*
* Not currently implented.
*
* Arguments:
*	wlandev		wlan device structure
*	none
*
* Returns: 
*	nothing
*
* Side effects:
*
* Call context:
*	process thread
----------------------------------------------------------------*/
void prism2sta_reset(wlandevice_t *wlandev )
{
	DBFENTER;
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_txframe
*
* Takes a frame from p80211 and queues it for transmission.
*
* Arguments:
*	wlandev		wlan device structure
*	pb		packet buffer struct.  Contains an 802.11
*			data frame.
* Returns: 
*	0		Success and more buffs available
*	1		Success but no more buffs
*	2		Allocation failure
*	4		Buffer full or queue busy
*
* Side effects:
*
* Call context:
*	process thread
----------------------------------------------------------------*/
int prism2sta_txframe(wlandevice_t *wlandev, wlan_pb_t *pb )
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	hfa384x_tx_frame_t	txdesc;
	int			result;
	DBFENTER;

#if defined(DETHER) 
printk(KERN_DEBUG"ethfrm[x] - [%d]:\n", pb->ethbuflen);
dmpmem(pb->ethbuf, pb->ethbuflen);
#endif

	/* Build Tx frame structure */
	/* Set up the control field */
	memset(&txdesc, 0, sizeof(txdesc));
	txdesc.tx_control = 
		HFA384x_TX_MACPORT_SET(0) | 
		HFA384x_TX_STRUCTYPE_SET(1) |
		HFA384x_TX_TXEX_SET(1) | 
		HFA384x_TX_TXOK_SET(1); 
	txdesc.tx_control = host2hfa384x_16(txdesc.tx_control);

	/* Set up the 802.11 header */
	if ( priv->priv_invoked ) {
		pb->p80211_hdr->a3.fc |= host2ieee16(WLAN_SET_FC_ISWEP(1));
	}
	memcpy(&(txdesc.frame_control), pb->p80211_hdr, WLAN_HDR_A3_LEN);

	/* Set the len, complicated because of pieces in pb */
	txdesc.data_len = pb->p80211buflen - WLAN_HDR_A3_LEN; /* llc+snap? */
	txdesc.data_len += pb->p80211_payloadlen;
	txdesc.data_len = host2hfa384x_16(txdesc.data_len);

	/* Copy descriptor part to FID */
#if defined(DDESC)
printk(KERN_DEBUG "Desc[0] - [%d]: \n", sizeof(txdesc));
dmpmem(&txdesc, sizeof(txdesc));
#endif
	result = hfa384x_copy_to_bap(hw, 
			hw->bap, 
			priv->txfid, 
			0, 
			&txdesc, 
			sizeof(txdesc));
	if ( result ) {
		WLAN_LOG_DEBUG3(1, "copy_to_bap(%04x, 0, %d) failed, result=0x%x\n", 
			priv->txfid, 
			sizeof(txdesc),
			result);
		result = 3;
		goto failed;
	}

	/* Copy 802.11 data to FID */
	if ( pb->p80211buflen > WLAN_HDR_A3_LEN ) { /* copy llc+snap hdr */
#if defined(DLLC)
printk(KERN_DEBUG "Hdr[%d] - [%d]:\n", sizeof(txdesc), pb->p80211buflen - WLAN_HDR_A3_LEN);
dmpmem(pb->p80211buf + WLAN_HDR_A3_LEN, pb->p80211buflen - WLAN_HDR_A3_LEN);
#endif
		
		result = hfa384x_copy_to_bap( hw, 
				     hw->bap, 
				     priv->txfid,
				     sizeof(txdesc),
				     pb->p80211buf + WLAN_HDR_A3_LEN,
				     pb->p80211buflen - WLAN_HDR_A3_LEN);
		if ( result ) {
			WLAN_LOG_DEBUG4(1,
				"copy_to_bap(%04x, %d, %d) failed, result=0x%x\n", 
				priv->txfid, 
				sizeof(txdesc),
				pb->p80211buflen - WLAN_HDR_A3_LEN,
				result);
			result = 3;
			goto failed;
		}
	}

#if defined(D80211)
printk(KERN_DEBUG "Data[%d] - [%d]:\n", sizeof(txdesc) + pb->p80211buflen - WLAN_HDR_A3_LEN, pb->p80211_payloadlen);
dmpmem(pb->p80211_payload, pb->p80211_payloadlen);
#endif

	result = hfa384x_copy_to_bap( hw, 
		     hw->bap, 
		     priv->txfid, 
		     sizeof(txdesc) + pb->p80211buflen - WLAN_HDR_A3_LEN, 
		     pb->p80211_payload, 
		     pb->p80211_payloadlen);
	
	if ( result ) {
		WLAN_LOG_DEBUG4(1,
			"copy_to_bap(%04x, %d, %d) failed, result=0x%x\n", 
			priv->txfid,
		 	sizeof(txdesc) + pb->p80211buflen - WLAN_HDR_A3_LEN, 
	 		pb->p80211_payloadlen,
			result);
		result = 3;
		goto failed;
	}

	/* Issue Tx command */
	if ( priv->qos_enable ) {
		if (priv->qos_staticlevel != 0) {
			/* static qos level, set via MIB */
			result = hfa384x_cmd_transmit(hw, 
				HFA384x_TXCMD_RECL, 
				priv->qos_staticlevel - 1, priv->txfid);
		} else {
			/* qos from skb */
			result = hfa384x_cmd_transmit(hw, 
				HFA384x_TXCMD_RECL, 
#if 0
/* skb->tc_index (per Gabe's suggestion) doesn't exist on my systems */
				((struct sk_buff*)pb->ethhostbuf)->tc_index & 
				(BIT0|BIT1),
#endif
				0,
				priv->txfid);

			/* TODO: use the two least significant bits of
			    skb->tc_index */


		}
	} else {
		result = hfa384x_cmd_transmit(hw,
				HFA384x_TXCMD_RECL, 0, priv->txfid);
		
	}

	if ( result != 0 ) {
		WLAN_LOG_DEBUG2(1,"cmd_tx(%04x) failed, result=%d", 
			priv->txfid, result);
		result = 2;
		goto failed;
	}
	result = 1;	/* indicate we haven't any buffers, int_alloc will clear */
failed:
	p80211pb_free(pb);

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2sta_mlmerequest
*
* wlan command message handler.  All we do here is pass the message
* over to the prism2sta_mgmt_handler.
*
* Arguments:
*	wlandev		wlan device structure
*	msg		wlan command message
* Returns: 
*	0		success
*	<0		successful acceptance of message, but we're
*			waiting for an async process to finish before
*			we're done with the msg.  When the asynch
*			process is done, we'll call the p80211 
*			function p80211req_confirm() .
*	>0		An error occurred while we were handling
*			the message.
*
* Side effects:
*
* Call context:
*	process thread
----------------------------------------------------------------*/
int prism2sta_mlmerequest(wlandevice_t *wlandev, p80211msg_t *msg)
{
	int result = 0;
	DBFENTER;

	switch( msg->msgcode )
	{
	case DIDmsg_dot11req_mibget :
		WLAN_LOG_DEBUG0(2,"Received mibget request\n");
		result = prism2mgmt_mibset_mibget(wlandev, msg);
		break;
	case DIDmsg_dot11req_mibset :
		WLAN_LOG_DEBUG0(2,"Received mibset request\n");
		result = prism2mgmt_mibset_mibget(wlandev, msg);
		break;
	case DIDmsg_dot11req_powermgmt :
		WLAN_LOG_DEBUG0(2,"Received powermgmt request\n");
		result = prism2mgmt_powermgmt(wlandev, msg);
		break;
	case DIDmsg_dot11req_scan :
		WLAN_LOG_DEBUG0(2,"Received scan request\n");
		result = prism2mgmt_scan(wlandev, msg);
		break;
	case DIDmsg_dot11req_scan_results :
		WLAN_LOG_DEBUG0(2,"Received scan_results request\n");
		result = prism2mgmt_scan_results(wlandev, msg);
		break;
	case DIDmsg_dot11req_join :
		WLAN_LOG_DEBUG0(2,"Received join request\n");
		result = prism2mgmt_join(wlandev, msg);
		break;
	case DIDmsg_dot11req_authenticate :
		WLAN_LOG_DEBUG0(2,"Received authenticate request\n");
		result = prism2mgmt_authenticate(wlandev, msg);
		break;
	case DIDmsg_dot11req_deauthenticate :
		WLAN_LOG_DEBUG0(2,"Received mlme deauthenticate request\n");
		result = prism2mgmt_deauthenticate(wlandev, msg);
		break;
	case DIDmsg_dot11req_associate :
		WLAN_LOG_DEBUG0(2,"Received mlme associate request\n");
		result = prism2mgmt_associate(wlandev, msg);
		break;
	case DIDmsg_dot11req_reassociate :
		WLAN_LOG_DEBUG0(2,"Received mlme reassociate request\n");
		result = prism2mgmt_reassociate(wlandev, msg);
		break;
	case DIDmsg_dot11req_disassociate :
		WLAN_LOG_DEBUG0(2,"Received mlme disassociate request\n");
		result = prism2mgmt_disassociate(wlandev, msg);
		break;
	case DIDmsg_dot11req_start :
		WLAN_LOG_DEBUG0(2,"Received mlme start request\n");
		result = prism2mgmt_start(wlandev, msg);
		break;
	case DIDmsg_lnxreq_wlansniff :
		WLAN_LOG_DEBUG0(2,"Received mlme wlansniff request\n");
		result = prism2mgmt_wlansniff(wlandev, msg);
		break;
	case DIDmsg_p2req_readpda :
		WLAN_LOG_DEBUG0(2,"Received mlme readpda request\n");
		result = prism2mgmt_readpda(wlandev, msg);
		break;
	case DIDmsg_p2req_auxport_state :
		WLAN_LOG_DEBUG0(2,"Received mlme auxport_state request\n");
		result = prism2mgmt_auxport_state(wlandev, msg);
		break;
	case DIDmsg_p2req_auxport_read :
		WLAN_LOG_DEBUG0(2,"Received mlme auxport_read request\n");
		result = prism2mgmt_auxport_read(wlandev, msg);
		break;
	case DIDmsg_p2req_auxport_write :
		WLAN_LOG_DEBUG0(2,"Received mlme auxport_write request\n");
		result = prism2mgmt_auxport_write(wlandev, msg);
		break;
	case DIDmsg_p2req_ramdl_state :
		WLAN_LOG_DEBUG0(2,"Received mlme ramdl_state request\n");
		result = prism2mgmt_ramdl_state(wlandev, msg);
		break;
	case DIDmsg_p2req_ramdl_write :
		WLAN_LOG_DEBUG0(2,"Received mlme ramdl_write request\n");
		result = prism2mgmt_ramdl_write(wlandev, msg);
		break;
	case DIDmsg_p2req_flashdl_state :
		WLAN_LOG_DEBUG0(2,"Received mlme flashdl_state request\n");
		result = prism2mgmt_flashdl_state(wlandev, msg);
		break;
	case DIDmsg_p2req_flashdl_write :
		WLAN_LOG_DEBUG0(2,"Received mlme flashdl_write request\n");
		result = prism2mgmt_flashdl_write(wlandev, msg);
		break;
	case DIDmsg_p2req_mm_state :
		WLAN_LOG_DEBUG0(2,"Received mlme mm_state request\n");
		result = prism2mgmt_mm_state(wlandev, msg);
		break;
	default:
		WLAN_LOG_WARNING1("Unknown mgmt request message 0x%08lx", msg->msgcode);
		break;
	}

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2sta_initmac
*
* Issue the commands to get the MAC controller into its intial
* state.
*
* Arguments:
*	wlandev		wlan device structure
*
* Returns: 
*	0	success
*	>0	f/w reported error
*	<0	driver reported error
*
* Side effects:
*
* Call context:
*	process thread
----------------------------------------------------------------*/
int prism2sta_initmac(wlandevice_t *wlandev)
{
	int 			result = 0;
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	UINT16			reg;
	int			i;
	UINT8			snum[12];
	DBFENTER;

	/* call initialize */
	result = hfa384x_cmd_initialize(hw);
	if (result != 0) {
		WLAN_LOG_ERROR0("Initialize command failed.\n");
		goto failed;
	}

	/* make sure interrupts are disabled and any layabout events cleared */
	outw( 0, HFA384x_INTEN(hw->iobase));
	outw( 0xffff, HFA384x_EVACK(hw->iobase));

	/* Read the PDA */
	result = hfa384x_drvr_readpda(hw, priv->pda, HFA384x_PDA_LEN_MAX);

	if ( result != 0) {
		WLAN_LOG_DEBUG0(2,"drvr_readpda() failed\n");
		goto failed;
	}

	/* Allocate tx and notify FIDs */
	/* First, tx */
	result = hfa384x_cmd_allocate(hw, PRISM2_TXBUF_MAX);
	if (result != 0) {
		WLAN_LOG_ERROR0("Allocate(tx) command failed.\n");
		goto failed;
	}
	i = 0;
	do {
		reg = hfa384x2host_16(inw(HFA384x_EVSTAT(hw->iobase)));
		udelay(10);
		i++;
	} while ( !HFA384x_INTEN_ISALLOC(reg) && i < 50); /* 50 is timeout */
	if ( i >= 50 ) {
		WLAN_LOG_ERROR0("Timed out waiting for evalloc(tx).\n");
		result = -ETIMEDOUT;
		goto failed;
	}
	priv->txfid = inw(HFA384x_ALLOCFID(hw->iobase));
	priv->txfid = hfa384x2host_16(priv->txfid);
	reg = host2hfa384x_16(HFA384x_EVACK_ALLOC_SET(1));
	outw( reg, HFA384x_EVACK(hw->iobase));
	WLAN_LOG_DEBUG1(1,"priv->txfid=0x%04x\n", priv->txfid);

	/* Now, the info frame fid */
	result = hfa384x_cmd_allocate(hw, PRISM2_INFOBUF_MAX);
	if (result != 0) {
		goto failed;
		WLAN_LOG_ERROR0("Allocate(tx) command failed.\n");
	}
	i = 0;
	do {
		reg = hfa384x2host_16(inw(HFA384x_EVSTAT(hw->iobase)));
		udelay(10);
		i++;
	} while ( !HFA384x_INTEN_ISALLOC(reg) && i < 50); /* 50 is timeout */
	if ( i >= 50 ) {
		WLAN_LOG_ERROR0("Timed out waiting for evalloc(info).\n");
		result = -ETIMEDOUT;
		goto failed;
	}
	priv->infofid = inw(HFA384x_ALLOCFID(hw->iobase));
	priv->infofid = hfa384x2host_16(priv->infofid);
	reg = host2hfa384x_16(HFA384x_EVACK_ALLOC_SET(1));
	outw( reg, HFA384x_EVACK(hw->iobase));
	WLAN_LOG_DEBUG1(1,"priv->infofid=0x%04x\n", priv->infofid);

	/* Collect version and compatibility info */
	/*  Some are critical, some are not */
	/* NIC identity */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_PRIIDENTITY,
			&priv->ident_nic, sizeof(hfa384x_compident_t));
	if ( result ) {
		WLAN_LOG_ERROR0("Failed to retrieve NICIDENTITY\n");
		goto failed;
	}
	WLAN_LOG_INFO4( "ident: nic h/w: id=0x%02x %d.%d.%d\n",
			priv->ident_nic.id, priv->ident_nic.major,
			priv->ident_nic.minor, priv->ident_nic.variant);

	/* Primary f/w identity */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_PRIIDENTITY,
			&priv->ident_pri_fw, sizeof(hfa384x_compident_t));
	if ( result ) {
		WLAN_LOG_ERROR0("Failed to retrieve PRIIDENTITY\n");
		goto failed;
	}
	WLAN_LOG_INFO4( "ident: pri f/w: id=0x%02x %d.%d.%d\n",
			priv->ident_pri_fw.id, priv->ident_pri_fw.major,
			priv->ident_pri_fw.minor, priv->ident_pri_fw.variant);

	/* Station (Secondary?) f/w identity */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_STAIDENTITY,
			&priv->ident_sta_fw, sizeof(hfa384x_compident_t));
	if ( result ) {
		WLAN_LOG_ERROR0("Failed to retrieve STAIDENTITY\n");
		goto failed;
	}
	if  ( priv->ident_sta_fw.id == 0x1f ) {
		WLAN_LOG_INFO4( 
			"ident: sta f/w: id=0x%02x %d.%d.%d\n",
			priv->ident_sta_fw.id, priv->ident_sta_fw.major,
			priv->ident_sta_fw.minor, priv->ident_sta_fw.variant);
	} else {
		WLAN_LOG_INFO4(
			"ident:  ap f/w: id=0x%02x %d.%d.%d\n",
			priv->ident_sta_fw.id, priv->ident_sta_fw.major,
			priv->ident_sta_fw.minor, priv->ident_sta_fw.variant);
	}

	/* Compatibility range, Modem supplier */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_MFISUPRANGE,
			&priv->cap_sup_mfi, sizeof(hfa384x_caplevel_t));
	if ( result ) {
		WLAN_LOG_ERROR0("Failed to retrieve MFISUPRANGE\n");
		goto failed;
	}
	WLAN_LOG_INFO5(
		"MFI:SUP:role=0x%02x:id=0x%02x:var=0x%02x:b/t=%d/%d\n",
		priv->cap_sup_mfi.role, priv->cap_sup_mfi.id,
		priv->cap_sup_mfi.variant, priv->cap_sup_mfi.bottom,
		priv->cap_sup_mfi.top);

	/* Compatibility range, Controller supplier */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_CFISUPRANGE,
			&priv->cap_sup_cfi, sizeof(hfa384x_caplevel_t));
	if ( result ) {
		WLAN_LOG_ERROR0("Failed to retrieve CFISUPRANGE\n");
		goto failed;
	}
	WLAN_LOG_INFO5( 
		"CFI:SUP:role=0x%02x:id=0x%02x:var=0x%02x:b/t=%d/%d\n",
		priv->cap_sup_cfi.role, priv->cap_sup_cfi.id,
		priv->cap_sup_cfi.variant, priv->cap_sup_cfi.bottom,
		priv->cap_sup_cfi.top);

	/* Compatibility range, Primary f/w supplier */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_PRISUPRANGE,
			&priv->cap_sup_pri, sizeof(hfa384x_caplevel_t));
	if ( result ) {
		WLAN_LOG_ERROR0("Failed to retrieve PRISUPRANGE\n");
		goto failed;
	}
	WLAN_LOG_INFO5(
		"PRI:SUP:role=0x%02x:id=0x%02x:var=0x%02x:b/t=%d/%d\n",
		priv->cap_sup_pri.role, priv->cap_sup_pri.id,
		priv->cap_sup_pri.variant, priv->cap_sup_pri.bottom,
		priv->cap_sup_pri.top);
	
	/* Compatibility range, Station f/w supplier */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_STASUPRANGE,
			&priv->cap_sup_sta, sizeof(hfa384x_caplevel_t));
	if ( result ) {
		WLAN_LOG_ERROR0("Failed to retrieve STASUPRANGE\n");
		goto failed;
	}
	if ( priv->cap_sup_sta.id == 0x04 ) {
		WLAN_LOG_INFO5(
		"STA:SUP:role=0x%02x:id=0x%02x:var=0x%02x:b/t=%d/%d\n",
		priv->cap_sup_sta.role, priv->cap_sup_sta.id,
		priv->cap_sup_sta.variant, priv->cap_sup_sta.bottom,
		priv->cap_sup_sta.top);
	} else {
		WLAN_LOG_INFO5(
		"AP:SUP:role=0x%02x:id=0x%02x:var=0x%02x:b/t=%d/%d\n",
		priv->cap_sup_sta.role, priv->cap_sup_sta.id,
		priv->cap_sup_sta.variant, priv->cap_sup_sta.bottom,
		priv->cap_sup_sta.top);
	}

	/* Compatibility range, primary f/w actor, CFI supplier */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_PRI_CFIACTRANGES,
			&priv->cap_act_pri_cfi, sizeof(hfa384x_caplevel_t));
	if ( result ) {
		WLAN_LOG_ERROR0("Failed to retrieve PRI_CFIACTRANGES\n");
		goto failed;
	}
	WLAN_LOG_INFO5(
		"PRI-CFI:ACT:role=0x%02x:id=0x%02x:var=0x%02x:b/t=%d/%d\n",
		priv->cap_act_pri_cfi.role, priv->cap_act_pri_cfi.id,
		priv->cap_act_pri_cfi.variant, priv->cap_act_pri_cfi.bottom,
		priv->cap_act_pri_cfi.top);
	
	/* Compatibility range, sta f/w actor, CFI supplier */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_STA_CFIACTRANGES,
			&priv->cap_act_sta_cfi, sizeof(hfa384x_caplevel_t));
	if ( result ) {
		WLAN_LOG_ERROR0("Failed to retrieve STA_CFIACTRANGES\n");
		goto failed;
	}
	WLAN_LOG_INFO5(
		"STA-CFI:ACT:role=0x%02x:id=0x%02x:var=0x%02x:b/t=%d/%d\n",
		priv->cap_act_sta_cfi.role, priv->cap_act_sta_cfi.id,
		priv->cap_act_sta_cfi.variant, priv->cap_act_sta_cfi.bottom,
		priv->cap_act_sta_cfi.top);

	/* Compatibility range, sta f/w actor, MFI supplier */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_STA_MFIACTRANGES,
			&priv->cap_act_sta_mfi, sizeof(hfa384x_caplevel_t));
	if ( result ) {
		WLAN_LOG_ERROR0("Failed to retrieve STA_MFIACTRANGES\n");
		goto failed;
	}
	WLAN_LOG_INFO5(
		"STA-MFI:ACT:role=0x%02x:id=0x%02x:var=0x%02x:b/t=%d/%d\n",
		priv->cap_act_sta_mfi.role, priv->cap_act_sta_mfi.id,
		priv->cap_act_sta_mfi.variant, priv->cap_act_sta_mfi.bottom,
		priv->cap_act_sta_mfi.top);

	/* Serial Number */
	/*TODO: print this out as text w/ hex for nonprint */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_NICSERIALNUMBER,
			snum, 12);
	if ( !result ) {
		int i;
		WLAN_LOG_INFO0("Prism2 card SN: ");
		for ( i=0; i < 12; i++) {
			printk("%02x ", snum[i]);
		}
		printk("\n");
	}

	/* Collect the MAC address */
	result = hfa384x_drvr_getconfig(hw, HFA384x_RID_CNFOWNMACADDR, 
		wlandev->netdev->dev_addr, WLAN_ADDR_LEN);
	if ( result != 0 ) {
		WLAN_LOG_ERROR0("Failed to retrieve mac address\n");
		goto failed;
	}

	/* Retrieve the maximum frame size */
	hfa384x_drvr_getconfig16(hw, HFA384x_RID_CNFMAXDATALEN, &reg);
	WLAN_LOG_DEBUG1(1,"Max frame data size=%d\n", reg);

	/* TODO: Set any internally managed config items */

	/* Set swsupport regs to magic # for card presence detection */
	outw( PRISM2STA_MAGIC, HFA384x_SWSUPPORT0(hw->iobase));

	goto done;
failed:
	WLAN_LOG_ERROR1("Failed, result=%d\n", result);
done:
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* prism2sta_inf_handover
*
* Handles the receipt of a Handover info frame. Should only be present
* in APs only.
*
* Arguments:
*	wlandev		wlan device structure
*	buf		ptr to buffer holding info frame
*
* Returns: 
*	nothing
*
* Side effects:
*
* Call context:
*	interrupt
----------------------------------------------------------------*/
void prism2sta_inf_handover(wlandevice_t *wlandev, void *buf)
{
	DBFENTER;
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_inf_tallies
*
* Handles the receipt of a CommTallies info frame. 
*
* Arguments:
*	wlandev		wlan device structure
*	buf		ptr to buffer holding info frame
*
* Returns: 
*	nothing
*
* Side effects:
*
* Call context:
*	interrupt
----------------------------------------------------------------*/
void prism2sta_inf_tallies(wlandevice_t *wlandev, void *buf)
{
	DBFENTER;
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_inf_scanresults
*
* Handles the receipt of a Scan Results info frame.
*
* Arguments:
*	wlandev		wlan device structure
*	buf		ptr to buffer holding info frame
*
* Returns: 
*	nothing
*
* Side effects:
*
* Call context:
*	interrupt
----------------------------------------------------------------*/
void prism2sta_inf_scanresults(wlandevice_t *wlandev, void *buf)
{
	DBFENTER;
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_inf_linkstatus
*
* Handles the receipt of a Link Status info frame.
*
* Arguments:
*	wlandev		wlan device structure
*	buf		ptr to buffer holding info frame
*
* Returns: 
*	nothing
*
* Side effects:
*
* Call context:
*	interrupt
----------------------------------------------------------------*/
void prism2sta_inf_linkstatus(wlandevice_t *wlandev, void *buf)
{
	prism2sta_priv_t	*priv = wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	hfa384x_LinkStatus_t	*inf = (hfa384x_LinkStatus_t*)buf;
	UINT16			portstatus;
	int			result;
	DBFENTER;
	/* Convert */
	inf->LinkStatus = hfa384x2host_16(inf->LinkStatus);
	/* Handle */
	switch (inf->LinkStatus) {
	case HFA384x_LINK_NOTCONNECTED:
		WLAN_LOG_DEBUG0(1,"LinkStatus=NOTCONNECTED (unhandled)\n");
		break;
	case HFA384x_LINK_CONNECTED:
		WLAN_LOG_DEBUG0(1,"LinkStatus=CONNECTED\n");
		/* Collect the BSSID, and set state to allow tx */
		result = hfa384x_drvr_getconfig(hw, 
				HFA384x_RID_CURRENTBSSID,
				wlandev->bssid, WLAN_BSSID_LEN);
		if ( result ) {
			WLAN_LOG_DEBUG2(1,
				"getconfig(0x%02x) failed, result = %d\n",
				HFA384x_RID_CURRENTBSSID, result);
			goto failed;
		}

		/* Collect the port status */
		result = hfa384x_drvr_getconfig16(hw, 
				HFA384x_RID_PORTSTATUS, &portstatus);
		if ( result ) {
			WLAN_LOG_DEBUG2(1,
				"getconfig(0x%02x) failed, result = %d\n",
				HFA384x_RID_PORTSTATUS, result);
			goto failed;
		}
		portstatus = hfa384x2host_16(portstatus);
		wlandev->macmode = 
			portstatus == HFA384x_PSTATUS_CONN_IBSS ?
			WLAN_MACMODE_IBSS_STA : WLAN_MACMODE_ESS_STA;
		break;
	case HFA384x_LINK_DISCONNECTED:
		WLAN_LOG_DEBUG0(1,"LinkStatus=DISCONNECTED (unhandled)\n");
		break;
	case HFA384x_LINK_AP_CHANGE:
		WLAN_LOG_DEBUG0(1,"LinkStatus=AP_CHANGE (unhandled)\n");
		break;
	case HFA384x_LINK_AP_OUTOFRANGE:
		WLAN_LOG_DEBUG0(1,"LinkStatus=AP_OUTOFRANGE (unhandled)\n");
		break;
	case HFA384x_LINK_AP_INRANGE:
		WLAN_LOG_DEBUG0(1,"LinkStatus=AP_INRANGE (unhandled)\n");
		break;
	case HFA384x_LINK_ASSOCFAIL:
		WLAN_LOG_DEBUG0(1,"LinkStatus=ASSOCFAIL (unhandled)\n");
		break;
	default:
		WLAN_LOG_WARNING1( 
			"unknown linkstatus=0x%02x\n", inf->LinkStatus);
		break;
	}

failed:
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_inf_assocstatus
*
* Handles the receipt of a Association Status info frame. Should 
* only be present in APs only.
*
* Arguments:
*	wlandev		wlan device structure
*	buf		ptr to buffer holding info frame
*
* Returns: 
*	nothing
*
* Side effects:
*
* Call context:
*	interrupt
----------------------------------------------------------------*/
void prism2sta_inf_assocstatus(wlandevice_t *wlandev, void *buf)
{
	DBFENTER;
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_interrupt
*
* Driver interrupt handler.
*
* Arguments:
*	irq		irq number
*	dev_id		pointer to the device
*	regs		registers
*
* Returns: 
*	nothing
*
* Side effects:
*	May result in a frame being passed up the stack or an info
*	frame being handled.  
*
* Call context:
*	Ummm, could it be interrupt?
----------------------------------------------------------------*/
void prism2sta_interrupt IRQ(int irq, void *dev_id, struct pt_regs *regs)
{
	int			reg;
	wlandevice_t		*wlandev = (wlandevice_t*)dev_id;
	prism2sta_priv_t	*priv = wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int			ev_read = 0;
	DBFENTER;

	/* Check swsupport reg magic # for card presence */
	reg = inw( HFA384x_SWSUPPORT0(hw->iobase));
	if ( reg != PRISM2STA_MAGIC) {
		WLAN_LOG_DEBUG1(2, "irq=%d, no magic.  Card removed?.\n", irq);
		return;
	}

	/* Set the BAP context */
	hw->bap = HFA384x_BAP_INT;

	/* read the EvStat register for interrupt enabled events */
	reg = hfa384x2host_16(inw(HFA384x_EVSTAT(hw->iobase)));
	ev_read++;

	do {

		/* Handle the events */
		if ( HFA384x_EVSTAT_ISINFDROP(reg) ){
			prism2sta_int_infdrop(wlandev);
			outw(host2hfa384x_16(HFA384x_EVACK_INFDROP_SET(1)), 
				HFA384x_EVACK(hw->iobase));
		}
	
		if ( HFA384x_EVSTAT_ISINFO(reg) ){
			prism2sta_int_info(wlandev);
			outw(host2hfa384x_16(HFA384x_EVACK_INFO_SET(1)),
				HFA384x_EVACK(hw->iobase));
		}
	
		if ( HFA384x_EVSTAT_ISTXEXC(reg) ){
			prism2sta_int_txexc(wlandev);
			outw(host2hfa384x_16(HFA384x_EVACK_TXEXC_SET(1)),
				HFA384x_EVACK(hw->iobase));
		}
	
		if ( HFA384x_EVSTAT_ISTX(reg) ){
			prism2sta_int_tx(wlandev);
			outw(host2hfa384x_16(HFA384x_EVACK_TX_SET(1)),
				HFA384x_EVACK(hw->iobase));
		}
	
		if ( HFA384x_EVSTAT_ISRX(reg) ){
			prism2sta_int_rx(wlandev);
			outw(host2hfa384x_16(HFA384x_EVACK_RX_SET(1)),
				HFA384x_EVACK(hw->iobase));
		}
		
		if ( HFA384x_EVSTAT_ISALLOC(reg) ){
			prism2sta_int_alloc(wlandev);
			outw(host2hfa384x_16(HFA384x_EVACK_ALLOC_SET(1)),
				HFA384x_EVACK(hw->iobase));
		}


		/* allow the evstat to be updated after the evack */
		udelay(20);

		/* Check swsupport reg magic # for card presence */
		reg = inw( HFA384x_SWSUPPORT0(hw->iobase));
		if ( reg != PRISM2STA_MAGIC) {
			WLAN_LOG_DEBUG1(2, "irq=%d, no magic.  Card removed?.\n", irq);
			return;
		}

		/* read the EvStat register for interrupt enabled events */
		reg = hfa384x2host_16(inw(HFA384x_EVSTAT(hw->iobase)));
		ev_read++;

	} while ( (HFA384x_EVSTAT_ISINFDROP(reg) || HFA384x_EVSTAT_ISINFO(reg) ||
		HFA384x_EVSTAT_ISTXEXC(reg) || HFA384x_EVSTAT_ISTX(reg) ||
		HFA384x_EVSTAT_ISRX(reg) || HFA384x_EVSTAT_ISALLOC(reg)) &&
		ev_read < prism2_irq_evread_max);

	/* Clear the BAP context */
	hw->bap = HFA384x_BAP_PROC;

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_int_infdrop
*
* Handles the InfDrop event.
*
* Arguments:
*	wlandev		wlan device structure
*
* Returns: 
*	nothing
*
* Side effects:
*
* Call context:
*	interrupt
----------------------------------------------------------------*/
void prism2sta_int_infdrop(wlandevice_t *wlandev)
{
#if 0
	prism2sta_priv_t	*priv = wlandev->priv;
	hfa384x_t		*hw = priv->hw;
#endif
	DBFENTER;
	WLAN_LOG_DEBUG0(3, "Info frame dropped due to card mem low.\n");
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_int_info
*
* Handles the Info event.
*
* Arguments:
*	wlandev		wlan device structure
*
* Returns: 
*	nothing
*
* Side effects:
*
* Call context:
*	interrupt
----------------------------------------------------------------*/
void prism2sta_int_info(wlandevice_t *wlandev)
{
	prism2sta_priv_t	*priv = wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	UINT16			reg;
	UINT8			buf[INFOFRM_LEN_MAX];
	hfa384x_InfFrame_t	inf;
	int			result;
	DBFENTER;
	/* Retrieve the FID */
	reg = hfa384x2host_16(inw(HFA384x_INFOFID(hw->iobase)));
	/* Retrieve the Inftype and length */
	result = hfa384x_copy_from_bap( hw, 
		hw->bap, reg, 0, &inf, sizeof(inf));
	if ( result ) {
		WLAN_LOG_DEBUG3(1, 
			"copy_from_bap(0x%04x, 0, %d) failed, result=0x%x\n", 
			reg, sizeof(inf), result);
		goto failed;
	}
	inf.framelen = hfa384x2host_16(inf.framelen);
	inf.infotype = hfa384x2host_16(inf.infotype);
	WLAN_LOG_DEBUG2(1, "rx infframe, len=%d type=0x%02x\n", inf.framelen, inf.infotype);
	/* Retrieve the rest */
	result = hfa384x_copy_from_bap( hw, 
		hw->bap, reg, 0, buf, 2 + (inf.framelen * 2));
	if ( result ) {
		WLAN_LOG_DEBUG3(1, 
			"copy_from_bap(0x%04x, 0, %d) failed, result=0x%x\n", 
			reg, sizeof(inf), result);
		goto failed;
	}
	/* Dispatch */
	switch ( inf.infotype ) {
		case HFA384x_IT_HANDOVERADDR:
			prism2sta_inf_handover(wlandev, buf);
			break;
		case HFA384x_IT_COMMTALLIES:
			prism2sta_inf_tallies(wlandev, buf);
			break;
		case HFA384x_IT_SCANRESULTS:
			prism2sta_inf_scanresults(wlandev, buf);
			break;
		case HFA384x_IT_LINKSTATUS:
			prism2sta_inf_linkstatus(wlandev, buf);
			break;
		case HFA384x_IT_ASSOCSTATUS:
			prism2sta_inf_assocstatus(wlandev, buf);
			break;
		default:
			WLAN_LOG_WARNING1(
				"Unknown info type=0x%02x", inf.infotype);
			break;
	}

failed:
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_int_txexc
*
* Handles the TxExc event.
*
* Arguments:
*	wlandev		wlan device structure
*
* Returns: 
*	nothing
*
* Side effects:
*
* Call context:
*	interrupt
----------------------------------------------------------------*/
void prism2sta_int_txexc(wlandevice_t *wlandev)
{
	prism2sta_priv_t	*priv = wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	UINT16			status;
	UINT16			fid;
	int			result = 0;
	DBFENTER;
	/* Collect the status and display */
	fid = hfa384x2host_16(inw(HFA384x_TXCOMPLFID(hw->iobase)));
	result = hfa384x_copy_from_bap(hw, hw->bap, fid, 0, &status, sizeof(status));
	if ( result ) {
		WLAN_LOG_DEBUG3(1, 
			"copy_from_bap(0x%04x, 0, %d) failed, result=0x%x\n", 
			fid, sizeof(status), result);
		goto failed;
	}
	WLAN_LOG_DEBUG1(3, "TxExc status=0x%x.\n", status);
failed:
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_int_tx
*
* Handles the Tx event.
*
* Arguments:
*	wlandev		wlan device structure
*
* Returns: 
*	nothing
*
* Side effects:
*
* Call context:
*	interrupt
----------------------------------------------------------------*/
void prism2sta_int_tx(wlandevice_t *wlandev)
{
	prism2sta_priv_t	*priv = wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	UINT16			fid;
	UINT16			status;
	DBFENTER;
	fid = hfa384x2host_16(inw(HFA384x_TXCOMPLFID(hw->iobase)));
	hfa384x_copy_from_bap(hw, hw->bap, fid, 0, &status, sizeof(status));
	status = hfa384x2host_16(status);
	WLAN_LOG_DEBUG1(4, "Tx Complete, status=0x%04x\n", status);
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_int_rx
*
* Handles the Rx event.
*
* Arguments:
*	wlandev		wlan device structure
*
* Returns: 
*	nothing
*
* Side effects:
*
* Call context:
*	interrupt
----------------------------------------------------------------*/
void prism2sta_int_rx(wlandevice_t *wlandev)
{
	prism2sta_priv_t	*priv = wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	UINT16			rxfid;
	hfa384x_rx_frame_t	rxdesc;
	wlan_pb_t		*pb;
	int			result;

	DBFENTER;

	/* Get the FID */
	rxfid = hfa384x2host_16(inw(HFA384x_RXFID(hw->iobase)));
	/* Get the descriptor (including headers) */
	result = hfa384x_copy_from_bap(hw, 
			hw->bap, 
			rxfid, 
			0, 
			&rxdesc, 
			sizeof(rxdesc));
	if ( result ) {
		WLAN_LOG_DEBUG4(1, 
			"copy_from_bap(0x%04x, %d, %d) failed, result=0x%x\n", 
			rxfid, 
			0, 
			sizeof(rxdesc),
			result);
		goto failed;
	}

	/* Byte order convert once up front. */
	rxdesc.status =	hfa384x2host_16(rxdesc.status);
	rxdesc.time =	hfa384x2host_32(rxdesc.time);
	rxdesc.data_len = hfa384x2host_16(rxdesc.data_len);

#if 0
printk(KERN_DEBUG"rxf(%d): ",rxlen);
for (i=0; i<pb->p80211frmlen; i++) {
	printk("%x ",pb->p80211buf[i]);
}
printk("\n");
#endif

	/* Now handle frame based on port# */
	switch( HFA384x_RXSTATUS_MACPORT_GET(rxdesc.status) )
	{
	case 0:
		/* Allocate the buffer, note CRC (aka FCS). pballoc */
		/* assumes there needs to be space for one */
		pb = p80211pb_alloc_p80211(NULL, 
			rxdesc.data_len + WLAN_HDR_A3_LEN + WLAN_CRC_LEN);
		if ( pb == NULL ) {
			WLAN_LOG_DEBUG0(1, "pballoc failed.\n");
			goto failed;
		}
	
		/* Copy the 802.11 hdr to the buffer */
		result = hfa384x_copy_from_bap(hw, hw->bap, rxfid, 
			HFA384x_RX_80211HDR_OFF, pb->p80211_hdr, WLAN_HDR_A3_LEN);
		if ( result ) {
			WLAN_LOG_DEBUG4(1, 
				"copy_from_bap(0x%04x, %d, %d) failed, result=0x%x\n", 
				rxfid, 
				HFA384x_RX_80211HDR_OFF, 
				WLAN_HDR_A3_LEN,
				result);
			p80211pb_free(pb);
			goto failed;
		}
	
		/* If exclude and we receive an unencrypted, drop it */
		if ( priv->exclude_unencrypt && 
			!WLAN_GET_FC_ISWEP(ieee2host16(pb->p80211_hdr->a3.fc))) {
			p80211pb_free(pb);
			goto failed;
		}

		/* Copy the payload data to the buffer */
		if ( rxdesc.data_len > 0 ) {
			result = hfa384x_copy_from_bap(hw, 
				hw->bap, rxfid, HFA384x_RX_DATA_OFF, 
				pb->p80211_payload, rxdesc.data_len);
			if ( result ) {
				WLAN_LOG_DEBUG4(1, 
					"copy_from_bap(0x%04x, %d, %d) failed, result=0x%x\n", 
					rxfid, 
					HFA384x_RX_DATA_OFF, 
					rxdesc.data_len,
					result);
				p80211pb_free(pb);
				goto failed;
			}
		}
		/* Set the length */
		pb->p80211frmlen = WLAN_HDR_A3_LEN + rxdesc.data_len + WLAN_CRC_LEN;
		/* Call p80211netdev_rx */
		p80211netdev_rx(wlandev, pb);
		break;
	case 7:
		if ( ! HFA384x_RXSTATUS_ISFCSERR(rxdesc.status) ) {
			/* Copy to wlansnif skb */
			prism2sta_int_rxmonitor( wlandev, rxfid, &rxdesc);
		} else {
			WLAN_LOG_DEBUG0(3,"Received monitor frame: FCSerr set\n");
		}
		break;
	default:
		WLAN_LOG_WARNING1("Received frame on unsupported port=%d\n",
			HFA384x_RXSTATUS_MACPORT_GET(rxdesc.status) );
		break;
	}

failed:
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_int_rxmonitor
*
* Helper function for int_rx.  Handles monitor frames.
* Note that this function allocates space for the FCS and sets it
* to 0xffffffff.  The hfa384x doesn't give us the FCS value but the
* higher layers expect it.  0xffffffff is used as a flag to indicate
* the FCS is bogus.
*
* Arguments:
*	wlandev		wlan device structure
*	rxfid		received FID
*	rxdesc		rx descriptor read from card in int_rx
*
* Returns: 
*	nothing
*
* Side effects:
*	Allocates an skb and passes it up via p80211ind_sniff()
* Call context:
*	interrupt
----------------------------------------------------------------*/
void prism2sta_int_rxmonitor( wlandevice_t *wlandev, UINT16 rxfid, hfa384x_rx_frame_t *rxdesc)
{
	prism2sta_priv_t		*priv = wlandev->priv;
	hfa384x_t			*hw = priv->hw;
	UINT				hdrlen = 0;
	UINT				datalen = 0;
	UINT				skblen = 0;
	p80211msg_lnxind_wlansniffrm_t	*msg;
	UINT8				*datap;
	UINT16				fc;
	struct sk_buff			*skb;

	DBFENTER;
	/* Don't forget the status, time, and data_len fields are in host order */
	/* Figure out how big the frame is */
	fc = ieee2host16(rxdesc->frame_control);
	switch ( WLAN_GET_FC_FTYPE(fc) )
	{
	case WLAN_FTYPE_DATA:
		if ( WLAN_GET_FC_TODS(fc) && WLAN_GET_FC_FROMDS(fc) ) {
			hdrlen = WLAN_HDR_A4_LEN;
		} else {
			hdrlen = WLAN_HDR_A3_LEN;
		}
		datalen = rxdesc->data_len;
		break;
	case WLAN_FTYPE_MGMT:
		hdrlen = WLAN_HDR_A3_LEN;
		datalen = rxdesc->data_len;
		break;
	case WLAN_FTYPE_CTL:
		switch ( WLAN_GET_FC_FSTYPE(fc) )
		{
		case WLAN_FSTYPE_PSPOLL:
		case WLAN_FSTYPE_RTS:
		case WLAN_FSTYPE_CFEND:
		case WLAN_FSTYPE_CFENDCFACK:
			hdrlen = 16;
			break;
		case WLAN_FSTYPE_CTS:
		case WLAN_FSTYPE_ACK:
			hdrlen = 10;
			break;
		}
		datalen = 0;
		break;
	default:
		WLAN_LOG_DEBUG1(1, "unknown frm: fc=0x%04x\n", fc);
		return;
	}

	/* Allocate an ind message+framesize skb */
	skblen = sizeof(p80211msg_lnxind_wlansniffrm_t) + 
			hdrlen + datalen + WLAN_CRC_LEN;
	/* sanity check the length */
	if ( skblen > 
		(sizeof(p80211msg_lnxind_wlansniffrm_t) + 
		WLAN_HDR_A4_LEN + WLAN_DATA_MAXLEN + WLAN_CRC_LEN) ) {
		WLAN_LOG_DEBUG1(1, "overlen frm: len=%d\n", 
			skblen - sizeof(p80211msg_lnxind_wlansniffrm_t));
	}
			
	if ( (skb = alloc_skb(skblen, GFP_ATOMIC)) == NULL ) {
		WLAN_LOG_DEBUG1(2,
			"alloc_skb failed trying to allocate %d bytes\n", skblen);
		return;
	}
	skb_put(skb, skblen);
	datap = skb->data + sizeof(p80211msg_lnxind_wlansniffrm_t);
	msg = (p80211msg_lnxind_wlansniffrm_t*)skb->data;

	/* Initialize the message members */
	msg->msgcode = DIDmsg_lnxind_wlansniffrm;
	msg->msglen = sizeof(p80211msg_lnxind_wlansniffrm_t);
	strcpy(msg->devname, wlandev->name);

	msg->hosttime.did = DIDmsg_lnxind_wlansniffrm_hosttime;
	msg->hosttime.status = 0;
	msg->hosttime.len = 4;
	msg->hosttime.data = jiffies;

	msg->mactime.did = DIDmsg_lnxind_wlansniffrm_mactime;
	msg->mactime.status = 0;
	msg->mactime.len = 4;
	msg->mactime.data = rxdesc->time;

	msg->channel.did = DIDmsg_lnxind_wlansniffrm_channel;
	msg->channel.status = P80211ENUM_msgitem_status_no_value;
	msg->channel.len = 4;
	msg->channel.data = 0;

	msg->rssi.did = DIDmsg_lnxind_wlansniffrm_rssi;
	msg->rssi.status = P80211ENUM_msgitem_status_no_value;
	msg->rssi.len = 4;
	msg->rssi.data = 0;

	msg->sq.did = DIDmsg_lnxind_wlansniffrm_sq;
	msg->sq.status = P80211ENUM_msgitem_status_no_value;
	msg->sq.len = 4;
	msg->sq.data = 0;

	msg->signal.did = DIDmsg_lnxind_wlansniffrm_signal;
	msg->signal.status = 0;
	msg->signal.len = 4;
	msg->signal.data = rxdesc->signal;

	msg->noise.did = DIDmsg_lnxind_wlansniffrm_noise;
	msg->noise.status = 0;
	msg->noise.len = 4;
	msg->noise.data = rxdesc->silence;

	msg->rate.did = DIDmsg_lnxind_wlansniffrm_rate;
	msg->rate.status = 0;
	msg->rate.len = 4;
	msg->rate.data = rxdesc->rate / 5; /* set to 802.11 units */

	msg->istx.did = DIDmsg_lnxind_wlansniffrm_istx;
	msg->istx.status = 0;
	msg->istx.len = 4;
	msg->istx.data = P80211ENUM_truth_false;

	msg->frmlen.did = DIDmsg_lnxind_wlansniffrm_frmlen;
	msg->frmlen.status = 0;
	msg->frmlen.len = 4;
	msg->frmlen.data = hdrlen + datalen + WLAN_CRC_LEN;

	/* Copy the 802.11 header to the skb (ctl frames may be less than a full header) */
	memcpy( datap, &(rxdesc->frame_control), hdrlen);

	/* If any, copy the data from the card to the skb */
	if ( datalen > 0 )
	{
		hfa384x_copy_from_bap(hw, 
			hw->bap, rxfid, HFA384x_RX_DATA_OFF, 
			datap + hdrlen, datalen);
	}

	/* Set the CRC */
	memset( ((UINT8 *)(skb->data)) + skb->len - WLAN_CRC_LEN, 0xff, WLAN_CRC_LEN);

	/* Pass it up */
	p80211ind_sniff(wlandev, skb);

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_int_alloc
*
* Handles the Alloc event.
*
* Arguments:
*	wlandev		wlan device structure
*
* Returns: 
*	nothing
*
* Side effects:
*
* Call context:
*	interrupt
----------------------------------------------------------------*/
void prism2sta_int_alloc(wlandevice_t *wlandev)
{
#if 0
	prism2sta_priv_t	*priv = wlandev->priv;
	hfa384x_t		*hw = priv->hw;
#endif
	DBFENTER;

	/* Handle the reclaimed FID (clear tbusy) */
	wlandev->netdev->tbusy = 0;
	mark_bh(NET_BH);

	DBFEXIT;
	return;

}


/*----------------------------------------------------------------
* prism2sta_attach
*
* Half of the attach/detach pair.  Creates and registers a device
* instance with Card Services.  In this case, it also creates the
* wlandev structure and device private structure.  These are 
* linked to the device instance via its priv member.
*
* Arguments:
*	none
*
* Returns: 
*	A valid ptr to dev_link_t on success, NULL otherwise
*
* Side effects:
*	
*
* Call context:
*	process thread (insmod/init_module/register_pccard_driver)
----------------------------------------------------------------*/
dev_link_t *prism2sta_attach(void)
{
	client_reg_t		client_reg;
	int			ret;
	dev_link_t		*link;
	wlandevice_t		*wlandev;
	prism2sta_priv_t	*priv;


	DBFENTER;

	/* Create the PC card device object. */
	link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
	if ( link == NULL ) {
		return NULL;
	}
	memset(link, 0, sizeof(struct dev_link_t));
	link->release.function = &prism2sta_release;
	link->release.data = (u_long)link;
	link->conf.IntType = INT_MEMORY_AND_IO;

	/* Create the network device object. */
	wlandev = kmalloc(sizeof(wlandevice_t), GFP_KERNEL);
	if ( wlandev == NULL ) {
		kfree_s(link, sizeof(dev_link_t));
		return NULL;
	}
	memset(wlandev, 0, sizeof(wlandevice_t));

	/* Make up a device private data structure. */
	wlandev->priv = kmalloc(sizeof(prism2sta_priv_t), GFP_KERNEL);
	if ( wlandev->priv == NULL ) {
		kfree_s(wlandev, sizeof(wlandevice_t));
		kfree_s(link, sizeof(dev_link_t));
		return NULL;
	}
	memset(wlandev->priv, 0, sizeof(prism2sta_priv_t));

	/* Make up a hw data structure. */
	priv = (prism2sta_priv_t*)wlandev->priv;
	priv->hw = kmalloc(sizeof(hfa384x_t), GFP_KERNEL);
	if ( priv->hw == NULL ) {
		kfree_s(wlandev->priv, sizeof(prism2sta_priv_t));
		kfree_s(wlandev, sizeof(wlandevice_t));
		kfree_s(link, sizeof(dev_link_t));
		return NULL;
	}
	memset(priv->hw, 0, sizeof(hfa384x_t));

	/* Set our entries in the wlandev */
	wlandev->open = &prism2sta_open;
	wlandev->close = &prism2sta_close;
	wlandev->reset = &prism2sta_reset;
	wlandev->txframe = &prism2sta_txframe;
	wlandev->mlmerequest = &prism2sta_mlmerequest;

	/* Set up the remaining entries in the wlan common way */
	wlandev->name = ((prism2sta_priv_t*)wlandev->priv)->node.dev_name;
	wlan_setup(wlandev);

	link->priv = wlandev;
#if CS_RELEASE_CODE > 0x2911
	link->irq.Instance = wlandev;
#endif

	/* Link in to the list of devices managed by this driver */
	link->next = dev_list;
	dev_list = link;	

	/* Register with Card Services */
	client_reg.dev_info = &dev_info;
	client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
	client_reg.EventMask =
		CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
		CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
		CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
	client_reg.event_handler = &prism2sta_event;
	client_reg.Version = 0x0210;
	client_reg.event_callback_args.client_data = link;

	ret = CardServices(RegisterClient, &link->handle, &client_reg);
	if (ret != 0) {
		cs_error(link->handle, RegisterClient, ret);
		prism2sta_detach(link);
		return NULL;
	}

	return link;
}


/*----------------------------------------------------------------
* prism2sta_detach
*
* Remove one of the device instances managed by this driver.
*   Search the list for the given instance, 
*   check our flags for a waiting timer'd release call
*   call release
*   Deregister the instance with Card Services
*   (netdevice) unregister the network device.
*   unlink the instance from the list
*   free the link, priv, and priv->priv memory
* Note: the dev_list variable is a driver scoped static used to
*	maintain a list of device instances managed by this
*	driver.
*
* Arguments:
*	link	ptr to the instance to detach
*
* Returns: 
*	nothing
*
* Side effects:
*	the link structure is gone, the netdevice is gone
*
* Call context:
*	Might be interrupt, don't block.
----------------------------------------------------------------*/
void prism2sta_detach(dev_link_t *link)
{
	dev_link_t		**linkp;
	UINT32			flags;
	wlandevice_t		*wlandev;
	prism2sta_priv_t	*priv;

	DBFENTER;

	/* Locate device structure */
	for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next) {
		if (*linkp == link) break;
	}

	if (*linkp != NULL) {
		/* Get rid of any timer'd release call */	
		save_flags(flags);
		cli();
		if (link->state & DEV_RELEASE_PENDING) {
			del_timer(&link->release);
			link->state &= ~DEV_RELEASE_PENDING;
		}
		restore_flags(flags);
		
		/* If link says we're still config'd, call release */
		if (link->state & DEV_CONFIG) {
			prism2sta_release((u_long)link);
			if (link->state & DEV_STALE_CONFIG) {
				link->state |= DEV_STALE_LINK;
				return;
			}
		}
		
		/* Tell Card Services we're not around any more */
		if (link->handle) {
			CardServices(DeregisterClient, link->handle);
		}	

		/* Unlink device structure, free bits */
		*linkp = link->next;
		if ( link->priv != NULL ) {
			wlandev = (wlandevice_t*)link->priv;
			if (link->dev != NULL) {
				unregister_wlandev(wlandev);
			}
			wlan_unsetup(wlandev);
			if (wlandev->priv) {
				priv = (prism2sta_priv_t*)wlandev->priv;
				if ( priv->hw )
					kfree_s(priv->hw, sizeof(hfa384x_t));
				kfree_s(wlandev->priv, sizeof(prism2sta_priv_t));
			}
			kfree_s(link->priv, sizeof(wlandevice_t));
		}
		kfree_s(link, sizeof(struct dev_link_t));
	}

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
* prism2sta_config
*
* Half of the config/release pair.  Usually called in response to
* a card insertion event.  At this point, we _know_ there's some
* physical device present.  That means we can start poking around
* at the CIS and at any device specific config data we want.
*
* Note the gotos and the macros.  I recoded this once without
* them, and it got incredibly ugly.  It's actually simpler with
* them.
*
* Arguments:
*	link	the dev_link_t structure created in attach that 
*		represents this device instance.
*
* Returns: 
*	nothing
*
* Side effects:
*	Resources (irq, io, mem) are allocated
*	The pcmcia dev_link->node->name is set
*	(For netcards) The device structure is finished and,
*	  most importantly, registered.  This means that there
*	  is now a _named_ device that can be configured from
*	  userland.
*
* Call context:
*	May be called from a timer.  Don't block!
----------------------------------------------------------------*/
#define CS_CHECK(fn, args...) \
while ((last_ret=CardServices(last_fn=(fn), args))!=0) goto cs_failed;

#if defined(WLAN_INCLUDE_DEBUG)
#define CFG_CHECK(fn, args...) \
if ((last_ret=CardServices(last_fn=(fn), args))!=0) {  \
	WLAN_LOG_DEBUG0(1,"CFG_CHECK failed\n"); \
	cs_error(link->handle, last_fn, last_ret); \
	goto next_entry; \
}
#else
#define CFG_CHECK(fn, args...) if (CardServices(fn, args)!=0) goto next_entry;
#endif 

void prism2sta_config(dev_link_t *link)
{
	client_handle_t		handle;
	wlandevice_t		*wlandev;
	prism2sta_priv_t	*priv;
	int			last_fn;
	int			last_ret;
	tuple_t			tuple;
	cisparse_t		parse;
	config_info_t		socket_config;
	UINT8			buf[64];
	int			i;

	DBFENTER;

	handle = link->handle;
	wlandev = (wlandevice_t*)link->priv;

	/* Collect the config register info */
	tuple.DesiredTuple = CISTPL_CONFIG;
	tuple.Attributes = 0;
	tuple.TupleData = buf;
	tuple.TupleDataMax = sizeof(buf);
	tuple.TupleOffset = 0;
	CS_CHECK(GetFirstTuple, handle, &tuple);
	CS_CHECK(GetTupleData, handle, &tuple);
	CS_CHECK(ParseTuple, handle, &tuple, &parse);
	link->conf.ConfigBase = parse.config.base;
	link->conf.Present = parse.config.rmask[0];
	
	/* Configure card */
	link->state |= DEV_CONFIG;

	/* Acquire the current socket config (need Vcc setting) */
	CS_CHECK(GetConfigurationInfo, handle, &socket_config);

	/* Loop through the config table entries until we find one that works */
	/* Assumes a complete and valid CIS */
	tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
	CS_CHECK(GetFirstTuple, handle, &tuple);
	while (1) {
		cistpl_cftable_entry_t dflt = { 0 };
		cistpl_cftable_entry_t *cfg = &(parse.cftable_entry);
		CFG_CHECK(GetTupleData, handle, &tuple);
		CFG_CHECK(ParseTuple, handle, &tuple, &parse);

		if (cfg->index == 0) goto next_entry;
		link->conf.ConfigIndex = cfg->index;

		/* Use power settings for Vcc and Vpp if present */
		/* Note that the CIS values need to be rescaled */
		if (cfg->vcc.present & (1<<CISTPL_POWER_VNOM)) {
			WLAN_LOG_DEBUG0(1, "vcc set from VNOM\n");
			link->conf.Vcc = cfg->vcc.param[CISTPL_POWER_VNOM]/10000;
		} else if (dflt.vcc.present & (1<<CISTPL_POWER_VNOM)) {
			WLAN_LOG_DEBUG0(1, "vcc set from VNOM\n");
			link->conf.Vcc = dflt.vcc.param[CISTPL_POWER_VNOM]/10000;
		} else if ((cfg->vcc.present & (1<<CISTPL_POWER_VMAX)) &&
			   (cfg->vcc.present & (1<<CISTPL_POWER_VMIN)) ) {
			WLAN_LOG_DEBUG0(1, "vcc set from avg(VMIN,VMAX)\n");
			link->conf.Vcc = 
				((cfg->vcc.param[CISTPL_POWER_VMIN] +
				cfg->vcc.param[CISTPL_POWER_VMIN]) / 2) / 10000;
		} else if ((dflt.vcc.present & (1<<CISTPL_POWER_VMAX)) &&
			   (dflt.vcc.present & (1<<CISTPL_POWER_VMIN)) ) {
			WLAN_LOG_DEBUG0(1, "vcc set from avg(VMIN,VMAX\n");
			link->conf.Vcc = 
				((dflt.vcc.param[CISTPL_POWER_VMIN] +
				dflt.vcc.param[CISTPL_POWER_VMIN]) / 2) / 10000;
		}

		if ( link->conf.Vcc >= 45 && link->conf.Vcc <= 55 ) {
			link->conf.Vcc = 50;
		} else {
			link->conf.Vcc = 33; /* default to 3.3v (safer)*/
		}

		if ( link->conf.Vcc != socket_config.Vcc ) {
			goto next_entry;
		}
			
		WLAN_LOG_DEBUG1(1, "link->conf.Vcc=%d\n", link->conf.Vcc);

		/* Do we need to allocate an interrupt? */
		if (cfg->irq.IRQInfo1 || dflt.irq.IRQInfo1)
			link->conf.Attributes |= CONF_ENABLE_IRQ;

		/* IO window settings */
		link->io.NumPorts1 = link->io.NumPorts2 = 0;
		if ((cfg->io.nwin > 0) || (dflt.io.nwin > 0)) {
			cistpl_io_t *io = (cfg->io.nwin) ? &cfg->io : &dflt.io;
			link->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;
			if (!(io->flags & CISTPL_IO_8BIT))
				link->io.Attributes1 = IO_DATA_PATH_WIDTH_16;
			if (!(io->flags & CISTPL_IO_16BIT))
				link->io.Attributes1 = IO_DATA_PATH_WIDTH_8;
			link->io.BasePort1 = io->win[0].base;
			if  ( link->io.BasePort1 != 0 ) {
				WLAN_LOG_WARNING1(
				"Brain damaged CIS: hard coded iobase="
				"0x%x, try letting pcmcia_cs decide...\n",
				link->io.BasePort1 );
				link->io.BasePort1 = 0;
			}
			link->io.NumPorts1 = io->win[0].len;
			if (io->nwin > 1) {
				link->io.Attributes2 = link->io.Attributes1;
				link->io.BasePort2 = io->win[1].base;
				link->io.NumPorts2 = io->win[1].len;
			}
		}

		/* This reserves IO space but doesn't actually enable it */
		CFG_CHECK(RequestIO, link->handle, &link->io);

		/* If we got this far, we're cool! */
		break;

next_entry:
		if (cfg->flags & CISTPL_CFTABLE_DEFAULT)
			dflt = *cfg;
		CS_CHECK(GetNextTuple, handle, &tuple);
	}

	/* Allocate an interrupt line.  Note that this does not assign a */
	/* handler to the interrupt, unless the 'Handler' member of the */
	/* irq structure is initialized. */
	if (link->conf.Attributes & CONF_ENABLE_IRQ)
	{
		link->irq.Attributes = IRQ_TYPE_EXCLUSIVE | IRQ_HANDLE_PRESENT;
		link->irq.IRQInfo1 = IRQ_INFO2_VALID | IRQ_LEVEL_ID;
		if (irq_list[0] == -1)
			link->irq.IRQInfo2 = irq_mask;
		else
			for (i=0; i<4; i++)
				link->irq.IRQInfo2 |= 1 << irq_list[i];
		link->irq.Handler = prism2sta_interrupt;
		link->irq.Instance = wlandev;
		CS_CHECK(RequestIRQ, link->handle, &link->irq);
	}

	/* This actually configures the PCMCIA socket -- setting up */
	/* the I/O windows and the interrupt mapping, and putting the */
	/* card and host interface into "Memory and IO" mode. */
	CS_CHECK(RequestConfiguration, link->handle, &link->conf);

	/* Fill the netdevice with this info */
	wlandev->netdev->irq = link->irq.AssignedIRQ;
	wlandev->netdev->base_addr = link->io.BasePort1;

	/* Report what we've done */
	printk(KERN_INFO "%s: index 0x%02x: Vcc %d.%d", 
		dev_info, link->conf.ConfigIndex, 
		link->conf.Vcc/10, link->conf.Vcc%10);
	if (link->conf.Vpp1)
		printk(", Vpp %d.%d", link->conf.Vpp1/10, link->conf.Vpp1%10);
	if (link->conf.Attributes & CONF_ENABLE_IRQ)
		printk(", irq %d", link->irq.AssignedIRQ);
	if (link->io.NumPorts1)
		printk(", io 0x%04x-0x%04x", link->io.BasePort1, link->io.BasePort1+link->io.NumPorts1-1);
	if (link->io.NumPorts2)
		printk(" & 0x%04x-0x%04x", link->io.BasePort2, link->io.BasePort2+link->io.NumPorts2-1);
	printk("\n");

	link->state &= ~DEV_CONFIG_PENDING;

	/* Register the network device and get assigned a name */
	if (register_wlandev(wlandev) != 0) {
		WLAN_LOG_NOTICE0("prism2sta_cs: register_wlandev() failed.\n");
		goto failed;
	}

	priv = (prism2sta_priv_t*)wlandev->priv;/* collect the device priv ptr */
	link->dev = &priv->node;		/* now pcmcia knows the device name */

	/* Any device custom config/query stuff should be done here */
	/* For a netdevice, we should at least grab the mac address */

	return;
cs_failed:
	cs_error(link->handle, last_fn, last_ret);

failed:
	prism2sta_release((UINT32)link);
	return;
}




/*----------------------------------------------------------------
* prism2sta_release
*
* Half of the config/release pair.  Usually called in response to 
* a card ejection event.  Checks to make sure no higher layers
* are still (or think they are) using the card via the link->open
* field.  
*
* NOTE: Don't forget to increment the link->open variable in the 
*  device_open method, and decrement it in the device_close 
*  method.
*
* Arguments:
*	arg	a generic 32 bit variable...we assume it's a 
*               ptr to a dev_link.
*
* Returns: 
*	nothing
*
* Side effects:
*	All resources should be released after this function
*	executes and finds the device !open.
*
* Call context:
*	Possibly in a timer context.  Don't do anything that'll
*	block.
----------------------------------------------------------------*/
void prism2sta_release(UINT32 arg)
{
        dev_link_t	*link = (dev_link_t *)arg;

	DBFENTER;

        if (link->open) {
                WLAN_LOG_DEBUG1(1, "prism2sta_cs: release postponed, '%s' still open\n",
                          link->dev->dev_name);
                link->state |= DEV_STALE_CONFIG;
                return;
        }

        CardServices(ReleaseConfiguration, link->handle);
        CardServices(ReleaseIO, link->handle, &link->io);
        CardServices(ReleaseIRQ, link->handle, &link->irq);

        link->state &= ~(DEV_CONFIG | DEV_RELEASE_PENDING);

	DBFEXIT;
}



/*----------------------------------------------------------------
* prism2sta_event
*
* Handler for card services events.
*
* Arguments:
*	event		The event code
*	priority	hi/low - REMOVAL is the only hi
*	args		ptr to card services struct containing info about
*			pcmcia status
*
* Returns: 
*	Zero on success, non-zero otherwise
*
* Side effects:
*	
*
* Call context:
*	Both interrupt and process thread, depends on the event.
----------------------------------------------------------------*/
static int prism2sta_event (event_t event, int priority, event_callback_args_t *args)
{
	int		result = 0;
	dev_link_t	*link = (dev_link_t *) args->client_data;
	wlandevice_t	*wlandev = (wlandevice_t*)link->priv;
	DBFENTER;

	switch (event)
	{
		case CS_EVENT_CARD_INSERTION:
			link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
			prism2sta_config(link);
			if ((link->state & DEV_CONFIG) == 0 ) {
				wlandev->netdev->irq = 0;
				WLAN_LOG_WARNING1("%s: Initialization failed!\n", dev_info);
			} else {
				hfa384x_inithw(
					((prism2sta_priv_t*)wlandev->priv)->hw, 
					wlandev->netdev->irq,
					wlandev->netdev->base_addr);
				if ( prism2sta_initmac(wlandev) != 0 ) {
					WLAN_LOG_ERROR0("MAC Initialization failed.\n");
				}
				result = 1;
			}
			break;

		case CS_EVENT_CARD_REMOVAL:
			link->state &= ~DEV_PRESENT;
			if (link->state & DEV_CONFIG)
			{
				wlandev->netdev->tbusy = 1;
				wlandev->netdev->start = 0;
				link->release.expires = RUN_AT(HZ/20);
				add_timer(&link->release);
			}
			break;
/* Not supported yet 
		case CS_EVENT_PM_SUSPEND:
			link->state |= DEV_SUSPEND;
		case CS_EVENT_RESET_PHYSICAL:
			if (link->state & DEV_CONFIG)
			{
				if (link->open)
				{
					dev->tbusy = 1;
					dev->start = 0;
				}
				CardServices(ReleaseConfiguration, link->handle);
			}
			break;

		case CS_EVENT_PM_RESUME:
			link->state &= ~DEV_SUSPEND;
		case CS_EVENT_CARD_RESET:
			if (link->state & DEV_CONFIG)
			{
				CardServices(RequestConfiguration, link->handle, &link->conf);
				if (link->open)
				{
					wvlan_hw_shutdown(dev);
					wvlan_hw_config(dev);
					dev->tbusy = 0;
					dev->start = 1;
				}
			}
			break;
*/
	}

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
* init_module
*
* Module initialization routine, called once at module load time.
* This one simulates some of the pcmcia calls.
*
* Arguments:
*	none
*
* Returns: 
*	0	- success 
*	~0	- failure, module is unloaded.
*
* Side effects:
*	TODO: define
*
* Call context:
*	process thread (insmod or modprobe)
----------------------------------------------------------------*/
int init_module(void)
{
	int		result = 0;
	servinfo_t	serv;

        DBFENTER;

        WLAN_LOG_NOTICE1("%s Loaded\n", version);
        WLAN_LOG_NOTICE1("dev_info is: %s\n", dev_info);

	CardServices(GetCardServicesInfo, &serv);
	if ( serv.Revision != CS_RELEASE_CODE )
	{
		printk(KERN_WARNING"%s: CardServices release does not match!\n", dev_info);
	}

	/* This call will result in a call to prism2sta_attach */
	/*   and eventually prism2sta_detach */
	register_pcmcia_driver( &dev_info, &prism2sta_attach, &prism2sta_detach);

        DBFEXIT;
        return result;
}


/*----------------------------------------------------------------
* cleanup_module
*
* Called at module unload time.  This is our last chance to
* clean up after ourselves.
*
* Arguments:
*	none
*
* Returns: 
*	nothing
*
* Side effects:
*	TODO: define
*
* Call context:
*	process thread
*
----------------------------------------------------------------*/
void cleanup_module(void)
{

        DBFENTER;

	unregister_pcmcia_driver( &dev_info);

/*
TODO: fix this to clean up leftover pcmcia and netdev instances
	dev_link_t *link = dev_list;
	while (link) {
		wlandevice_t *wlandev = (wlandevice_t *)link->priv;
		if (link->dev != NULL) {
		unregister_wlandev(wlandev);
		}
		wlan_unsetup(wlandev);
		if (wlandev->priv) {
			kfree_s(wlandev->priv, sizeof(prism2sta_priv_t));
		}
		kfree_s(wlandev, sizeof(wlandevice_t));
		kfree_s(link, sizeof(struct dev_link_t));
	}
*/
        printk(KERN_NOTICE "%s Unloaded\n", version);

        DBFEXIT;
        return;
}
