/* src/prism2/driver/prism2mib.c
*
* Management request for mibset/mibget
*
* Copyright (C) 1999 AbsoluteValue Systems, Inc.  All Rights Reserved.
* --------------------------------------------------------------------
*
* linux-wlan
*
*   The contents of this file are subject to the Mozilla Public
*   License Version 1.1 (the "License"); you may not use this file
*   except in compliance with the License. You may obtain a copy of
*   the License at http://www.mozilla.org/MPL/
*
*   Software distributed under the License is distributed on an "AS
*   IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*   implied. See the License for the specific language governing
*   rights and limitations under the License.
*
*   Alternatively, the contents of this file may be used under the
*   terms of the GNU Public License version 2 (the "GPL"), in which
*   case the provisions of the GPL are applicable instead of the
*   above.  If you wish to allow the use of your version of this file
*   only under the terms of the GPL and not to allow others to use
*   your version of this file under the MPL, indicate your decision
*   by deleting the provisions above and replace them with the notice
*   and other provisions required by the GPL.  If you do not delete
*   the provisions above, a recipient may use your version of this
*   file under either the MPL or the GPL.
*
* --------------------------------------------------------------------
*
* Inquiries regarding the linux-wlan Open Source project can be
* made directly to:
*
* AbsoluteValue Systems Inc.
* info@linux-wlan.com
* http://www.linux-wlan.com
*
* --------------------------------------------------------------------
*
* Portions of the development of this software were funded by 
* Intersil Corporation as part of PRISM(R) chipset product development.
*
* --------------------------------------------------------------------
*
* The functions in this file handle the mibset/mibget management
* functions.
*
* --------------------------------------------------------------------
*/

/*================================================================*/
/* System Includes */

#define __NO_VERSION__

#include <linux/config.h>
#define WLAN_DBVAR	prism2_debug
#include <wlan/wlan_compat.h>

#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/malloc.h>
#include <linux/netdevice.h>
#include <asm/io.h>
#include <asm/delay.h>
#include <asm/byteorder.h>

#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>
#include <pcmcia/cisreg.h>
#include <pcmcia/driver_ops.h>

/*================================================================*/
/* Project Includes */

#include <wlan/version.h>
#include <wlan/p80211types.h>
#include <wlan/p80211hdr.h>
#include <wlan/p80211mgmt.h>
#include <wlan/p80211conv.h>
#include <wlan/p80211msg.h>
#include <wlan/p80211netdev.h>
#include <wlan/p80211metadef.h>
#include <wlan/p80211metastruct.h>
#include <prism2/hfa384x.h>
#include <prism2/prism2mgmt.h>

/*================================================================*/
/* Local Constants */


/*================================================================*/
/* Local Macros */

/*================================================================*/
/* Local Types */


/*================================================================*/
/* Local Static Definitions */


/*================================================================*/
/* Local Function Declarations */


/*================================================================*/
/* Function Definitions */


/*----------------------------------------------------------------
* prism2mgmt_mibset_mibget
*
* Set the value of a mib item.
*
* Arguments:
*	wlandev		wlan device structure
*	msgp		ptr to msg buffer
*
* Returns: 
*	0	success and done
*	<0	success, but we're waiting for something to finish.
*	>0	an error occurred while handling the message.
* Side effects:
*
* Call context:
*	process thread  (usually)
*	interrupt
----------------------------------------------------------------*/
#define MIB_TMP_MAXLEN		100
int prism2mgmt_mibset_mibget(wlandevice_t *wlandev, void *msgp)
{
	prism2sta_priv_t	*priv = (prism2sta_priv_t*)wlandev->priv;
	hfa384x_t		*hw = priv->hw;
	int 			result = 0;
	int 			isget;

	/* Get the pieces out of the msg */
	p80211msg_dot11req_mibset_t	*msg = msgp;
	p80211itemd_t		*mibitem = (p80211itemd_t*)msg->mibattribute.data;

	/* These two are pointers to the mib data, use whichever */
	/*  one is appropriate for the given mib item */
	p80211pstrd_t		*pstr = (p80211pstrd_t*)mibitem->data;	
	UINT32			*uint32 = (UINT32*)mibitem->data;

	/* The following two are a temporary buffer to use in conversions */
	UINT8			bytebuf[MIB_TMP_MAXLEN];
	UINT16			*wordbuf = (UINT16*)bytebuf;
	hfa384x_bytestr_t	*p2bytestr = (hfa384x_bytestr_t*)bytebuf;

	DBFENTER;

	memset( bytebuf, 0, MIB_TMP_MAXLEN );

	msg->resultcode.status = P80211ENUM_msgitem_status_data_ok;
	msg->resultcode.data = P80211ENUM_resultcode_success;

	/* Determine if message code is mibget; otherwise, it's mibset */
	if ( msg->msgcode == DIDmsg_dot11req_mibget ) {
		isget = 1;
	} else {
		isget = 0;
	}
	
/* Ok, here's the list,(it just didn't make sense to indent the case statement */
switch( mibitem->did ) 
{
/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11StationID:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, HFA384x_RID_CNFOWNMACADDR,
			bytebuf, HFA384x_RID_CNFOWNMACADDR_LEN);
		prism2mgmt_bytearea2pstr(bytebuf, pstr,
			HFA384x_RID_CNFOWNMACADDR_LEN);
	} else {
		prism2mgmt_pstr2bytearea(bytebuf, pstr);
		result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFOWNMACADDR,
			bytebuf, HFA384x_RID_CNFOWNMACADDR_LEN);
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11MediumOccupancyLimit:
#if defined(WLAN_STA)
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	result = hfa384x_drvr_getconfig( hw, HFA384x_RID_CNFAPPCFINFO, bytebuf, HFA384x_RID_CNFAPPCFINFO_LEN);
	if ( result ) {
		WLAN_LOG_DEBUG0(1,"Unable to read CNFAPPCFINFO\n");
		break;
	}
	if ( isget ) {
		*uint32 = hfa384x2host_16(((hfa384x_PCFInfo_data_t *)bytebuf)->MediumOccupancyLimit);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		((hfa384x_PCFInfo_data_t*)bytebuf)->MediumOccupancyLimit = *wordbuf;
		result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFAPPCFINFO,
						bytebuf, HFA384x_RID_CNFAPPCFINFO_LEN);
		if ( result ) {
			WLAN_LOG_DEBUG0(1,"Unable to write CNFAPPCFINFO\n");
		}
	}
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11CFPollable:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, HFA384x_RID_CFPOLLABLE,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* this is read only established by the MAC */
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
#elif defined(WLAN_AP)
	/* MIB not supported on APs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11CFPPeriod:
#if defined(WLAN_STA)
	/* MIB not supported on STAs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	result = hfa384x_drvr_getconfig( hw, HFA384x_RID_CNFAPPCFINFO, bytebuf, HFA384x_RID_CNFAPPCFINFO_LEN);
	if ( result ) {
		WLAN_LOG_DEBUG0(1,"Unable to read CNFAPPCFINFO\n");
		break;
	}
	if ( isget ) {
		*uint32 = hfa384x2host_32(((hfa384x_PCFInfo_data_t *)bytebuf)->CFPPeriod);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		((hfa384x_PCFInfo_data_t*)bytebuf)->CFPPeriod = *wordbuf;
		result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFAPPCFINFO,
						bytebuf, HFA384x_RID_CNFAPPCFINFO_LEN);
		if ( result ) {
			WLAN_LOG_DEBUG0(1,"Unable to write CNFAPPCFINFO\n");
		}
	}
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11CFPMaxDuration:
#if defined(WLAN_STA)
	/* MIB not supported on STAs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	result = hfa384x_drvr_getconfig( hw, HFA384x_RID_CNFAPPCFINFO, bytebuf, HFA384x_RID_CNFAPPCFINFO_LEN);
	if ( result ) {
		WLAN_LOG_DEBUG0(1,"Unable to read CNFAPPCFINFO\n");
		break;
	}
	if ( isget ) {
		*uint32 = hfa384x2host_32(((hfa384x_PCFInfo_data_t *)bytebuf)->CFPMaxDuration);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		((hfa384x_PCFInfo_data_t*)bytebuf)->CFPMaxDuration = *wordbuf;
		result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFAPPCFINFO,
						bytebuf, HFA384x_RID_CNFAPPCFINFO_LEN);
		if ( result ) {
			WLAN_LOG_DEBUG0(1,"Unable to write CNFAPPCFINFO\n");
		}
	}
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11AuthenticationResponseTimeOut:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, HFA384x_RID_CNFAUTHRSPTIMEOUT,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, HFA384x_RID_CNFAUTHRSPTIMEOUT,
			wordbuf);
	}
	break;
/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11PrivacyOptionImplemented:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, HFA384x_RID_PRIVACYOPTIMP,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* this is read only established by the MAC */
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;
/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11PowerManagementMode:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, HFA384x_RID_CNFPMENABLED,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
		/* Prism2 defines this item to be boolean; 802.11 defines
		it to be 1=active, 2=powersave */
		if ( (*uint32) == 0 ) { 
			*uint32 = 1; /* if false, set to active */
		} else {
			*uint32 = 2; /* if true, set to powersave */
		}
	} else {
		/* Prism2 defines this item to be boolean; 802.11 defines
		it to be 1=active, 2=powersave */

		UINT32	mode;

		if ( (*uint32) == 1 ) { 
			mode = (UINT32)0; /* if active, set to false */
		} else {
			mode = (UINT32)1; /* if powersave, set to true */
		}
		prism2mgmt_p80211int2prism2int(wordbuf, &mode);
		result = hfa384x_drvr_setconfig16( hw, HFA384x_RID_CNFPMENABLED,
			wordbuf);
	}
#elif defined(WLAN_AP)
	/* MIB not supported on APs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11DesiredSSID:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, HFA384x_RID_CNFDESIREDSSID,
			bytebuf, HFA384x_RID_CNFDESIREDSSID_LEN);
		prism2mgmt_bytestr2pstr(p2bytestr, pstr);
	} else {
		prism2mgmt_pstr2bytestr(p2bytestr, pstr);
		result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFDESIREDSSID,
			bytebuf, HFA384x_RID_CNFDESIREDSSID_LEN);
	}
#elif defined(WLAN_AP)
	/* MIB not supported on APs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11DesiredBSSType:
#if defined(WLAN_STA)
	if ( isget ) {
		*uint32 = priv->dot11_desired_bss_type;
	} else {
		priv->dot11_desired_bss_type = *uint32;
	}
#elif defined(WLAN_AP)
	/* MIB not supported on APs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11OperationalRateSet:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_TXRATECNTL, wordbuf);
		prism2mgmt_get_oprateset(wordbuf, pstr);
	} else {
		prism2mgmt_set_oprateset(wordbuf, pstr);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_TXRATECNTL, wordbuf);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFSUPPRATES, wordbuf);
	}
#elif defined(WLAN_AP)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
			HFA384x_RID_TXRATECNTL0, wordbuf);
		prism2mgmt_get_oprateset(wordbuf, pstr);
	} else {
		prism2mgmt_set_oprateset(wordbuf, pstr);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_TXRATECNTL0, wordbuf);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFSUPPRATES, wordbuf);
	}
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11BeaconPeriod:
#if defined(WLAN_STA)
	/* MIB not supported on STAs */
	/* TODO: this will need to be supported when Ad-Hoc is supported */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CNFAPBCNINT, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFAPBCNINT, wordbuf);
	}
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11DTIMPeriod:
#if defined(WLAN_STA)
	/* MIB not supported on STAs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
			HFA384x_RID_CNFOWNDTIMPER, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
			HFA384x_RID_CNFOWNDTIMPER, wordbuf);
	}
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11AssociationResponseTimeOut:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_PROTOCOLRSPTIME, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* this is read only established by the MAC */
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;


/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11DisassociateReason:
	/* Supported in both STA and AP */
	if ( isget ) {
		*uint32 = priv->dot11_disassoc_reason;
	} else {
		msg->resultcode.data = 
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;


/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11DisassociateStation:
	/* Supported in both STA and AP */
	if ( isget ) {
		memcpy(pstr->data, priv->dot11_disassoc_station, WLAN_ADDR_LEN);
		pstr->len = WLAN_ADDR_LEN;
	} else {
		msg->resultcode.data = 
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11DeauthenticateReason:
	/* Supported in both STA and AP */
	if ( isget ) {
		*uint32 = priv->dot11_deauth_reason;
	} else {
		msg->resultcode.data = 
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;


/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11DeauthenticateStation:
	/* Supported in both STA and AP */
	if ( isget ) {
		memcpy(pstr->data, priv->dot11_deauth_station, WLAN_ADDR_LEN);
		pstr->len = WLAN_ADDR_LEN;
	} else {
		msg->resultcode.data = 
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11AuthenticateFailStatus:
	/* Supported in both STA and AP */
	if ( isget ) {
		*uint32 = priv->dot11_auth_fail_status;
	} else {
		msg->resultcode.data = 
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;


/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11StationConfigTable_dot11AuthenticateFailStation:
	/* Supported in both STA and AP */
	if ( isget ) {
		memcpy(pstr->data, priv->dot11_auth_fail_station, WLAN_ADDR_LEN);
		pstr->len = WLAN_ADDR_LEN;
	} else {
		msg->resultcode.data = 
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11mac_dot11OperationTable_dot11MACAddress:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, 
				HFA384x_RID_CNFOWNMACADDR, bytebuf, 
				HFA384x_RID_CNFOWNMACADDR_LEN);
		prism2mgmt_bytearea2pstr(bytebuf, 
			pstr, HFA384x_RID_CNFOWNMACADDR_LEN);
	} else {
		prism2mgmt_pstr2bytearea(bytebuf, pstr);
		result = hfa384x_drvr_setconfig( hw, 
				HFA384x_RID_CNFOWNMACADDR, bytebuf, 
				HFA384x_RID_CNFOWNMACADDR_LEN);
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11mac_dot11OperationTable_dot11RTSThreshold:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_RTSTHRESH, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_RTSTHRESH, wordbuf);
	}
#elif defined(WLAN_AP)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_RTSTHRESH0, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_RTSTHRESH0, wordbuf);
	}
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11mac_dot11OperationTable_dot11ShortRetryLimit:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_SHORTRETRYLIMIT, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* this is read only established by the MAC */
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11mac_dot11OperationTable_dot11LongRetryLimit:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, HFA384x_RID_LONGRETRYLIMIT,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* this is read only established by the MAC */
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11mac_dot11OperationTable_dot11FragmentationThreshold:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, HFA384x_RID_FRAGTHRESH,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* only even numbers supported */
		if ( !(*uint32 % 2) ) {
			prism2mgmt_p80211int2prism2int(wordbuf, uint32);
			result = hfa384x_drvr_setconfig16( hw,
				HFA384x_RID_FRAGTHRESH, wordbuf);
		} else {
			WLAN_LOG_WARNING0("Attempt to set odd number "
				"FragmentationThreshold\n"); 
			msg->resultcode.data =
				P80211ENUM_resultcode_not_supported;
		}
	}
#elif defined(WLAN_AP)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_FRAGTHRESH0, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* only even numbers supported */
		if ( !(*uint32 % 2) ) {
			prism2mgmt_p80211int2prism2int(wordbuf, uint32);
			result = hfa384x_drvr_setconfig16( hw,
				HFA384x_RID_FRAGTHRESH0, wordbuf);
		} else {
			WLAN_LOG_WARNING0("Attempt to set odd number "
				"FragmentationThreshold\n"); 
			msg->resultcode.data =
				P80211ENUM_resultcode_not_supported;
		}
	}
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11mac_dot11OperationTable_dot11MaxTransmitMSDULifetime:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_MAXTXLIFETIME, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* this is read only established by the MAC */
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11mac_dot11OperationTable_dot11MaxReceiveLifetime:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_MAXRXLIFETIME, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* this is read only established by the MAC */
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address1:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address2:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address3:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address4:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address5:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address6:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address7:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address8:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address9:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address10:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address11:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address12:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address13:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address14:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address15:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address16:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address17:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address18:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address19:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address20:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address21:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address22:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address23:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address24:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address25:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address26:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address27:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address28:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address29:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address30:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address31:
case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address32:
	/* Supported in both STA and AP */
	/* TODO: fix this.  f/w doesn't support mcast filters */
	/* TODO: Add support to p80211 for the linux set_multicast...() */
	/*         interface function.                                  */
	if ( isget ) {
		prism2mgmt_get_grpaddr(mibitem->did,
			pstr, priv );
	} else {
		if ( (result = prism2mgmt_set_grpaddr(mibitem->did, bytebuf,
			pstr, priv)) == 0 ) {
			if ( (priv->dot11_grpcnt) <= MAX_PRISM2_GRP_ADDR ) {
				memcpy( bytebuf,
					priv->dot11_grp_addr[0],
					((priv->dot11_grpcnt) * WLAN_ADDR_LEN));
				result = hfa384x_drvr_setconfig( hw,
					HFA384x_RID_GROUPADDR,
					bytebuf,
					((priv->dot11_grpcnt) * WLAN_ADDR_LEN));

				/* turn off promiscuous mode if count is
				equal to MAX; we may have been at a higher
				count in promiscuous mode and need to turn
				it off. */
				if ( (priv->dot11_grpcnt) == MAX_PRISM2_GRP_ADDR ) {
					*wordbuf = 0;
					result = hfa384x_drvr_setconfig16( hw,
						HFA384x_RID_PROMISCMODE,
						wordbuf);
				}
			} else {
				/* clear group addresses in card
				and set to promiscuous mode */
				memset( bytebuf, 0, sizeof(bytebuf));
				result = hfa384x_drvr_setconfig( hw,
					HFA384x_RID_GROUPADDR,
					bytebuf, 0);
				if ( result == 0 )
				{
					*wordbuf = 1;
					result = hfa384x_drvr_setconfig16( hw,
						HFA384x_RID_PROMISCMODE,
						wordbuf);
				}
			}

		} else {
			msg->resultcode.data = P80211ENUM_resultcode_not_supported;
		}
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11PhyOperationTable_dot11PHYType:
	/* Supported in both STA and AP */
	/* This is an enumerated type. The Prism2 enumerated values
	are a 1-to-1 match against the wlan enumerated values for this item */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
			HFA384x_RID_PHYTYPE, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* this is read only established by the MAC */
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11PhyOperationTable_dot11CurrentRegDomain:
	/* TODO: wlan has this set up as an enumerated type.  However,
	it can have multiple values, i.e. a list of domains.  Need
	to add support.  For now, it's not supported */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11PhyOperationTable_dot11TempType:
	/* Supported in both STA and AP */
	/* This is an enumerated type. The Prism2 enumerated values
	are a 1-to-1 match against the wlan enumerated values for this item */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, HFA384x_RID_TEMPTYPE,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* this is read only established by the MAC */
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11PhyOperationTable_dot11ChannelAgilityPresent:
	/* Supported in both STA and AP */
	/*TODO: we need to locate RID and place it here */
	/* This is an enumerated type. The Prism2 enumerated values
	are a 1-to-1 match against the wlan enumerated values for this item */
	if ( isget ) {
		/*
		result = hfa384x_drvr_getconfig16( hw, PLACE_RID_HERE,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
		*/
	} else {
	/* Need to determine if this is read only or if we do something
	here; it depends on the RID */
	}
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11PhyOperationTable_dot11ChannelAgilityEnabled:
	/*TODO: we need to locate RID and place it here */
	/* This is an enumerated type. The Prism2 enumerated values
	are a 1-to-1 match against the wlan enumerated values for this item */
	if ( isget ) {
		/*
		result = hfa384x_drvr_getconfig16( hw, PLACE_RID_HERE,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
		*/
	} else {
	/* Need to determine if this is read only or if we do something
	here; it depends on the RID */
	}
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11PhyDSSSTable_dot11CurrentChannel:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CURRENTCHANNEL, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* this is read only established by the MAC */
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
#elif defined(WLAN_AP)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CNFOWNCHANNEL, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* this is read only established by the MAC */
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11PhyDSSSTable_dot11CurrentCCAMode:
	/* This is an enumerated type. The Prism2 enumerated values
	are a 1-to-1 match against the wlan enumerated values for this item */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, HFA384x_RID_CCAMODE,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		/* this is read only established by the MAC */
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11PhyDSSSTable_dot11ShortPreambleOptionImplemented:
	/* TODO: Find out the firmware version number(s) for identifying 
	whether the firmware is capable of short preamble. TRUE or FALSE
	will be returned based on the version of the firmware.  For now,
	set resultcode.data to "not supported" */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11PhyDSSSTable_dot11PBCCOptionImplemented:
	/* TODO: Need to locate RID and place it here */
	if ( isget ) {
		/*
		result = hfa384x_drvr_getconfig16( hw, PLACE_RID_HERE,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
		*/
	} else {
	/* Need to determine if this is read only or if we do something
	here; it depends on the RID */
	}
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11RegDomainsSupportedTable_dot11RegDomainsSupportIndex:
	/* TODO: add support for this data item. For now, set
	resultcode.data to "not supported" */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11RegDomainsSupportedTable_dot11RegDomainsSupportValue:
	/* TODO: need to investigate why wlan has this as enumerated and
	Prism2 has this as btye str. For now, not supported */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11SupportedDataRatesTxTable_dot11SupportedDataRatesTxIndex:
	/* TODO: add support for this data item. For now, set
	resultcode.data to "not supported" */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11SupportedDataRatesTxTable_dot11SupportedDataRatesTxValue:
	/* TODO: add support for this data item. For now, set
	resultcode.data to "not supported" */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11SupportedDataRatesRxTable_dot11SupportedDataRatesRxIndex:
	/* TODO: add support for this data item. For now, set
	resultcode.data to "not supported" */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11phy_dot11SupportedDataRatesRxTable_dot11SupportedDataRatesRxValue:
	/* TODO: add support for this data item. For now, set
	resultcode.data to "not supported" */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11WEPDefaultKeysTable_dot11WEPDefaultKey0:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, HFA384x_RID_CNFWEPDEFAULTKEY0,
			bytebuf, HFA384x_RID_CNFWEPDEFAULTKEY_LEN);
		prism2mgmt_bytearea2pstr(bytebuf, pstr,
			HFA384x_RID_CNFWEPDEFAULTKEY_LEN - 1);
	} else {
		prism2mgmt_pstr2bytearea(bytebuf, pstr);
		bytebuf[HFA384x_RID_CNFWEPDEFAULTKEY_LEN - 1] = 0x00;
		result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFWEPDEFAULTKEY0,
			bytebuf, HFA384x_RID_CNFWEPDEFAULTKEY_LEN);
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11WEPDefaultKeysTable_dot11WEPDefaultKey1:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, HFA384x_RID_CNFWEPDEFAULTKEY1,
			bytebuf, HFA384x_RID_CNFWEPDEFAULTKEY_LEN);
		prism2mgmt_bytearea2pstr(bytebuf, pstr,
			HFA384x_RID_CNFWEPDEFAULTKEY_LEN - 1);
	} else {
		prism2mgmt_pstr2bytearea(bytebuf, pstr);
		bytebuf[HFA384x_RID_CNFWEPDEFAULTKEY_LEN - 1] = 0x00;
		result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFWEPDEFAULTKEY1,
			bytebuf, HFA384x_RID_CNFWEPDEFAULTKEY_LEN);
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11WEPDefaultKeysTable_dot11WEPDefaultKey2:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, HFA384x_RID_CNFWEPDEFAULTKEY2,
			bytebuf, HFA384x_RID_CNFWEPDEFAULTKEY_LEN);
		prism2mgmt_bytearea2pstr(bytebuf, pstr,
			HFA384x_RID_CNFWEPDEFAULTKEY_LEN - 1);
	} else {
		prism2mgmt_pstr2bytearea(bytebuf, pstr);
		bytebuf[HFA384x_RID_CNFWEPDEFAULTKEY_LEN - 1] = 0x00;
		result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFWEPDEFAULTKEY2,
			bytebuf, HFA384x_RID_CNFWEPDEFAULTKEY_LEN);
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11WEPDefaultKeysTable_dot11WEPDefaultKey3:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, HFA384x_RID_CNFWEPDEFAULTKEY3,
			bytebuf, HFA384x_RID_CNFWEPDEFAULTKEY_LEN);
		prism2mgmt_bytearea2pstr(bytebuf, pstr,
			HFA384x_RID_CNFWEPDEFAULTKEY_LEN - 1);
	} else {
		prism2mgmt_pstr2bytearea(bytebuf, pstr);
		bytebuf[HFA384x_RID_CNFWEPDEFAULTKEY_LEN - 1] = 0x00;
		result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFWEPDEFAULTKEY3,
			bytebuf, HFA384x_RID_CNFWEPDEFAULTKEY_LEN);
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11PrivacyTable_dot11PrivacyInvoked:
	/* Supported in both STA and AP */
	{
		UINT32	flags;

		result = hfa384x_drvr_getconfig16( hw, HFA384x_RID_CNFWEPFLAGS,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, &flags);

		if ( isget ) {
			if ( (flags & HFA384x_WEPFLAGS_PRIVINVOKED) ) { 
				*uint32 = P80211ENUM_truth_true;
			} else {
				*uint32 = P80211ENUM_truth_false;
			}
		} else {
			if ( (*uint32) == P80211ENUM_truth_true ) { 
				flags |= HFA384x_WEPFLAGS_PRIVINVOKED;
				priv->priv_invoked = 1;
			} else {
				flags&=~(HFA384x_WEPFLAGS_PRIVINVOKED);
				priv->priv_invoked = 0;
			}
			prism2mgmt_p80211int2prism2int(wordbuf, &flags);
			result = hfa384x_drvr_setconfig16( hw, HFA384x_RID_CNFWEPFLAGS,
				wordbuf);
		}
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11PrivacyTable_dot11WEPDefaultKeyID:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, HFA384x_RID_CNFWEPDEFAULTKEYID,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, HFA384x_RID_CNFWEPDEFAULTKEYID,
			wordbuf);
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_dot11smt_dot11PrivacyTable_dot11ExcludeUnencrypted:
{
	/* Supported in both STA and AP */
	UINT32	flags;

	result = hfa384x_drvr_getconfig16( hw, 
			HFA384x_RID_CNFWEPFLAGS, wordbuf);
	prism2mgmt_prism2int2p80211int(wordbuf, &flags);

	if ( isget ) {
		if ( (flags & HFA384x_WEPFLAGS_EXCLUDE) ) { 
			*uint32 = P80211ENUM_truth_true;
		} else {
			*uint32 = P80211ENUM_truth_false;
		}
	} else {
		if ( (*uint32) == 1 ) { 
			priv->exclude_unencrypt = 1;
			flags |= HFA384x_WEPFLAGS_EXCLUDE;
		} else {
			priv->exclude_unencrypt = 0;
			flags &= ~(HFA384x_WEPFLAGS_EXCLUDE);
		}
		prism2mgmt_p80211int2prism2int(wordbuf, &flags);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFWEPFLAGS, wordbuf);
	}
	break;
}


/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2EarlyBeacon:
{
#if defined(WLAN_STA)
	/* MIB not supported on STAs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	hfa384x_PCFInfo_data_t	*data = (hfa384x_PCFInfo_data_t*)bytebuf;
	UINT16			word;

	result = hfa384x_drvr_getconfig( hw, 
			HFA384x_RID_CNFAPPCFINFO, 
			bytebuf, HFA384x_RID_CNFAPPCFINFO_LEN);
	if ( result ) {
		WLAN_LOG_DEBUG0(1,"Unable to read CNFAPPCFINFO\n");
		break;
	}
	if ( isget ) {
		*uint32 = hfa384x2host_16(data->CFPFlags) & BIT7 ?
			P80211ENUM_truth_true: P80211ENUM_truth_false;

	} else {
		word = hfa384x2host_16(data->CFPFlags);
		word = (*uint32 == P80211ENUM_truth_true) ?
			word | BIT7 : 
			word & ~BIT7;
		data->CFPFlags = host2hfa384x_16(word);
		result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFAPPCFINFO,
						bytebuf, HFA384x_RID_CNFAPPCFINFO_LEN);
		if ( result ) {
			WLAN_LOG_DEBUG0(1,"Unable to write CNFAPPCFINFO\n");
		}
	}
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;
}

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2ReserveDuration:
{
#if defined(WLAN_STA)
	/* MIB not supported on STAs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	hfa384x_PCFInfo_data_t	*data = (hfa384x_PCFInfo_data_t*)bytebuf;
	UINT16			word;
	result = hfa384x_drvr_getconfig( hw, HFA384x_RID_CNFAPPCFINFO, 
			bytebuf, HFA384x_RID_CNFAPPCFINFO_LEN);
	if ( result ) {
		WLAN_LOG_DEBUG0(1,"Unable to read CNFAPPCFINFO\n");
		break;
	}
	if ( isget ) {
		*uint32 = hfa384x2host_16(data->CFPFlags) & BIT1 ?
			P80211ENUM_truth_true: P80211ENUM_truth_false;
	} else {
		word = hfa384x2host_16(data->CFPFlags);
		word = (*uint32 == P80211ENUM_truth_true) ?
			word | BIT1 : 
			word & ~BIT1;
		data->CFPFlags = host2hfa384x_16(word);
		result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFAPPCFINFO,
						bytebuf, HFA384x_RID_CNFAPPCFINFO_LEN);
		if ( result ) {
			WLAN_LOG_DEBUG0(1,"Unable to write CNFAPPCFINFO\n");
		}
	}
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;
}

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2MMLife:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
			HFA384x_RID_CNFMMLIFE,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFMMLIFE,
				wordbuf);
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2AltRetryCount:
	/* Supported in both STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
			HFA384x_RID_CNFALTRETRYCNT,
			wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFALTRETRYCNT,
				wordbuf);
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2MMTx:
	/* Supported in both STA and AP */
	if ( isget ) {
		*uint32 = priv->qos_enable ?
			P80211ENUM_truth_true:
			P80211ENUM_truth_false;
		result = 0;
	} else {
		priv->qos_enable = (*uint32 == P80211ENUM_truth_true);
		result = 0;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2TxControlDepth:
#if defined(WLAN_STA)
	/* MIB not supported on STAs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CNFTXCONTROL, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFTXCONTROL, wordbuf);
	}
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2RoamingMode:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CNFROAMINGMODE, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFROAMINGMODE, wordbuf);
	}
#elif defined(WLAN_AP)
	/* MIB not supported on APs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;


/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2CurrBSSID:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, 
				HFA384x_RID_CURRENTBSSID, bytebuf, 
				HFA384x_RID_CNFOWNMACADDR_LEN);
		prism2mgmt_bytearea2pstr(bytebuf, 
			pstr, HFA384x_RID_CNFOWNMACADDR_LEN);
	} else {
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
#elif defined(WLAN_AP)
	/* MIB not supported on APs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#else
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2SystemScale:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CNFSYSSCALE, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFSYSSCALE, wordbuf);
	}
#elif defined(WLAN_AP)
	/* MIB not supported on APs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2MaxDataLen:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CNFMAXDATALEN, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFMAXDATALEN, wordbuf);
	}
#elif defined(WLAN_AP)
	/* MIB not supported on APs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
/* NOTE: cnfPMEnable should be implemented as part of the dot11req_powermgmt */

case DIDmib_p2_p2Table_p2EnhancePowerSaveMode:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CNFPMEPS, wordbuf);
		*uint32 = *wordbuf ?
			P80211ENUM_truth_true: P80211ENUM_truth_false;
	} else {
		*wordbuf = *uint32 == P80211ENUM_truth_true ? 1 : 0;
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFPMEPS, wordbuf);
		if ( result ) {
			WLAN_LOG_DEBUG0(1,"Unable to write CNFPMEPS");
		}
	}
#elif defined(WLAN_AP)
	/* MIB not supported on APs */
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
/* Tells the MAC to 'rx mcast when PM is enabled'.  I'm assuming
 * this is identical to the 802.11 powermgt.request argument 
 * "ReceiveDTIMS".  Therefore, this one is implemented via the 
 * powermgmt.request primitive.
 */
#if 0
case DIDmib_p2_p2Table_p2MulticastRx:
#if defined(WLAN_STA)
#elif defined(WLAN_AP)
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;
#endif

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2MaxSleepDuration:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CNFMAXSLEEPDUR, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFMAXSLEEPDUR, wordbuf);
	}
#elif defined(WLAN_AP)
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2PMHoldoverDuration:
#if defined(WLAN_STA)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CNFPMHOLDDUR, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFPMHOLDDUR, wordbuf);
	}
#elif defined(WLAN_AP)
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2OwnName:
	/* Defined for STA and AP */
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, HFA384x_RID_CNFOWNNAME,
			bytebuf, HFA384x_RID_CNFDESIREDSSID_LEN);
		prism2mgmt_bytestr2pstr(p2bytestr, pstr);
	} else {
		prism2mgmt_pstr2bytestr(p2bytestr, pstr);
		result = hfa384x_drvr_setconfig( hw, HFA384x_RID_CNFOWNNAME,
			bytebuf, HFA384x_RID_CNFDESIREDSSID_LEN);
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2McastPMBuffer:
#if defined(WLAN_STA)
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
#elif defined(WLAN_AP)
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CNFMCASTPMBUFF, wordbuf);
		*uint32 = *wordbuf ?
			P80211ENUM_truth_true: P80211ENUM_truth_false;
	} else {
		*wordbuf = *uint32 == P80211ENUM_truth_true ? 1 : 0;
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_CNFMCASTPMBUFF, wordbuf);
		if ( result ) {
			WLAN_LOG_DEBUG0(1,"Unable to write CNFMCASTPMBUFF");
		}
	}
#elif
#error "WLAN_STA or WLAN_AP not defined!"
#endif
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2TickTime:
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_ITICKTIME, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		prism2mgmt_p80211int2prism2int(wordbuf, uint32);
		result = hfa384x_drvr_setconfig16( hw, 
				HFA384x_RID_ITICKTIME, wordbuf);
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2MaxLoadTime:
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_MAXLOADTIME, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2DLBufferPage:
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, 
				HFA384x_RID_DOWNLOADBUFFER, bytebuf, 
				HFA384x_RID_DOWNLOADBUFFER_LEN);
		*uint32 = hfa384x2host_16(((hfa384x_downloadbuffer_t *)bytebuf)->page);
	} else {
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2DLBufferOffset:
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, 
				HFA384x_RID_DOWNLOADBUFFER, bytebuf, 
				HFA384x_RID_DOWNLOADBUFFER_LEN);
		*uint32 = hfa384x2host_16(((hfa384x_downloadbuffer_t *)bytebuf)->offset);
	} else {
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2DLBufferLength:
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, 
				HFA384x_RID_DOWNLOADBUFFER, bytebuf, 
				HFA384x_RID_DOWNLOADBUFFER_LEN);
		*uint32 = hfa384x2host_16(((hfa384x_downloadbuffer_t *)bytebuf)->len);
	} else {
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2PortStatus:
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_PORTSTATUS, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;
	
/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2CommsQualityCQ:
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, 
				HFA384x_RID_COMMSQUALITY, bytebuf, 
				HFA384x_RID_COMMSQUALITY_LEN);
		*uint32 = hfa384x2host_16(((hfa384x_commsquality_t *)bytebuf)->CQ_currBSS);
	} else {
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2CommsQualityASL:
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, 
				HFA384x_RID_COMMSQUALITY, bytebuf, 
				HFA384x_RID_COMMSQUALITY_LEN);
		*uint32 = hfa384x2host_16(((hfa384x_commsquality_t *)bytebuf)->ASL_currBSS);
	} else {
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2CommsQualityANL:
	if ( isget ) {
		result = hfa384x_drvr_getconfig( hw, 
				HFA384x_RID_COMMSQUALITY, bytebuf, 
				HFA384x_RID_COMMSQUALITY_LEN);
		*uint32 = hfa384x2host_16(((hfa384x_commsquality_t *)bytebuf)->ANL_currFC);
	} else {
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2CurrentTxRate:
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CURRENTTXRATE, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2EDThreshold:
case DIDmib_p2_p2Table_p2CDThreshold:
case DIDmib_p2_p2Table_p2DeferThreshold:
case DIDmib_p2_p2Table_p2CellSearchThreshold:
case DIDmib_p2_p2Table_p2OORThreshold:
	/*
	 *  HFA384x_RID_CURRENTSCALETHRESH
	 */

/*----------------------------------------------------------------*/
case DIDmib_p2_p2Table_p2CurrentPowerState:
	if ( isget ) {
		result = hfa384x_drvr_getconfig16( hw, 
				HFA384x_RID_CURRENTPOWERSTATE, wordbuf);
		prism2mgmt_prism2int2p80211int(wordbuf, uint32);
	} else {
		msg->resultcode.data =
			P80211ENUM_resultcode_cant_set_readonly_mib;
	}
	break;

/*----------------------------------------------------------------*/
/* These items are not supported at this time */

/* TODO: implement these in a sane way.  Unfortunately, they're a pain. */
case DIDmib_dot11smt_dot11AuthenticationAlgorithmsTable_dot11AuthenticationAlgorithm1:
case DIDmib_dot11smt_dot11AuthenticationAlgorithmsTable_dot11AuthenticationAlgorithmsEnable1:
case DIDmib_dot11smt_dot11AuthenticationAlgorithmsTable_dot11AuthenticationAlgorithm2:
case DIDmib_dot11smt_dot11AuthenticationAlgorithmsTable_dot11AuthenticationAlgorithmsEnable2:
case DIDmib_dot11smt_dot11AuthenticationAlgorithmsTable_dot11AuthenticationAlgorithm3:
case DIDmib_dot11smt_dot11AuthenticationAlgorithmsTable_dot11AuthenticationAlgorithmsEnable3:
case DIDmib_dot11smt_dot11AuthenticationAlgorithmsTable_dot11AuthenticationAlgorithm4:
case DIDmib_dot11smt_dot11AuthenticationAlgorithmsTable_dot11AuthenticationAlgorithmsEnable4:
case DIDmib_dot11smt_dot11AuthenticationAlgorithmsTable_dot11AuthenticationAlgorithm5:
case DIDmib_dot11smt_dot11AuthenticationAlgorithmsTable_dot11AuthenticationAlgorithmsEnable5:
case DIDmib_dot11smt_dot11AuthenticationAlgorithmsTable_dot11AuthenticationAlgorithm6:
case DIDmib_dot11smt_dot11AuthenticationAlgorithmsTable_dot11AuthenticationAlgorithmsEnable6:

/* WEP Key mappings aren't supported in the f/w */
case DIDmib_dot11smt_dot11WEPKeyMappingsTable_dot11WEPKeyMappingIndex:
case DIDmib_dot11smt_dot11WEPKeyMappingsTable_dot11WEPKeyMappingAddress:
case DIDmib_dot11smt_dot11WEPKeyMappingsTable_dot11WEPKeyMappingWEPOn:
case DIDmib_dot11smt_dot11WEPKeyMappingsTable_dot11WEPKeyMappingValue:
case DIDmib_dot11smt_dot11PrivacyTable_dot11WEPKeyMappingLength:

/* TODO: implement counters */
case DIDmib_dot11smt_dot11PrivacyTable_dot11WEPICVErrorCount:
case DIDmib_dot11smt_dot11PrivacyTable_dot11WEPExcludedCount:
case DIDmib_dot11mac_dot11CountersTable_dot11TransmittedFragmentCount:
case DIDmib_dot11mac_dot11CountersTable_dot11MulticastTransmittedFrameCount:
case DIDmib_dot11mac_dot11CountersTable_dot11FailedCount:
case DIDmib_dot11mac_dot11CountersTable_dot11RetryCount:
case DIDmib_dot11mac_dot11CountersTable_dot11MultipleRetryCount:
case DIDmib_dot11mac_dot11CountersTable_dot11FrameDuplicateCount:
case DIDmib_dot11mac_dot11CountersTable_dot11RTSSuccessCount:
case DIDmib_dot11mac_dot11CountersTable_dot11RTSFailureCount:
case DIDmib_dot11mac_dot11CountersTable_dot11ACKFailureCount:
case DIDmib_dot11mac_dot11CountersTable_dot11ReceivedFragmentCount:
case DIDmib_dot11mac_dot11CountersTable_dot11MulticastReceivedFrameCount:
case DIDmib_dot11mac_dot11CountersTable_dot11FCSErrorCount:
case DIDmib_dot11mac_dot11CountersTable_dot11TransmittedFrameCount:
case DIDmib_dot11mac_dot11CountersTable_dot11WEPUndecryptableCount:

/* TODO: implement sane values for these */
case DIDmib_dot11mac_dot11OperationTable_dot11ManufacturerID:
case DIDmib_dot11mac_dot11OperationTable_dot11ProductID:

/* Not too worried about these at the moment */
case DIDmib_dot11phy_dot11PhyAntennaTable_dot11CurrentTxAntenna:
case DIDmib_dot11phy_dot11PhyAntennaTable_dot11DiversitySupport:
case DIDmib_dot11phy_dot11PhyAntennaTable_dot11CurrentRxAntenna:
case DIDmib_dot11phy_dot11PhyTxPowerTable_dot11NumberSupportedPowerLevels:
case DIDmib_dot11phy_dot11PhyTxPowerTable_dot11TxPowerLevel1:
case DIDmib_dot11phy_dot11PhyTxPowerTable_dot11TxPowerLevel2:
case DIDmib_dot11phy_dot11PhyTxPowerTable_dot11TxPowerLevel3:
case DIDmib_dot11phy_dot11PhyTxPowerTable_dot11TxPowerLevel4:
case DIDmib_dot11phy_dot11PhyTxPowerTable_dot11TxPowerLevel5:
case DIDmib_dot11phy_dot11PhyTxPowerTable_dot11TxPowerLevel6:
case DIDmib_dot11phy_dot11PhyTxPowerTable_dot11TxPowerLevel7:
case DIDmib_dot11phy_dot11PhyTxPowerTable_dot11TxPowerLevel8:
case DIDmib_dot11phy_dot11PhyTxPowerTable_dot11CurrentTxPowerLevel:

/* Ummm, FH and IR don't apply */
case DIDmib_dot11phy_dot11PhyFHSSTable_dot11HopTime:
case DIDmib_dot11phy_dot11PhyFHSSTable_dot11CurrentChannelNumber:
case DIDmib_dot11phy_dot11PhyFHSSTable_dot11MaxDwellTime:
case DIDmib_dot11phy_dot11PhyFHSSTable_dot11CurrentDwellTime:
case DIDmib_dot11phy_dot11PhyFHSSTable_dot11CurrentSet:
case DIDmib_dot11phy_dot11PhyFHSSTable_dot11CurrentPattern:
case DIDmib_dot11phy_dot11PhyFHSSTable_dot11CurrentIndex:
case DIDmib_dot11phy_dot11PhyDSSSTable_dot11CCAModeSupported:
case DIDmib_dot11phy_dot11PhyDSSSTable_dot11EDThreshold:
case DIDmib_dot11phy_dot11PhyIRTable_dot11CCAWatchdogTimerMax:
case DIDmib_dot11phy_dot11PhyIRTable_dot11CCAWatchdogCountMax:
case DIDmib_dot11phy_dot11PhyIRTable_dot11CCAWatchdogTimerMin:
case DIDmib_dot11phy_dot11PhyIRTable_dot11CCAWatchdogCountMin:

/* We just don't have enough antennas right now to worry about this */
case DIDmib_dot11phy_dot11AntennasListTable_dot11AntennaListIndex:
case DIDmib_dot11phy_dot11AntennasListTable_dot11SupportedTxAntenna:
case DIDmib_dot11phy_dot11AntennasListTable_dot11SupportedRxAntenna:
case DIDmib_dot11phy_dot11AntennasListTable_dot11DiversitySelectionRx:

/* TODO: Prism2 extension items.  Implement as needed. */
case DIDmib_p2_p2Table_p2PRIIdentity:
case DIDmib_p2_p2Table_p2PRISUPRange:
case DIDmib_p2_p2Table_p2PRI_CFI_ACTRange:
case DIDmib_p2_p2Table_p2SerialNum:
case DIDmib_p2_p2Table_p2NicIdentity:
case DIDmib_p2_p2Table_p2MFISupRange:
case DIDmib_p2_p2Table_p2CFISupRange:
case DIDmib_p2_p2Table_p2SuppChannelList:
case DIDmib_p2_p2Table_p2STAIdentity:
case DIDmib_p2_p2Table_p2StaSupRange:
case DIDmib_p2_p2Table_p2STA_MFI_ActRange:
case DIDmib_p2_p2Table_p2STA_CFI_ActRange:
case DIDmib_p2_p2Table_p2IVPeriod:
case DIDmib_p2_p2Table_p2HostDecrypt:
case DIDmib_p2_p2Table_p2HostEncrypt:
case DIDmib_p2_p2Table_p2MaxAssociations:
case DIDmib_p2_p2Table_p2HostAuth:
case DIDmib_p2_p2Table_p2CRCErrAll:
case DIDmib_p2_p2Table_p2CRCErrMM:
case DIDmib_p2_p2Table_p2LongPreamble:
case DIDmib_p2_p2Table_p2ShortPreamble:
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
	break;
default:
	msg->resultcode.data = P80211ENUM_resultcode_not_supported;
}

	if ( msg->resultcode.data == P80211ENUM_resultcode_success )
	{
		if ( result == 0 ) {
			msg->resultcode.data =
				P80211ENUM_resultcode_success;
			if ( isget ) {
				/* set the status for the mibattribute and for the mib item
				within the data portion of the mibattribute */
				msg->mibattribute.status = P80211ENUM_msgitem_status_data_ok;
				((p80211itemd_t *)(msg->mibattribute.data))->status =
					P80211ENUM_msgitem_status_data_ok;
			}
		} else {
			WLAN_LOG_DEBUG1(1,"get/set failure, result=%d\n", result);
			msg->resultcode.data =
				P80211ENUM_resultcode_implementation_failure;
		}
	}

	DBFEXIT;
	return 0;
}


/*----------------------------------------------------------------
* prism2mgmt_pstr2bytestr
*
* Convert the pstr data in the WLAN message structure into an hfa384x
* byte string format.
*
* Arguments:
*	bytestr		hfa384x byte string data type
*	pstr		wlan message data
*
* Returns: 
*	Nothing
*
----------------------------------------------------------------*/

void prism2mgmt_pstr2bytestr(hfa384x_bytestr_t *bytestr, p80211pstrd_t *pstr)
{
	bytestr->len = host2hfa384x_16((UINT16)(pstr->len));
	memcpy(bytestr->data, pstr->data, pstr->len);
}


/*----------------------------------------------------------------
* prism2mgmt_pstr2bytearea
*
* Convert the pstr data in the WLAN message structure into an hfa384x
* byte area format.
*
* Arguments:
*	bytearea	hfa384x byte area data type
*	pstr		wlan message data
*
* Returns: 
*	Nothing
*
----------------------------------------------------------------*/

void prism2mgmt_pstr2bytearea(UINT8 *bytearea, p80211pstrd_t *pstr)
{
	memcpy(bytearea, pstr->data, pstr->len);
}


/*----------------------------------------------------------------
* prism2mgmt_bytestr2pstr
*
* Convert the data in an hfa384x byte string format into a
* pstr in the WLAN message.
*
* Arguments:
*	bytestr		hfa384x byte string data type
*	msg		wlan message
*
* Returns: 
*	Nothing
*
----------------------------------------------------------------*/

void prism2mgmt_bytestr2pstr(hfa384x_bytestr_t *bytestr, p80211pstrd_t *pstr)
{
	pstr->len = (UINT8)(hfa384x2host_16((UINT16)(bytestr->len)));
	memcpy(pstr->data, bytestr->data, pstr->len);
}


/*----------------------------------------------------------------
* prism2mgmt_bytearea2pstr
*
* Convert the data in an hfa384x byte area format into a pstr
* in the WLAN message.
*
* Arguments:
*	bytearea	hfa384x byte area data type
*	msg		wlan message
*
* Returns: 
*	Nothing
*
----------------------------------------------------------------*/

void prism2mgmt_bytearea2pstr(UINT8 *bytearea, p80211pstrd_t *pstr, int len)
{
	pstr->len = (UINT8)len;
	memcpy(pstr->data, bytearea, len);
}


/*----------------------------------------------------------------
* prism2mgmt_prism2int2p80211int
*
* Convert an hfa384x integer into a wlan integer
*
* Arguments:
*	prism2enum	pointer to hfa384x integer
*	wlanenum	pointer to p80211 integer
*
* Returns: 
*	Nothing
*
----------------------------------------------------------------*/

void prism2mgmt_prism2int2p80211int(UINT16 *prism2int, UINT32 *wlanint)
{
	*wlanint = (UINT32)(*prism2int);
}


/*----------------------------------------------------------------
* prism2mgmt_p80211int2prism2int
*
* Convert a wlan integer into an hfa384x integer
*
* Arguments:
*	prism2enum	pointer to hfa384x integer
*	wlanenum	pointer to p80211 integer
*
* Returns: 
*	Nothing
*
----------------------------------------------------------------*/

void prism2mgmt_p80211int2prism2int(UINT16 *prism2int, UINT32 *wlanint)
{
	*prism2int = (UINT16)(*wlanint);
}


/*----------------------------------------------------------------
* prism2mgmt_prism2enum2p80211enum
*
* Convert the hfa384x enumerated int into a p80211 enumerated int
*
* Arguments:
*	prism2enum	pointer to hfa384x integer
*	wlanenum	pointer to p80211 integer
*	rid		hfa384x record id
*
* Returns: 
*	Nothing
*
----------------------------------------------------------------*/
void prism2mgmt_prism2enum2p80211enum(UINT16 *prism2enum, UINT32 *wlanenum, UINT16 rid)
{
	/* At the moment, the need for this functionality hasn't
	presented itself. All the wlan enumerated values are
	a 1-to-1 match against the Prism2 enumerated values*/
	return;
}


/*----------------------------------------------------------------
* prism2mgmt_p80211enum2prism2enum
*
* Convert the p80211 enumerated int into an hfa384x enumerated int
*
* Arguments:
*	prism2enum	pointer to hfa384x integer
*	wlanenum	pointer to p80211 integer
*	rid		hfa384x record id
*
* Returns: 
*	Nothing
*
----------------------------------------------------------------*/
void prism2mgmt_p80211enum2prism2enum(UINT16 *prism2enum, UINT32 *wlanenum, UINT16 rid)
{
	/* At the moment, the need for this functionality hasn't
	presented itself. All the wlan enumerated values are
	a 1-to-1 match against the Prism2 enumerated values*/
	return;
}



/*----------------------------------------------------------------
* prism2mgmt_get_oprateset
*
* Convert the hfa384x bit area into a wlan octet string.
*
* Arguments:
*	rate		Prism2 bit area
*	pstr		wlan octet string
*
* Returns: 
*	Nothing
*
----------------------------------------------------------------*/
void prism2mgmt_get_oprateset(UINT16 *rate, p80211pstrd_t *pstr)
{
	UINT8	len;
	UINT8	*datarate;

	len = 0;
	datarate = pstr->data;

 	/* 1 Mbps */
	if ( BIT0 & (*rate) ) {
		len += (UINT8)1;
		*datarate = (UINT8)2;
		datarate++;
	}

 	/* 2 Mbps */
	if ( BIT1 & (*rate) ) {
		len += (UINT8)1;
		*datarate = (UINT8)4;
		datarate++;
	}

 	/* 5.5 Mbps */
	if ( BIT2 & (*rate) ) {
		len += (UINT8)1;
		*datarate = (UINT8)11;
		datarate++;
	}

 	/* 11 Mbps */
	if ( BIT3 & (*rate) ) {
		len += (UINT8)1;
		*datarate = (UINT8)22;
		datarate++;
	}

	pstr->len = len;

	return;
}



/*----------------------------------------------------------------
* prism2mgmt_set_oprateset
*
* Convert the wlan octet string into an hfa384x bit area.
*
* Arguments:
*	rate		Prism2 bit area
*	pstr		wlan octet string
*
* Returns: 
*	Nothing
*
----------------------------------------------------------------*/
void prism2mgmt_set_oprateset(UINT16 *rate, p80211pstrd_t *pstr)
{
	UINT8	*datarate;
	int	i;

	*rate = 0;

	datarate = pstr->data;

	for ( i=0; i < pstr->len; i++, datarate++ ) {
		switch (*datarate) {
		case 2: /* 1 Mbps */
			*rate |= BIT0;
			break;
		case 4: /* 2 Mbps */
			*rate |= BIT1;
			break;
		case 11: /* 5.5 Mbps */
			*rate |= BIT2;
			break;
		case 22: /* 11 Mbps */
			*rate |= BIT3;
			break;
		default:
			WLAN_LOG_DEBUG1(1, "Unrecoginzed Rate of %d\n",
				*datarate);
			break;
		}
	}

	return;
}



/*----------------------------------------------------------------
* prism2mgmt_get_grpaddr
*
* Retrieves a particular group address from the list of
* group addresses.
*
* Arguments:
*	did		mibitem did
*	pstr		wlan octet string
*	priv		prism2 driver private data structure
*
* Returns: 
*	Nothing
*
----------------------------------------------------------------*/
void prism2mgmt_get_grpaddr(UINT32 did, p80211pstrd_t *pstr,
	prism2sta_priv_t *priv )
{
	int	index;

	index = prism2mgmt_get_grpaddr_index(did);

	if ( index >= 0 ) {
		pstr->len = WLAN_ADDR_LEN;
		memcpy(pstr->data, priv->dot11_grp_addr[index],
			WLAN_ADDR_LEN);
	}

	return;
}



/*----------------------------------------------------------------
* prism2mgmt_set_grpaddr
*
* Convert the wlan octet string into an hfa384x bit area.
*
* Arguments:
*	did		mibitem did
*	buf		
*	groups
*
* Returns: 
*	0	Success
*	!0	Error
*
----------------------------------------------------------------*/
int prism2mgmt_set_grpaddr(UINT32 did, UINT8 *prism2buf,
	p80211pstrd_t *pstr, prism2sta_priv_t *priv )
{
	UINT8	no_addr[WLAN_ADDR_LEN];
	int	index;
	int	result;

	memset(no_addr, 0, WLAN_ADDR_LEN);
	result = 0;

	/* here we're "adding" an address to the group
	address list.  Check to make sure we aren't
	trying to add more than the maximum allowed
	number of group addresses in the list */
	if ( memcmp(no_addr, pstr->data, WLAN_ADDR_LEN) != 0) {
		if ( priv->dot11_grpcnt < MAX_GRP_ADDR ) {
			memcpy( priv->dot11_grp_addr[priv->dot11_grpcnt],
				pstr->data, WLAN_ADDR_LEN);
			priv->dot11_grpcnt += 1;
		} else {
			result = -1;
		}
	} else {
	/* here we're "deleting" an address from the
	group address list.  Check to make sure we're
	not trying to delete from an empty list */
		if ( priv->dot11_grpcnt > 0 ) {
			index = prism2mgmt_get_grpaddr_index(did);

			if ( index >= 0 ) {
				/* check to make sure there is an address
				to delete at the requested group address */
				if ( memcmp(priv->dot11_grp_addr[index],
					no_addr,
					WLAN_ADDR_LEN) != 0) {

					priv->dot11_grpcnt -= 1;
					/* shift all the addresses following
						the one to delete */
					memcpy( priv->dot11_grp_addr[index],
						priv->dot11_grp_addr[index + 1],
						((priv->dot11_grpcnt)-index) * WLAN_ADDR_LEN);
					/* erase the last address that now appears
					twice */
					memcpy( priv->dot11_grp_addr[priv->dot11_grpcnt],
						no_addr,
						WLAN_ADDR_LEN);
				}
			}
		} else {
			result = -1;
		}
	}

	return result;
}



/*----------------------------------------------------------------
* prism2mgmt_get_grpaddr_index
*
* Gets the index in the group address list based on the did.
*
* Arguments:
*	did		mibitem did
*
* Returns: 
*	>= 0	If valid did
*	< 0	If not valid did
*
----------------------------------------------------------------*/
int prism2mgmt_get_grpaddr_index( UINT32 did )
{
	int	index;

	index = -1;

	switch (did) {
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address1:
		index = 0;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address2:
		index = 1;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address3:
		index = 2;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address4:
		index = 3;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address5:
		index = 4;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address6:
		index = 5;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address7:
		index = 6;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address8:
		index = 7;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address9:
		index = 8;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address10:
		index = 9;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address11:
		index = 10;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address12:
		index = 11;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address13:
		index = 12;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address14:
		index = 13;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address15:
		index = 14;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address16:
		index = 15;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address17:
		index = 16;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address18:
		index = 17;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address19:
		index = 18;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address20:
		index = 19;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address21:
		index = 20;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address22:
		index = 21;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address23:
		index = 22;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address24:
		index = 23;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address25:
		index = 24;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address26:
		index = 25;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address27:
		index = 26;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address28:
		index = 27;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address29:
		index = 28;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address30:
		index = 29;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address31:
		index = 30;
		break;
	case DIDmib_dot11mac_dot11GroupAddressesTable_dot11Address32:
		index = 31;
		break;
	}

	return index;
}
