/* prismdump.c - Utility to dump 802.11 frames to a file. 
 * 
 * This utility is written for use with IEEE 802.11 adapters based
 * on Intersil's PRISM II chipset (PCMCIA). 
 * The linux driver for these cards can be found on www.linux-wlan.com
 * It has been verified with a Compaq WL100 IEEE 802.11 adapter. 
 * 
 * Copyright (c)2000 by Axis Communications AB, LUND, SWEDEN
 * Comments/Bugreports should be sent to: johan.jorgensen@axis.com
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#include <asm/types.h>
#include <linux/netlink.h>
#include <netinet/in.h>
#include <wtap.h>



#define MAX_BUFFER_SIZE 4000	/* Size of receive buffer */
#define DEVNAME_LEN 16
#define __WLAN_ATTRIB_PACK__       __attribute__ ((packed))
#define MCAST_GRP_SNIFF 0x00000002	/* Value used to notify netlink  */

typedef struct
{
  __u32 did __attribute__ ((packed));
  __u16 status __attribute__ ((packed));
  __u16 len __attribute__ ((packed));
  __u32 data __attribute__ ((packed));
}
p80211item_t;

typedef struct
{
  __u32 msgcode __attribute__ ((packed));
  __u32 msglen __attribute__ ((packed));
  __u8 devname[DEVNAME_LEN] __attribute__ ((packed));
  p80211item_t hosttime __attribute__ ((packed));
  p80211item_t mactime __attribute__ ((packed));
  p80211item_t channel __attribute__ ((packed));
  p80211item_t rssi __attribute__ ((packed));
  p80211item_t sq __attribute__ ((packed));
  p80211item_t signal __attribute__ ((packed));
  p80211item_t noise __attribute__ ((packed));
  p80211item_t rate __attribute__ ((packed));
  p80211item_t istx __attribute__ ((packed));
  p80211item_t frmlen __attribute__ ((packed));
}
sniff_packet_t;

typedef struct
{
  __u16 frame_control __attribute__ ((packed));
  __u16 duration_id __attribute__ ((packed));
  __u8 mac1[6] __attribute__ ((packed));
  __u8 mac2[6] __attribute__ ((packed));
  __u8 mac3[6] __attribute__ ((packed));
  __u16 sequence __attribute__ ((packed));
  __u8 mac4[6] __attribute__ ((packed));
}
wlan_hdr_t;

static unsigned int stop_sniffing = 0;

void
stop_signal ()
{
  fprintf (stderr,"Received CTRL-C - sniffing aborted\n");
  stop_sniffing = 1;
}


int
main (int argc, char **argv)
{
  struct sockaddr_nl nl_sk_addr;
  int nlfd, n, oi;
  int recvlen;
  unsigned char msgbuf[MAX_BUFFER_SIZE];
  sniff_packet_t *sniff_info;
  wlan_hdr_t *packet_hdr;
  wtap_dumper *dump_file;
  struct wtap_pkthdr packet_hdr_info;
  int wtap_error;

  fprintf (stderr, "Prismdump 1.0.0 (C)2000 Axis Communications AB\n");

  setpriority (PRIO_PROCESS, 0, -20);

  signal (SIGINT, stop_signal);
  fprintf (stderr, "Use CTRL-C to stop sniffing\n");
  dump_file = wtap_dump_fdopen (fileno(stdout), WTAP_FILE_PCAP, 
				WTAP_ENCAP_IEEE_802_11,
				2344, &wtap_error);

  if (!dump_file)
    {
      perror ("Cannot open the dump file!");
      return 1;
    }

#ifdef DEBUG
  fprintf (stderr,"Capture file was successfully opened\n");
#endif

  if ((nlfd = socket (PF_NETLINK, SOCK_RAW, MCAST_GRP_SNIFF)) < 0)
    {
      perror ("Could not create socket");
      wtap_dump_close (dump_file, &wtap_error);
    }

  memset (&nl_sk_addr, 0, sizeof (struct sockaddr_nl));
  nl_sk_addr.nl_family = (sa_family_t) PF_NETLINK;
  nl_sk_addr.nl_pid = (unsigned int) getpid ();
  nl_sk_addr.nl_groups = MCAST_GRP_SNIFF;

  if (bind
      (nlfd, (struct sockaddr *) &nl_sk_addr,
       sizeof (struct sockaddr_nl)) < 0)
    {
      perror ("Call to bind failed");
      return 1;
    }

  n = 0;

  fprintf (stderr, "Capture loop entered\n");

  do
    {

      memset (msgbuf, 0, MAX_BUFFER_SIZE);

      if ((recvlen = recv (nlfd, msgbuf, MAX_BUFFER_SIZE, 0)) < 0)
	{
	  fprintf (stderr,"Error in recvfrom (%d)", recvlen);
	  perror ("");
	  exit (0);
	}

      else
	{

	  sniff_info = (sniff_packet_t *) msgbuf;

	  /* Setup Wiretap packet header */
	  gettimeofday (&packet_hdr_info.ts, NULL);
	  packet_hdr_info.caplen = sniff_info->frmlen.data;
	  packet_hdr_info.len = sniff_info->frmlen.data;
	  packet_hdr_info.pkt_encap = WTAP_ENCAP_IEEE_802_11;

	  oi = sizeof (sniff_packet_t);

	  /* Convert Intersils stupid solution to a proper Axis one */
	  packet_hdr = (wlan_hdr_t *) & msgbuf[oi];

	  /* change duration_id to proper 802.11 standard */
	  packet_hdr->duration_id = htons (packet_hdr->duration_id);

	  /* change sequence number to proper 802.11 standard */
	  packet_hdr->sequence = htons (packet_hdr->sequence);

	  /* Replace data in frame header with the updated info! */
	  memcpy (&msgbuf[oi], packet_hdr, sizeof (wlan_hdr_t));


	  /* Now we can save the frame to the capture file */
	  wtap_dump (dump_file, &packet_hdr_info, NULL, &msgbuf[oi],
		     &wtap_error);

	  if (wtap_error < 0)
	    {
	      perror ("Error in call to wtap_dump");
	      exit (0);
	    }
	}

      n++;
    }
  while (stop_sniffing == 0);

  fprintf (stderr,
	   "Exiting - %d packets were successfully captured\n", n);
  wtap_dump_close (dump_file, &wtap_error);

  return 0;
}
