/*
 * Linux-Mobile-IP
 *            An implementation of Mobile IP for the LINUX operating system
 *            developed at the State University of New York, Binghamton
 *            (with support from the Center for Computing Technologies).
 *            The implementation uses the message formats described in the
 *            Internet Engineering Task Force (IETF) mobile-ip draft
 *            but is not (yet) fully compliant.
 *
 *            Recent drafts of the IETF Mobile IP proposal are
 *            available at ftp://software.watson.ibm.com/pub/mobile-ip/ 
 *
 *            Permission is hereby granted to redistribute this code and/or
 *            modify it under the terms of the GNU Genral Public License
 *            as published by the Free Software Foundation; either version
 *            two or (at your option) any later version.
 *
 *            The authors request that their contribution be appropriately 
 *            acknowledged in any derived work. 
 *
 *            agent.c - Design of the Mobile Agent (either home agent or 
 *                      foreign agent).
 * 
 * Version:   0.90     08/31/1995
 * 
 * Authors:   Vipul Gupta <vgupta@cs.binghamton.edu>
 *            Benjamin Lancki <ben@anchor.cs.binghamton.edu>
 * 
 * */  

/*
 *  Command line arguments:
 *    -a : agent's IP address on the network on which it supports mobility
 *    -m : network mask corresponding to the IP address above
 *    -h : hardware address of the mobility interface (used for proxies
 *	   and used only if this agent is a home agent
 *    -w : the agent's IP address on a wired interface (this is useful
 *         in a setting where all mobility agents and mobile hosts have
 *         been assigned IP addresses on a common subnet even though
 *         they may not all have link level connectivity -- similar
 *         to the intracampus addressing scheme used in the Columbia
 *         proposal)
 *
 *
 * To do:  
 *   1. Currently supports agent behaviour on only one network interface.
 *
 *   2. The part where an mh is registered from a foreign network and
 *       does its own decapsulation is currently untested.
 *
 * History of Changes:
 *     Jul 25, 1995              First version coded  - V.G.
 *     Jul 31, 1995              Removed calls to GETRT - V.G.
 *     Aug  1, 1995              Removed bug in sendRefusal and reorganized
 *				 it to improve efficiency - V.G.
 *     Aug  9, 1995              Added support for MD5 authentication in
 *                               processing the registration and registration 
 *                               reply - B.L.
 *     Aug 13, 1995              Agent now reads config file /etc/mip-mh.ok to
 *                               initialize mhinfo struct - B.L.
 *     Aug 14, 1995              Agent keeps a log in event of system crash
 *                               or shutdown and recovers on startup - B.L.
 * */
   
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>

#include "messages.h"
#include "low.h"
#include "agent.h"

int debug = 2;

extern void 
testprint(char *,int);

void 
sendReply(unsigned short, int, int, unsigned long, 
	  unsigned long, unsigned long, struct id);
void
sendConfirm(unsigned short, int, int, unsigned long, 
	    unsigned long, unsigned long, int);
void
sendRefusal(unsigned short, int, int, unsigned long, 
	       unsigned long, unsigned long, int);
extern void 
appendauth(char *,int *,char *,int);

extern int 
authextfound(char *,int,int);

extern int 
authok(char *,int,int,char *,int);

extern void 
printext(char *,int,int);

extern void 
printtime();

extern void readlog(char *), writelog(char *);
void usage(char *), cleanup();
int URheresid, Regreplysid; /* for sending, these could be the same */
int WhereAmIsid, RegisterMesid; /* for listening to requests, these are
				 * bound to different ports */ 
int ioctlsid; /* for all kinds of low level ioctl calls */

unsigned long tunnelbitvec = 0; /* a bit vector to track tunnels in use */
char supportedMHnum = 0; /* # MHs we are configured to serve in mip-mh.ok */ 
char proxyHwAddr[7];  /* hw address to use for proxy ARP */
unsigned long haAddr; /* home agent address */
unsigned long haNetmask; /* home agent's network mask */
unsigned long haWiredAddr; /* a wired address usable as tunnel end point
			    * advertised in URhere messages */
unsigned long haAdvertAddr;

struct sockaddr_in URhereto = 
     {  (short) AF_INET, (unsigned short) ADVERTISETO, 
	(unsigned long) 0, (unsigned long) 0, (unsigned long) 0 };

mhdata mhinfo[MAXMHNUM+1]; /* zeroth mh is not really used */ 

struct stats  {
   unsigned long badwhereami;
   unsigned long badregisterme;
   unsigned long mhprohibited;
   unsigned long idmismatch;
   unsigned long authfailed;
   unsigned long noresources;
   unsigned long successfulreg;
} agentstats =  { (long) 0, (long) 0, (long) 0, (long) 0, 
                  (long) 0, (long) 0, (long) 0 };

void
initsockets()  {
   struct sockaddr_in sa;
   int enable = 1;
   
   Regreplysid = URheresid = socket(AF_INET, SOCK_DGRAM, 0);
   WhereAmIsid = socket(AF_INET, SOCK_DGRAM, 0);
   RegisterMesid = socket(AF_INET, SOCK_DGRAM, 0);
   ioctlsid = socket(AF_INET, SOCK_DGRAM, 0);
   if (Regreplysid < 0 || URheresid < 0 || 
       WhereAmIsid < 0 || RegisterMesid < 0 || ioctlsid < 0)  {
	  fprintf(stderr, "initsockets(): could not create sockets.\n");
	   exit(-1);
       }
   sa.sin_family = AF_INET; sa.sin_addr.s_addr = htonl(INADDR_ANY); 
   sa.sin_port = 0; 
   if (bind(URheresid, (struct sockaddr *) &sa, sizeof(sa)) < 0)  {
      perror("initsockets(): Bind URheresid failed.\n");
      cleanup(); exit(-1);
   }
   if (setsockopt(URheresid, SOL_SOCKET, SO_BROADCAST, 
		  (char *)&enable, sizeof(int)) < 0)  {
	  perror("initsockets(): setsockopt SO_BROADCAST failed");
	  cleanup(); exit(-1);
       }
   sa.sin_port = WHEREAMIPORT;
   if (bind(WhereAmIsid, (struct sockaddr *) &sa, sizeof(sa)) < 0)  {
      perror("initsockets(): Bind WhereAmIsid failed.\n");
      cleanup(); exit(-1);
   }
   sa.sin_port = MIPREGPORT;
   if (bind(RegisterMesid, (struct sockaddr *) &sa, sizeof(sa)) < 0)  {
      perror("initsockets(): Bind RegisterMesid failed.\n");
      cleanup(); exit(-1);
   }
   if (debug > 0)
     fprintf(stderr, "Initialized sockets ...\n");
}

void
cleanup()  {
   /* down all tunnels that we could have created and remove
    * proxy arps for all MHs we support if the arp HWaddr in
    * those entries is our own */
   int i;
   
   for (i = 0; i < MAXTUNNELS; i++)
     lowifacereq(DOWN, i, 0, 0); 
   for (i = 0; i < MAXMHNUM; i++)
     lowARPreq(REM0, mhinfo[i].ipaddr, proxyHwAddr);
   if (debug > 1)
     fprintf(stderr, "Done cleaning up ...\n");
}

void
graceful_exit() {
   cleanup();
   writelog("/usr/adm/mip-ha.log");
   exit(0);
}

void
newId(struct id *id1, struct id *id2)  {
   static int count = 0;
   struct timeval tv;
   
   if (count++ == 0)  {
      gettimeofday(&tv, NULL);
      srandom((int) tv.tv_usec);
   };
   id1->high = (unsigned long) random();
   id1->low = id2->low;
}
	
void 
sendURhere(struct sockaddr_in *toaddr)  {
   static struct urhmsg  {
      struct youarehere part1;
      unsigned long addr1, mask1, addr2, mask2; 
   } mesg;
	
   if (debug > 2)
     fprintf(stderr, "Sending a URhere message to %s at port %d.\n",
	     inet_ntoa(toaddr->sin_addr), (unsigned int) toaddr->sin_port);
   else if (debug > 0)
     fprintf(stderr, ".");
   /* fill the buffer appropriately and send */
   mesg.part1.code = YOUAREHERETYPE;
   mesg.part1.addrnum = 2;
   mesg.part1.flags = 0x0002;
   mesg.addr1 = haAddr;
   mesg.mask1 = haNetmask;
   mesg.addr2 = haWiredAddr;
   /* Change to take from commandline */
   mesg.mask2 = inet_addr("255.255.0.0"); 
   
   if (sendto(URheresid, (char *) &mesg, sizeof(mesg), 0, 
	      (struct sockaddr *) toaddr, sizeof(struct sockaddr))
       < 0)  {
	  perror("sendURhere(): sendto failed.\n");
	  cleanup(); exit(-1);
       }
}

void
update_lifetimes()  {
   /* Walk through the list of MH for which TUNNELUP is set and
    * decrement lifetimes.
    * If lifetime becomes zero or less, stop supporting this
    * MH i.e. 
    * (a) remove its routing table entry, (route del ipaddr)
    * (b) destroy its tunnel name (ifconfig tunnelname down)
    * (c) stop proxying for this MH by removing its arp entry
    *     (arp -d ipaddr) 
    * (d) reset the TUNNELUP flag, timeleft, coaddr and 
    *     are set to 0. Tunnel name becomes a null string */
   int i;
   
   for (i = 1; i <= supportedMHnum; i++)  {
      if ((mhinfo[i].status & TUNNELUP) == 0) continue; 
      if (--mhinfo[i].timeleft <= 0)  {
	 mhinfo[i].status &= ~TUNNELUP;
	 mhinfo[i].timeleft = 0;
	 mhinfo[i].coaddr = 0;
	 newId(&(mhinfo[i].RegistrationId), &(mhinfo[i].RegistrationId));
	 if (debug > 1)  {
	    fprintf(stderr, "\n--Registration expired for %lx\n",
		   htonl(mhinfo[i].ipaddr));
	 };
	 lowrtreq(DELRT, mhinfo[i].ipaddr, &(mhinfo[i].tunnelnum));
	 lowifacereq(DOWN, mhinfo[i].tunnelnum, 0, 0);
	 lowARPreq(REM1, mhinfo[i].ipaddr, "000000");
	 mhinfo[i].tunnelnum = 0;
      }
   }
}

void
alarm_handler()  {
   /* Check for expired regitrations and send out URhere broadcast */
   update_lifetimes();
   sendURhere(&URhereto);
   
   /* set things up to come back to this procedure sometime soon */ 
   signal(SIGALRM, alarm_handler);
   alarm(1);
}

void
init()  {
   initsockets(); /* set up sockets for listening to requests and
		   * sending messages */
   cleanup(); /* cleanup relies on ioctlsid */
   URhereto.sin_addr.s_addr = inet_addr(ADVERTISEON);
   URhereto.sin_addr.s_addr = haAdvertAddr;
   URhereto.sin_port = ADVERTISETO;
   
   readmhdata("/etc/mip-mh.ok");
   /* read the ipaddr, authtype and 
				  * secret for first MAXMHNUM mobile
				  * hosts */ 
   readlog("/usr/adm/mip-ha.log"); /* restore any state from a previous
				    * incarnation of this daemon */ 
   signal(SIGTERM, graceful_exit); /* terminate gracefully */
   signal(SIGINT, graceful_exit); /* terminate gracefully */
   signal(SIGALRM, alarm_handler); /* alarm_handler is the function
				    * that gets called when alarm
				    * goes off */ 
   alarm(1);
}

void
processWhereAmI(char *msg, int len, struct sockaddr_in *from)  {
   /* if message is not of type WhereAmI, record this in a
    * debug log and ignore msg. Otherwise call sendURhere()
    * with the address and port obtained from the from
    * structure */
   struct whereami *where;
   
   if (debug > 1)
     fprintf(stderr, "\n-- WHEREAMI from %s\n", inet_ntoa(from->sin_addr));
   where = (struct whereami *) msg;
   if (where->code != WHEREAMITYPE || len < sizeof(struct whereami))
     agentstats.badwhereami++;
   else
     sendURhere(from);
}

int
IsOurAddress(unsigned long addr)  {
   return((haAddr == addr) ? 1 : 0);
}

char
newtunnel() {
   char i;

   for (i = 1; i < MAXTUNNELS; i++)  {
      if ((tunnelbitvec & (1 << i)) == 0)
      	return(i);
   }
   return(0);
}

int 
whichmh(unsigned long addr)  {
   int i;
   
   mhinfo[0].ipaddr = addr;
   for (i = supportedMHnum; i >= 0; i--)
     if (mhinfo[i].ipaddr == addr) break;
   
   return(i);
}

int
havetunnel(int mh, unsigned long coaddr)  {
   /* Return 0 (false) or 1 (true) depending on whether we can find
    * an existing tunnel as required set up for the mobile host whose
    * mhinfo index is given in mh. This tunnel should end at
    * coaddr and a route to homeaddr must use the tunnel */

   if ((mhinfo[mh].status & TUNNELUP) && (mhinfo[mh].coaddr == coaddr))
     return (1);
   else
     return(0);
}

int 
maIdcmp(struct id *id1, struct id *id2)  {
   return((id1->high == id2->high) ? 0:1);
}

void
processRegisterMe(char *msg, int len, struct sockaddr_in *from)  {
   struct registerme *reg;
   int code = 0;
   int mh;
   int authoffset;
   char temp = 0;
   unsigned short fromport;
   
   if (debug > 2) 
      fprintf(stderr, "<<<<<<  Entering processRegisterMe \n");
   
   if (debug > 0) 
     {
       if (len < sizeof(struct registerme)) 
	 fprintf(stderr,"msg length short of registerme size.\n");
     }
   
   reg = (struct registerme *) msg;
   fromport = (unsigned short) from->sin_port;
   
   if (debug > 0)  {
      fprintf(stderr, "\n==========================\n");
      printtime();
      if (reg->lifetime == 0) 
      	fprintf(stderr, "-- DEREGISTERME from %s\n",
		inet_ntoa(from->sin_addr));
      else
      	fprintf(stderr, "-- REGISTERME from %s\n",
		inet_ntoa(from->sin_addr));
      if (debug > 1)  {
	 fprintf(stderr, "[%8lx:%8lx] Type %2d Flags %2x Lifetime %8d\n",
	      reg->Id.high, reg->Id.low, reg->type, reg->flags, reg->lifetime);
	 fprintf(stderr, "Homeaddr: %8lx, Homeagent: %8lx, Careof: %8lx",
	      htonl(reg->homeaddr), htonl(reg->ha), htonl(reg->coaddr));
	 printext(msg, len, 0);
	 if (debug > 2) {
	   fprintf(stderr, "\n--------------------\n");
	   testprint(msg, len);
	   fprintf(stderr, "--------------------");
	 }
      }
      fprintf(stderr, "\n==========================\n");
   }

   mh = 0;
   if (len < sizeof(struct registerme) || reg->type != REGISTERMETYPE || 
       !IsOurAddress(reg->ha) || !(authoffset=authextfound(msg, len, 0)))  {
      code = 38; /* badly formed request */
      agentstats.badregisterme++;
   } else if ((mh = whichmh(reg->homeaddr)) == 0)  {
      code = 33; /* administratvely prohibited */
      agentstats.mhprohibited++;
   } else if (maIdcmp(&mhinfo[mh].RegistrationId, &(reg->Id)) != 0)  {
      code = 37; /* id is not quite what we expected */ 
      agentstats.idmismatch++;
   } else if (!authok(msg, len, authoffset, mhinfo[mh].secret, 
              mhinfo[mh].keylen))  {
      code = 35; /* failed authentication */
      agentstats.authfailed++;
   }

   
   /* create a new id we can use for our reply */ 
   newId(&(mhinfo[mh].RegistrationId), &(reg->Id));

   if (code)  {
      /* refusals may require setting up a temporary tunnel if we use
       * the columbia setup */ 
   /*    if coaddr
    *    is same as permanent addr (could happen if a MH at home sends a
    *	 deregistration then no need for tunnel) or a tunnel already
    *    exists send directly to perm home addr else create temp tunnel
    *    to the given coaddr, a temporary route through the tunnel
    *    and send the refusal of request to perm home addr. */

      sendRefusal(fromport, code, 0, reg->homeaddr, reg->coaddr, haAddr, mh);
      if (debug > 2)
      	fprintf(stderr, "Refusal sent.\n");
      return;
   }

   /* Now we know that we are looking at a good request */

   if (reg->lifetime == 0) /* We got a deregistration request */
     {
   /* if this is a degistration request, we can always oblige. 
    *    i. make lifetime field zero, if there is a name in the
    *       tunnel field, down it clear name Change status. Clear
    *       coaddr field also. We can leave the RegistrationId field
    *       unchanged. Update any statistics we maintain. code = 1
    *       lifetime = 0  */
	mhinfo[mh].timeleft = 0;
	code = 1;
	mhinfo[mh].status &= ~TUNNELUP;
	mhinfo[mh].coaddr = 0;
	lowrtreq(DELRT, mhinfo[mh].ipaddr, &temp);
	lowifacereq(DOWN, mhinfo[mh].tunnelnum, 0, 0);
	mhinfo[mh].tunnelnum = 0;
	agentstats.successfulreg++;
	sendConfirm(fromport, code, 0, reg->homeaddr, reg->coaddr, haAddr, mh);
      } else if (mhinfo[mh].status & TUNNELUP)  {
	mhinfo[mh].timeleft = (reg->lifetime > MAXLIFETIME) ?
	      MAXLIFETIME : (int) reg->lifetime;
	code = 1;
	if (reg->coaddr == mhinfo[mh].coaddr)  {
	   /* we have a renewal */
	   agentstats.successfulreg++;
	   sendConfirm(fromport, code, mhinfo[mh].timeleft, reg->homeaddr,
		       reg->coaddr, haAddr, mh);
	} else  {
	   /* the tunnel needs to be redirected */
	   mhinfo[mh].coaddr = reg->coaddr;
	   lowifacereq(REDIRECT, mhinfo[mh].tunnelnum, 0, reg->coaddr);
	   agentstats.successfulreg++;
	   sendConfirm(fromport, code, mhinfo[mh].timeleft, reg->homeaddr,
		       reg->coaddr, haAddr, mh);
	}
     } else { /* A tunnel does not already exist */ 
	mhinfo[mh].timeleft = (reg->lifetime > MAXLIFETIME) ?
	      MAXLIFETIME : (int) reg->lifetime;
	code = 1;
	if (mhinfo[mh].tunnelnum = newtunnel())  {
	   /* we were able to find an unused tunnel name */
	   mhinfo[mh].status |= TUNNELUP;
	   mhinfo[mh].coaddr = reg->coaddr;
	   if (haWiredAddr == 0)
	     lowifacereq(MKPT2PT, mhinfo[mh].tunnelnum, haAddr, reg->coaddr);
	   else
	     lowifacereq(MKPT2PT, mhinfo[mh].tunnelnum, haWiredAddr, reg->coaddr);
	   lowrtreq(ADDRT, mhinfo[mh].ipaddr, &(mhinfo[mh].tunnelnum));
	   lowARPreq(PROXY, mhinfo[mh].ipaddr, proxyHwAddr);
	   agentstats.successfulreg++;
	   sendConfirm(fromport, code, mhinfo[mh].timeleft, reg->homeaddr,
		       reg->coaddr, haAddr, mh);
	} else  {
	   code = 34; /* insufficient resources */
	   agentstats.noresources++;
	   sendRefusal(fromport, code, 0, reg->homeaddr, reg->coaddr, 
		       haAddr, mh);
	   if (debug > 2)
	     fprintf(stderr, "Refusal sent due to insuff resources.\n");
	}
     }

}

void 
main(int argc, char **argv)  {
   struct sockaddr_in from;
   static char inmsg[1024];
   char *cmd;
   int len, fromlen, c;
   extern char *optarg;
   extern int optind;
   unsigned char argsfound = 0;
   fd_set fdvec;
   struct timeval tv;
   
   cmd = argv[0];
   while ((c = getopt(argc, argv, "a:m:h:w:")) != -1)  {
      switch ((char) c)  {
       case 'a':
	 argsfound |= 0x01;
	 if ((haAddr = inet_addr(optarg)) == -1)  {
	    fprintf(stderr, "Bad address passed to -a.\n");
	    exit(-1);
	 };
	 break;
       case 'm':
	 argsfound |= 0x02;
	 if ((haNetmask = inet_addr(optarg)) == -1)  {
	    fprintf(stderr, "Bad netmask passed to -m.\n");
	    exit(-1);
	 };
	 break;
       case 'h':
	 argsfound |= 0x04;
	 if (hwaddread(proxyHwAddr,optarg) == -1)  {
	    fprintf(stderr, "Bad hwaddr passed to -h.\n");
	    exit(-1);
	 };
	 break;
       case 'w': /* address on a wired interface */ 
	 if ((haWiredAddr = inet_addr(optarg)) == -1)  {
	    fprintf(stderr, "Bad address passed to -w.\n");
	    exit(-1);
	 };
	 break;
       case '?':
	 usage(cmd);
	 exit(1);
	 break;
      }
   }
   if (argsfound != 0x07)  {
      usage(cmd);
      exit(1);
   }
    
   haAdvertAddr=(haAddr&haNetmask)|~haNetmask;

   if (debug > 1)  {
      fprintf(stderr, "IPaddr: %x, Mask: %x, Wired: %x ",
	      htonl(haAddr), htonl(haNetmask), htonl(haWiredAddr));
      fprintf(stderr, "HWaddr: %x:%x:%x:%x:%x:%x\n", 
	      (unsigned char) proxyHwAddr[0],
	      (unsigned char) proxyHwAddr[1],
	      (unsigned char) proxyHwAddr[2],
	      (unsigned char) proxyHwAddr[3],
	      (unsigned char) proxyHwAddr[4],
	      (unsigned char) proxyHwAddr[5]);
   }

   /* Make sure we ARE on the network we are supposed to act as an 
    * agent for and we do have tunnel support in /proc/net/dev
    * else complain and exit. Also make sure we have our effective
    * uid set to root since we will be manipulating routing tables and such.
    * We currently do not perform this check */
   
   init();
   while (1)  {
      
      FD_ZERO(&fdvec); FD_SET(WhereAmIsid, &fdvec); 
      FD_SET(RegisterMesid, &fdvec); tv.tv_sec = 0; tv.tv_usec = 0;
      if (select(16, &fdvec, NULL, NULL, &tv) < 0)  {
	 perror("select() failed in main loop.\n");
	 cleanup(); exit(-1);
      }
      /* listen for a WhereAmI message on WhereAmIsid without 
       * blocking and if there is a message, process it */
      len = 1024; fromlen = sizeof(struct sockaddr_in);
      if (FD_ISSET(WhereAmIsid, &fdvec))  {
	 if (debug > 2) 
	   fprintf(stderr, "Data on WhereAmIsid.\n");
	 if ((len = recvfrom(WhereAmIsid, inmsg, len, 0,
			     (struct sockaddr *) &from, &fromlen)) < 0)  {
	   perror("recvfrom() on WhereAmIsid failed.\n");
 	   cleanup(); exit(-1);
	 };
	 processWhereAmI(inmsg, len, &from);
      }
      len = 1024; fromlen = sizeof(struct sockaddr_in);
      if (FD_ISSET(RegisterMesid, &fdvec))  {
	 if (debug > 2) 
	   fprintf(stderr, "Data on RegisterMesid.\n");
	 if ((len = recvfrom(RegisterMesid, inmsg, len, 0,
			     (struct sockaddr *) &from, &fromlen)) < 0)  {
	   perror("recvfrom() on RegisterMesid failed.\n");
 	   cleanup(); exit(-1);
	 };
	 processRegisterMe(inmsg, len, &from);
	 if (debug > 2) 
	   fprintf(stderr, "In main finished processing RegisterMe.\n");
	 
      }
   }
}

void
usage(char *cmd)  {
   fprintf(stderr, 
	   "Usage is: %s -a ipaddr -m netmask -h hwaddr [-w wiredaddr]\n",
	   cmd);
}

int 
hwaddread(char s[], char *hwadr)  {
   int a, b, c, d, e, f;
   if (sscanf(hwadr,"%x:%x:%x:%x:%x:%x", &a, &b, &c, &d, &e, &f) != 6) 
     return(-1);
   else  {
      s[0] = (unsigned char) a;
      s[1] = (unsigned char) b;
      s[2] = (unsigned char) c;
      s[3] = (unsigned char) d;
      s[4] = (unsigned char) e;
      s[5] = (unsigned char) f;
     return(0);
   }
}
 
void
sendReply(unsigned short fromport, int code, int time, 
	  unsigned long homeaddr, unsigned long coaddr, 
	  unsigned long haAddr, struct id replyid)  
{
   struct sockaddr_in sock;
   static char outmsg[1024];
   struct regreply *reply;
   int len=sizeof(struct regreply);
   int mh;

   sock.sin_family = AF_INET;
   sock.sin_addr.s_addr = homeaddr;
   sock.sin_port = fromport;
   reply = (struct regreply *) outmsg;
   reply->type = 3;
   reply->code = (unsigned char) code;
   reply->lifetime = (unsigned short) time;
   reply->homeaddr = homeaddr;
   reply->ha = haAddr;
   reply->Id.high = replyid.high;
   reply->Id.low = replyid.low;

   mh = whichmh(homeaddr);
   appendauth(outmsg, &len, mhinfo[mh].secret, mhinfo[mh].keylen);

   if (debug > 0)  {
      fprintf(stderr, "\n==========================\n");
      printtime();
      fprintf(stderr, "-- REPLY to %s at port %d ",
	      inet_ntoa(sock.sin_addr), sock.sin_port);
      if ((reply->code != 0) && (reply->code != 1))
      	fprintf(stderr, "(Rejection)\n");
      else
      	fprintf(stderr, "(Acceptance)\n");
      
      if (debug > 1)  {
	 fprintf(stderr, "[%8lx:%8lx] Type %2d Code %3d Lifetime %8d\n",
		 reply->Id.high, reply->Id.low, reply->type, reply->code,
		 reply->lifetime);
	 fprintf(stderr, "Homeaddr: %8lx, Homeagent: %8lx",
		 htonl(reply->homeaddr), htonl(reply->ha));
         printext(outmsg, len, 1);
	 if (debug > 2) {
	   fprintf(stderr, "\n--------------------\n");
	   testprint(outmsg, len);
	   fprintf(stderr, "--------------------");
	 }
      }
      fprintf(stderr, "\n==========================\n");
   }
   
   if (sendto(Regreplysid, (char *) outmsg, len, 0,
	      (struct sockaddr *) &sock, sizeof(struct sockaddr)) < 0)  {
		 perror("sendReply(): send to failed.\n");
		 cleanup(); exit(-1);
	      }
}

void
sendConfirm(unsigned short fromport, int code, int time, 
	    unsigned long homeaddr, unsigned long coaddr,
	    unsigned long haAddr, int mh)  {
  sendReply(fromport, code, time, homeaddr, coaddr, haAddr, 
	    mhinfo[mh].RegistrationId);	       
}

void
delay()  {
   int i; /* artificial delay in an attempt to ensure that packet is
	   * sent before we change the kernel routing tables again */
   
   for (i = 0; i <= 100000; i++) ;
}

void
sendRefusal(unsigned short fromport, int code, int time, 
	    unsigned long homeaddr, unsigned long coaddr,
	    unsigned long haAddr, int mh)  {

   char newtunl = 0;

   if (coaddr == homeaddr)  {
      if ((mhinfo[mh].status & TUNNELUP) == 0)  {
	 sendReply(fromport, code, time, homeaddr, coaddr, 
		   haAddr, mhinfo[mh].RegistrationId);
      } else  {
	 lowrtreq(DELRT, mhinfo[mh].ipaddr, &(mhinfo[mh].tunnelnum));
	 sendReply(fromport, code, time, homeaddr, coaddr, 
		   haAddr, mhinfo[mh].RegistrationId);
	 delay();
	 lowrtreq(ADDRT, mhinfo[mh].ipaddr, &(mhinfo[mh].tunnelnum));
      }
   } else  { /* coaddr != homeaddr */
      if (havetunnel(mh, coaddr))  {
	 sendReply(fromport, code, time, homeaddr, coaddr, 
		   haAddr, mhinfo[mh].RegistrationId);
      } else  {
	 if (mhinfo[mh].tunnelnum != 0)
	   lowrtreq(DELRT, mhinfo[mh].ipaddr, &(mhinfo[mh].tunnelnum));
	 /* we delete existing route and create a new one through tunl0 */ 
	 if (haWiredAddr == 0)
	   lowifacereq(MKPT2PT, (char) 0, haAddr, coaddr);
	 else
	   lowifacereq(MKPT2PT, (char) 0, haWiredAddr, coaddr);
	 lowrtreq(ADDRT, homeaddr, &newtunl);
	 sendReply(fromport, code, time, homeaddr, coaddr, 
		   haAddr, mhinfo[mh].RegistrationId);
	 delay();
	 lowrtreq(DELRT, homeaddr, &newtunl);
	 lowifacereq(DOWN, (char) 0, (unsigned long) 0, (unsigned long) 0);
	 /* restore the previous route */
	 if (mhinfo[mh].tunnelnum != 0)
	   lowrtreq(ADDRT, mhinfo[mh].ipaddr, &(mhinfo[mh].tunnelnum));
      }
   }
}

