/*
 * Linux-Mobile-IP
 *            An implementation of Mobile IP for the LINUX operating system
 *            developed at the State University of New York, Binghamton
 *            (with support from the Center for Computing Technologies).
 *            The implementation uses the message formats described in the
 *            Internet Engineering Task Force (IETF) mobile-ip draft
 *            but is not (yet) fully compliant.
 *
 *            Recent drafts of the IETF Mobile IP proposal are
 *            available at ftp://software.watson.ibm.com/pub/mobile-ip/ 
 *
 *            Permission is hereby granted to redistribute this code and/or
 *            modify it under the terms of the GNU Genral Public License
 *            as published by the Free Software Foundation; either version
 *            two or (at your option) any later version.
 *
 *            The authors request that their contribution be appropriately 
 *            acknowledged in any derived work. 
 *
 *            fileio.c - Routines to read initialization file and read/write
 *                       recovery log for Agent.
 * 
 * Version:   0.90     08/31/1995
 * 
 * Authors:   Vipul Gupta <vgupta@cs.binghamton.edu>
 *            Benjamin Lancki <ben@anchor.cs.binghamton.edu>
 * 
 * */  

#include <stdio.h>
#include <sys/time.h>
#include <errno.h>
#include "agent.h"
#include "low.h"

extern unsigned long tunnelbitvec;
extern unsigned long haAddr;
extern char supportedMHnum;
extern mhdata mhinfo[];
extern int debug;
extern char proxyHwAddr[7];

void 
readmhdata(char *file)  {
  int i,j;
  char stringbuf[256];
  FILE *fileptr;

  /* initialize values for dummy mh */
  mhinfo[0].ipaddr = mhinfo[0].coaddr = 0;
  mhinfo[0].status = 0; mhinfo[0].timeleft = 0;
  mhinfo[0].tunnelnum = 0;
  strcpy(mhinfo[0].authtype, "md5");
  strcpy(mhinfo[0].secret, "                ");
  mhinfo[0].keylen=16;

  /* open config file for supported mobile hosts */
  if((fileptr=fopen(file,"r"))==NULL) {
    printf("\nError opening configuration file: %s\n",file);
    cleanup();
    exit(-1);
  }

  if(debug>0) {
    printtime();
    fprintf(stderr,"Reading configuration file...\n"); 
  }

  /* read number of supported hosts first */
  if(fscanf(fileptr,"%d",&supportedMHnum)!=1) {
    printf("\nError reading configuration file: %s\n",file);
    cleanup();
    exit(-1);
  }

  /* read and initialize values for each mobile host in init file */
  for(j=1;j<=supportedMHnum;j++) {
    fscanf(fileptr,"%s%s%d",stringbuf,mhinfo[j].authtype,&(mhinfo[j].keylen));
    for(i=0;i<mhinfo[j].keylen;i++)
      fscanf(fileptr,"%x",&(mhinfo[j].secret[i]));
    mhinfo[j].ipaddr=inet_addr(stringbuf);

    if(debug>1)  {
      fprintf(stderr,"\naddr:%lx authtype:%s keylength:%d key: ",
	      ntohl(mhinfo[j].ipaddr),mhinfo[j].authtype,mhinfo[j].keylen);
      for(i=0;i<mhinfo[j].keylen;i++)
        fprintf(stderr,"%x ",mhinfo[j].secret[i]);
      fprintf(stderr,"\n");
    }

    mhinfo[j].status = 0;
    mhinfo[j].timeleft = 0;
    mhinfo[j].tunnelnum = 0;
    mhinfo[j].coaddr = 0;
    newId(&(mhinfo[j].RegistrationId), &(mhinfo[j].RegistrationId));
  }

  /* close config file */
  if(fclose(fileptr)!=0) {
    printf("\nError closing configuration file: %s\n",file);
    cleanup();
    exit(-1);
  }
}

void
readlog(char *file)  {
  int i;
  FILE *fileptr;
  struct timeval clock;
  struct timezone zone;
  long int time_elapsed;

  /* open log file for writing */
  if((fileptr=fopen(file,"r"))==NULL) {
    if(errno!=2) {
      printf("\nError opening log file for reading: %s\n",file);
      cleanup();
      exit(-1);
    }
  }

  /* if file exists, then process it, otherwise, this is the first time
     the agent has been executed */
  if(fileptr!=NULL) {
    /* figure out how much time has passed since system went down */
    gettimeofday(&clock,&zone);    
    fscanf(fileptr,"%lx",&time_elapsed);
    time_elapsed=clock.tv_sec-time_elapsed;

    /* read tunnel routing datafor each mh, if elapsed time has exceeded 
       timeout for mh, return it to init values */
    for(i=1;i<=supportedMHnum;i++) {
      fscanf(fileptr,"%x %x %x %lx %lx %lx\n",&(mhinfo[i].status),
        &(mhinfo[i].timeleft),&(mhinfo[i].tunnelnum),&(mhinfo[i].coaddr),
        &(mhinfo[i].RegistrationId.high),&(mhinfo[i].RegistrationId.low));
      if(time_elapsed>=mhinfo[i].timeleft) {
        mhinfo[i].status = 0;
        mhinfo[i].timeleft = 0;
        mhinfo[i].tunnelnum = 0;
        mhinfo[i].coaddr = 0;
      }
      else {
        mhinfo[i].timeleft-= (int) time_elapsed;
	tunnelbitvec |= (1 << mhinfo[i].tunnelnum);
        lowifacereq(MKPT2PT,mhinfo[i].tunnelnum,haAddr,mhinfo[i].coaddr);
	lowrtreq(ADDRT, mhinfo[i].ipaddr, &(mhinfo[i].tunnelnum));
	lowARPreq(PROXY, mhinfo[i].ipaddr, proxyHwAddr);
      }

      if(debug>0) {
        printtime();
        fprintf(stderr,"Reading recovery log...\n");
        if(debug>1)
          fprintf(stderr,
		  "status:%x time:%x tunnel:%x coaddr:%lx id:[%8lx:%8lx]\n",
		  mhinfo[i].status, mhinfo[i].timeleft, mhinfo[i].tunnelnum, 
		  htonl(mhinfo[i].coaddr), mhinfo[i].RegistrationId.high,
                  mhinfo[i].RegistrationId.low);
      }         
    }

    /* close config file */
    if(fclose(fileptr)!=0) {
      printf("\nError closing log file: %s\n",file);
      cleanup();
      exit(-1);
    }
  }
}

void
writelog(char *file)  {
  int i;
  FILE *fileptr;
  struct timeval clock;
  struct timezone zone;

  /* open log file for writing */
  if((fileptr=fopen(file,"w"))==NULL) {
    printf("\nError opening log file for writing: %s\n",file);
    cleanup();
    exit(-1);
  }
 
  /* place timestamp at beginning of file */
  gettimeofday(&clock,&zone);
  fprintf(fileptr,"%lx\n",clock.tv_sec);

  /* write tunnel routing and timeout info for each mh */
  for(i=1;i<=supportedMHnum;i++)
    fprintf(fileptr,"%x %x %x %lx %lx %lx\n",mhinfo[i].status,
      mhinfo[i].timeleft,
      mhinfo[i].tunnelnum,mhinfo[i].coaddr,mhinfo[i].RegistrationId.high,
      mhinfo[i].RegistrationId.low);
 
  if(debug>0) {
    printtime();
    fprintf(stderr,"Writing recovery log...\n");
     if(debug>1)
       fprintf(stderr,
	       "status:%x time:%x tunnel:%x coaddr:%lx id:[%8lx:%8lx]\n",
	       mhinfo[i].status, mhinfo[i].timeleft, mhinfo[i].tunnelnum, 
	       htonl(mhinfo[i].coaddr), mhinfo[i].RegistrationId.high,
               mhinfo[i].RegistrationId.low);
  }         

  /* close config file */
  if(fclose(fileptr)!=0) {
    printf("\nError closing log file: %s\n",file);
    cleanup();
    exit(-1);
  }
}



