/*
 * Linux-Mobile-IP
 *            An implementation of Mobile IP for the LINUX operating system
 *            developed at the State University of New York, Binghamton
 *            (with support from the Center for Computing Technologies).
 *            The implementation uses the message formats described in the
 *            Internet Engineering Task Force (IETF) mobile-ip draft
 *            but is not (yet) fully compliant.
 *
 *            Recent drafts of the IETF Mobile IP proposal are
 *            available at ftp://software.watson.ibm.com/pub/mobile-ip/ 
 *
 *            Permission is hereby granted to redistribute this code and/or
 *            modify it under the terms of the GNU Genral Public License
 *            as published by the Free Software Foundation; either version
 *            two or (at your option) any later version.
 *
 *            The authors request that their contribution be appropriately 
 *            acknowledged in any derived work. 
 *
 *            mh.c - Design of the Mobile Host.
 * 
 * Version:   0.90     08/31/1995
 * 
 * Authors:   Vipul Gupta <vgupta@cs.binghamton.edu>
 *            Benjamin Lancki <ben@anchor.cs.binghamton.edu>
 * 
 * */  

/*
 *  Command line arguments:
 *    -a : mobile host's IP address on its home network 
 *    -m : network mask corresponding to the IP address above
 *    -g : IP address on the mobility interface of the homeagent.
 *    -o : The home agent's IP address on a wired interface (this is useful
 *         in a setting where all mobility agents and mobile hosts have
 *         been assigned IP addresses on a common subnet even though
 *         they may not all have link level connectivity -- similar
 *         to the intracampus addressing scheme used in the Columbia
 *         proposal)
 *
 *
 * To do:  
 *   1. Currently supports only one home agent and assumes the mobile has only
 *      one network interface which is named eth0.
 *
 *   2. The part where an mh is registered from a foreign network and
 *       does its own decapsulation is currently untested.
 *
 * History of Changes:
 *     Jul 25, 1995              First version coded  - V.G.
 *     Aug  1, 1995              Removed bug in processURhere. Now we
 *                               turn off REGAWAY when we move home and
 *                               turn off REGHOME when we move away - V.G.
 *     Aug  2, 1995              Moved URhere gathering in alarm_handler()
 *                               as part of preventing a flip-flop effect
 *                               when a mobile host hears URhere messages
 *                               from multiple mobility agents 
 *                               alternately - V.G.
 *     Aug  9, 1995              Added support for MD5 authentication in
 *                               processing the registration and registration 
 *                               reply - B.L.
 *     Aug 13, 1995              Agent now reads config file /etc/mip-mh.ok to
 *                               initialize mhinfo struct - B.L.
 *     Aug 14, 1995              Agent keeps a log in event of system crash
 *                               or shutdown and recovers on startup - B.L.
 * */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include "mh.h"

int debug = 2;

extern void 
testprint(char *,int);

extern void 
appendauth(char *,int *,char *,int);

extern int 
authextfound(char *,int,int);

extern int 
authok(char *,int,int,char *,int);

extern void 
printext(char *,int,int);

extern void 
printtime();

extern unsigned long getipaddr(unsigned long *);
extern unsigned long getnetmask(unsigned long *);

void sendRegisterMe(int, unsigned char, unsigned short, unsigned long,
		    unsigned long, unsigned long);
int WhereAmIsid, URheresid, Regreplysid, RegisterMesid;
int ioctlsid;

void usage(char *), cleanup();

struct sockaddr_in Whereamito =
     {  (short) AF_INET, (unsigned short) WHEREAMIPORT,
	(unsigned long) 0, (unsigned long) 0 };

char lastreq[1024]; 
/* this is the last request on which we are awaiting a reply */

myinfo Myinfo;

unsigned long mhAdvertAddr;

struct stats  {
   unsigned long badurhere;
   unsigned long badregreply;
   unsigned long idmismatch;
   unsigned long authfailed;
   unsigned long regsent;
   unsigned long regfailed;
   unsigned long regsuccessful;
} mhstats =  { (long) 0, (long) 0,  (long) 0, (long) 0,
               (long) 0, (long) 0,  (long) 0 };

void
initsockets()  {
   struct sockaddr_in sa;
   int enable = 1;
   
   WhereAmIsid = socket(AF_INET, SOCK_DGRAM, 0);
   Regreplysid = RegisterMesid = socket(AF_INET, SOCK_DGRAM, 0);
   URheresid = socket(AF_INET, SOCK_DGRAM, 0);
   ioctlsid = socket(AF_INET, SOCK_DGRAM, 0);
   if (Regreplysid < 0 || URheresid < 0 || 
       WhereAmIsid < 0 || RegisterMesid < 0 || ioctlsid < 0)  {
	  fprintf(stderr, "initsockets(): could not create sockets.\n");
	   exit(-1);
       }
   sa.sin_family = AF_INET; sa.sin_addr.s_addr = htonl(INADDR_ANY); 
   sa.sin_port = YOUAREHEREPORT; 
   if (bind(URheresid, (struct sockaddr *) &sa, sizeof(sa)) < 0)  {
      perror("initsockets(): Bind URheresid failed.\n");
      cleanup(); exit(-1);
   }

   sa.sin_port = 0; /* can send whereamis from any port */
   if (bind(WhereAmIsid, (struct sockaddr *) &sa, sizeof(sa)) < 0)  {
      perror("initsockets(): Bind WhereAmIsid failed.\n");
      cleanup(); exit(-1);
   }
   if (setsockopt(WhereAmIsid, SOL_SOCKET, SO_BROADCAST, &enable, sizeof(int))
       < 0)  {
	  perror("initsockets(): setsockopt SO_BROADCAST failed");
	  cleanup(); exit(-1);
       }
   /* This is where we receive regreplies and send registermes from */
   sa.sin_port = 0; 
   if (bind(RegisterMesid, (struct sockaddr *) &sa, sizeof(sa)) < 0)  {
      perror("initsockets(): Bind RegisterMesid failed.\n");
      cleanup(); exit(-1);
   }
   if (debug > 0)
     fprintf(stderr, "Initialized sockets ...\n");
}

void
cleanup()  {
   lowifacedown("dummy0"); /* if dummy0 is in use down it */
   Myinfo.status &= ~DUMMYUP;
   if (debug > 1)
     fprintf(stderr, "Done cleaning up ... \n");
}

void
graceful_exit()  {
   struct sockaddr_in sock;
   unsigned char flags = 0x00;
   
   cleanup();
   if (Myinfo.status & REGAWAY)  {
      Myinfo.retries = 0;
      sendRegisterMe((int) 0, flags, (unsigned short) 0, Myinfo.homeaddr, 
		     Myinfo.homeagent, Myinfo.coaddr);
   }
   /* deregister ourselves if we were registered or this will
    * tie up a tunnel for atmost MAXLIFETIME period */ 
   exit(0);
}

void
sendWhereAmI(struct sockaddr_in *toaddr)  {
   struct whereami msg;
   
   if (debug > 0)
     fprintf(stderr, "\n-- WHEREAMI to %s at port %d.\n",
	     inet_ntoa(toaddr->sin_addr), (unsigned int) toaddr->sin_port);
   msg.code = WHEREAMITYPE;
   if (sendto(WhereAmIsid, (char *) &msg, sizeof(msg), 0, 
	      (struct sockaddr *) toaddr, sizeof(struct sockaddr)) < 0)  {
		 perror("sendWhereAmI(): send to failed.\n");
		 cleanup(); exit(-1);
	      }
}

void
sendRegisterMe(int attempt, unsigned char flags, unsigned short lifetime, 
	       unsigned long homeaddr, unsigned long homeagent,
	       unsigned	long coaddr)
{
   struct sockaddr_in toaddr;
   struct registerme *req;
   int len=sizeof(struct registerme);   

   toaddr.sin_family = AF_INET;
   toaddr.sin_port = MIPREGPORT;
/*   toaddr.sin_addr.s_addr = htonl(homeagent); */
    toaddr.sin_addr.s_addr = homeagent; 
   if ((Myinfo.closeagentaddr != homeagent) && 
       ((Myinfo.homenetmask & (Myinfo.homeagent ^ Myinfo.closeagentaddr)) == 0))
     {
	/* we have a Columbia style campus arrangement so we cannot 
	 * directly use the homeagent address, we need its other address */
	if (Myinfo.o_homeagent)
/*	  toaddr.sin_addr.s_addr = htonl(Myinfo.o_homeagent); */
	  toaddr.sin_addr.s_addr = Myinfo.o_homeagent;
	else
	  fprintf(stderr, "Potential problems while preparing toaddr.\n");
     }
   req = (struct registerme *) lastreq;
   req->type = REGISTERMETYPE;
   req->flags = flags;
   req->lifetime = lifetime;
   req->homeaddr = homeaddr;
   req->ha = homeagent;
   req->coaddr = coaddr;
   /* we generate a new 64-bit Id unless this is a retransmission */
   if (attempt == 0)
     NewId(&Myinfo.RequestId, &Myinfo.RequestId);
   req->Id.high = Myinfo.RequestId.high;
   req->Id.low = Myinfo.RequestId.low;

   appendauth(lastreq, &len, Myinfo.secret, Myinfo.secretlen);

   if (debug > 0)  {
      fprintf(stderr, "\n==========================\n");
      printtime();
      if (req->lifetime == 0) 
      	fprintf(stderr, "-- DEREGISTERME to %s at port %d (attempt %d).\n",
		inet_ntoa(toaddr.sin_addr), (unsigned int) toaddr.sin_port,
		attempt);
      else
      	fprintf(stderr, "-- REGISTERME to %s at port %d (attempt %d).\n",
		inet_ntoa(toaddr.sin_addr), (unsigned int) toaddr.sin_port,
		attempt);
      if (debug > 1)  {
	 fprintf(stderr, "[%8lx:%8lx] Type %2d Flags %2x Lifetime %8d\n",
	      req->Id.high, req->Id.low, req->type, req->flags, req->lifetime);
	 fprintf(stderr, "Homeaddr: %8lx, Homeagent: %8lx, Careof: %8lx",
	      htonl(req->homeaddr), htonl(req->ha), htonl(req->coaddr));
	 printext(lastreq, len, 0);
	 if (debug > 2) {
	   fprintf(stderr, "\n--------------------\n");
	   testprint(lastreq, len);
	   fprintf(stderr, "--------------------");
	 }
      }
      fprintf(stderr, "\n==========================\n");
   }
   
   Myinfo.retries = attempt;
   Myinfo.rextimeleft = (1 << attempt) * RETRYINTERVAL;
   if (sendto(RegisterMesid, (char *) lastreq, len, 0, 
	      (struct sockaddr *) &toaddr, sizeof(struct sockaddr)) < 0)  {
		 perror("sendRegisterMe(): send to failed.\n");
		 cleanup(); exit(-1);
	      };
   Myinfo.status |= AWAITINGREPLY;
}

void
pickupURheres(int sockid)  {
   /* Here we shall try to pick up all URheres received since
    * the last call to this procedure. If we get more than one, we try to
    * pick one that will prevent a flip-flop behavior */
   fd_set fdvec;
   struct timeval tv;
   int i, len, fromlen, foundURhere;
   static char incoming[256];
   static struct sockaddr_in fromsock;
   static struct youarehere *yah;
   unsigned long *addr;
   char *p;
   int ready;
   
   foundURhere = 0; ready = 0;
   for (i = 0; i < MAXOVERLAP; i++)  {
      FD_ZERO(&fdvec); FD_SET(sockid, &fdvec); 
      tv.tv_sec = 0; tv.tv_usec = 0;
      if (select(16, &fdvec, NULL, NULL, &tv) < 0)  {
	 perror("select() failed in pickupURhere(0.\n");
	 cleanup(); exit(-1);
      }
      if (FD_ISSET(sockid, &fdvec) == 0) continue;
      len = 256; fromlen = sizeof(struct sockaddr_in);
      if ((len = recvfrom(sockid, incoming, len, 0,
			  (struct sockaddr *) &fromsock, 
			  &fromlen)) < 0)  {
           perror("recvfrom() failed in pickupURheres.\n");
 	   cleanup(); exit(-1);
	 };
      yah = (struct youarehere *) incoming;
      if (yah->code != YOUAREHERETYPE || yah->addrnum < 1 ||
	  len != yah->addrnum*2*sizeof(long) + sizeof(struct youarehere))  {
      	mhstats.badurhere++;
	continue;
      }
      if (Myinfo.haveUR != 0)  {
	 /* previous URhere has not been processed so we pick it up from
	  * the socket but then simply discard it. In debug mode we print
	  * a * instead of . to indicate this discarding */
	 if (debug > 0) fprintf(stderr, "*");
	 continue;
      } else  {
	 foundURhere++;
	 if (debug > 0) fprintf(stderr, ".");
	 p = incoming + sizeof(struct youarehere);
	 addr = (unsigned long *) p;
	 if (foundURhere == 1 || ((*addr != (unsigned long) 0) &&
				  (*addr == Myinfo.closeagentaddr)))  {
	  /* we always pick the first URhere but if later we find another
	   * that matches our current state, we prefer that and overwrite
	   * the first one */
	     memcpy(Myinfo.URheremsg, incoming, len);
   	     Myinfo.URherelen = len;
             memcpy(&Myinfo.URheresock, &fromsock, sizeof(struct sockaddr_in));
	  }
      }
   }
   /* We wait till we have finished looping before setting Myinfo.haveUR
    * in order to avoid a potential race situation with the main loop */ 
   if (foundURhere) 
     Myinfo.haveUR = 1;
   else if (debug > 0) fprintf(stderr, "?");
   /* we print a "?" to indicate we did not hear any URhere messages */
}

void
alarm_handler()  {
   struct registerme *req;
   unsigned char flags = 0x00;

   /* we look for "you are here" messages */
   pickupURheres(URheresid);
   
   /* we take care of our registration renewals */
   if (Myinfo.status & REGAWAY || Myinfo.status & REGHOME)  {
      --Myinfo.regtimeleft;
      if (Myinfo.regtimeleft == THRESHOLD1 ||
	  Myinfo.regtimeleft == THRESHOLD2 ||
	  Myinfo.regtimeleft == THRESHOLD3)  {
	     /* transmit a new registration request here */
	     /* clear out the outstanding request and reset 
	      * the timer rextimeleft , reset retries */
	     Myinfo.retries = 0;
	     sendRegisterMe((int) 0, flags, 
			    (Myinfo.status & REGAWAY) ? MAXLIFETIME : 0, 
			    Myinfo.homeaddr, 
			    Myinfo.homeagent, Myinfo.coaddr);
	  }
      else if (Myinfo.regtimeleft <= 0)  {
	 Myinfo.regtimeleft = 0;
	 if (Myinfo.status & DUMMYUP) lowifacedown("dummy0");
	 NewId(&Myinfo.RequestId, &Myinfo.RequestId);
	 Myinfo.status &= ~DUMMYUP; 
	 Myinfo.status &= ~AWAITINGREPLY; 
	 Myinfo.status &= ~REGHOME;
	 Myinfo.status &= ~REGAWAY;
	 Myinfo.closeagentaddr = 0;
	 Myinfo.itsotheraddr = 0;
	 if (debug > 1)  {
	    fprintf(stderr, "Registration expired ... \n");
	 }
	 /* Put ourselves in the state when we just started out */
      }
   }

   /* and finally we take care of an request retransmissions */
   if (Myinfo.status & AWAITINGREPLY)  {
      if (--Myinfo.rextimeleft < 0)  {
	 Myinfo.rextimeleft = 0;
	 if (++Myinfo.retries < MAXRETRIES)  {
	    req = (struct registerme *) lastreq;
	    /* resend request which will again restore rextimeleft */
	    sendRegisterMe(Myinfo.retries, req->flags, req->lifetime, 
			   req->homeaddr, req->ha, req->coaddr);
	 } else {
	    /* We have retransmitted this request enough times and
	     * we now wait till a move generates a new request 
	     * and expect an Id that we never sent out so all replies
	     * will be rejected till we first send out a request */
	    Myinfo.status &= ~AWAITINGREPLY;
	    NewId(&Myinfo.RequestId, &Myinfo.RequestId);
	    if (debug > 1)  {
	       fprintf(stderr, "We have retransmitted enough ... \n");
	    }
	 }
      }
   }
   signal(SIGALRM, alarm_handler);
   alarm(1);
}

void
init()  {
   Myinfo.coaddr = Myinfo.closeagentaddr = Myinfo.itsotheraddr = 0;
   Myinfo.rextimeleft = Myinfo.regtimeleft = 0;
   Myinfo.status = 0;
   Myinfo.haveUR = 0;
   Myinfo.retries = 0;
   NewId(&Myinfo.RequestId, &Myinfo.RequestId);
   initsockets();
   cleanup();  /* basically undo stuff e.g. if dummy0 was in use stop that */
   Whereamito.sin_addr.s_addr = inet_addr(ADVERTISEON);
   Whereamito.sin_addr.s_addr = mhAdvertAddr;
   Whereamito.sin_port = WHEREAMIPORT;
   readhadata("/etc/mip-ha.dat");
   signal(SIGTERM, graceful_exit);
   signal(SIGINT, graceful_exit);
   signal(SIGALRM, alarm_handler);
   alarm(1);
}
   
int
IsOurAddress(unsigned long addr)  {
   return((Myinfo.homeaddr == addr) ? 1:0);
}

int
IsOurAgent(unsigned long addr)  {
   return((Myinfo.homeagent == addr) ? 1:0);
}

int
mhIdcmp(struct id *id1, struct id *id2)  {
   return((id1->low == id2->low) ? 0:1);
}

int
NewId(struct id *id1, struct id *id2)  {
   static count = 0;
   struct timeval tv;
   
   if (count++ == 0)  {
      gettimeofday(&tv, NULL);
      srandom((int) tv.tv_usec);
   };
   id1->high = id2->high;
   id1->low = (unsigned long) random();
}

void
processURhere(char *msg, int len, struct sockaddr_in *from)  {
   struct youarehere *urh;
   unsigned long *addr;
   char *p;
   int i;
   
   if (debug > 2) 
     fprintf(stderr, "Processing a URhere from %s\n",
	     inet_ntoa(from->sin_addr));

   urh = (struct youarehere *) msg;
   if (urh->code != YOUAREHERETYPE || len != urh->addrnum*2*sizeof(long) +
       sizeof(struct youarehere) || urh->addrnum < 1)  {
	  mhstats.badurhere++;
	  return;
       };
   /* we have a good URhere message */
   p = msg + sizeof(struct youarehere);
   addr = (unsigned long *) p;

   if (debug > 2)  {
      fprintf(stderr, "Myinfo.homeagent = %lx\n", htonl(Myinfo.homeagent));
      fprintf(stderr, "Myinfo.homenetmask = %lx\n", htonl(Myinfo.homenetmask));
      fprintf(stderr, "Myinfo.closeagent = %lx\n", htonl(Myinfo.closeagentaddr));
      fprintf(stderr, "URhere addrnum = %d, flags = %x\n",
	      urh->addrnum, urh->flags);
      for (i = 0; i < urh->addrnum; i++)
      	fprintf(stderr, "Address %lx Mask %lx\n", htonl(*(addr + 2*i)),
		htonl(*(addr + 2*i + 1)));
   }

   if (((Myinfo.status & REGHOME) || (Myinfo.status & REGAWAY)) &&
       (Myinfo.closeagentaddr == *addr))  {
	  return; /* No action is required ... we are already registered
		   * here */
       };
   
   Myinfo.closeagentaddr = *addr;

   if (*addr == Myinfo.homeagent)  {
      /* we are home and we should unregister. However, if we are already
       * waiting to hear from a previous unregister request, we need not
       * do anything */
      if (debug > 2)  {
	 fprintf(stderr, "Processing URhere at home.\n");
	 fprintf(stderr, "Myinfo.coaddr = %lx\n", htonl(Myinfo.coaddr));
	 fprintf(stderr, "Myinfo.homeaddr = %lx\n", htonl(Myinfo.homeaddr));
         fprintf(stderr, "Myinfo.status = %d\n", (int) Myinfo.status);
      }

      if ((Myinfo.coaddr == Myinfo.homeaddr) &&  
	  (Myinfo.status & AWAITINGREPLY))
	return;
	
      Myinfo.coaddr = Myinfo.homeaddr;
      lowifaceset("eth0", Myinfo.homeaddr, Myinfo.homenetmask);
      /* set default route through closeagentaddr */
      lowrtdefault(*addr);
      Myinfo.status &= ~REGAWAY;
      sendRegisterMe((int) 0, (unsigned char) 0, (unsigned short) 0, 
		     Myinfo.homeaddr, Myinfo.homeagent, Myinfo.homeaddr);
      return;
   }
   
   if (((*addr ^ Myinfo.homeagent) & Myinfo.homenetmask) == 0)  {
      if (debug > 2)  {
	 fprintf(stderr, "We are using a columbia style arrangement\n");
      }
      /* We are using a Columbia style campus arrangement */
      if (urh->addrnum < 2)  {
	 fprintf(stderr, "Need a wired address for a coaddr.\n");
	 cleanup(); exit(-1);
      }
      addr += 2;

      if ((Myinfo.coaddr == *addr) &&  
	  (Myinfo.status & AWAITINGREPLY))
	return;  /* we already have an outstanding request */

      Myinfo.coaddr = *addr;
      lowifaceset("eth0", Myinfo.homeaddr, Myinfo.homenetmask);
      /* set default route through closeagentaddr */
      lowrtdefault(Myinfo.closeagentaddr); 
      Myinfo.status &= ~REGHOME;
      sendRegisterMe((int) 0, (unsigned char) 0, (unsigned short) MAXLIFETIME,
		     Myinfo.homeaddr, Myinfo.homeagent, Myinfo.coaddr);
      return;
   }
      
  /***** This last part has not been tested **************/
   
   if ((Myinfo.coaddr == getipaddr(addr)) &&  
       (Myinfo.status & AWAITINGREPLY))
     return;  /* we already have an outstanding request */
   Myinfo.coaddr = getipaddr(addr); /* my temp ip addr on the foreign
				     * network becomes the coaddr so I can
				     * decapsulate my own packets */
   /* set up dummy, assign an ip address on eth0 belonging to this foreign
    * network and set default router as well */
   lowifaceset("eth0", Myinfo.coaddr, getnetmask(addr));
   lowrtdefault(*addr);
   /* need an interface so mh can assign its permanent address to it. This
    * will let the mh recognize packets destined to its perm address as its
    * own */
   lowifaceset("dummy0", Myinfo.homeaddr, inet_addr("255.255.255.254"));
   Myinfo.status |= DUMMYUP;
   lowrtset(Myinfo.homeaddr, "dummy0");
   sendRegisterMe((int) 0, (unsigned char) 0, MAXLIFETIME, Myinfo.homeaddr, 
		  Myinfo.homeagent, Myinfo.coaddr);
   return; 
}

void 
processRegreply(char *msg, int len, struct sockaddr_in *from)  {
   struct regreply *reply;
   int authoffset;
   
   reply = (struct regreply *) msg;
   if (debug > 0)  {
      fprintf(stderr, "\n==========================\n");
      printtime();
      fprintf(stderr, "-- REPLY from %s ", inet_ntoa(from->sin_addr));
      if ((reply->code != 0) && (reply->code != 1))
      	fprintf(stderr, "(Rejection)\n");
      else
      	fprintf(stderr, "(Acceptance)\n");
      if (debug > 1)  {
	 fprintf(stderr, "[%8lx:%8lx] Type %2d Code %3d Lifetime %8d\n",
		 reply->Id.high, reply->Id.low, reply->type, reply->code,
		 reply->lifetime);
	 fprintf(stderr, "Homeaddr: %8lx, Homeagent: %8lx",
		 htonl(reply->homeaddr), htonl(reply->ha));
	 printext(msg, len, 1);
	 if (debug > 2) {
	   fprintf(stderr, "\n--------------------\n");
	   testprint(msg, len);
	   fprintf(stderr, "--------------------");
	 }
      }
      fprintf(stderr, "\n==========================\n");
   }
   
   if (reply->type != REGREPLYTYPE || len < sizeof(struct regreply) ||
       !IsOurAgent(reply->ha) || !IsOurAddress(reply->homeaddr) ||
       !(authoffset=authextfound(msg, len, 1))) {
	  mhstats.badregreply++;
	  if (debug > 0)  {
	   fprintf(stderr, "Bad structure -- Reply ignored.\n");
	  }
	  return;
       }

   Myinfo.RequestId.high = reply->Id.high;
   if (mhIdcmp(&Myinfo.RequestId, &(reply->Id)) != 0)  {
      mhstats.idmismatch++;
      if (debug > 0)  {
	 fprintf(stderr, "Id mismatch -- Reply ignored.\n");
      }
      return;
   } else if (!authok(msg, len, authoffset, Myinfo.secret, 
              Myinfo.secretlen))  {
      mhstats.authfailed++;
      if (debug > 0)  {
	 fprintf(stderr, "Authentication failed -- Reply ignored.\n");
      }
      return;
   }

   if (Myinfo.status & AWAITINGREPLY == 0)  {
      fprintf(stderr, "Got a reply when we weren't expecting it.\n");
      /* This should never happen (probability is very very low) because 
	 after we get a good reply we change the Id that we now expect 
	 to a value we never sent */
   }
     
   Myinfo.status &= ~AWAITINGREPLY;
   NewId(&Myinfo.RequestId, &Myinfo.RequestId);
   if (reply->code == 0 || reply->code == 1)  {
      /* Our service request was approved */
      mhstats.regsuccessful++;
      if (reply->lifetime == 0)  {
	 /* deregistration approved */
	 Myinfo.regtimeleft = MAXLIFETIME;
	 Myinfo.status |= REGHOME;
	 Myinfo.status &= ~REGAWAY; /* sanity check */
      } else  {
	 Myinfo.regtimeleft = reply->lifetime;
	 Myinfo.status |= REGAWAY;
	 Myinfo.status &= ~REGHOME; /* sanity check */
      }
   } else  {
      /* Request was disapproved */
      mhstats.regfailed++;
   }
   /* See if we had an outstanding message and it is coming from
    * the agent we expect it to be from and that the id matches 
    * if one of these is not true, we simply ignore the reply
    * update log to indicate a bad reply
    * If we have a reply from where we expect, remove the outstanding
    * request and see what the reply says
    * was our request accepted or rejected (by looking at the code)
    * If we had sent out a registration request and it has been
    * accepted, set the REGAWAY flag and update the registration timer.
    * If we sent a deresgistration, it should always get accepted so
    * we set the REGATHOME flag */
}

main(int argc, char **argv)  {
   struct sockaddr_in from;
   char *cmd, msg[1024];
   int len, fromlen, c;
   extern char *optarg;
   extern int optind;
   unsigned char argsfound = 0;
   fd_set fdvec;
   struct timeval tv;
   
   cmd = argv[0];
   while ((c = getopt(argc, argv, "a:m:g:o:")) != -1)  {
      switch ((char) c)  {
       case 'a':
	 argsfound |= 0x01;
	 if ((Myinfo.homeaddr = inet_addr(optarg)) == -1)  {
	    fprintf(stderr, "Bad address passed to -a.\n");
	    exit(-1);
	 };
	 break;
       case 'm':
	 argsfound |= 0x02;
	 if ((Myinfo.homenetmask = inet_addr(optarg)) == -1)  {
	    fprintf(stderr, "Bad netmask passed to -m.\n");
	    exit(-1);
	 };
	 break;
       case 'g':
	 argsfound |= 0x04;
	 if ((Myinfo.homeagent = inet_addr(optarg)) == -1)  {
	    fprintf(stderr, "Bad address passed to -g.\n");
	    exit(-1);
	 };
       case 'o':
	 /* argsfound |= 0x08; */
	 if ((Myinfo.o_homeagent = inet_addr(optarg)) == -1)  {
	    fprintf(stderr, "Bad address passed to -o.\n");
	    exit(-1);
	 };
	 break;
       case '?':
	 usage(cmd);
	 exit(1);
	 break;
      }
   }

   if (argsfound != 0x07)  {
      usage(cmd);
      exit(1);
   }
     
   mhAdvertAddr= (Myinfo.homeaddr & Myinfo.homenetmask) | ~Myinfo.homenetmask;

   if (debug > 1)  {
      fprintf(stderr, "IPaddr: %lx, Mask: %lx, HomeAgent: %lx, Otheraddr: %lx\n",
	      htonl(Myinfo.homeaddr), htonl(Myinfo.homenetmask),
	      htonl(Myinfo.homeagent), htonl(Myinfo.o_homeagent));
   }

   init();
   sendWhereAmI(&Whereamito);
   while (1)  {
      if (Myinfo.haveUR == 1)  {
	 processURhere(Myinfo.URheremsg, Myinfo.URherelen, 
		       &Myinfo.URheresock);
	 Myinfo.haveUR = 0; /* we are ready to process more URhere messages */
      };
      FD_ZERO(&fdvec); FD_SET(Regreplysid, &fdvec); 
      tv.tv_sec = 0; tv.tv_usec = 0;
      if (select(16, &fdvec, NULL, NULL, &tv) < 0)  {
	 perror("select() failed in main loop.\n");
	 cleanup(); exit(-1);
      }
      len = 1024; fromlen = sizeof(struct sockaddr_in);
      if (FD_ISSET(Regreplysid, &fdvec))  {
	 if ((len = recvfrom(Regreplysid, msg, len, 0,
			     (struct sockaddr *) &from, &fromlen)) < 0)  {
	   perror("recvfrom() on RegisterMesid failed.\n");
 	   cleanup(); exit(-1);
	 };
	 processRegreply(msg, len, &from);
      }
   }
}

void
usage(char *cmd)  {
   fprintf(stderr, 
   "Usage is: %s -a homeaddr -m homenetmask -g haaddr [-o otherhaaddr]\n",
	   cmd);
}

