/*
 * Linux-Mobile-IP
 *            An implementation of Mobile IP for the LINUX operating system
 *            developed at the State University of New York, Binghamton
 *            (with support from the Center for Computing Technologies).
 *            The implementation uses the message formats described in the
 *            Internet Engineering Task Force (IETF) mobile-ip draft
 *            but is not (yet) fully compliant.
 *
 *            More information can be obtained from:
 *            http://anchor.cs.binghamton.edu/~mobileip/      
 *           
 *
 *            Permission is hereby granted to redistribute this code and/or
 *            modify it under the terms of the GNU Genral Public License
 *            as published by the Free Software Foundation; either version
 *            two or (at your option) any later version.
 *
 *            The authors request that their contribution be appropriately 
 *            acknowledged in any derived work. 
 *
 *            mhlow.c - Collection of low level routines to manipulate 
 *                      routing tables, arp caches and network interfaces
 *                      used by the mobile host.
 * 
 * Version:   0.95     11/24/1995
 * 
 * Authors:   Abhijit Dixit <abhijit@cs.binghamton.edu>
 *            Vipul Gupta <vgupta@cs.binghamton.edu>
 *            Benjamin Lancki <ben@anchor.cs.binghamton.edu>
 *            
 * */  

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#include <linux/route.h>
#include <linux/if.h>
#include <linux/if_arp.h>
#include <errno.h>
#include <sys/file.h>
#include <sys/param.h>
#include <netdb.h>

#include "low.h"

extern int ioctlsid;
extern int debug;

lowifacedown(char *name)  {
 /* down the interface whose name is given */
   struct ifreq ifr;

   if (debug > 0)  {
      fprintf(stderr, "DOWNiface %s.\n", name);
   };
   strcpy(ifr.ifr_name, name);
   ifr.ifr_flags &= ~IFF_UP;
   if (ioctl(ioctlsid, SIOCSIFFLAGS, (char *) &ifr) < 0)  {
      perror("lowifacedown failed SIOCSIFFLAGS");
   };
}

lowifaceset(char *name, unsigned long ipaddr, unsigned long mask)  {
   struct ifreq ifr;
   struct sockaddr_in *sock;
   
   if (debug > 0)  {
      fprintf(stderr, "lowifaceset called on %s addr %lx mask %lx.\n", 
	      name, htonl(ipaddr), htonl(mask));
   };
   
   strcpy(ifr.ifr_name, name);

   if (ioctl(ioctlsid, SIOCGIFFLAGS, (char *) &ifr) < 0 )  {
      fprintf(stderr, "SIOCGIFFLAGS failed on %s\n", name);
      exit(-1);
   };
   ifr.ifr_flags |= IFF_UP;
   if (ioctl(ioctlsid, SIOCSIFFLAGS, (char *) &ifr) < 0)  {
      perror("lowifaceset failed SIOCSIFFLAGS");
   };
   sock = (struct sockaddr_in *) &ifr.ifr_addr;
   sock->sin_family = AF_INET;
   sock->sin_addr.s_addr = ipaddr;
   if (ioctl(ioctlsid, SIOCSIFADDR, (char *) &ifr) < 0)  {
      perror("lowifaceset failed SIOCSIFADDR");
   }
   sock = (struct sockaddr_in *) &ifr.ifr_broadaddr;
   sock->sin_family = AF_INET;
   sock->sin_addr.s_addr = (ipaddr & mask) | (0xffffffff & ~mask);
   if (ioctl(ioctlsid, SIOCSIFBRDADDR, (char *) &ifr) < 0)  {
      perror("lowifaceset failed SIOCSIFBRDADDR");
   }
   sock = (struct sockaddr_in *) &ifr.ifr_netmask;
   sock->sin_family = AF_INET;
   sock->sin_addr.s_addr = mask;
    if (ioctl(ioctlsid, SIOCSIFNETMASK, (char *) &ifr) < 0)  {
      perror("lowifaceset failed SIOCSIFNETMASK");
   };
}

lowrtdefault(unsigned long ipaddr)  {
   struct rtentry rtent;
   struct sockaddr_in *sock;
   
   if (debug > 0)  {
      fprintf(stderr, "ROUTEdefault gw %lx.\n", 
	      htonl(ipaddr));
   };
   bzero((char *) &rtent, sizeof(struct rtentry));
   sock = (struct sockaddr_in *) &rtent.rt_dst;
   sock->sin_family = AF_INET;
   sock = (struct sockaddr_in *) &rtent.rt_gateway;
   sock->sin_family = AF_INET;
   sock->sin_addr.s_addr = ipaddr;
   rtent.rt_flags = RTF_GATEWAY | RTF_UP;
   if (ioctl(ioctlsid, SIOCADDRT, (char *) &rtent) == -1)  {
      perror("SIOCADDRT failed");
   }
}

lowrtset(unsigned long dest, char *devname, int type)  {
   struct rtentry rtent;
   struct sockaddr_in *sock;
   
   if (debug > 0)  {
      fprintf(stderr, "ROUTEset dest %8lx dev %s Type(6:DELRT 7:ADDRT) %2d.\n",
 	      htonl(dest), devname, type);
   };
   bzero((char *) &rtent, sizeof(struct rtentry));
   sock = (struct sockaddr_in *) &rtent.rt_dst;
   sock->sin_family = AF_INET;
   sock->sin_addr.s_addr = dest;
   switch(type){
   case ADDRT:
     rtent.rt_flags = RTF_UP | RTF_HOST;
     rtent.rt_dev = devname;
     if (ioctl(ioctlsid, SIOCADDRT, (char *) &rtent) == -1)  {
      perror("SIOCADDRT failed");
    }
    break;
   
   case DELRT:
     if (ioctl(ioctlsid, SIOCDELRT, (char *) &rtent) < 0 &&
	 errno != EFAULT)  {
       perror("lowrtreq DELRT failed SIOCDELRT");
     };
     break;
   }  
}

lowrtnetset(unsigned long net,char *devname, unsigned long netmask,int type)  {
   struct rtentry rtent;
   struct sockaddr_in *sock , *socknm;
   
   if (debug > 0)  {
      fprintf(stderr, "ROUTEset dest net %8lx dev %s 
              Type(6:DELRT 7:ADDRT) %2d\n",htonl(net), devname,type);
     };
   bzero((char *) &rtent, sizeof(struct rtentry));
   sock = (struct sockaddr_in *) &rtent.rt_dst;
   sock->sin_family = AF_INET;
   sock->sin_addr.s_addr = net;
   
   switch(type){
   case ADDRT:
     socknm = (struct sockaddr_in *) &rtent.rt_genmask;
     socknm->sin_family = AF_INET;
     socknm->sin_addr.s_addr = netmask;
     rtent.rt_flags = RTF_UP;
     rtent.rt_dev =  devname;
     if (ioctl(ioctlsid, SIOCADDRT, (char *) &rtent) == -1)  {
       perror("SIOCADDRT failed");
       }
     break;
     
   case DELRT:
      if (ioctl(ioctlsid, SIOCDELRT, (char *) &rtent) < 0 &&
	       errno != EFAULT)  {
	      perror("lowrtreq DELRT failed SIOCDELRT");
	   };
	   break;
   }
   if(debug>2)fprintf(stderr,"Finished adding/del route for net %lx\n"
		      ,htonl(net));
}



/* delete is used in lowflshARP() */

int deleteARP(char *host)
	
{
	struct arpreq ar;
	struct hostent *hp;
	struct sockaddr_in *sin;
	int s;

 	bzero((caddr_t)&ar, sizeof ar);
	ar.arp_pa.sa_family = AF_INET;
	sin = (struct sockaddr_in *)&ar.arp_pa;
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = inet_addr(host);
	if ((long)sin->sin_addr.s_addr == -1) {
		if (!(hp = gethostbyname(host))) {
			fprintf(stderr, "arp: %s: ", host);
			herror((char *)NULL);
			return(-1);
		}
		bcopy((char *)hp->h_addr, (char *)&sin->sin_addr,
		    sizeof sin->sin_addr);
	}
	s = socket(AF_INET, SOCK_DGRAM, 0);
	if (s < 0) {
		perror("arp: socket");
		return(-1);
	}
	if (ioctl(s, SIOCDARP, (caddr_t)&ar) < 0) {
		if (errno == ENXIO && debug >2)
			fprintf(stderr,"%s (%s) -- no entry\n",
			    host, inet_ntoa(sin->sin_addr));
		else
			perror("SIOCDARP");
		return(-1);
	}
	close(s);
	
      }


lowflshARP() {

  int i,fd;
  char buf[1024];
  char host[20];

  if((fd = open("/proc/net/arp",O_RDONLY,0))<0)
    {
      perror("arp: error opening arptab");
      return;};

  
  if((i = read(fd,buf,sizeof(buf)))<0)
	{
		perror("arp: error reading arptab");
		exit(1);
	      }

  close(fd);
  buf[i]='\0';

for(i=0;buf[i]!='\0';i++)if(buf[i] == '\n')
{
  if(sscanf(&buf[i+1],"%s",host)!=0)
    { if(debug >2) 
      {
	fprintf(stderr,"%s\n",host);
	fprintf(stderr,"Deleting arp entry...\n");
      }
      deleteARP(host);
     }
  
}

/* The following routines should return an appropriate netmask and ip address
   that a mobile host can use while it is visiting a network outside
   its original administrative domain. Until we can use something better
   like DHCP, individual users will have to tailor these functions to
   their needs. The ip-addr of a nearby mobility agent is passed as
   argument and may be used to look up a table for the appropriate
   choice */


}
getnetmask(unsigned long addr)  {
   return(inet_addr("255.255.255.224"));
}

unsigned long
getipaddr(unsigned long addr)  {
   return(inet_addr("131.205.4.4"));
}

