/*----------------------------------------------------------------------------

  Copyright (C) 1995-6, Vipul Gupta and Abhijit Dixit. Created 1995-6. All
  rights reserved.

  Linux-Mobile-IP
            An implementation of Mobile IP for the LINUX operating system
            developed at the State University of New York, Binghamton
            (with partial support from the Center for Computing Technologies).
            Except as noted in the accompanying documentation, this
            implementation complies with revision 16 of the Internet 
            Engineering Task Force (IETF) Mobile-IP draft.

            More information can be obtained from:
                  http://anchor.cs.binghamton.edu/~mobileip/  

  Version:   1.00     05/23/1996
 
  Authors:   Abhijit Dixit <abhijit@cs.binghamton.edu> 
             Vipul Gupta <vgupta@cs.binghamton.edu>
             Benjamin Lancki <ben@anchor.cs.binghamton.edu>

  Permission is hereby granted to redistribute this code and/or
  modify it under the terms of the GNU Genral Public License
  as published by the Free Software Foundation; either version
  two or (at your option) any later version provided this ENTIRE
  notice is retained in any copies or any part of this software.

  A copy of the GNU General Public License can be obtained by contacting
  the Free Software Foundation, Inc. 675 Mass Ave, Cambridge, MA 02139, USA.
             
-----------------------------------------------------------------------------*/

/*     agent.c - Design of the Mobile Agent (either home agent or 
 *                                 foreign agent).
 *
 *  Command line arguments:
 *    -a : agent's IP address on the network on which it supports mobility
 *    -m : network mask corresponding to the IP address above
 *    -h : hardware address of the mobility interface (used for proxies
 *	   and used only if this agent is a home agent
 *    -i : Network interface name (eg eth0) on which MHs are supported. 
 *
 *
 *
 *
 *
 * History of Changes:
 *     Jul 25, 1995              First version coded  - V.G.
 *     Jul 31, 1995              Removed calls to GETRT - V.G.
 *     Aug  1, 1995              Removed bug in sendRefusal and reorganized
 *				 it to improve efficiency - V.G.
 *     Aug  9, 1995              Added support for MD5 authentication in
 *                               processing the registration and registration 
 *                               reply - B.L.
 *     Aug 13, 1995              Agent now reads config file /etc/mip-mh.ok to
 *                               initialize mhinfo struct - B.L.
 *     Aug 14, 1995              Agent keeps a log in event of system crash
 *                               or shutdown and recovers on startup - B.L.
 * 
 *     Sept 22, 1995             Foreign agent has become more active.
 *                               Registerme and Regreply are sent to
 *             			 foreign agent and foreign agent forwards 
 *				 it -A.D.
 *
 *     Sept 30, 1995             Agent does not exit when select
 *                               fails on INTR -A.D.
 *       
 *     Oct 2, 1995               huphandler added to toggle between
 *                               sending URHere and not sending URhere
 *				 on SIGHUP signal-A.D.
 *				 
 *     Oct 3, 1995               URhere mesgs are now sent to link level
 *                               broadcast address -A.D.
 *
 *
 *     Oct 6, 1995               Command line arguments changed. 
 *                               Wired address is no more required so 
 *                               removed from command line.
 *                               network interface is now command line 
 *                               parameter -A.D.
 *
 *     Oct 8, 1995               Agent does not exit when sendto fails
 *                               because of network or host unreachable
 *                               condition. -A.D.
 *
 *     Nov 10, 1995              Due to addition of a 'special sleep'
 *                               agent consumes very little cpu time. -A.D.    
 *
 *     Dec 5, 1995	         Router discovery messages are changed to use 			 
 *                               ICMP. -A.D.
 *
 *     May    1996               Agent now sends gratuitous ARP when it
 *                               starts Proxy ARPing for the MH. -A.D.
 *
 *     May    1996               Agent now sends rejection to Deregistration
 *                               also on local network. -A.D
 * */
   
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <netinet/ip.h>

#include "messages.h"
#include "low.h"
#include "agent.h"

#define SLEEPINTERVAL 0.5 /* second */
#define ALARMINTERVAL 1   

char DEVICE[32];

int debug = 2;

unsigned int sendURhere_flag = 1;
 
extern void 
testprint(char *,int);

extern int in_cksum(u_short *addr,int len);

void 
sendReply(unsigned short, int, int, unsigned long, 
	  unsigned long, unsigned long, struct id);
void
sendConfirm(unsigned short, int, int, unsigned long, 
	    unsigned long, unsigned long, int);
void
sendRefusal(unsigned short, int, int, unsigned long, 
	       unsigned long, unsigned long, int);

void
sendRefusalVMH(unsigned short fromport,int code,int time,
	       unsigned long homeaddr,unsigned long ha,int vmh);

void delay();

extern void 
appendauth(char *,int *,char *,int,unsigned long);

extern int 
authextfound(char *,int,int);

extern int 
authok(char *,int,int,char *,int,unsigned long);

extern void 
printext(char *,int,int);

extern void 
printtime();

extern int errno;

extern void readvmhdata(char *); 
extern void readlog(char *), writelog(char *);
void usage(char *), cleanup();
int URheresid, Regreplysid; /* for sending, these could be the same */
int WhereAmIsid; 
int RegisterMesid; /* for listening to requests, these are
				 * bound to different ports */ 
int RegMeVMHid, RegReplyHAid;          /* for sending and receiving regme for 
				 visiting MH and reg reply from 
				 HA of visiting MH respectively */

int ioctlsid; /* for all kinds of low level ioctl calls */

unsigned long tunnelbitvec = 0; /* a bit vector to track tunnels in use */
char supportedMHnum = 0; /* # MHs we are configured to serve in mip-mh.ok */ 
char supportedVMHnum = 0; /* # VMHs we are configured to serve in mip-vmh.ok */
char proxyHwAddr[7];  /* hw address to use for proxy ARP */
unsigned long haAddr; /* home agent address */
unsigned long haNetmask; /* home agent's network mask */

struct sockaddr_in URhereto = 
     {  (short) AF_INET, (unsigned short) 0, 
	(unsigned long) 0, (unsigned long) 0, (unsigned long) 0 };

mhdata mhinfo[MAXMHNUM+1]; /* zeroth mh is not really used */ 
vmhdata vmhinfo[MAXMHNUM+1];

struct stats  {
   unsigned long badwhereami;
   unsigned long badregisterme;
   unsigned long mhprohibited;
   unsigned long idmismatch;
   unsigned long authfailed;
   unsigned long noresources;
   unsigned long successfulreg;
} agentstats =  { (long) 0, (long) 0, (long) 0, (long) 0, 
                  (long) 0, (long) 0, (long) 0 };

void
initsockets()  {
   struct sockaddr_in sa;
   int enable = 1;
   
   Regreplysid = socket(AF_INET, SOCK_DGRAM, 0);
   WhereAmIsid = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP);
   RegisterMesid = socket(AF_INET, SOCK_DGRAM, 0);
   ioctlsid = socket(AF_INET, SOCK_DGRAM, 0);
   RegReplyHAid = RegMeVMHid = socket(AF_INET, SOCK_DGRAM, 0); 
   URheresid = socket(AF_INET,SOCK_RAW,IPPROTO_ICMP);

   if (Regreplysid < 0 || URheresid < 0 || 
       WhereAmIsid < 0 || RegisterMesid < 0 || 
         RegReplyHAid <0 ||ioctlsid < 0)  {
	  fprintf(stderr, "initsockets(): could not create sockets.\n");
	   exit(-1);
	}              
   sa.sin_family = AF_INET; sa.sin_addr.s_addr = htonl(INADDR_ANY); 
   sa.sin_port = 0; 
   if (bind(Regreplysid, (struct sockaddr *) &sa, sizeof(sa)) < 0)  {
      perror("initsockets(): Bind Regreplysid failed.\n");
      cleanup(); exit(-1);
   }
   if (setsockopt(URheresid, SOL_SOCKET, SO_BROADCAST, 
		  (char *)&enable, sizeof(int)) < 0)  {
	  perror("initsockets(): setsockopt SO_BROADCAST failed");
	  cleanup(); exit(-1);
       }

   sa.sin_port = htons(MIPREGPORT);
   if (bind(RegisterMesid, (struct sockaddr *) &sa, sizeof(sa)) < 0)  {
      perror("initsockets(): Bind RegisterMesid failed.\n");
      cleanup(); exit(-1);
    }
   sa.sin_port = 0; /* FA can send to HA on any port */
   sa.sin_addr.s_addr = haAddr;
   if (bind(RegReplyHAid, (struct sockaddr *) &sa, sizeof(sa)) < 0)  {
      perror("initsockets(): Bind RegisterMesid failed.\n");
      cleanup(); exit(-1);    

    }
   if (debug > 0)
     fprintf(stderr, "Initialized sockets ...\n");
}

void
cleanup()  {
   /* down all tunnels that we could have created and remove
    * proxy arps for all MHs we support if the arp HWaddr in
    * those entries is our own */
   int i;
   
   for (i = 0; i < MAXTUNNELS; i++)
     lowifacereq(DOWN, i, 0, 0); 
   for (i = 0; i < MAXMHNUM; i++)
     lowARPreq(REM0, mhinfo[i].ipaddr, proxyHwAddr);
   if (debug > 1)
     fprintf(stderr, "Done cleaning up ...\n");
}

void
graceful_exit() {
   cleanup();
   writelog("/usr/adm/mip-ha.log");
   exit(0);
}

void
newId(struct id *id1, struct id *id2)  {
   static int count = 0;
   struct timeval tv;
   
   if (count++ == 0)  {
      gettimeofday(&tv, NULL);
      srandom((int) tv.tv_usec);
   };
   id1->high = (unsigned long) random();
   id1->low = id2->low;
}
	
void 
sendURhere(struct sockaddr_in *toaddr)  {
   static struct urhmsg  {
      struct icmp icp;
      struct youarehere URh;
      unsigned long addr1; 
   } mesg;

   static u_short seqno = 0;

   if (debug > 2)
     fprintf(stderr, "Sending a URhere message to %s \n",
	     inet_ntoa(toaddr->sin_addr));
   else if (debug > 0)
     fprintf(stderr, ".");
   /* fill the buffer appropriately and send */
   
   mesg.icp.type = ROUTERADVTYPE; /* router advertisement */
   mesg.icp.code = 0;
   mesg.icp.cksum = 0;
   mesg.icp.addrnum = 0;
   mesg.icp.addr_entry_size =0;
   mesg.icp.lifetime = 6 ; /* 3*period of URhere mesg. */
   mesg.URh.type = YOUAREHERETYPE;  /* Mobile serv ext */
   mesg.URh.length = 10; /* 6 + 4*N bytes where N is # of addresses(1) */
   mesg.URh.seqno = seqno++;
   mesg.URh.lifetime = MAXLIFETIME;
   mesg.URh.flags = MSRFLAG|MSHFLAG|MSFFLAG; 

   mesg.addr1 = haAddr;

   mesg.icp.cksum = in_cksum((u_short*)&mesg,sizeof(struct urhmsg));

   if(!seqno)seqno = 256;
   
   if (sendto(URheresid, (char *) &mesg, sizeof(mesg), 0, 
	      (struct sockaddr *) toaddr, sizeof(struct sockaddr))
       < 0)  {
          
	  perror("sendURhere(): sendto failed.\n");
	  fprintf(stderr,"Send Addr: %s at %d\n",
		  inet_ntoa(toaddr->sin_addr), 
		  (unsigned int) toaddr->sin_port);
	  if(errno == ECONNREFUSED || 
	     errno == ENETUNREACH || errno == EHOSTUNREACH);
	  else
	    {
	      cleanup(); exit(-1);
	    }
	}
 }

void
update_lifetimes()  {
   /* Walk through the list of MH for which TUNNELUP is set and
    * decrement lifetimes.
    * If lifetime becomes zero or less, stop supporting this
    * MH i.e. 
    * (a) remove its routing table entry, (route del ipaddr)
    * (b) destroy its tunnel name (ifconfig tunnelname down)
    * (c) stop proxying for this MH by removing its arp entry
    *     (arp -d ipaddr) 
    * (d) reset the TUNNELUP flag, timeleft, coaddr and 
    *     are set to 0. Tunnel name becomes a null string */
   int i;
   char devname[10];

   for (i = 1; i <= supportedMHnum; i++)  {
      if ((mhinfo[i].status & TUNNELUP) == 0) continue; 
      if ((mhinfo[i].timeleft -= 2) <= 0)  {
	 mhinfo[i].status &= ~TUNNELUP;
	 mhinfo[i].timeleft = 0;
	 mhinfo[i].coaddr = 0;
	 newId(&(mhinfo[i].RegistrationId), &(mhinfo[i].RegistrationId));
	 if (debug > 1)  {
	    fprintf(stderr, "\n--Registration expired for %lx\n",
		   htonl(mhinfo[i].ipaddr));
	 };
	 lowrtreq(DELRT, mhinfo[i].ipaddr, &(mhinfo[i].tunnelnum));
	 lowifacereq(DOWN, mhinfo[i].tunnelnum, 0, 0);
	 lowARPreq(REM1, mhinfo[i].ipaddr, "000000");
	 mhinfo[i].tunnelnum = 0;
       }
    }

   /* In this part we check for lifetime of visiting 
    *  mobile hosts. If lifetime becomes zero and  
    *  status = confirmed then delet existing route 
    * for VMH */
   strcpy(devname,DEVICE);
   for(i = 1; i<= supportedVMHnum; i++) {
     if(vmhinfo[i].status == 0||vmhinfo[i].status == PENDING) continue;
     if((vmhinfo[i].timeleft -=2) <=0){
       vmhinfo[i].status &= 0x0;
       vmhinfo[i].timeleft = 0;
       lowroutset(vmhinfo[i].ipaddr,devname,DELRT);
     }
   }
}


void 
hup_handler(){
sendURhere_flag = sendURhere_flag ? 0 : 1;
signal(SIGHUP, hup_handler);
}


void
alarm_handler()  {
   /* Check for expired regitrations and send out URhere broadcast */
   update_lifetimes();

   if (sendURhere_flag)
   sendURhere(&URhereto);
   
   /* set things up to come back to this procedure sometime soon */ 
   signal(SIGALRM, alarm_handler);
   alarm(ALARMINTERVAL);
}

void
init()  {
   initsockets(); /* set up sockets for listening to requests and
		   * sending messages */
   cleanup(); /* cleanup relies on ioctlsid */
   URhereto.sin_addr.s_addr = inet_addr(ADVERTISEON);

   /* No port # for ICMP delete later 
      URhereto.sin_port = htons(ADVERTISETO);
   */
   readmhdata("/etc/mip-mh.ok");
   readvmhdata("/etc/mip-vmh.ok");  /* read data for visiting MH */
   /* read the ipaddr, authtype and 
				  * secret for first MAXMHNUM mobile
				  * hosts */ 
   readlog("/usr/adm/mip-ha.log"); /* restore any state from a previous
				    * incarnation of this daemon */ 
   signal(SIGTERM, graceful_exit); /* terminate gracefully */
   signal(SIGINT, graceful_exit); /* terminate gracefully */
   signal(SIGALRM, alarm_handler); /* alarm_handler is the function
				    * that gets called when alarm
				    * goes off */ 

   signal(SIGHUP, hup_handler);  /* Added to toggle URH messages */
   
   alarm(ALARMINTERVAL);
}


slp(double sec)
{
  struct timeval tv;
  
  tv.tv_sec = 0; tv.tv_usec = sec*1000000 ;

  if((select(FD_SETSIZE, NULL, NULL, NULL, &tv)) < 0)  {
    if(errno == EINTR ||  errno == ERESTART);
    else  
      perror("select() failed in main loop.\n");
    
  }
 
}

void
processWhereAmI(char *msg, int len, struct sockaddr_in *from)  {
   /* if message is not of type WhereAmI, record this in a
    * debug log and ignore msg. Otherwise call sendURhere()
    * with the address and port obtained from the from
    * structure */
   
  struct icmp *where;
  struct iphdr *ip;
  int iphdrlen;

  ip = (struct iphdr*)msg;
  
  iphdrlen = ip->ihl << 2; /* Bytes */
      
  where = (struct icmp*)(msg+iphdrlen);
   
  
  if (where->type != ROUTERSOLTYPE || len < (iphdrlen + sizeof(struct icmp)))
     agentstats.badwhereami++;
  else
    {
      if (debug > 1)
	fprintf(stderr, "\n-- WHEREAMI from %s\n", inet_ntoa(from->sin_addr));
   
      sendURhere(from);}
}    
 
int
IsOurAddress(unsigned long addr)  {
   return((haAddr == addr) ? 1 : 0);
}

char
newtunnel() {
   char i;

   for (i = 1; i < MAXTUNNELS; i++)  {
      if ((tunnelbitvec & (1 << i)) == 0)
      	return(i);
   }
   return(0);
}

int 
whichmh(unsigned long addr)  {
   int i;
   
   mhinfo[0].ipaddr = addr;
   for (i = supportedMHnum; i >= 0; i--)
     if (mhinfo[i].ipaddr == addr) break;
   
   return(i);
}

int
whichvmh(unsigned long addr) {
  
  int i;
  
  vmhinfo[0].ipaddr = addr;
  for(i= supportedVMHnum; i>=0; i--)
    if(vmhinfo[i].ipaddr == addr) break;
  return(i);
}

int
havetunnel(int mh, unsigned long coaddr)  {
   /* Return 0 (false) or 1 (true) depending on whether we can find
    * an existing tunnel as required set up for the mobile host whose
    * mhinfo index is given in mh. This tunnel should end at
    * coaddr and a route to homeaddr must use the tunnel */

   if ((mhinfo[mh].status & TUNNELUP) && (mhinfo[mh].coaddr == coaddr))
     return (1);
   else
     return(0);
}

int 
maIdcmp(struct id *id1, struct id *id2)  {
   return((id1->high == id2->high) ? 0:1);
}

void
processRegisterMe(char *msg, int len, struct sockaddr_in *from)  {
   struct registerme *reg;
   int code = 0;
   int mh, vmh;
   int authoffset;
   char temp = 0;
   unsigned short fromport;
   unsigned long fromaddr; /* for sending back reg reply */
   struct sockaddr_in toaddr; /* for sending mesg to HA */
   
   if (debug > 2) 
      fprintf(stderr, "<<<<<<  Entering processRegisterMe \n");
   
   if (debug > 0) 
     {
       if (len < sizeof(struct registerme)) 
	 fprintf(stderr,"msg length short of registerme size.\n");
     }
   
   reg = (struct registerme *) msg;
   fromport = (unsigned short) from->sin_port;
   fromaddr = (unsigned long) from->sin_addr.s_addr;

   if (debug > 0)  {
      fprintf(stderr, "\n==========================\n");
      printtime();
      if (reg->lifetime == 0) 
      	fprintf(stderr, "-- DEREGISTERME from %s Port %2d\n",
		inet_ntoa(from->sin_addr),fromport);
      else
      	fprintf(stderr, "-- REGISTERME from %s Port %2d\n",
		inet_ntoa(from->sin_addr), fromport);
      if (debug > 1)  {
	 fprintf(stderr, "[%8lx:%8lx] Type %2d Flags %2x Lifetime %8d\n",
	      reg->Id.high, reg->Id.low, reg->type, reg->flags, reg->lifetime);
	 fprintf(stderr, "Homeaddr: %8lx, Homeagent: %8lx, Careof: %8lx",
	      htonl(reg->homeaddr), htonl(reg->ha), htonl(reg->coaddr));
	 printext(msg, len, 0);
	 if (debug > 2) {
	   fprintf(stderr, "\n--------------------\n");
	   testprint(msg, len);
	   fprintf(stderr, "--------------------");
	 }
      }
      fprintf(stderr, "\n==========================\n");
   }
   

   if(IsOurAddress(reg->ha)) {

     if(debug > 1)fprintf(stderr,"Acting as home agent...\n");
     mh = 0;
   if (len < sizeof(struct registerme) || (reg->flags & 0x03)|| reg->type != REGISTERMETYPE || 
       !IsOurAddress(reg->ha) || !(authoffset=authextfound(msg, len, 0)))  {
      code = 134; /* badly formed request */
      agentstats.badregisterme++;
   } else if ((mh = whichmh(reg->homeaddr)) == 0)  {
      code = 129; /* administratvely prohibited */
      agentstats.mhprohibited++;
   } else if (reg->lifetime == 0 && 
	      (reg->coaddr != mhinfo[mh].coaddr && reg->coaddr != mhinfo[mh].ipaddr)){
     code = 134;
     agentstats.badregisterme++;
   } else if (maIdcmp(&mhinfo[mh].RegistrationId, &(reg->Id)) != 0)  {
      code = 133; /* id is not quite what we expected */ 
      agentstats.idmismatch++;
   } else if (reg->flags & 0x5c) {
     code = 128; /* reason unspecified  no broadcast support 
		    no GRE encap no Van Jacobson header compession */
   } else if (!authok(msg, len, authoffset, mhinfo[mh].secret, 
              mhinfo[mh].keylen,mhinfo[mh].SPIval))  {
      code = 131; /* failed authentication */
      agentstats.authfailed++;
   }

   
   /* create a new id we can use for our reply */ 
   newId(&(mhinfo[mh].RegistrationId), &(reg->Id));

   if (code)  { /* Refusal */
     
      sendRefusal(fromport, code, 0, reg->homeaddr, reg->coaddr, haAddr, mh);
      if (debug > 2)
      	fprintf(stderr, "Refusal sent.\n");

      /* If it was deregistration request rejection due to ID mismatch
	 when mobile host is away, send that rejection also on local net*/
 
      if(code == 133 && reg->lifetime == 0 && (mhinfo[mh].status & TUNNELUP))
	{
	  lowARPreq(REM0, mhinfo[mh].ipaddr, proxyHwAddr);
	  lowrtreq(DELRT, mhinfo[mh].ipaddr, &temp);
	  sendRefusal(fromport, code, 0, reg->homeaddr, reg->coaddr, haAddr, mh);
	  lowARPreq(PROXY, mhinfo[mh].ipaddr, proxyHwAddr);
	  lowrtreq(ADDRT, mhinfo[mh].ipaddr, &(mhinfo[mh].tunnelnum));
	  if(debug >2)
	    fprintf(stderr, "Refusal sent on home network.\n");
	}
	  
      return;
   }

   /* Now we know that we are looking at a good request */

   if (reg->lifetime == 0) /* We got a deregistration request */
     {
   /* if this is a degistration request, we can always oblige. 
    *    i. make lifetime field zero, if there is a name in the
    *       tunnel field, down it clear name Change status. Clear
    *       coaddr field also. We can leave the RegistrationId field
    *       unchanged. Update any statistics we maintain. code = 1
    *       lifetime = 0  */
	mhinfo[mh].timeleft = 0;
	code = 1;
	mhinfo[mh].status &= ~TUNNELUP;
	mhinfo[mh].coaddr = 0;
	lowrtreq(DELRT, mhinfo[mh].ipaddr, &temp);
	lowifacereq(DOWN, mhinfo[mh].tunnelnum, 0, 0);
	lowARPreq(REM0, mhinfo[mh].ipaddr, proxyHwAddr);

	mhinfo[mh].tunnelnum = 0;
	agentstats.successfulreg++;
	sendConfirm(fromport, code, 0, reg->homeaddr, reg->coaddr, haAddr, mh);
   } 
   else if (mhinfo[mh].status & TUNNELUP)  {
	mhinfo[mh].timeleft = (reg->lifetime > MAXLIFETIME) ?
	      MAXLIFETIME : (int) reg->lifetime;
	/* If S bit is set we send code =1 since this agent does not support 
	   simultaneous bindings */
	if(reg->flags & 0x80) 
	  code = 1;
	else 
	  code = 0;
	
	if (reg->coaddr == mhinfo[mh].coaddr)  {
	   /* we have a renewal */
	   agentstats.successfulreg++;
	   sendConfirm(fromport, code, mhinfo[mh].timeleft, reg->homeaddr,
		       reg->coaddr, haAddr, mh);
	} 
	else  {
	   /* the tunnel needs to be redirected */
	   mhinfo[mh].coaddr = reg->coaddr;
	   lowifacereq(REDIRECT, mhinfo[mh].tunnelnum, 0, reg->coaddr);
	   agentstats.successfulreg++;
	   sendConfirm(fromport, code, mhinfo[mh].timeleft, reg->homeaddr,
		       reg->coaddr, haAddr, mh);
	}
     }
   else { /* A tunnel does not already exist */ 
	mhinfo[mh].timeleft = (reg->lifetime > MAXLIFETIME) ?
	      MAXLIFETIME : (int) reg->lifetime;
	/* If S bit is set we send code =1 since this agent does not support 
	   simultaneous bindings */
	if(reg->flags & 0x80) 
	  code = 1;
	else 
	  code = 0;
	if (mhinfo[mh].tunnelnum = newtunnel())  {
	   /* we were able to find an unused tunnel name */
	   mhinfo[mh].status |= TUNNELUP;
	   mhinfo[mh].coaddr = reg->coaddr;
	   lowifacereq(MKPT2PT, mhinfo[mh].tunnelnum, haAddr, reg->coaddr);
	   lowrtreq(ADDRT, mhinfo[mh].ipaddr, &(mhinfo[mh].tunnelnum));
	   lowARPreq(PROXY, mhinfo[mh].ipaddr, proxyHwAddr);
	   /* Send gratuitous ARP */
	   lowarpsend(DEVICE,mhinfo[mh].ipaddr,proxyHwAddr,mhinfo[mh].ipaddr);
	   agentstats.successfulreg++;
	   sendConfirm(fromport, code, mhinfo[mh].timeleft, reg->homeaddr,
		       reg->coaddr, haAddr, mh);
	} else  {
	   code = 130; /* insufficient resources */
	   agentstats.noresources++;
	   sendRefusal(fromport, code, 0, reg->homeaddr, reg->coaddr, 
		       haAddr, mh);
	   if (debug > 2)
	     fprintf(stderr, "Refusal sent due to insuff resources.\n");
	}
     }
   }
else /* Now agent has to act as FA */
  {
     code =0;
    if(debug>1)fprintf(stderr,"Acting as a foreign agent...\n");
    vmh = 0;
   if (len < sizeof(struct registerme) || reg->type != REGISTERMETYPE)  {
      code = 70; /* badly formed request */
      agentstats.badregisterme++;
   } else if ((vmh = whichvmh(reg->homeaddr)) == 0)  {
      code = 65; /* administratvely prohibited */
      agentstats.mhprohibited++;
   } else if (vmhinfo[vmh].ha != reg->ha)
      code = 65; /* check for home agent  address if it is the 
		    same as present in FA database */ 
     else if (reg->flags & 0x08) /* No GRE encap */
       code = 72;
     else if (reg->flags & 0x04) /* No Van Jacobson header comp.*/
       code = 73;
     else if (reg->flags & 0x10) /* No minimal encap */
       code = 72;

vmhinfo[vmh].RegId.high = reg->Id.high;
vmhinfo[vmh].RegId.low = reg->Id.low;

if(code) /* refusal */
    {
      sendRefusalVMH(fromport,code,0,reg->homeaddr,reg->ha,vmh);
      if(debug>2)
	fprintf(stderr,"Refusal sent to VMH\n");
    
      return;
    }

/* Now we have valid registerme (as far as FA is concerned ) */

     if(vmhinfo[vmh].status != CONFIRMED){
      vmhinfo[vmh].status |= PENDING;
      vmhinfo[vmh].timeleft = 0;}

     vmhinfo[vmh].RegId.high = reg->Id.high;
     vmhinfo[vmh].RegId.low = reg->Id.low;
     vmhinfo[vmh].fromport = fromport;

     toaddr.sin_family = AF_INET;
     toaddr.sin_port = htons(MIPREGPORT);
     toaddr.sin_addr.s_addr = vmhinfo[vmh].ha;

     reg = (struct registerme *) msg;
     
     errno = 0;

     if(sendto(RegMeVMHid,(char *)msg,len,0,
	      (struct sockaddr *) &toaddr, sizeof(struct sockaddr))<0)
      {
	perror("FA: sendRegisterme for HA failed in sendto");

	 if(errno == ECONNREFUSED || 
	     errno == ENETUNREACH || errno == EHOSTUNREACH);
	  else{
	    cleanup(); exit(-1);
	  }
      }

    if(debug > 0){
      fprintf(stderr,"====================================\n");
      if(!errno)
	fprintf(stderr,"REGISTERME FORWARDED TO %s PORT %d\n",
	      inet_ntoa(toaddr.sin_addr),(unsigned int) toaddr.sin_port);
    }
    
  } /* End of else part */

 } /* End of ProcessRegisterme */

/* Handle ragister reply from HA  */

void processRegReply(char *inmsg, int len, struct sockaddr_in *from)
  {
     struct regreply *rreply; 
     int code = 0;
     int vmh;
     int authoffset;
     char temp = 0;
     unsigned short fromport;
     struct sockaddr_in toaddr;
     char devname[10];

     strcpy(devname,DEVICE);
     rreply = (struct regreply*) inmsg;
     fromport = (unsigned short)from->sin_port;
     
     if(debug>0)
       {
	 fprintf(stderr,"================================\n");
	 fprintf(stderr,"REPLY from HA %8lx for VMH %8lx Code %2d\n",
		 htonl(rreply->ha),htonl(rreply->homeaddr),rreply->code);

	 if(len < sizeof(struct regreply))
	   fprintf(stderr,"msg length short for register reply...ignoring\n");
	 fprintf(stderr,"===============================\n");
       }

     vmh = whichvmh(rreply->homeaddr);
     if(vmh == 0)
       {
	 if(debug>0)
	   fprintf(stderr,"Regreply for unsupported VMH..ignoring\n");
	 return;
       }
     else if(!(vmhinfo[vmh].status & CONFIRMED))
       {
	 if(debug>0)
	   fprintf(stderr,"Unexpected regreply...ignoring\n");
	 return;
       }
     
     if(rreply->type != REGREPLYTYPE || len < sizeof(struct registerme)){
       if(debug >0) fprintf(stderr,"Bad regreply...ignoring\n");
       return;
     }

     if(debug >0){
       if(rreply->code != 0 && rreply->code != 1)
	 fprintf(stderr,"Rejection from HA\n");
       else
	 fprintf(stderr,"Acceptance from HA\n");
       fprintf(stderr,"==============================\n");
     }

     /* If reply is rejection from HA */
     if(rreply->code != 0 && rreply->code != 1){
       
       if(vmhinfo[vmh].status == PENDING) {
	 vmhinfo[vmh].status &= 0x0;
	 vmhinfo[vmh].timeleft = 0;

	 toaddr.sin_family = AF_INET;
	 toaddr.sin_addr.s_addr = vmhinfo[vmh].ipaddr;
	 toaddr.sin_port = vmhinfo[vmh].fromport;

	 lowroutset(vmhinfo[vmh].ipaddr,devname,ADDRT);

	 if(sendto(Regreplysid,(char *)inmsg,len,0,
	      (struct sockaddr *) &toaddr, sizeof(struct sockaddr))<0)
	   {
	     perror("FA: processRegreply for VMH failed in sendto");
	     if(errno == ECONNREFUSED || 
	     errno == ENETUNREACH || errno == EHOSTUNREACH);
	     else
	       {
	       cleanup();
	       exit(-1);
	     }
	   }
       
	 delay();

	 lowroutset(vmhinfo[vmh].ipaddr,devname,DELRT);
       }

       else if(vmhinfo[vmh].status == CONFIRMED)
	 {
	   toaddr.sin_family = AF_INET;
	   toaddr.sin_addr.s_addr = vmhinfo[vmh].ipaddr;
	   toaddr.sin_port = vmhinfo[vmh].fromport;
	   
	   if(sendto(Regreplysid,(char *)inmsg,len,0,
		     (struct sockaddr *) &toaddr, sizeof(struct sockaddr))<0)
	     {
	       perror("FA: processRegreply for VMH failed in sendto");
	       if(errno == ECONNREFUSED || 
		  errno == ENETUNREACH || errno == EHOSTUNREACH);
	       else{
		 cleanup();
		 exit(-1);
	       }
	     }
	 }
     
     }     

/* Accept from home agent */

else
   {
     if(debug >0)fprintf(stderr,"Processing Aceept...\n");

     vmhinfo[vmh].status |= CONFIRMED;
     vmhinfo[vmh].timeleft = rreply->lifetime;

     toaddr.sin_family = AF_INET;
     toaddr.sin_addr.s_addr = vmhinfo[vmh].ipaddr;
     toaddr.sin_port = vmhinfo[vmh].fromport;
     
     lowroutset(vmhinfo[vmh].ipaddr,devname,ADDRT);
     
     if(debug>1){
       fprintf(stderr,"Sending Acceptance from HA...\n");
       fprintf(stderr,"VMHAddr: %s Port %2d \n", inet_ntoa(toaddr.sin_addr),
	       toaddr.sin_port);
     }
     if(sendto(Regreplysid,(char *)inmsg,len,0,
	      (struct sockaddr *) &toaddr, sizeof(struct sockaddr))<0)
	   {
	     perror("FA: processRegreply for VMH failed in sendto");
	     if(errno == ECONNREFUSED || 
		errno == ENETUNREACH || errno == EHOSTUNREACH);
	     else{
	       cleanup();
	       exit(-1);
	     }
	   }

     if(debug>1)fprintf(stderr,"Sent Reply to MH...\n");
     

   }/* End of Accept part */
     
   }/* End of processRegReply */
void 
main(int argc, char **argv)  {
   struct sockaddr_in from;
   static char inmsg[1024];
   char *cmd;
   int len, fromlen, c;
   extern char *optarg;
   extern int optind;
   unsigned char argsfound = 0;
   fd_set fdvec;
   struct timeval tv;
   
   cmd = argv[0];
   while ((c = getopt(argc, argv, "a:m:h:i:")) != -1)  {
      switch ((char) c)  {
       case 'a':
	 argsfound |= 0x01;
	
	 if ((haAddr = inet_addr(optarg)) == -1)  {
	    fprintf(stderr, "Bad address passed to -a.\n");
	    exit(-1);
	 };
	 break;
       case 'm':
	 argsfound |= 0x02;
  	 if ((haNetmask = inet_addr(optarg)) == -1)  {
	    fprintf(stderr, "Bad netmask passed to -m.\n");
	    exit(-1);
	 };
	 break;
       case 'h':
	 argsfound |= 0x04;
	 if (hwaddread(proxyHwAddr,optarg) == -1)  {
	    fprintf(stderr, "Bad hwaddr passed to -h.\n");
	    exit(-1);
	 };
	 break;
        /* Network interface to be used */
       case 'i':
	 argsfound |= 0x08;
	 if((strcpy(DEVICE,optarg))<0) { 
	   fprintf(stderr,"Bad device name\n");exit(-1);
	 };
	 break; 
       case '?':
	 usage(cmd);
	 exit(1);
	 break;
      }
   }
   if (argsfound != 0x0F)  {
      usage(cmd);
      exit(1);
   }
    

   if (debug > 1)  {
      fprintf(stderr, "IPaddr: %x, Mask: %x",
	      htonl(haAddr), htonl(haNetmask));
      fprintf(stderr, "HWaddr: %x:%x:%x:%x:%x:%x\n", 
	      (unsigned char) proxyHwAddr[0],
	      (unsigned char) proxyHwAddr[1],
	      (unsigned char) proxyHwAddr[2],
	      (unsigned char) proxyHwAddr[3],
	      (unsigned char) proxyHwAddr[4],
	      (unsigned char) proxyHwAddr[5]);
   }

   /* Make sure we ARE on the network we are supposed to act as an 
    * agent for and we do have tunnel support in /proc/net/dev
    * else complain and exit. Also make sure we have our effective
    * uid set to root since we will be manipulating routing tables and such.
    * We currently do not perform this check */
   
   
   init();

   while (1)  { int selcode;
      
      FD_ZERO(&fdvec); FD_SET(WhereAmIsid, &fdvec); 
      FD_SET(RegisterMesid, &fdvec); 
      FD_SET(RegReplyHAid, &fdvec); /* For reg replies from HA to 
				       be fwd to MH */ 
      tv.tv_sec = 0; tv.tv_usec = 0;
       while((selcode = select(FD_SETSIZE, &fdvec, NULL, NULL, &tv)) < 0)  {
	 if(errno == EINTR ||  errno == ERESTART);
	 else  
	 perror("select() failed in main loop.\n");
	  
       }
      /* listen for a WhereAmI message on WhereAmIsid without 
       * blocking and if there is a message, process it */
      len = 1024; fromlen = sizeof(struct sockaddr_in);
      if (FD_ISSET(WhereAmIsid, &fdvec))  {
	 if (debug > 2) 
	   fprintf(stderr, "Data on WhereAmIsid.\n");
	 if ((len = recvfrom(WhereAmIsid, inmsg, len, 0,
			     (struct sockaddr *) &from, &fromlen)) < 0)  {
	   
	   if(errno == ECONNREFUSED || 
	      errno == ENETUNREACH || errno == EHOSTUNREACH);
	   else
	     { perror("recvfrom() on WhereAmIsid failed.\n");
	       cleanup(); exit(-1);}
	 };
	 processWhereAmI(inmsg, len, &from);
      }
      len = 1024; fromlen = sizeof(struct sockaddr_in);
      if (FD_ISSET(RegisterMesid, &fdvec))  {
	 if (debug > 2) 
	   fprintf(stderr, "Data on RegisterMesid.\n");
	 if ((len = recvfrom(RegisterMesid, inmsg, len, 0,
			     (struct sockaddr *) &from, &fromlen)) < 0)  {
	   
	   perror("recvfrom() on RegisterMesid failed.\n");
	   if(errno == ECONNREFUSED || 
	      errno == ENETUNREACH || errno == EHOSTUNREACH);
	  
	   else
	     
	   {
	     cleanup(); exit(-1);
	   }
	 };
	 processRegisterMe(inmsg, len, &from);
	 if (debug > 2) 
	   fprintf(stderr, "In main finished processing RegisterMe.\n");
	 
	 
       }

      /* For register replies from HA */
      
      if (FD_ISSET(RegReplyHAid, &fdvec)){
	if(debug>2)
	  fprintf(stderr,"Data on RegReplyHAid\n");

	if ((len = recvfrom(RegReplyHAid, inmsg, len, 0,
			     (struct sockaddr *) &from, &fromlen)) < 0)  {
	   perror("recvfrom() on RegReplyidid failed.\n");
	   if(errno == ECONNREFUSED || 
	      errno == ENETUNREACH || errno == EHOSTUNREACH);
	   else{
	     cleanup(); exit(-1);
	 
	   }	 
	 };
	processRegReply(inmsg, len, &from);
      }
	slp(SLEEPINTERVAL); 	
	      }
   
 }

void
usage(char *cmd)  {
   fprintf(stderr, 
	   "Usage is: %s -a ipaddr -m netmask -h hwaddr -i interface\n",
	   cmd);
}

int 
hwaddread(char s[], char *hwadr)  {
   int a, b, c, d, e, f;
   if (sscanf(hwadr,"%x:%x:%x:%x:%x:%x", &a, &b, &c, &d, &e, &f) != 6) 
     return(-1);
   else  {
      s[0] = (unsigned char) a;
      s[1] = (unsigned char) b;
      s[2] = (unsigned char) c;
      s[3] = (unsigned char) d;
      s[4] = (unsigned char) e;
      s[5] = (unsigned char) f;
     return(0);
   }
}
 
void
sendReply(unsigned short fromport, int code, int time, 
	  unsigned long homeaddr, unsigned long coaddr, 
	  unsigned long haAddr, struct id replyid)  
{
   struct sockaddr_in sock;
   static char outmsg[1024];
   struct regreply *reply;
   int len=sizeof(struct regreply);
   int mh;

   sock.sin_family = AF_INET;
   if(coaddr != 0)
     sock.sin_addr.s_addr = coaddr;
   else
     sock.sin_addr.s_addr =homeaddr;
   sock.sin_port = fromport;
   reply = (struct regreply *) outmsg;
   reply->type = 3;
   reply->code = (unsigned char) code;
   reply->lifetime = (unsigned short) time;
   reply->homeaddr = homeaddr;
   reply->ha = haAddr;
   reply->Id.high = replyid.high;
   reply->Id.low = replyid.low;

   if(coaddr != 0){  /* for diff bet FA and HA reply */
   mh = whichmh(homeaddr);
   appendauth(outmsg, &len, mhinfo[mh].secret, mhinfo[mh].keylen,mhinfo[mh].SPIval);
 }
   if (debug > 0)  {
      fprintf(stderr, "\n==========================\n");
      printtime();
      fprintf(stderr, "-- REPLY to %s at port %d ",
	      inet_ntoa(sock.sin_addr), sock.sin_port);
      if ((reply->code != 0) && (reply->code != 1))
      	fprintf(stderr, "(Rejection)\n");
      else
      	fprintf(stderr, "(Acceptance)\n");
      
      if (debug > 1)  {
	 fprintf(stderr, "[%8lx:%8lx] Type %2d Code %3d Lifetime %8d\n",
		 reply->Id.high, reply->Id.low, reply->type, reply->code,
		 reply->lifetime);
	 fprintf(stderr, "Homeaddr: %8lx, Homeagent: %8lx",
		 htonl(reply->homeaddr), htonl(reply->ha));
         printext(outmsg, len, 1);
	 if (debug > 2) {
	   fprintf(stderr, "\n--------------------\n");
	   testprint(outmsg, len);
	   fprintf(stderr, "--------------------");
	 }
      }
      fprintf(stderr, "\n==========================\n");
    }
   
   if (sendto(Regreplysid, (char *) outmsg, len, 0,
	      (struct sockaddr *) &sock, sizeof(struct sockaddr)) < 0)  {
		 perror("sendReply(): send to failed.\n");

		 if(errno == ECONNREFUSED || 
		    errno == ENETUNREACH || errno == EHOSTUNREACH);
		 else
		 
		   {
		     cleanup(); exit(-1);
		   }
	       }
 }



/* send refusal from FA to MH */
void
sendRefusalVMH(unsigned short fromport,int code,int time,
	       unsigned long homeaddr,unsigned long ha,int vmh)
{
  char devname[10];
  
  strcpy(devname,DEVICE);

/* set host specific route to visiting MH to send refusal */ 
  lowroutset(homeaddr,devname,ADDRT);

  sendReply(fromport, code, time, homeaddr, 0, ha, vmhinfo[vmh].RegId);
  delay();

  /* delete the route */
  lowroutset(homeaddr, devname, DELRT);

}

void
sendConfirm(unsigned short fromport, int code, int time, 
	    unsigned long homeaddr, unsigned long coaddr,
	    unsigned long haAddr, int mh)  {
  sendReply(fromport, code, time, homeaddr, coaddr, haAddr, 
	    mhinfo[mh].RegistrationId);	       
}

void
delay()  {
   int i; /* artificial delay in an attempt to ensure that packet is
	   * sent before we change the kernel routing tables again */
   
   for (i = 0; i <= 100000; i++) ;
}

void
sendRefusal(unsigned short fromport,int code,int time,
	    unsigned long homeaddr,unsigned long coaddr,
	    unsigned long ha,int mh)
{
  
  sendReply(fromport, code, time, homeaddr, coaddr, 
		   ha, mhinfo[mh].RegistrationId);
}


