/*----------------------------------------------------------------------------

  Copyright (C) 1995-6, Vipul Gupta and Abhijit Dixit. Created 1995-6. All
  rights reserved.

  Linux-Mobile-IP
            An implementation of Mobile IP for the LINUX operating system
            developed at the State University of New York, Binghamton
            (with partial support from the Center for Computing Technologies).
            Except as noted in the accompanying documentation, this
            implementation complies with revision 16 of the Internet 
            Engineering Task Force (IETF) Mobile-IP draft.

            More information can be obtained from:
                  http://anchor.cs.binghamton.edu/~mobileip/  

  Version:   1.00     05/23/1996
 
  Authors:   Abhijit Dixit <abhijit@cs.binghamton.edu> 
             Vipul Gupta <vgupta@cs.binghamton.edu>
             Benjamin Lancki <ben@anchor.cs.binghamton.edu>

  Permission is hereby granted to redistribute this code and/or
  modify it under the terms of the GNU Genral Public License
  as published by the Free Software Foundation; either version
  two or (at your option) any later version provided this ENTIRE
  notice is retained in any copies or any part of this software.

  A copy of the GNU General Public License can be obtained by contacting
  the Free Software Foundation, Inc. 675 Mass Ave, Cambridge, MA 02139, USA.
             
-----------------------------------------------------------------------------*/ 
/*
 *  
 *                      mh.c : Design of the Mobile Host
 *
 *  Command line arguments:
 *    -a : mobile host's IP address on its home network 
 *    -m : network mask corresponding to the IP address above
 *    -g : IP address on the mobility interface of the homeagent.
 *    
 *
 *
 * History of Changes:
 *     Jul 25, 1995              First version coded  - V.G.
 *     Aug  1, 1995              Removed bug in processURhere. Now we
 *                               turn off REGAWAY when we move home and
 *                               turn off REGHOME when we move away - V.G.
 *     Aug  2, 1995              Moved URhere gathering in alarm_handler()
 *                               as part of preventing a flip-flop effect
 *                               when a mobile host hears URhere messages
 *                               from multiple mobility agents 
 *                               alternately - V.G.
 *     Aug  9, 1995              Added support for MD5 authentication in
 *                               processing the registration and registration 
 *                               reply - B.L.
 *     Aug 13, 1995              Agent now reads config file /etc/mip-mh.ok to
 *                               initialize mhinfo struct - B.L.
 *     Aug 14, 1995              Agent keeps a log in event of system crash
 *                               or shutdown and recovers on startup - B.L.
 * 
 *
 *     Sept 20, 1995             Low level routine added for adding and 
 *                               deleting route to a network -A.D.
 *
 *
 *
 *     Sept 22, 1995             Changed the code so Registerme mesgs are
 *                               now sent to FA instead of home agent;
 *				 because of this change , reply from home 
 *                               agent forwarded by foreign agent.
 *				 Separate mobile subnet is 
 *				 not required. -A.D.
 *
 *     Sept 30, 1995             mh does not exit when select fails on 
 *                               INTR -A.D
 *
 *     Oct 5, 1995               ARP cache is flushed when MH moves to
 *                               new network.-A.D.
 *      
 *     Oct 10, 1995              mh does not exit when sendto fails
 *                               because of network or host unreachable
 *                               condition. -A.D.   
 *				 
 *     Nov 10, 1995              Due to addition of a 'special sleep'
 *                               mh consumes very little cpu time.- A.D.    
 *
 *     Dec 5, 1995               Router Solicitation
 *                               Messages changed to use ICMP -A.D.
 * 
 *     April  1996               Complete redesign of the Mobile Host. 
 *                               MH now operates without Foreign Agent (Self 
 *                               Decapsulation) and some of the other 
 *                               inefficiencies are removed. 
 *                               - V.G. and A.D.
 *     
 *     May    1996               MH now sends gratuitous ARP after to coming to home network.
 *                               -A.D.
*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/ip.h>

#include "mh.h"
#include "low.h"

#define did_reboot(new,prev) (((new)<(prev))&((new)<256))
/* Application required flags */
#define APREF 0x00

int debug = 2;
short doit  = 0;

char *state[] =  { "INIT", "ATHOME", "ATFOREIGN", "ASPOPUP"};
extern int errno;

extern void 
testprint(char *,int);

extern void 
appendauth(char *,int *,char *,int,unsigned long);

extern int 
authextfound(char *,int,int);

extern int 
authok(char *,int,int,char *,int,unsigned long);

extern void 
printext(char *,int,int);

extern void 
printtime();

extern int  
in_cksum(u_short*,int);

extern int 
lowdefifcreq(u_long*,u_long*,char*);

void sendRegisterMe(int, unsigned char, unsigned short, unsigned long,
		    unsigned long, unsigned long);
int routersolId, URheresid, Regreplysid, RegisterMesid;
int ioctlsid;

void usage(char *), cleanup();

struct sockaddr_in routersolto =
     {  (short) AF_INET, (unsigned short) 0,
	(unsigned long) 0, (unsigned long) 0 };

char lastreq[1024]; 
/* this is the last request on which we are awaiting a reply */

void init();
myinfo Myinfo;
dm_info Dminfo;

short backoff[] = {1,2,4,8,16,32,64};
short renewal[] = {2,4,8};

struct stats  {
   unsigned long badurhere;
   unsigned long badregreply;
   unsigned long idmismatch;
   unsigned long authfailed;
   unsigned long regsent;
   unsigned long regfailed;
   unsigned long regsuccessful;
} mhstats =  { (long) 0, (long) 0,  (long) 0, (long) 0,
               (long) 0, (long) 0,  (long) 0 };

void
initsockets()  {
   struct sockaddr_in sa;
   int enable = 1;
   
   bzero((void*)&sa,sizeof(sa));
   routersolId = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP);
   Regreplysid = RegisterMesid = socket(AF_INET, SOCK_DGRAM, 0);
   URheresid = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP);
   ioctlsid = socket(AF_INET, SOCK_DGRAM, 0);
   if (Regreplysid < 0 || URheresid < 0 || 
       routersolId < 0 || RegisterMesid < 0 || ioctlsid < 0)  {
	  fprintf(stderr, "initsockets(): could not create sockets.\n");
	   exit(-1);
       }

   sa.sin_family = AF_INET; sa.sin_addr.s_addr = htonl(INADDR_ANY); 

   if (setsockopt(routersolId, SOL_SOCKET, SO_BROADCAST, &enable, sizeof(int))
       < 0)  {
	  perror("initsockets(): setsockopt SO_BROADCAST failed");
	  cleanup(); exit(-1);
       }

   /* This is where we receive regreplies and send registermes from */
   sa.sin_port = 0; 
   if (bind(RegisterMesid, (struct sockaddr *) &sa, sizeof(sa)) < 0)  {
      perror("initsockets(): Bind RegisterMesid failed.\n");
      cleanup(); exit(-1);
   }
   if (debug > 0)
     fprintf(stderr, "Initialized sockets ...\n");
}

void
cleanup()  {
   
   if (debug > 1)
     fprintf(stderr, "Done cleaning up ... \n");
}

void
graceful_exit()  {
   struct sockaddr_in sock;
   unsigned char flags = 0x00;
   
   cleanup();
   if ((Myinfo.status & 0x03)== ATFOREIGN || 
       (Myinfo.status & 0x03)  == ASPOPUP)  {
      Myinfo.retries = 0;
      sendRegisterMe((int) 0, flags, (unsigned short) 0, Myinfo.homeaddr, 
		     Myinfo.homeagent, Myinfo.coaddr);
   }
   /* deregister ourselves if we were registered or this will
    * tie up a tunnel for atmost MAXLIFETIME period */ 
   exit(0);
}

void
sendroutersol(struct sockaddr_in *toaddr)
{
  struct icmp msg;
  memset((char*)&msg,0,sizeof(struct icmp));
  
  if (debug > 0)
    fprintf(stderr, "\n-- RouterSol to  %s.\n",
	     inet_ntoa(toaddr->sin_addr));
  msg.type = ROUTERSOLTYPE;
  msg.code = 0;
  msg.cksum = 0;
  msg.cksum = in_cksum((u_short*)&msg,sizeof(struct icmp));
  if (sendto(routersolId, (char *) &msg, sizeof(msg), 0, 
	     (struct sockaddr *) toaddr, sizeof(struct sockaddr)) < 0)  {
    perror("sendroutersol(): send to failed.\n");
    
    if(errno == ECONNREFUSED || 
       errno == ENETUNREACH || errno == EHOSTUNREACH);
    else
		 
      {
	cleanup(); exit(-1);
      }
  }
}

void
sendRegisterMe(int attempt, unsigned char flags, unsigned short lifetime, 
	       unsigned long homeaddr, unsigned long homeagent,
	       unsigned	long coaddr)
{
   struct sockaddr_in toaddr;
   struct registerme *req;
   static int len;

   toaddr.sin_family = AF_INET;
   toaddr.sin_port = htons(MIPREGPORT);
   
   /* Depending upon ASPOPUP and ATFOREIGN states toaddr is selected */
   if(Myinfo.status & 0x03 == ASPOPUP)
     toaddr.sin_addr.s_addr = Myinfo.homeagent;
   else
     toaddr.sin_addr.s_addr = Myinfo.closeagentaddr; 

   if(attempt != 0)
     {
       fprintf(stderr,"Retransmitting the same (earlier) request\n");
       if (sendto(RegisterMesid, (char *) lastreq, len, 0, 
	      (struct sockaddr *) &toaddr, sizeof(struct sockaddr)) < 0)  {
		 perror("sendRegisterMe(): send to failed.\n");
		 if(errno == ECONNREFUSED || 
		    errno == ENETUNREACH || errno == EHOSTUNREACH);
		 else
		 
		   {
		     cleanup(); exit(-1);
		   }
	      };
       return;
       
     }

   len = sizeof(struct registerme);
   req = (struct registerme *) lastreq;
   req->type = REGISTERMETYPE;
   req->flags = flags;
   req->lifetime = lifetime;
   req->homeaddr = homeaddr;
   req->ha = homeagent;
   req->coaddr = coaddr;
   /* we generate a new 64-bit Id unless this is a retransmission */
   if (attempt == 0)
     NewId(&Myinfo.RequestId, &Myinfo.RequestId);
   req->Id.high = Myinfo.RequestId.high;
   req->Id.low = Myinfo.RequestId.low;

   appendauth(lastreq, &len, Myinfo.secret, Myinfo.secretlen,Myinfo.SPIval);

   if (debug > 0)  {
     fprintf(stderr, "\n==========================\n");
     printtime();
     if (req->lifetime == 0) 
       fprintf(stderr, "-- DEREGISTERME to %s at port %d (attempt %d).\n",
	       inet_ntoa(toaddr.sin_addr), (unsigned int) toaddr.sin_port,
	       attempt);
     else
       fprintf(stderr, "-- REGISTERME to %s at port %d (attempt %d).\n",
	       inet_ntoa(toaddr.sin_addr), (unsigned int) toaddr.sin_port,
	       attempt);
     if (debug > 1)  {
       fprintf(stderr, "[%8lx:%8lx] Type %2d Flags %2x Lifetime %8d\n",
	       req->Id.high, req->Id.low, req->type, req->flags, req->lifetime);
       fprintf(stderr, "Homeaddr: %8lx, Homeagent: %8lx, Careof: %8lx",
	       htonl(req->homeaddr), htonl(req->ha), htonl(req->coaddr));
	 if (debug > 2) {
	   fprintf(stderr, "\n--------------------\n");
	   printext(lastreq, len, 0);
	   testprint(lastreq, len);
	   fprintf(stderr, "--------------------");
	 }
      }
      fprintf(stderr, "\n==========================\n");
   }
   
   if (sendto(RegisterMesid, (char *) lastreq, len, 0, 
	      (struct sockaddr *) &toaddr, sizeof(struct sockaddr)) < 0)  {
		 perror("sendRegisterMe(): send to failed.\n");
		 if(errno == ECONNREFUSED || 
		    errno == ENETUNREACH || errno == EHOSTUNREACH);
		 else
		 
		   {
		     cleanup(); exit(-1);
		   }
	      };
}

void dsn_maker(int);

void
alarm_handler()  {
   struct registerme *req;
   
   if(debug >2)
     fprintf(stderr,"The current state is %s\n",state[Myinfo.status & 0x03]);
   
   if((Myinfo.status & 0x03)== INIT)
     sendroutersol(&routersolto); 
   dsn_maker(URheresid);
   
   /* we take care of our registration renewals */
   if (!((Myinfo.status & 0x03) == ATHOME && 
	 (Myinfo.status & DEREGISTERED)||Myinfo.status == INIT)) 
     {
       Myinfo.timeleft-=2;
      
       if(Myinfo.status & REGISTERED)
	 {
	   if(Myinfo.retries <3){
	     if(Myinfo.timeleft <= Myinfo.lifetime/renewal[Myinfo.retries])
	       {
		 /* Send new registration request */
		 Myinfo.status |= AWAITINGREPLY;
		 sendRegisterMe((int) 0, Myinfo.flags,Myinfo.lifetime,Myinfo.homeaddr, 
			    Myinfo.homeagent, Myinfo.coaddr);
		 Myinfo.retries++;
	       }
	   }
	   else if (Myinfo.timeleft <= 0)  {
	     Myinfo.timeleft = 0;
	     /* Restore normal POPUP state so we ca have at least simple
		connectivity */
	     if((Myinfo.status & 0x03)==ASPOPUP)
	       {
		 lowifaceset(Myinfo.infname, Myinfo.coaddr, Myinfo.foreignnetmask);
		 lowifacedown("dummy");
	       }
	     init((int)1);
	 
	     if (debug > 1)  {
	       fprintf(stderr, "Registration expired ... \n");
	     }
	
	   }
	 }

       /* Take care of an request Retransmissions if NOT YET registered */
       else {
	 if (Myinfo.timeleft <= 0)  {
	   Myinfo.timeleft = 0;
	   if(Myinfo.retries < 6)
	     {
	       Myinfo.timeleft = (backoff[Myinfo.retries++])*RETRYINTERVAL;
	       req = (struct registerme *) lastreq;
	   
	       sendRegisterMe(Myinfo.retries, req->flags, req->lifetime, 
			   req->homeaddr, req->ha, req->coaddr);
	     }
	  else {
	    /* MH has retransmitted this request enough times and
	       and now it goes back to INIT state */
	    init((int)1);
	    if (debug > 1)  {
	      fprintf(stderr, "We have retransmitted enough ... \n");
	    }
	  }
	 }
       }
     }
   signal(SIGALRM, alarm_handler);
   alarm(1);
}

slp(double sec)
{
  struct timeval tv;
  
  tv.tv_sec = 0; tv.tv_usec = sec*1000000 ;

  if((select(FD_SETSIZE, NULL, NULL, NULL, &tv)) < 0)  {
    if(errno == EINTR ||  errno == ERESTART);
    else  
      perror("select() failed in main loop.\n");
    
  }
 
}
void
init(int level)  {
   
   Myinfo.timeleft = Myinfo.closeagentaddr = Myinfo.retries =0; 
   Myinfo.agentflags = Myinfo.seqno = Myinfo.lifetime =0;
   Myinfo.coaddr = 0;
   Myinfo.status = 0x00; /*INIT*/
   
   if(level == 0){
     bzero((void*)&Dminfo,sizeof(dm_info));
     initsockets();
     routersolto.sin_addr.s_addr = inet_addr(ADVERTISEON);
     readhadata("/etc/mip-ha.dat");
     signal(SIGTERM, graceful_exit);
     signal(SIGINT, graceful_exit);
     signal(SIGALRM, alarm_handler);
     alarm(1);
   }
}
   
int
IsOurAddress(unsigned long addr)  {
   return((Myinfo.homeaddr == addr) ? 1:0);
}

int
IsOurAgent(unsigned long addr)  {
   return((Myinfo.homeagent == addr) ? 1:0);
}

int
mhIdcmp(struct id *id1, struct id *id2)  {
   return((id1->low == id2->low) ? 0:1);
}

int
NewId(struct id *id1, struct id *id2)  {
   static count = 0;
   struct timeval tv;
   
   if (count++ == 0)  {
      gettimeofday(&tv, NULL);
      srandom((int) tv.tv_usec);
   };
   id1->high = id2->high;
   id1->low = (unsigned long) random();
}


void 
processRegreply(char *msg, int len, struct sockaddr_in *from)  {
   struct regreply *reply;
   int authoffset;
   char hwaddr[10];

   reply = (struct regreply *) msg;

   /* This should never happen (probability is very very low) because 
      after we get a good reply we change the Id that we now expect 
      to a value we never sent 
      */
   if (Myinfo.status & AWAITINGREPLY == 0)  {
      fprintf(stderr, "Got a reply when we weren't expecting it.\n");
      return;
    }

   /* Print contents of received message */
   if (debug > 0)  
     {
       fprintf(stderr, "\n==========================\n");
       printtime();
       fprintf(stderr, "-- REPLY from %s ", inet_ntoa(from->sin_addr));
       if ((reply->code != 0) && (reply->code != 1))
	 fprintf(stderr, "(Rejection)\n");
       else
	 fprintf(stderr, "(Acceptance)\n");
       if (debug > 1)  {
	 fprintf(stderr, "[%8lx:%8lx] Type %2d Code %3d Lifetime %8d\n",
		 reply->Id.high, reply->Id.low, reply->type, reply->code,
		 reply->lifetime);
	 fprintf(stderr, "Homeaddr: %8lx, Homeagent: %8lx",
		 htonl(reply->homeaddr), htonl(reply->ha));
	 if (debug > 2) {
	   fprintf(stderr, "\n--------------------\n");
	   printext(msg, len, 1);
	   testprint(msg, len);
	   fprintf(stderr, "--------------------");
	 }
       }
       fprintf(stderr, "\n==========================\n");
     }
   
   Myinfo.RequestId.high = reply->Id.high;
   Myinfo.retries = 0;

   if (mhIdcmp(&Myinfo.RequestId, &(reply->Id)) != 0)  {
     mhstats.idmismatch++;
     if (debug > 0)  {
       fprintf(stderr, "Id mismatch -- Reply ignored.\n");
     }
      return;
   }

   /* Check whether a valid reply */
   if (reply->type != REGREPLYTYPE || len < sizeof(struct regreply) ||
       !IsOurAgent(reply->ha) || !IsOurAddress(reply->homeaddr) ||
     ((reply->code<16 || reply->code>127) && 
      !(authoffset=authextfound(msg, len, 1)))) {
	  mhstats.badregreply++;
	  if (debug > 0)  {
	   fprintf(stderr, "Bad structure -- Reply ignored.\n");
	  }
	  return;
       }

   if ((reply->code<16 || reply->code>127) && 
	      !authok(msg, len, authoffset, Myinfo.secret, Myinfo.secretlen,Myinfo.SPIval))
     {
       mhstats.authfailed++;
       if (debug > 0)  {
	 fprintf(stderr, "Authentication failed -- Reply ignored.\n");
       }
       return;
     }
   
    

   Myinfo.status &= ~AWAITINGREPLY;
  
   if (reply->code == 0 || reply->code == 1)  {
      /* Our service request was approved */
     
     mhstats.regsuccessful++;
     if (reply->lifetime == 0)  
       {
	 /* deregistration approved */
	 /* Timer won't be running now */
	 Myinfo.timeleft = 0;
	 Myinfo.status |= DEREGISTERED;
       } 
     else  
       {
	 Myinfo.timeleft = reply->lifetime;
	 Myinfo.status |= REGISTERED;
	 
	 if((Myinfo.status & 0x03) == ASPOPUP)
	   {
	     lowifaceset(Myinfo.infname,Myinfo.homeaddr,Myinfo.homenetmask);
	     lowifaceset("dummy",Myinfo.coaddr,Myinfo.foreignnetmask);
	     lowhwaddrreq(Myinfo.infname,hwaddr);
	     lowARPreq(PROXY, Myinfo.coaddr, hwaddr);
	     lowrtset(Myinfo.coaddr, "dummy", ADDRT);
	   }
       }
   }
 
   else /* Rejection */  
     {
       /* If we are not yet registered then rejection might be due to ID mismatch, 
	  so without waiting for timer to go off,send registration */
       if(!(Myinfo.status & REGISTERED))
	 {
	   Myinfo.status |= AWAITINGREPLY;
	   Myinfo.timeleft = RETRYINTERVAL;
	   sendRegisterMe((int) 0, Myinfo.flags,Myinfo.lifetime,Myinfo.homeaddr, 
			    Myinfo.homeagent, Myinfo.coaddr);
	 }
       
       mhstats.regfailed++;
   }
   
}


/*Check whether received router discovery message is valid
  and cannot be ignored
  */
int cannotignore(char *msg, u_long *src, int len)
{
  struct youarehere *urh;
  char *p;
  struct icmp *icp;
  unsigned long *addr;
  
  /* We first check weather message is not corrupt or invalid */
  icp = (struct icmp*)msg;

  if(icp->type != ROUTERADVTYPE)
    {
      if(debug>1)
	fprintf(stderr,"Not a router discovery message\n");
      return FALSE;
    }

  urh = (struct youarehere*)(msg + SIZE_ICMP_HDR + 
			     4*(icp->addr_entry_size)*(icp->addrnum));
  len -= SIZE_ICMP_HDR +  4*(icp->addr_entry_size)*(icp->addrnum);

  
  if(urh->type != YOUAREHERETYPE || len <(sizeof(struct youarehere) + (urh->length-6)))
     {
       fprintf(stderr,"Rt discovery message is too short or type mismatch\n");
       return FALSE;
     }
      
  p = msg + SIZE_ICMP_HDR +  4*(icp->addr_entry_size)*(icp->addrnum)
    + sizeof(struct youarehere);
          
  addr = (unsigned long *) p;

  if (debug > 2)  {
    int i;   
    fprintf(stderr, "Rt. discovery  addrnum = %d, flags = %x  seqno = %d lifetime = %d\n",
	      (urh->length-6)/4, urh->flags,urh->seqno,urh->lifetime);
      for (i = 0; i < (urh->length-6)/4; i++)
      	fprintf(stderr, "Address %lx\n", htonl(*(addr + i)));
   }

  if((IsOurAgent(*addr) && (urh->flags & MSHFLAG))
     ||(!IsOurAgent(*addr) && (urh->flags & MSFFLAG)&&
	(!(urh->flags & MSBFLAG))&& urh->lifetime !=0))
    {
      *src = *addr;
      return TRUE;
    }

  else return FALSE;
}



/* This function decides which action should be taken: HOMEACTION,
   FOREIGNACTION or POPUPACTION. It uses PickupURheres() to 
   get any received router die]scovery message */

void dsn_maker(int sockid)
{
  unsigned long ipaddr,src,gw;
  short i,heard_no_adv;
  char dev[10], *p;
  static char msg[256];
  unsigned int msglen,fromlen;
  struct sockaddr_in from;
  struct youarehere *urh;
  fd_set fdvec;
  static struct timeval tv;
  static struct icmp *icp;
  static struct iphdr *ip;
  u_short iphdrlen;
  
  msglen = 0;
  bzero(msg,sizeof(msg));
  bzero((void*)&from,sizeof(struct sockaddr_in));
  if(!doit)Dminfo.action = 0;
  
  if(debug >2)
    fprintf(stderr,"Decision Maker with doit = %d\n",doit);

  if(lowdefifcreq(&ipaddr,&gw,dev)<0)
    {
      fprintf(stderr,"lowdefifcreq failed\n");
      return;
    }

  if(debug >2)
    {
       fprintf(stderr,"Default interface address %lx Interface name %s Gateway : %lx\n",
	      ipaddr, dev, gw);
    }
  

  heard_no_adv = TRUE;

  for(i=0;i<MAXOVERLAP;i++)
    {
      FD_ZERO(&fdvec);
      FD_SET(sockid,&fdvec);
      tv.tv_sec = 0;
      tv.tv_usec = 0;
      
      if(select(FD_SETSIZE,&fdvec,NULL,NULL,&tv)<0)
	{
	  if(errno == EINTR); /* This is expected since alarm is ruuning so do nothing */
	  else
	    perror("select() failed in dsn_maker()");
	}
	
      if(FD_ISSET(sockid,&fdvec)){
            
	msglen = sizeof(msg);
	fromlen = sizeof(struct sockaddr_in);
	if((msglen = recvfrom(sockid,msg,msglen,0,(struct sockaddr*)&from, &fromlen))<0)
	  {
	    if(errno == ECONNREFUSED||errno == ENETUNREACH || errno == EHOSTUNREACH);
	    else
	      perror("recvfrom() failed in dsn_maker");
	  }

	if(msglen < 0)continue;

	ip = (struct iphdr *)msg;
	iphdrlen = ip->ihl <<2; /*Bytes */
	icp = (struct icmp*)(msg + iphdrlen); /* remove IP header */
	msglen -= iphdrlen; 
	
	if(debug >2)
	  fprintf(stderr,"Recvd message on rtdisc socket from %s \n",inet_ntoa(from.sin_addr));
	else
	  fprintf(stderr,".");

	if(cannotignore((char*)icp,&src,msglen) && (heard_no_adv || src == Myinfo.closeagentaddr))
	  {
	    heard_no_adv = FALSE;
	    p = (msg + iphdrlen + SIZE_ICMP_HDR + 
				       4*(icp->addr_entry_size)*(icp->addrnum));
	    msglen -= SIZE_ICMP_HDR +4*(icp->addr_entry_size)*(icp->addrnum);
	    /*Store only router discovery message */
	    memcpy(Dminfo.rtdismsg,p,msglen);
	    memcpy((char*)&Dminfo.from,(char*)&from,sizeof(struct sockaddr_in));
	  }
      }
      
    }

  /* Indicate in log that no adv is heard*/
  if(heard_no_adv)
    fprintf(stderr,"?");

  if(doit)return;

  if(!IsOurAddress(ipaddr)&& ipaddr != 0)
    {
      /* This indicates that MH interface addr is
	 changed. So go for POPUPACTION */
      
      Dminfo.action = POPUPACTION;
      Dminfo.coaddr = ipaddr;
      Dminfo.gw = gw;
      Dminfo.closeagentaddr = 0;
      strcpy(Dminfo.infname,dev);
      Dminfo.seqno = 0;
      Dminfo.lifetime = MAXLIFETIME;
      Dminfo.flags = 0; 
      /*
      bzero(Dminfo.rtdismsg,sizeof(Dminfo.rtdismsg));
      bzero(&(Dminfo.from),sizeof(struct sockaddr_in));
      */
      }

  if(heard_no_adv && Dminfo.action == POPUPACTION)
    {
      /* This indicates that MH has acquired new IP address
	 due to PPP or DHCP and no router discovery messages are
	 heard.
	 */
       
      doit =1;
    }

  else if(heard_no_adv == FALSE)
    { 
      urh = (struct youarehere*)Dminfo.rtdismsg;
      if(IsOurAgent(src))
	{
	  Dminfo.action = HOMEACTION;
	  Dminfo.closeagentaddr = src;
	  Dminfo.coaddr = Myinfo.homeaddr;
	  Dminfo.gw = gw;
	  strcpy(Dminfo.infname,dev);
	  Dminfo.seqno = urh->seqno;
	  Dminfo.lifetime = urh->lifetime;
	  Dminfo.flags = urh->flags;
	  doit = 1;
	}
      else /* Not at Home */
	{
	  Dminfo.action = FOREIGNACTION;
	  Dminfo.closeagentaddr = src;
	  Dminfo.coaddr = src;
	  Dminfo.gw =gw;
	  strcpy(Dminfo.infname,dev);
	  Dminfo.seqno = urh->seqno;
	  Dminfo.lifetime = urh->lifetime;
	  Dminfo.flags = urh->flags;
	  doit = 1;
	}
    }
}

void copy_new()
{
  Myinfo.coaddr = Dminfo.coaddr;
  Myinfo.closeagentaddr = Dminfo.closeagentaddr;
  Myinfo.gw = Dminfo.gw;
  strcpy(Myinfo.infname,Dminfo.infname);
  Myinfo.seqno = Dminfo.seqno;
  Myinfo.lifetime = Dminfo.lifetime;
  Myinfo.agentflags = Dminfo.flags;
}

doer()
{
  struct sockaddr_in from;
  char msg[1024];
  int len, fromlen;
  fd_set fdvec;
  struct timeval tv;

  if(doit)
    switch(Dminfo.action){
    case FOREIGNACTION :
      if(Dminfo.coaddr != Myinfo.coaddr || did_reboot(Dminfo.seqno,Myinfo.seqno))
	{
	  if(debug>1)
	    fprintf(stderr,"Doer processing FOREIGNACTION with Coaddr %lx\n",
		    Dminfo.coaddr);

	  /* Flush all ARP entries */
	  lowflshARP();
	  /* Down dummy (if present) */
	  lowifacedown("dummy");
	  /* Delete earlier default route */
	  lowrtset(0x00000000,Myinfo.infname,DELRT);
	  /* Delete route to home network if present */
	   lowrtnetset((Myinfo.homeagent & Myinfo.homenetmask),Myinfo.infname,0,DELRT);
	  /* Delete network route if earlier state was ASPOPUP */
	  if((Myinfo.status & 0x03)==ASPOPUP)
	    lowrtnetset((Myinfo.coaddr & Myinfo.foreignnetmask),Myinfo.infname,0,DELRT);
	  /* Delete route to existing default gateway*/
	  lowrtset(Myinfo.gw,Myinfo.infname,DELRT);
	  /* If it is INIT delete route to home agent anyway */
	  if((Myinfo.status & 0x03) == INIT)
	    lowrtset(Myinfo.homeagent,Myinfo.infname,DELRT);
	  /* Clear Myinfo*/
	  init(1);
	  /* Copy new info */ 
	  copy_new();
	  
	  /* Set new routes */
	  if(debug>2)fprintf(stderr,"Setting new routes...\n");
	  lowrtset(Myinfo.closeagentaddr,Myinfo.infname,ADDRT);
	  lowrtdefault(Myinfo.closeagentaddr);
	  Myinfo.timeleft = RETRYINTERVAL;
	  Myinfo.flags = 0x00|APREF; /* No Broadcast , FA will decapsulate, etc */
	  Myinfo.status = ATFOREIGN | WAITINGTOREGISTER | AWAITINGREPLY;
	  
	  sendRegisterMe(Myinfo.retries, Myinfo.flags, Myinfo.lifetime,
			 Myinfo.homeaddr, Myinfo.homeagent, Myinfo.coaddr);
	  if(debug >1)
	    fprintf(stderr,"Done with FOREIGNACTION\n");
			 
	}
      doit = 0;
      break;

    case HOMEACTION :
      if(Dminfo.coaddr != Myinfo.coaddr || did_reboot(Dminfo.seqno,Myinfo.seqno))
	{ 
	  char hwaddr[6];
	  if(debug>1){
	    fprintf(stderr,"Doer processing HOMEACTION\n");
	  }
	  
	  /* Flush all ARP entries */
	  lowflshARP();
	  /* Down dummy (if present) */
	  lowifacedown("dummy");
	  /* Delete earlier default route */
	  lowrtset(0x00000000,Myinfo.infname,DELRT); 
	  /* Delete network route if earlier state was ASPOPUP */
	  if((Myinfo.status & 0x03)==ASPOPUP)
	    lowrtnetset((Myinfo.coaddr & Myinfo.foreignnetmask),Myinfo.infname,0,DELRT);
	  /* Delete route to existing default gateway*/
	  lowrtset(Myinfo.gw,Myinfo.infname,DELRT);
	  /* Clear Myinfo*/
	  init(1);
	  /* Copy new info */
	  copy_new();
	  
	  /* Set new routes */
	  if(debug>2)fprintf(stderr,"Setting new routes...\n");
	  lowrtnetset((Myinfo.homeagent & Myinfo.homenetmask),Myinfo.infname,Myinfo.homenetmask,ADDRT);
	  lowrtdefault(Myinfo.closeagentaddr);
	  /* Send gratuitous ARP*/
	  lowhwaddrreq(Myinfo.infname,hwaddr); /* get hwaddr */
	  lowarpsend(Myinfo.infname,Myinfo.homeaddr,hwaddr,Myinfo.homeaddr);

	  Myinfo.timeleft =  RETRYINTERVAL;
	  Myinfo.lifetime = 0; /*Deregistration */
	  Myinfo.flags = 0x00|APREF;
	  Myinfo.status = ATHOME | WAITINGTODEREGISTER| AWAITINGREPLY;
	  sendRegisterMe(Myinfo.retries, Myinfo.flags, Myinfo.lifetime,
		     Myinfo.homeaddr, Myinfo.homeagent, Myinfo.coaddr);
	  if(debug >1)
	    fprintf(stderr,"Done with HOMEACTION\n");
	  
	}
      doit =0;
      break;
     
    case POPUPACTION:
      if(Dminfo.coaddr != Myinfo.coaddr)
	{
	  if(debug>1)
	    fprintf(stderr,"Doer processing POPUPACTION\n");
	  lowflshARP();
	  /* Make dummy down */
	  lowifacedown("dummy");
	  if((Myinfo.status & 0x03)==ASPOPUP)
	    lowrtnetset((Myinfo.coaddr & Myinfo.foreignnetmask),Myinfo.infname,0,DELRT);
	  /* Delete route to home network if present */
	   lowrtnetset((Myinfo.homeagent & Myinfo.homenetmask),Myinfo.infname,0,DELRT);
	  /* Delete route to existing default gateway*/
	  lowrtset(Myinfo.gw,Myinfo.infname,DELRT);
	  init(1);
	  /* Copy new info */ 
	  copy_new();
	  
	  /* Set new routes */
	  Myinfo.timeleft = RETRYINTERVAL;
	  Myinfo.flags = 0x20|APREF; /* MH will decapsulate */
	  lownetmaskreq(Myinfo.infname,&Myinfo.foreignnetmask);
	  Myinfo.status = ASPOPUP | WAITINGTOREGISTER| AWAITINGREPLY;
	  
	  sendRegisterMe(Myinfo.retries, Myinfo.flags, Myinfo.lifetime,
		     Myinfo.homeaddr, Myinfo.homeagent, Myinfo.coaddr);
	  if(debug >1)
	    fprintf(stderr,"Done with POPUPACTION\n");

	}
      doit =0;
      break;
      
    default:
      break;
    } 


  /* Check for any registration reply */

  FD_ZERO(&fdvec); FD_SET(Regreplysid, &fdvec); 
  tv.tv_sec = 0; tv.tv_usec = 0;

  while(( select(FD_SETSIZE, &fdvec, NULL, NULL, &tv)) < 0)  {
    if(errno == EINTR);
    else 
      perror("select() failed in Doer\n");
  }
     
  len = 1024; fromlen = sizeof(struct sockaddr_in);
  if (FD_ISSET(Regreplysid, &fdvec))  
    {
      if ((len = recvfrom(Regreplysid, msg, len, 0,
			  (struct sockaddr *) &from, &fromlen)) < 0)  
	{
	  perror("recvfrom() on RegisterMesid failed.\n");
	  if(errno == ECONNREFUSED || 
	     errno == ENETUNREACH || errno == EHOSTUNREACH);
	  else
	
	    {
	      cleanup(); exit(-1);
	    }
	};
      processRegreply(msg, len, &from);
    }

}/* End of Doer */

main(int argc, char **argv)  
{
  char *cmd;
  int c;
  extern char *optarg;
  extern int optind;
  unsigned char argsfound = 0;
    
  cmd = argv[0];
  while ((c = getopt(argc, argv, "a:m:g:")) != -1)  {
      switch ((char) c)  {
       case 'a':
	 argsfound |= 0x01;
	 if ((Myinfo.homeaddr = inet_addr(optarg)) == -1)  {
	    fprintf(stderr, "Bad address passed to -a.\n");
	    exit(-1);
	 };
	 break;
       case 'm':
	 argsfound |= 0x02;
	 if ((Myinfo.homenetmask = inet_addr(optarg)) == -1)  {
	    fprintf(stderr, "Bad netmask passed to -m.\n");
	    exit(-1);
	 };
	 break;
       case 'g':
	 argsfound |= 0x04;
	 if ((Myinfo.homeagent = inet_addr(optarg)) == -1)
	   {
	    fprintf(stderr, "Bad address passed to -g.\n");
	    exit(-1);
	 };
	 break;
       
      case '?':
	 usage(cmd);
	 exit(1);
	 break;
      }
   }

   if (argsfound != 0x07)  {
      usage(cmd);
      exit(1);
   }
     

   if (debug > 1)  {
    fprintf(stderr, "IPaddr: %lx, Mask: %lx, HomeAgent: %lx\n",
	      htonl(Myinfo.homeaddr), htonl(Myinfo.homenetmask),
	      htonl(Myinfo.homeagent));
   }

   init(0);
   sendroutersol(&routersolto);
   while (1)  
     { 
       doer();
       slp(0.5);
     }
}

void
usage(char *cmd)  {
   fprintf(stderr, 
   "Usage is: %s -a homeaddr -m homenetmask -g haaddr \n",
	   cmd);
}

