/* route.c		Route file reader
 *
 * Copyright 1991, Michael Westerhof, Sun Microsystems, Inc.
 * This software may be freely used, distributed, or modified, providing
 * this header is not removed.
 *
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <memory.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>

#include "ipip.h"

static void rerr();		/* General error printer */
static void init_route();
static void r_route();
static void print_routes();

static int rlineno;
static int rerrflag;

/*
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * open and read the route file
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 */
int
read_routes(f)
char *f;
{
	FILE *cf;
	char buf[256], *p;

	init_route();

/* Open the route file */
	if((cf = fopen(f,"r"))==NULL){
		(void)fprintf(stderr,"%sRoute file %s not found or could not be opened\n",progname,f);
		return -1;
	}

	while(fgets(buf, 255, cf)!=NULL){
		rlineno++;
		if((p = strtok(buf, " \t\n\r"))==NULL)continue;
		if(*p=='#' || *p==';')continue;

		if(strcmp(p,"route")==0)r_route();
/*		else if(strcmp(p,"whatever")==0)r_dowhatever(); */
		else rerr("Unrecognized command: %s",p);
	}

	if(rts_top==0)rerr("No routes defined","");

	if(debugd)print_routes();

	return rerrflag;
}
/*
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Initialize defaults and null route entries
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 */
static void
init_route()
{
	int i;

	for(i=0;i<MAX_ROUTES;i++){
		rts[i].ipaddr = 0;
		rts[i].mask = 0xffffffff;
		rts[i].destif = NULL;
		rts[i].destaddr = 0;
		rts[i].destport = 0;
		rts[i].hits = 0;
	}
	rts_top = 0;

	rlineno = 0;
	rerrflag = 0;
}

/*
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Print out the route table (DEBUG)
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 */
static void
print_routes()
{
	int i, port, type;
	unsigned char *p, *q;
	char *id;
	unsigned int m;

	if(rerrflag==0)(void)syslog(LOG_DEBUG,"%d routes:", rts_top);
	else (void)syslog(LOG_DEBUG,"%d routes (there are errors):", rts_top);
	for(i=0;i<rts_top;i++){
		p = (unsigned char *)&rts[i].ipaddr;		
		q = (unsigned char *)&rts[i].destaddr;		
		if(rts[i].destif != NULL){
			id = rts[i].destif->id;
			type = rts[i].destif->type;
		} else {
			id = "unknown";
			type = IF_TYPE_NONE;
		}
		port = ntohs(rts[i].destport);
		m = ntohl(rts[i].mask);

		if((type == IF_TYPE_SLIP)||
		   (type == IF_TYPE_TUN)){
			(void)syslog(LOG_DEBUG,"ip %d.%d.%d.%d mask 0x%08x interface %s\n",
				p[0],p[1],p[2],p[3],m,id);
		} else if(type == IF_TYPE_IPIP){
			(void)syslog(LOG_DEBUG,"ip %d.%d.%d.%d mask 0x%08x interface %s ip %d.%d.%d.%d\n",
				p[0],p[1],p[2],p[3],m,id,
				q[0],q[1],q[2],q[3]);
		} else {
			(void)syslog(LOG_DEBUG,"ip %d.%d.%d.%d mask 0x%08x interface %s ip %d.%d.%d.%d port %d\n",
				p[0],p[1],p[2],p[3],m,id,
				q[0],q[1],q[2],q[3],port);
		}
	}
}

/*
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Print a general route file error
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 */
static void
rerr(s,a)
char *s;
char *a;
{
	(void)syslog(LOG_ERR,"Route file error at line %d:\n",rlineno);
	(void)syslog(LOG_ERR,s,a);
	rerrflag--;
}

/*
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Handle the "route" command
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 */
static void
r_route()
{
	int i, syntax;
	char *q;
	struct hostent *he;

	if(rts_top>=MAX_ROUTES){
		rerr("Too many routes defined","");
		return;
	}

	syntax = 0;

	if(q = strtok((char *)NULL, " \t\n\r")){
		if((he = gethostbyname(q))){
			(void)memcpy((char *)&rts[rts_top].ipaddr,
					he->h_addr_list[0], 4);
		} else {
			if((rts[rts_top].ipaddr = inet_addr(q)) == 0xffffffff)
				rerr("Bad IP address: %s",q);
		}
	} else syntax++;

	if(q = strtok((char *)NULL, " \t\n\r")){
		rts[rts_top].mask = htonl(strtoul(q, NULL, 0));
	} else syntax++;

	if(q = strtok((char *)NULL, " \t\n\r")){
		for(i=0;i<ifs_top;i++){
			if(strcmp(q,ifs[i].id)==0)rts[rts_top].destif = &ifs[i];
		}
		if(rts[rts_top].destif == NULL)
			rerr("Interface %s not defined",q);
	} else syntax++;

	if(q = strtok((char *)NULL, " \t\n\r")){
		if((he = gethostbyname(q))){
			(void)memcpy((char *)&rts[rts_top].destaddr,
					he->h_addr_list[0], 4);
		} else {
			if((rts[rts_top].destaddr = inet_addr(q)) == (unsigned long)-1)
				rerr("Bad destination IP address: %s",q);
		}
	}

	if(q = strtok((char *)NULL, " \t\n\r")){
		rts[rts_top].destport = htons((unsigned short)atoi(q));
	}

	if(syntax)rerr("Syntax error (route <ipaddr> <mask> <iface> [<destipaddr> [<destport>]]","");

	rts_top++;
}
