/*-------------------------------------------------------------------------*/
/* schema.c --- xcircuit routines specific to the schematic capture system */
/* Copyright (c) 1998  Tim Edwards, Johns Hopkins University        	   */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      written by Tim Edwards, 10/13/97    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "cursors.h"
#include "colordefs.h"
#include "xcircuit.h"

/*------------------------------------------------------------------------*/
/* External Variable definitions                                          */
/*------------------------------------------------------------------------*/

extern Clientdata areastruct;
extern short eventmode;

/*-------------------------------------------------------------------------*/
#ifdef SCHEMA

/*--------------------------------------------------------*/
/* Menu calls (procedure wrappers)			  */
/*--------------------------------------------------------*/

XtCallbackProc callgennet(Widget w, unsigned int mode, caddr_t calldata)
{
   if (mode == 0) {
      gennet("spice", "spc");
   }
   else if (mode == 1) {
      gennet("sim", "sim");
   }
}

/*--------------------------------------------------------*/
/* Change a pinlabel's type to GLOBAL			  */
/*--------------------------------------------------------*/

XtCallbackProc dopinglobal(Widget w, unsigned int mode, caddr_t calldata)
{
   short *gsel;
   labelptr glab;

   if (areastruct.selects == 0) {
      Wprintf("Must first select a label to make global");
      return;
   }

   for (gsel = areastruct.selectlist; gsel < areastruct.selectlist +
		areastruct.selects; gsel++) {
      if (SELECTTYPE(gsel) == LABEL) {
	 glab = SELTOLABEL(gsel);
	 if (glab->pin == GLOBAL) {
	    glab->pin = LOCAL;
	    Wprintf("Changed pin to type LOCAL");
	 }
	 else if (glab->pin == LOCAL) {
	    glab->pin = GLOBAL;
	    Wprintf("Changed pin to type GLOBAL");
	 }
      }
   }
}

/*--------------------------------------------------------*/
/* Change an object's type to Fundamental or back again	  */
/*--------------------------------------------------------*/

XtCallbackProc makefund(Widget w, unsigned int mode, caddr_t calldata)
{
   short *fsel;
   objinstptr fobj;

   if (areastruct.selects == 0) {
      if (objectdata->schemtype == SYMBOL) {
	 objectdata->schemtype = FUNDAMENTAL;
	 Wprintf("Changed top-level object to type FUNDAMENTAL");
      }
      else if (objectdata->schemtype == FUNDAMENTAL) {
	 if (objectdata->symschem == NULL) {
	    objectdata->schemtype = SYMBOL;
	    Wprintf("Changed top-level object to type SYMBOL");
	 }
	 else {
	    if (objectdata->symschem->schemtype == SYMBOL) {
	       objectdata->schemtype = SCHEMATIC;
	       Wprintf("Changed top-level object to type SCHEMATIC");
	    }
	    else if (objectdata->symschem->schemtype == SCHEMATIC) {
	       objectdata->schemtype = SYMBOL;
	       Wprintf("Changed top-level object to type SYMBOL");
	    }
	 }
      }
      return;
   }

   for (fsel = areastruct.selectlist; fsel < areastruct.selectlist +
		areastruct.selects; fsel++) {
      if (SELECTTYPE(fsel) == OBJECT) {
	 fobj = SELTOOBJINST(fsel);
	 if (fobj->thisobject->schemtype == SYMBOL) {
	    fobj->thisobject->schemtype = FUNDAMENTAL;
	    Wprintf("Changed object to type FUNDAMENTAL");
	 }
	 else if (fobj->thisobject->schemtype == FUNDAMENTAL) {
	    if (fobj->thisobject->symschem != NULL) {
	       if (fobj->thisobject->symschem->schemtype == SYMBOL) {
		  fobj->thisobject->schemtype = SCHEMATIC;
	          Wprintf("Changed object to type SCHEMATIC");
	       }
	       else if (fobj->thisobject->symschem->schemtype == SCHEMATIC) {
		  fobj->thisobject->schemtype = SYMBOL;
	          Wprintf("Changed object to type SYMBOL");
	       }
	    }
	    else {
	       fobj->thisobject->schemtype = SYMBOL;
	       Wprintf("Changed object to type SYMBOL");
	    }
	 }
      }
   }
}

/*--------------------------------------------------------*/
/* Find the page number for an object			  */
/*--------------------------------------------------------*/

int findpageobj(pobj)
  objectptr pobj;
{
   int tpage;
   objectptr chkpage;

   for (tpage = 0; tpage < areastruct.pages; tpage++) {
      if (*(areastruct.pagelist + tpage) == pobj) return tpage;
   }
   return -1;
}

/*--------------------------------------------------------*/
/* Recursively find all sub-circuits associated with the  */
/* top-level circuit and set their filenames to be the	  */
/* same.						  */
/*--------------------------------------------------------*/

findsubschems(toppage, cschem)
  int toppage;
  objectptr cschem;
{
   genericptr *cgen;

   for (cgen = cschem->plist; cgen < cschem->plist + cschem->parts; cgen++) {
      if ((*cgen)->type == OBJECT) {
	 objectptr cobj = TOOBJINST(cgen)->thisobject;

	 if (cobj->symschem != NULL) {
	    int pageno = findpageobj(cobj->symschem);
	    printf("Symbol %s has schematic at page %d\n", cobj->name, pageno);
	    if (pageno < 0) {	/* This shouldn't happen (hopefully) */
	       Wprintf("Error: disconnected schematic!\n");
	    }
	    else {
	       areastruct.filename[pageno] = areastruct.filename[toppage];
	    }

	    /* A symbol on its own schematic page is allowed for clarity */
	    /* of the schematic, but this cannot be a functional part of */
	    /* the schematic circuit!					 */
	    
	    if (cobj->symschem != cschem) {
	       printf("Searching for dependencies of symbol %s\n", cobj->name);
	       findsubschems(toppage, cobj->symschem);
	    }
	    else {
	       printf("Notice:  Symbol on its own schematic\n");
	    }
	 }
      }
   }
}

/*--------------------------------------------------------*/
/* Copy label to corresponding schematic/symbol		  */
/*--------------------------------------------------------*/

copypinlabel(pinlab)
  labelptr pinlab;
{
   labelptr *newlabel;

   if (areastruct.schemon && objectdata->symschem != NULL && pinlab->pin) {
      NEW_LABEL(newlabel, objectdata->symschem);
      (*newlabel)->pin = pinlab->pin;
      (*newlabel)->rotation = pinlab->rotation;
      (*newlabel)->justify = pinlab->justify;
      (*newlabel)->color = pinlab->color;
      (*newlabel)->scale = pinlab->scale;
      (*newlabel)->string = pinlab->string;

      /* place label just outside bounding box, then recompute bbox */

      (*newlabel)->position.x = objectdata->symschem->lowerleft.x + 
	   (objectdata->symschem->width >> 1);
      (*newlabel)->position.y = objectdata->symschem->lowerleft.y - TEXTHEIGHT
	   * (*newlabel)->scale;
      objectdata->symschem->parts++;
      calcbbox(objectdata->symschem);
   }
}

/*--------------------------------------------------------*/
/* Check if top-level page is the same name as a library  */
/* object; if so, connect it.				  */
/*--------------------------------------------------------*/

int checkschem()
{
   objectptr *tlib;

   if (areastruct.schemon == False || objectdata->symschem != NULL) return 0;

   for (tlib = areastruct.library; tlib != areastruct.userlib +
	  areastruct.userobjs; tlib++) {
      if (tlib == areastruct.library + areastruct.builtins) {
	 tlib = areastruct.userlib;
	 if (areastruct.userobjs == 0) break;
      }
      if (!strcmp(objectdata->name, (*tlib)->name)) {
	 objectdata->symschem = (*tlib);
	 objectdata->schemtype = SCHEMATIC;
	 (*tlib)->symschem = objectdata;
	 (*tlib)->schemtype = SYMBOL;
	 return 1;
      }
   }
  return 0;
}

/*--------------------------------------------------------*/
/* Find location of corresponding pin in symbol/schematic */
/*--------------------------------------------------------*/

labelptr findotherpin(curlabel)
  labelptr curlabel;
{
   objectptr other = objectdata->symschem;
   genericptr *tgen;
   labelptr tlab;

   if (areastruct.schemon == False || other == NULL || curlabel->pin == False)
      return NULL;

   for (tgen = other->plist; tgen < other->plist + other->parts; tgen++) {
      if ((*tgen)->type == LABEL) {
	 tlab = TOLABEL(tgen);
	 if (tlab->string == curlabel->string) return tlab;
      }
   }
   return NULL;
}

/*--------------------------------------------------------*/
/* Delete label in corresponding schematic/symbol	  */
/*--------------------------------------------------------*/

deletepinlabel(pinlab)
  labelptr pinlab;
{
   labelptr cplab = findotherpin(pinlab);
   genericptr *genobj;

   if (cplab != NULL) {
      for (genobj = objectdata->symschem->plist; genobj <
	     objectdata->symschem->plist + objectdata->symschem->parts; genobj++)
	 if ((*genobj) == (genericptr)cplab) break;

      free(cplab);
      for (++genobj; genobj < objectdata->symschem->plist +
	      objectdata->symschem->parts; genobj++)
         *(genobj - 1) = *genobj;

      objectdata->symschem->parts--;
   }
}

/*------------------------------------------*/
/* Swap object schematic and symbol pages   */
/*------------------------------------------*/

XtCallbackProc swapschem(Widget w, caddr_t clientdata, caddr_t calldata)
{
   objectptr savepage = objectdata;
   labelptr  *pinlab;
   genericptr *plab;

   if (areastruct.schemon == False) return;

   if (eventmode == PRESS_MODE || eventmode == COPY2_MODE)
      objectdelete(NORMAL);

   if (objectdata->symschem == NULL) {  /* create symbol or schematic */
      if (objectdata->schemtype == SYMBOL) {
	 int tpage;

	 /* create a new page for the new schematic */

	 for (tpage = 0; tpage < areastruct.pages; tpage++)
	    if (*(areastruct.pagelist + tpage) == NULL) break;
         newpage(tpage);
      }
      else {
	 objectptr *newobject;

	 /* create a new library object for the new symbol */

	 areastruct.userlib = (objectptr *) realloc(areastruct.userlib,
		++areastruct.userobjs * sizeof(objectptr));
	 newobject = areastruct.userlib + areastruct.userobjs - 1;
         *newobject = (objectptr) malloc(sizeof(object));
	 initmem(*newobject);
	 (*newobject)->schemtype = SYMBOL;
	 objectdata = *newobject;
      }

      /* set links between the two objects */

      savepage->symschem = objectdata;
      objectdata->symschem = savepage;

      /* make the name of the new object equal to that of the old */

      strcpy(objectdata->name, savepage->name);

      /* copy all pin labels into the new object */

      for (plab = savepage->plist; plab < savepage->plist + savepage->parts;
		plab++)
	 if ((*plab)->type == LABEL) {
	    labelptr lpin = (labelptr)*plab;

	    if (lpin->pin) {
	       NEW_LABEL(pinlab, objectdata);
	       (*pinlab)->pin = lpin->pin;
	       (*pinlab)->color = lpin->color;
	       (*pinlab)->rotation = 1;
	       (*pinlab)->scale = 1.0;
	       (*pinlab)->justify = areastruct.justify; 
	       (*pinlab)->string = lpin->string;
	       (*pinlab)->position.x = 0;
	       (*pinlab)->position.y = objectdata->parts * (TEXTHEIGHT + 10);
	       objectdata->parts++;
	    }
         }
   }
   else {
      objectdata = objectdata->symschem;
   }

   if (areastruct.selects > 0) free(areastruct.selectlist);
   areastruct.selects = 0;

   setvscale();
   transferselects();
   
   zoomview(NULL, Number(1), NULL);
   printname(objectdata);
}

#endif
/*-------------------------------------------------------------------------*/
