#include <stdio.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdarg.h>
#include <netdb.h>
#include <fcntl.h>
#include <errno.h>
#include "ftp.h"

void ftp_init(void)
{
    memset(&ftp, 0, sizeof(ftp));
    ftp.port = 7350;
}

void ftp_shutdown(void)
{
    ftp_send_cmd(0, "QUIT\n");
    close(ftp.fd);
}

int ftp_login(char *username, char *password, char *ip, short port)
{
   struct sockaddr_in server;

    if (!port) port = 21;
    memset(&server, 0, sizeof(server));
    server.sin_family = AF_INET;
    server.sin_port = htons(port);
    server.sin_addr.s_addr = host2addr(ip);

    if ((ftp.fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        error_errno("socket");
        return -1;
    }
    if (connect(ftp.fd, (struct sockaddr *)&server, sizeof(server)) < 0) {
        error_errno("connect");
        return -1;
    }
    if (ftp_get_reply() < 0)
	return -1;

    if (ftp_send_cmd(331, "USER %s\n", username) < 0)
        return -1;

    if (ftp_send_cmd(230, "PASS %s\n", password) < 0) 
        return -1;

    return 0;
}

int ftp_get_data(int type)
{
    fd_set rset;
    struct timeval tv;
    struct sockaddr_in peer;
    int fd, len, fdset, total_size = 0, size = 0;

    len = sizeof(peer);

    switch (type) {
    case ACTIVE:
        if ((fd = accept(ftp.dfd, (struct sockaddr *)&peer, &len)) < 0) {
            error_errno("accept");
            return -1;
        }
        close(ftp.dfd);
	break;
    case PASSIVE:
	fd = ftp.dfd;
	break;
    }

    tv.tv_sec  = 1;
    tv.tv_usec = 0;
 
    while (total_size < DATASIZ)  { 
	FD_ZERO(&rset);
        FD_SET(fd, &rset);

        if ((fdset = select(fd + 1, &rset, NULL, NULL, &tv)) < 0) {
	    error_errno("select");
	    return -1;
	}
	if (!fdset)
	    break;

	if (FD_ISSET(fd, &rset)) {
    	    if ((size = recv(fd, &ftp.data[total_size], DATASIZ-total_size, 0)) < 0) {
        	error_errno("recv");
        	return -1;
	    }
	    if (!size) 
		break;
	    total_size += size;
	}
    }
    ftp.data[total_size] = '\0';
    close(fd);

    return 0;
}

int ftp_data_conn(short type, short port)
{
    struct linger lin;
    int len, lopt = 1;
    struct sockaddr_in dataconn, local;

    memset(&local, 0, sizeof(local));
    memset(&dataconn, 0, sizeof(dataconn));
    dataconn.sin_family = AF_INET;

    if (!port) port = 7350;

    if ((ftp.dfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        error_errno("socket");
        return -1;
    }
    if (setsockopt(ftp.dfd, SOL_SOCKET, SO_REUSEADDR, &lopt, sizeof(lopt)) < 0) {
        error_errno("setsockopt");
        return -1;
    }
    lin.l_onoff  = 1;
    lin.l_linger = 0;
    if (setsockopt(ftp.dfd, SOL_SOCKET, SO_LINGER, &lin, sizeof(lin)) < 0) {
        error_errno("setsockopt");
        return -1;
    }

    switch (type) {
    case ACTIVE:
	len = sizeof(local);
	dataconn.sin_port = htons(port);
	dataconn.sin_addr.s_addr = htonl(INADDR_ANY);

	if (bind(ftp.dfd, (struct sockaddr *)&dataconn, sizeof(dataconn)) < 0) {
	    error_errno("bind");
	    return -1;
	}
	if (listen(ftp.dfd, 1) < 0) {
	    error_errno("listen");
	    return -1;
	}
	if (getsockname(ftp.fd, (struct sockaddr *)&local, &len) < 0) {
	    error_errno("getsockname");
	    return -1;
	}

	if (ftp_send_cmd(200, "PORT %s\n", addr2str(ntohl(local.sin_addr.s_addr), port)) < 0)
	    return -1;

	break;

    case PASSIVE:
        if (ftp_send_cmd(227, "PASV\n") < 0)
            return -1;

        if (str2addr(ftp.reply, (u_long *)&dataconn.sin_addr.s_addr, &dataconn.sin_port) < 0)
            return -1;

        if (connect(ftp.dfd, (struct sockaddr *)&dataconn, sizeof(dataconn)) < 0) {
            error_errno("connect");
            return -1;
        }

        break;
    }

    return 0;
}

int ftp_send_data_cmd(short code, short type, char *fmt, ...)
{
    va_list ap;

    if (ftp_data_conn(type, ftp.port++) < 0)
	return -1;

    va_start(ap, fmt);
    vsnprintf(ftp.cmd, CMDSIZ - 1, fmt, ap);
    va_end(ap);

    if (send(ftp.fd, ftp.cmd, strlen(ftp.cmd), 0) != strlen(ftp.cmd)) {
        error_errno("send");
        return -1;
    }
    ftp.cmd[strlen(ftp.cmd)-1] = '\0';
    if (ftp_get_reply() < 0)
        return -1;

    if (code && ftp.reply_code != code) {
        error("Cmd: %s: Reply: %s\n", ftp.cmd, ftp.reply);
        return -1;
    }

    if (ftp_get_data(type) < 0)
	return -1;

    if (ftp_get_reply() < 0)
	return -1;

    return 0;
}

int ftp_send_cmd(short code, char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    vsnprintf(ftp.cmd, CMDSIZ - 1, fmt, ap);
    va_end(ap);

    if (send(ftp.fd, ftp.cmd, strlen(ftp.cmd), 0) != strlen(ftp.cmd)) {
        error_errno("send");
        return -1;
    }
    ftp.cmd[strlen(ftp.cmd)-1] = '\0';
    if (ftp_get_reply() < 0)
        return -1;

    if (code && ftp.reply_code != code) {
        error("Cmd: %s: Reply: %s\n", ftp.cmd, ftp.reply);
        return -1;
    }

    return 0;
}

int ftp_get_reply(void)
{
    char *p;
    int more = 0, total_size = 0, size = 0;

    p = ftp.reply;
    if ((size = read(ftp.fd, p, 4)) != 4) {
	error_errno("read");
	return -1;
    }
    p[size] = '\0';
    ftp.reply_code = atoi(p);
    if (p[0] == '2' && p[3] == '-')
	more++;
    p += size;
    total_size += size;
    
    while (total_size < CMDSIZ && (read(ftp.fd, p, 1) == 1))
    {
	if (p[0] == '\n') {
	    if (p[-1] == '\r')
		p[-1] = '\n';
	    if (more) {
		if ((size = read(ftp.fd, p, 4)) != 4) {
		    error_errno("read");
		    return -1;
		}
		p[size] = '\0';
		ftp.reply_code = atoi(p);

		if (p[0] != '2' || p[3] != '-')
		    more--;

		p += size;
		total_size += size;
		continue;
	    }
	    p[-1] = '\0';
	    break;
	}
	p++;
	total_size++;
    }
    return total_size;
}

char * addr2str(u_long ip, u_short port)
{
    int b[6];
    static char string[BUFSIZ];
  
    b[0] = (ip & 0xff000000) >> 24;
    b[1] = (ip & 0xff0000) >> 16;
    b[2] = (ip & 0xff00) >> 8;
    b[3] = (ip & 0xff);
    b[4] = (port & 0xff00) >> 8;
    b[5] = port & 0xff;
    snprintf(string, BUFSIZ - 1, "%d,%d,%d,%d,%d,%d", b[0], b[1], b[2], b[3], b[4], b[5]);

    return string;
}

int str2addr(char *string, u_long *ip, u_short *port)
{
    char *p;
    int i = 0, b[6];

    if ((p = (char *)strchr(string, '(')) == NULL) {
        string[strlen(string)-2] = '\0';
        error("Could not find \"(\" in string: %s", string);
        return -1;
    }
    p++;
    while (i != 6) {
        b[i++] = atoi(p);
        p = (char *)strchr(p, (int)',');
        p++;
    }
    b[0] <<= 24, b[1] <<= 16, b[2] <<=  8;
    *ip = htonl(b[0] | b[1] | b[2] | b[3]);
    b[4] <<= 8;
    *port =htons(b[4] | b[5]);

    return 0;
}

u_long host2addr(char *name)
{
    struct hostent *host;

    if ((host = gethostbyname(name)) == NULL)
        error_die("failed: gethostbyname: %s", name);
        
    return *(long *)host->h_addr;
}

int ftp_interactive(char *cmd, char *reply)
{
    int size;
    fd_set rset;
    char buf[1];

    if (write(ftp.fd, cmd, strlen(cmd)) != strlen(cmd)) {
	error_errno("write");
	return -1;
    }
    if (read(ftp.fd, ftp.reply, 5) != 5) {
	error_errno("read");
	return -1;
    }
    ftp.reply[5] = '\0';
    
    if (!strstr(ftp.reply, reply)) {
	error("command: %s failed", cmd);
	return -1;
    }

    for (;;) {
        FD_ZERO(&rset);
        FD_SET(0, &rset);
        FD_SET(ftp.fd, &rset);

        select(ftp.fd + 1, &rset, NULL, NULL, NULL);

        if (FD_ISSET(0, &rset)) {
            if (read(0, buf, 1) != 1) {
	        error_errno("read");
		return -1;
	    }
            if (write(ftp.fd, buf, 1) != 1) {
		error_errno("write");
		return -1;
	    }
        }
        if (FD_ISSET(ftp.fd, &rset)) {
            if (read(ftp.fd, buf, 1) != 1) {
	        error_errno("read");
		return -1;
	    }
            if (write(0, buf, 1) != 1) {
		error_errno("write");
		return -1;
	    }
        }
    }
    return 0;
}
