/*
        Title:  Brainwave Machine Controller

        Desc.:  Mode selection LEDs are on PORTC
                Goggle LEDs are on PORTB
                Button on INT0, with 100k pullup resistor
                4MHz Clock Speed (adjustable, see #define HZ)
*/

#include <io-avr.h>
#include <io.h>
#include <wdt.h>
#include <interrupt.h>
#include <sig-avr.h>

#define HZ              4000000L        /* clock freq */
#define MS_CLOCKDIV     64              /* approx. millisecond divisor */

/* MODES enum */
enum MODES
{
 MODE_MEDITATION = 1,
 MODE_INDUCE_SLEEP,
 MODE_INDUCE_SLEEP2,
 MODE_STRESS_REDUCE,
 MODE_INTUITION,
};
volatile short delay_done;
unsigned int current_led = 1, previous_led = 4,
        chosen_mode = 0, delay_start[6], delay_end[6],
        next_mode[6], session_time, current_led_status,
        adder, reset_mode, interrupt_wait2;
short led_counter, freq_counter, interrupt_wait1;
long freq_period;

SIGNAL(SIG_OVERFLOW0) {
 /* int called when timer resets from 255 to 0 */
 if(delay_done) delay_done--;
}

SIGNAL(SIG_INTERRUPT0) {
 /* int0 called when a button is pushed */

 cli();

 reset_mode++;
 if(reset_mode > 5) reset_mode = 0;

 switch(reset_mode)
 {
  case 1:
   cbi(PORTC, PC5);
   sbi(PORTC, PC4);
   sbi(PORTC, PC3);
   sbi(PORTC, PC2);
   sbi(PORTC, PC1);
   break;
  case 2:
   sbi(PORTC, PC5);
   cbi(PORTC, PC4);
   sbi(PORTC, PC3);
   sbi(PORTC, PC2);
   sbi(PORTC, PC1);
   break;
  case 3:
   sbi(PORTC, PC5);
   sbi(PORTC, PC4);
   cbi(PORTC, PC3);
   sbi(PORTC, PC2);
   sbi(PORTC, PC1);
   break;
  case 4:
   sbi(PORTC, PC5);
   sbi(PORTC, PC4);
   sbi(PORTC, PC3);
   cbi(PORTC, PC2);
   sbi(PORTC, PC1);
   break;
  case 5:
   sbi(PORTC, PC5);
   sbi(PORTC, PC4);
   sbi(PORTC, PC3);
   sbi(PORTC, PC2);
   cbi(PORTC, PC1);
   break;
  default:
   outp(0xFF, PORTC);
   break;
 }

 interrupt_wait2 = 60;
 while(interrupt_wait2>0)
 {
  interrupt_wait2--;
  interrupt_wait1=10000;
  while(interrupt_wait1>0)
  {
   interrupt_wait1--;
  }
 }

 sei();
}

void led_on(unsigned int lednum)
{
 /* turn on a given led (1 - 4, 0xFF[ALL]) */
 switch(lednum)
 {
  case 0: cbi(PORTB, PB5); break;     /* current_led-1 fix */
  case 1: cbi(PORTB, PB0); break;
  case 2: cbi(PORTB, PB1); break;
  case 3: cbi(PORTB, PB2); break;
  case 4: cbi(PORTB, PB5); break;
  case 5: cbi(PORTC, PC5); break;
  case 6: cbi(PORTC, PC4); break;
  case 7: cbi(PORTC, PC3); break;
  case 8: cbi(PORTC, PC2); break;
  case 9: cbi(PORTC, PC1); break;
  case 0xFF:
   cbi(PORTB, PB0);  cbi(PORTB, PB1);
   cbi(PORTB, PB2);  cbi(PORTB, PB5);
   break;
 }
}

void led_off(unsigned int lednum)
{
 /* turn off a given led (1 - 4, 0xFF[ALL]) */
 switch(lednum)
 {
  case 0: sbi(PORTB, PB5); break;     /* current_led-1 fix */
  case 1: sbi(PORTB, PB0); break;
  case 2: sbi(PORTB, PB1); break;
  case 3: sbi(PORTB, PB2); break;
  case 4: sbi(PORTB, PB5); break;
  case 5: sbi(PORTC, PC5); break;
  case 6: sbi(PORTC, PC4); break;
  case 7: sbi(PORTC, PC3); break;
  case 8: sbi(PORTC, PC2); break;
  case 9: sbi(PORTC, PC1); break;
  case 0xFF:
   sbi(PORTB, PB0);  sbi(PORTB, PB1);
   sbi(PORTB, PB2);  sbi(PORTB, PB5);
   break;
 }
}

void increment_led(void)
{
 current_led++;
 if(current_led > 4) current_led = 1;
 previous_led = current_led - 1;
 led_off(previous_led);
 led_on(current_led);
}

void led_status(int led1, int led2, int led3, int led4)
{
 if(led1) led_on(1); else led_off(1);
 if(led2) led_on(2); else led_off(2);
 if(led3) led_on(3); else led_off(3);
 if(led4) led_on(4); else led_off(4);
}

void led_status_portc(int led_status_c)
{
 switch(led_status_c)
 {
  case 1: led_on(5);
   led_off(6); led_off(7); led_off(8); led_off(9);
   break;
  case 2: led_on(6);
   led_off(5); led_off(7); led_off(8); led_off(9);
   break;
  case 3: led_on(7);
   led_off(5); led_off(6); led_off(8); led_off(9);
   break;
  case 4: led_on(8);
   led_off(5); led_off(6); led_off(7); led_off(9);
   break;
  case 5: led_on(9);
   led_off(5); led_off(6); led_off(7); led_off(8);
   break;
  default:
   break;
 }
}

void stop_timer0(void) {
 /* stop the timer */
 outp(0x00, TCCR0);
 cbi(TIMSK, TOIE0);
}

void start_timer0(short clockdiv, int initial_value)
{
 /* start timer and initialize counter */
 cli();
 stop_timer0();

 /* set timer to initial value */
 outp(initial_value, TCNT0);

 /* enable timer overflow interrupt */
 sbi(TIMSK, TOIE0);

 /* set the timer clock divisor to start timer */
 switch(clockdiv)
 {
  case 1:    outp(1, TCCR0); break;
  case 8:    outp(2, TCCR0); break;
  case 64:   outp(3, TCCR0); break;
  case 256:  outp(4, TCCR0); break;
  case 1024: outp(5, TCCR0); break;
  default:   outp(4, TCCR0); break;
 }
 sei();
}

void msdelay(short ms)
{
 /* millisecond accuracy delay */
 if(ms == 0) return;
 long wait_cycles = ((HZ/MS_CLOCKDIV) * ms) / 100;
 int wait_low = wait_cycles % 255;
 short wait_high = wait_cycles / 255;

 delay_done = wait_high;
 start_timer0(MS_CLOCKDIV, (255-wait_low));
 while (delay_done != 0);
 stop_timer0();
}

void control_brainwaves(int chosen_mode)
{
 /* make sure we have a mode set */
if(chosen_mode!=0)
{

 /* setup standard session times (minutes) */
 switch(chosen_mode)
 {
  case MODE_INDUCE_SLEEP:
   session_time = 25;
   freq_period = 38000;
   adder = 1;
   break;
  case MODE_INDUCE_SLEEP2:
   session_time = 45;
   freq_period = 6000;
   adder = 1;
   break;
  default:
   session_time = 30;
   freq_period = 18000;
   adder = -1;
   break;
 }

 /* standard init for times */
 int led_period;
 led_period = delay_start[chosen_mode];
 led_counter = 0;

 while(led_period != delay_end[chosen_mode]) {
  msdelay(1);

  freq_counter++;
  led_counter++;
  if(led_counter >= led_period)
  {
   if(current_led_status == 1) {
    led_status(1,0,1,0);
    current_led_status = 0;
   } else {
    led_status(0,1,0,1);
    current_led_status = 1;
   }
   led_counter = 0;
  }
  if(freq_counter >= freq_period)
  {
   freq_counter = 0;
   if(chosen_mode != MODE_STRESS_REDUCE) led_period += adder;
  }

  /* make sure they havent pressed the button */
  if(chosen_mode != reset_mode)
  {
   led_status_portc(reset_mode);
   led_status(0,0,0,0);
   return;
  }
 }
 /* don't accidentally change reset_mode */
 if(chosen_mode == reset_mode)
 {
  reset_mode = next_mode[chosen_mode];
 }
 led_status(0,0,0,0);
}
}

int main(void)
{
 wdt_disable();

 /* set PORTB & PORTC as output */
 outp(0xFF, DDRB);
 outp(0xFF, DDRC);
 /* enable INT0 */
 outp(0x00, DDRD);
 outp((1<<INT0), GIMSK);
 outp((0<<ISC01)|(0<<ISC00), MCUCR);

 sei();

 /* setup delay variables */
 delay_start[MODE_MEDITATION] =         25;      /* (in [[1/Hz] * 100]) */
 delay_end[MODE_MEDITATION] =           15;
 next_mode[MODE_MEDITATION] =           0;

 delay_start[MODE_INDUCE_SLEEP] =       25;
 delay_end[MODE_INDUCE_SLEEP] =         29;
 next_mode[MODE_INDUCE_SLEEP] =         MODE_INDUCE_SLEEP2;

 delay_start[MODE_INDUCE_SLEEP2] =      29;
 delay_end[MODE_INDUCE_SLEEP2] =        67;
 next_mode[MODE_INDUCE_SLEEP2] =        0;

 delay_start[MODE_STRESS_REDUCE] =      9;
 delay_end[MODE_STRESS_REDUCE] =        9;
 next_mode[MODE_STRESS_REDUCE] =        0;

 delay_start[MODE_INTUITION] =          25;
 delay_end[MODE_INTUITION] =            15;
 next_mode[MODE_INTUITION] =            0;

 while(1)
 {
  control_brainwaves(reset_mode);
 }
}
