package Freenet;
import Freenet.support.*;
import java.io.*;
import java.util.*;
import java.math.*;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

/**
 * This class is used to store data.  The object creates a file on disk
 * in a (statically specified) directory to store the data in.  In general
 * the data is accessed using streams.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 **/

public class Data implements Serializable
{

  // Public Fields

  /** The path where files should be stored **/
  public static File path;
  /** Indicates whether this class is ready for reading **/
  public boolean readyToRead = false;

  // Protected Fields
  protected File name;
  protected boolean delFile = true;

  // Constructors
  /**
   * This creates a data item from the specified file.
   * NOTE: The file <B>will</B> be deleted when this object
   * is finalized unless the fileName method is called
   * @param name The file in which the data is stored
   * @see fileName
   **/
  public Data(File name)
  {
    this.name = name;
  }

  /** 
   * Creates a new data item reading the data from the InputStream 
   * @param data The stream from where the data will be read
   **/
    public Data(InputStream data)
    {
      try
	{
	  Conduit c = new Conduit(data, newFile());
	  c.syncFeed();
	  readyToRead = true;
	}
      catch (Exception e)
	{
	  Logger.log("Data.java","Exception creating file to read data to!",Logger.ERROR);
	}
    }


    /**
     *
     *
     **/
    public Data(SplitOutputStream datatunnel)
    {
	try {
	    datatunnel.addOutput(newFile(),
				 new DataCB(this));
	} catch (Exception e) {
	    Logger.log("Data.java","Exception creating file for cacheing stream!",Logger.ERROR);
	}
    }


    /*
     * Creates a new file for this data object. Will return null if
     * there already is a file
     */
    private FileOutputStream newFile() throws IOException
    {
	if (name != null)
	    return null; // only run if data object has no file
	String tname;
	// Get a list of files in this directory so that there is no duplication
	String[] filelist = path.list();
	// Choose a name for this data item
	Random r = new Random();
	boolean used;
	do
	    {
		tname = "t"+Math.abs(r.nextInt() % 10000000);
		used = false;
		if(filelist!=null)
		    for (int x=0; x<filelist.length; x++)
			{
			    if (tname.equals(filelist[x]))
				used = true;
			}
	    } while (used);
	name = new File(path, tname);
	Logger.log("Data.java","Created new file " + name,Logger.DEBUGGING);
	return new FileOutputStream(name);
    }    

  
  /**
   * Returns the length of the data in bytes
   **/
  public long length()
  	{
	    return name.length();
	}

  /**
   * Returns a File containing the data.  After calling this
   * method the File will not be deleted should this Data item
   * be finalized - thus the user takes on responsibility for
   * deleting this file.
   * @return A File containing the data
   **/

  public File fileName()
  {
    delFile = false;
    return name;
  }

  /**
   * Copies the data into the specified file, returns when copy complete
   * @param f The file that the data should be copied into
   **/
  public void CopyToFile(File f) 
    throws DataNotReadyException, IOException
    {
      Conduit c = new Conduit(new FileInputStream(name), 
			      new FileOutputStream(f));
      c.syncFeed();
    }


  /**
   * Returns a BigInteger which specifies the length of the data stored in this
   * object
   **/
  public BigInteger getLength() 
    throws DataNotReadyException, IOException
    {
      if (!readyToRead)
	throw new DataNotReadyException();
      return new BigInteger(new Long(name.length()).toString());
    }

  /**
   * Returns an InputStream which supplies the data stored in this
   * object
   **/
  public InputStream toInputStream() 
    throws DataNotReadyException, IOException
    {
      if (!readyToRead)
	throw new DataNotReadyException();
      return new FileInputStream(name);
    }

  protected void finalize() throws Throwable
    {
      if (delFile)
	name.delete();
      super.finalize();
    }

	public String toString()
	{
		return name.toString();
	}
}

/**
 * A callback which will set the readyToRead field when the object
 * is properly initiated.
 **/
class DataCB implements Callback
{
  public Data d;
  public DataCB(Data d)
    {
      this.d = d;
    }
  public void callback()
    {
      d.readyToRead = true;
    }
}
