
/*
 *$Id: DataStore.java,v 1.17 2000/03/22 18:19:36 blanu Exp $

  This code is part of the Java Adaptive Network Client by Ian Clarke.
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

  Explanation of Code Versions:
    0.0.0      = Initial Description
    0.0.1      = API Specified
    0.x (x>0)  = Partial Implementation
    x.0 (x>0)  = Operational

  Requires Classes: Address
                    Data
 */

/**
 * DataStore.java
 *
 * This class is used by a node to store references and data.
 *
 * @version 0.0
 * @author Ian Clarke (I.Clarke@strs.co.uk)
 **/

package Freenet;
import Freenet.support.*;
import java.io.*;
import java.util.*;

public class DataStore implements Serializable, Callback
{
  static private long write_interval;

  // Debugging stuff to test this class
    /* not working since Ian removed test code from data class
  public static void main(String[] args) {
    System.out.println("DataStore test routine\n-------");
    System.out.println("Creating datastore size 20 with room for 30 bytes of data");
    DataStore ds = new DataStore(30, 20);
    System.out.println("Adding 10 data items");
    for (int x = 0; x < 100; x += 5)
      ds.put(new StringKey("Key "+x), new Address("local/"+x),
             new Data("Data "+x));
    System.out.println("Adding 10 reference items");
    for (int x = 100; x < 200; x += 5)
      ds.put(new StringKey("Key "+x), new Address("local/"+x), null);
    Key last = null;
    for (int x = 0; x < 10; x++) {
      System.out.println("Finding closest key to Key 138 with mask "+
                         last);
      last = ds.findClosestKey(new StringKey("Key 138"), last);
      System.out.println("Found "+last);
      }
    }
    */
  public static DataStore makeDataStore(Node node, String filename, long maxData, int maxItems)
  {
    Object o=null;

    try
    {
      FileInputStream stream=new FileInputStream(filename);
      ObjectInputStream ostream=new ObjectInputStream(stream);
      o=ostream.readObject();
      ostream.close();
      if(!(o instanceof DataStore))
        o=new DataStore(maxData, maxItems);
    }
    catch(Exception e) 
	{
	    Logger.log("DataStore.java", 
		       "Couldn't load DataStore from "+filename, 
		       Logger.NORMAL);
	    o=new DataStore(maxData, maxItems); 
	}
    finally
    { 
      write_interval=1000*Node.params.getlong("checkpointInterval",5*60);
      Node.timer.add(23, write_interval, (Callback)o);
      return (DataStore)o;
    }
  }

  public void tofile(String filename) throws IOException
  {
      Logger.log("DataStore.java","Writing datastore to disk",Logger.MINOR);
      FileOutputStream stream=new FileOutputStream(filename);
      ObjectOutputStream ostream=new ObjectOutputStream(stream);
      ostream.writeObject(this);
      ostream.close();
  }

  public void callback()
  {
    try {
      tofile(".freenet/store");
      Node.timer.add(23, write_interval, this);
    } catch(Exception e) {e.printStackTrace();}
  }

  // Public Fields
  public long maxData;

  // Protected Fields
  protected CyclicArray ar;
  protected Hashtable h;

  // Constructor

  public DataStore(long maxData, int maxItems) {
    ar = new CyclicArray(maxItems);
    h = new Hashtable();
    this.maxData = maxData;
    }

  // Public Methods

  public Enumeration keys()
  {
    return h.keys();
  }

  /**
   * Searches for a key in the DataStore and if found returns
   * any data associated with that key.  If none is found,
   * null is returned.
   * @param k The key to search for
   * @return The data associated with the key or null if not found
   **/

  public Data searchData(Key k) {
      Logger.log("DataStore.java", "searchData("+k + ")",
                 Logger.MINOR);
    synchronized (this) {
/*      for (int x = 0; x < ar.length(); x++) {
        DataStoreItem dsi = (DataStoreItem) ar.get(x);
        if ((dsi != null) && (dsi.key.equals(k)) &&
            (dsi.data != null))
          return dsi.data;*/

        DataStoreItem dsi = (DataStoreItem)h.get(k);
//      }
      if(dsi==null || dsi.data==null)
      {
        Logger.log("DataStore.java", "Not found.", Logger.MINOR);
        return null;
      }
      return dsi.data;
      }
    }

  /**
    * Searches for a key in the DataStore and if found returns
    * any reference associated with that key.  If none is found,
    * null is returned.
    * @param k The key to search for
    * @return The reference associated with the key or null
    *         if not found
    **/

  public Address searchRef(Key k) {
    Logger.log("DataStore.java", "DS:searchRef("+k + ")",
               Logger.MINOR);
    synchronized (this) {
      for (int x = 0; x < ar.length(); x++) {
        DataStoreItem dsi = (DataStoreItem) ar.get(x);
        if (dsi != null)
          if (dsi.key.equals(k))// && (dsi.data != null))
            return dsi.ref;
        }
      }
      return null;
    }

  /**
      * Removes an item from the DataStore given its key
    * @param k The key of the item to remove
    **/

  public void remove(Key k) {
    synchronized (this) {
      for (int x = 0; x < ar.length(); x++) {
        DataStoreItem dsi = (DataStoreItem) ar.get(x);
        Logger.log("DataStore.java",
                   "Testing "+dsi + " against "+k, Logger.DEBUGGING);
        if (dsi != null)
          if (dsi.key.equals(k)) {
            ar.remove(x);
            h.remove(k);
            return;
            }
        }
      }
    }

  /**
    * Returns the reference associated with the closest key
    * to k.
    * @param k The key to search for
    * @return The closest Key to k
    **/
  public Key findClosestKey(Key k) {
    return findClosestKey(k, null);
    }

  /**
    * Returns the reference associated with the closest key
    * to k.  The masking key allows the 'next best' key to
    * be found.  Imagine the entries in the datastore being
    * sorted in terms of closeness to the key you specify.
    * With no mask key the reference associated with the
    * top-most item will be returned.  However, with a mask
    * key any references associated with keys above the mask
    * key (including the mask key itself) will be ignored.
    * This facility allows best-first backtracking.
    * @param k The key to search for
    * @param maskKey The masking key (or null for no mask)
    * @return The closest Key to k (excluding masked keys) or
    *         null if all keys are masked.
    **/

  public Key findClosestKey(Key k, Key maskKey) {
    int m = 0;
    synchronized (this) {
      // Ensure that maskKey exists
      if (searchRef(maskKey) == null)
      {
        maskKey = null;
      }
      for (int x = 0; x < ar.length(); x++) {
        DataStoreItem dsi = (DataStoreItem) ar.get(x);
        if (dsi != null)
          dsi.ignore = false;
        }
      // Ignore all DataStoreItems up to maskKey
      if (maskKey != null) {
        int p;
        do {
          p = findClosestPos(k);
          ((DataStoreItem) ar.get(p)).ignore = true;
          m++;
          if (m == ar.length()) { // We have ignored ALL keys, return null
            return null;
            }
          } while (
            !(((DataStoreItem) ar.get(p)).key.equals(maskKey)))
          ;
        }
      DataStoreItem dsi = (DataStoreItem) ar.get(findClosestPos(k));
      if ((dsi != null) && (!dsi.ignore))
        return dsi.key;
      return null;
      }
    }

  /**
    * Adds an item to the DataStore
    * @param k The key to add
    * @param r The address to which requests should be directed for
    *          this key if the data is removed
    * @param d The data associated with this key
    **/

  public void put(Key k, Address r, Data d) {
    synchronized (this) {
      DataStoreItem dsi = new DataStoreItem(k, r, d);
      ar.put(dsi);
      h.put(k, dsi);
      cleanUpData();
      }
    }

  // Protected Methods

  /**
   * Will return the position in ar of the closest key to k
   * ignoring any data items with their ignore flags set
   * @param k Key to search for
   * @return Position in ar of closest key
   **/
  protected int findClosestPos(Key k) {
    synchronized (this) {
      int cl = 0;
      for (int x = 1; x < ar.length(); x++) {
        DataStoreItem bs = (DataStoreItem) ar.get(cl);
	if ((bs!=null)&&(bs.ignore)) bs = (DataStoreItem) ar.get(++cl);
        DataStoreItem ds = (DataStoreItem) ar.get(x);
        if ((bs != null) && (ds != null) && (!ds.ignore) &&
            (k.compare(ds.key, bs.key)))
          cl = x;
        }
      return cl;
      }
    }

  /**
    * Removes excess data (ie. if there is more data than is
    * permitted by the maxData field).
    **/
  protected void cleanUpData() {
    long total = 0;
    for (int x = 0; x < ar.length(); x++) {
      Object o = ar.get(x);
      if (o != null) {
        DataStoreItem dsi = (DataStoreItem) o;
        if (dsi.data != null) {
          if (total + dsi.data.length() > maxData)
            dsi.data = null;
          else
            total += dsi.data.length();
          }
        }
      }
    }

  // Mainly for debugging
  public String toString() {
    String t = "DataStore: maxData="+maxData + " bytes";
    for (int x = 0; x < ar.length(); x++) {
      t = t + "\n "+x + ")\t"+ar.get(x);
      }
    return t;
    }
  }

class DataStoreItem implements Serializable {
  public Key key;
  public Address ref;
  public Data data;
  public boolean ignore = false;

  public DataStoreItem(Key k, Address a, Data d) {
    key = k;
    ref = a;
    data = d;
    }

  public String toString() {
    return "Key: "+key + " Ref: "+ref + " Data:"+data;
    }
  }

/*
 *$Log: DataStore.java,v $
 *Revision 1.17  2000/03/22 18:19:36  blanu
 *Added Hashtable for keys for faster lookup.
 *
 *Revision 1.16  2000/03/17 10:06:02  blanu
 *Changed Freenet.support.Timer to Freenet.support.Ticker to avoid JDK1.3 name conflict.
 *
 *Revision 1.15  2000/02/23 20:52:42  blanu
 *Fixed timeout in client.
 *
 *Revision 1.14  2000/02/14 21:08:14  blanu
 *The datastore is now written to disk every checkpointInterval seconds as
 *per the configuration file, with a default of every 5 minutes.
 *
 *Revision 1.13  2000/02/14 00:20:31  blanu
 *Periodic writing of DataStore information to .freenet/store. It doesn't
 *work yet. I think it might be something wrong with Ticker.
 *
 *Revision 1.12  2000/02/13 12:11:57  sanity
 *Fixed numerous syntax errors.
 *
 *Revision 1.11  2000/02/09 23:56:39  hobbex
 *Preliminary tunneling of DataReplies and DataInserts (for now)
 *
 *Revision 1.10  2000/02/08 16:22:06  sanity
 *Removed printing of exception when datastore file is not found, replaced
 *with logger message.
 *
 *Revision 1.9  2000/02/02 21:03:06  blanu
 *Changed the fromfile method so that it should work without a store file.
 *
 *Revision 1.8  2000/02/02 11:26:40  sanity
 *Merged changes.
 *
 *Revision 1.7  2000/02/02 05:13:28  blanu
 *Datastore can now be read to / written from a file. Params can be fetched as various types.
 *
 *Revision 1.6  2000/01/03 10:20:48  hobbex
 *Removed deleting of MM on Loop and Failed Request
 *
 *Revision 1.5  2000/01/03 00:09:03  michaels
 *findClosestPos didn't respect a set ignore flag on first dsi in array (cl==0)
 *now it does.
 *
 *Revision 1.4  2000/01/02 14:10:15  michaels
 *findClosestKey doesnt't return a dsi anymore if its ignore-flag is set to true
 *
 *Revision 1.3  2000/01/02 13:05:06  michaels
 *searchRef is used for checking that a maskKey exists (not searchData)
 *
 */
