package Freenet;
import Freenet.message.TimedOut;
import Freenet.support.*;
import java.lang.reflect.*;
import java.net.*;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

  Explanation of Code Versions: 
    0.0.0      = Initial Description
    0.0.1      = API Specified
    0.x (x>0)  = Partial Implementation
    x.0 (x>0)  = Operational
                
  Requires Classes: Node (1.0)
                    Address (1.0)
 */

/**
 * This is the abstract superclass of all messages
 *
 * @see Node
 * @see Address
 * @author Brandon Wiley (blanu@uts.cc.utexas.edu)
 * @author Ian Clarke (I.Clarke@strs.co.uk)
 **/

public class Message
{
  /** The Address of the node where this message originated **/
  public Address source;
    /** Address of this node on the network that this message came from **/
    public Address receivedAt = null;
  /** The number of times this message should be forwarded before
      generating a TimedOutMessage **/
  public long hopsToLive;
    /** The number number of hops this message is from its originating node **/
    public long depth;
  /** A randomly generated unique ID used to identify related messages **/
  public long id;

  public Message(long idnum, long htl, long dpth)
    {
      id=idnum;
      hopsToLive=htl;
      depth=dpth;
    }

  public Message(RawMessage raw) throws InvalidMessageException
  {
    id=hextolong(raw.readField("UniqueID"));
    hopsToLive=new Long(raw.readField("HopsToLive")).longValue();
    depth=new Long(raw.readField("Depth")).longValue();
    source=new Address(raw.readField("Source"));
  }

/* I know, you're thinking Brandon, why don't you use Long.valueOf(hex,
16). Well, because Long.toHexString() generates unsigned values and
Long.valueOf expects a signed value. So if you know the elegant way to do
this, please tell me. It's one line in C and Python. */
  private long hextolong(String hex) throws NumberFormatException
  {
    long l=0;
    char c;

    int len=hex.length();
    int p=len-1;

    for (int i=0; i<len; i++) {
      c=hex.charAt(i);
      if((c>='0')&&(c<='9'))       l=l+((c-'0')*(long)Math.pow(16,p-i));
      else if ((c>='a')&&(c<='f')) l=l+((c-'a'+10)*(long)Math.pow(16,p-i));
      else if ((c>='A')&&(c<='F')) l=l+((c-'A'+10)*(long)Math.pow(16,p-i));
      else throw new NumberFormatException();
    }
    return l;
  }

  public RawMessage toRawMessage()
    {
      RawMessage r = new RawMessage("");
      r.trailingFieldName="EndMessage";
      r.setField("UniqueID", Long.toHexString(id));
      r.setField("HopsToLive", String.valueOf(hopsToLive));
      r.setField("Depth", String.valueOf(depth));
      r.setField("Source", source.toString());
      return r;
    }


    /** 
     * Called by the connection handler after right constructing
     * the message, while still in touch with the connection
     * @param me The address of this node that the message was
     *           received on.
     * @param peer The address of the node the message was recieved
     *             from
     **/

    public void initSources(Address me, Address peer)
    {
	receivedAt = me;
	source = peer;
    }

  /**
   * Called by a node after this message is received.  This allows
   * message handling code to be packaged within the message class
   * that is being handled.
   * @param n The node that called this message.  This should be used
   *          to make any nescessary modifications to the DataStore etc.
   * @param sb Null unless this node has been seen before.  If non-null it
   *           is the Object returned by the received method of the
   *           last message seen with this ID.
   * @return The object to be passed to any messages received in the near
   *         future with the same ID as this one.  Null if no object 
   *         should be passed.
   **/
  public final MessageMemory received(Node n, MessageMemory sb)
    { 
	// Reduce the lifespan of this message
	hopsToLive--;
	
	if(hopsToLive<0) { // Message expired. Inform sender.
		Logger.log("Message.java","Message timed out, sending timeout to original source",Logger.MINOR);
	    return timeOut(n, sb);
	} else {
	    // Increase the depth
	    depth++;
	    return pReceived(n, sb); // Actually do something with the message
	}
    }

    /**
     * Called just before the message is sent, and makes any last minute
     * changes.
     *
     * @param n          The node that is sending the message
     * @param peer       The address to which the message is being sent
     * @param myAddress  The sending nodes address with respect to peer
     **/

    public void sending(Node n, Address peer, Address myAddress) throws SendFailedException
    {
	if (myAddress.equals(peer))
	    throw new SendFailedException(peer); // don't send to yourself
	source = myAddress;
    }

  /**
   * An overridable version of the received message.
   * @param n The node that called this message.  This should be used
   *          to make any nescessary modifications to the DataStore etc.
   * @param sb Null unless this node has been seen before.  If non-null it
   *           is the Object returned by the received method of the
   *           last message seen with this ID.
   * @return The object to be passed to any messages received in the near
   *         future with the same ID as this one.  Null if no object 
   *         should be passed.
   **/

    protected MessageMemory pReceived(Node n, MessageMemory sb)
    {
	return null;
    }

    protected MessageMemory timeOut(Node n, MessageMemory sb)
    {
	TimedOut to=new TimedOut(id, depth); // this message is going in
    
	try {
	    n.sendMessage(to,source);
	} catch(SendFailedException sfe) {
	    Logger.log("Message.java","Send of timeout failed",Logger.NORMAL);
	}
	
	return sb;
    }
    
    public String toString()
    {
	return "Src:"+source+" htl:"+hopsToLive+
	    " depth:" + depth + " id:"+id;
    }
}


