/*
 *$Id: RawMessage.java,v 1.12 2000/02/29 06:46:56 blanu Exp $

  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU General Public Licence (GPL) 
  version 2.  See http://www.gnu.org/ for further details of the GPL.
 */

/**
 * This class represents a raw message in the general format
 * expected by Freenet.  It can be created either from
 * an InputStream or manually created from scratch.  It can
 * then by piped out to an OutputStream.  Methods are provided
 * for manipulation of normal fields, however the type of the
 * message and the trailing field should be set by direct
 * manipulation of fields.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author <A HREF="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</A>
 **/
package Freenet;

import Freenet.support.*;
import java.io.*;
import java.util.*;
import java.math.BigInteger;
 

public class RawMessage
{
  public static final boolean debug = true;

  /**
   * Test class
   **/
  public static void main(String[] args)
    throws Exception
    {
      RawMessage r = new RawMessage(System.in);
      InputStream i = r.toInputStream();
      int t=0;
      while (t != -1)
	{
	  t = i.read();
	  System.out.write(t);
	}
    }

  // Public Fields
  /** The type of the message **/
  public String messageType;
  public BigInteger trailingFieldLength;
  public String trailingFieldName = null;
  public InputStream trailingFieldStream = null;
  
  // Protected/Private Fields
  protected Hashtable fields = new Hashtable();
  protected static final char[] crlf = {'\n', '\r'};
  protected static final char[] endfn = {'=', '\n', '\r'};
  // Constructors
  public RawMessage(InputStream i) throws InvalidMessageException
    {
      try
	{
	  PushbackInputStream in = new PushbackInputStream(i);
	  // Read message type
	  messageType = readTo(in, crlf);
	  // Now read field/data pairs
	  while(true)
	    {
	      String name, data;
	      long length=0;

	      // Read field name
	      remCRLF(in);
	      if (testEOF(in))
		  break;
	      name = readTo(in, endfn);

	      if(name.equals("EndMessage")) break;

	      if (testEOF(in))
		  break;
	      // If the next value is a CR then this is a
	      // trailing field
	      if (testEOF(in))
		  break;

	      if (in.read() == '\n')
		  {
		      // Removing LF after CR
		      char r = (char) in.read();
		      if (r != 10)
			  in.unread ((int)r);

		      Logger.log("RawMessage.java","Found trailing field",Logger.MINOR);
		      trailingFieldName = name;
		      String dl=readField("DataLength");
                      if(dl!=null)
		        trailingFieldLength = new BigInteger(dl);
                      trailingFieldStream=in;
		      break;
		  }
	      else
		  { // This is not a trailing field
		              data = readTo(in, crlf);
		  }
	      setField(name, data);
	    }
	}
      catch (IOException e)
	{
	  throw new InvalidMessageException("Could not parse message from stream.");
	}
      catch (EOFException e)
	{
	}
      catch(Exception e)
        {
          Logger.log("RawMessage.java", "Exception in RawMessage()", Logger.ERROR);
          e.printStackTrace();
        }
    }

  public RawMessage(String name)
    {
      messageType = name;
    }

  // Public Methods
  public void deleteField(String name)
    {
      fields.remove(name);
    }

  public String readField(String name)
    {
      return (String) fields.get(name);
    }

  public void setField(String name, String value)
    {
      fields.put(name, value);
    }


    /**
     * @depreaciated This method could lock on large messages. Use
     * writeMessage instead.
     **/
    public InputStream toInputStream() throws IOException
    {
	PipedOutputStream po=new PipedOutputStream();
	BufferedInputStream pi=new BufferedInputStream(new PipedInputStream(po));
	writeMessage(po);
	po.flush();
	po.close();

	return (trailingFieldStream != null) ? (InputStream)(new SequenceInputStream(pi, trailingFieldStream)) : (InputStream)pi;
    }

    public InputStream getTrailing() {
	return trailingFieldName != null ? trailingFieldStream : null;
    }


    public void writeMessage(OutputStream out) throws IOException
    {
	String key, value;

	PrintWriter writer=new PrintWriter(out);
      
	// Output message type
	writer.println(messageType);

	// Output non-trailing fields
	Enumeration iterator=fields.keys();
	while(iterator.hasMoreElements()) {
	    key=(String)iterator.nextElement();
		value=(String)fields.get(key);
		writer.print(key+"="+value+"\n");
	    }

      // Output trailing field
      if(trailingFieldName!=null)
      {
	if(trailingFieldLength!=null)
          writer.print("DataLength="+trailingFieldLength+"\n");
        writer.print(trailingFieldName+"\n");
      }

      writer.flush();
    }

    public void writeTrailing(OutputStream out) 
    {
	if (trailingFieldStream!=null && out!=null) {
	    Conduit c = new Conduit(trailingFieldStream,out);
	    c.syncFeed();
	}
    }

  public String toString()
    {
      return messageType + "\n" + fields.toString();
    }

  // Protected/Private Methods

  /**
   * Removes any CRLFs from the part of the stream to be
   * read next
   **/
  protected void remCRLF(PushbackInputStream i)
    throws IOException
    {
      char r;
      do
	{
	  r = (char) i.read();
	} while ((r == '\n') || (r == '\r'));
      i.unread((int) r);
    }

    /** Checks non-destructively whether we have received an EOF **/
    protected boolean testEOF(PushbackInputStream i)
	throws IOException
    {
	int c;
	c = i.read();
	if (c==-1)
	    return true;
	else
	    {
		i.unread(c);
		return false;
	    }
    }
  
  /**
   * Will return String from InputStream until one of
   * the characters in tms is about to be read.
   **/
  protected String readTo(PushbackInputStream i, char[] tms)
    throws IOException, EOFException
    {
      StringBuffer tmp = new StringBuffer();
      char r=' ';
      int ir;
      while (true)
	{
	    ir = i.read();
	    if (ir == -1)
		break;
	    r = (char) ir;
	    if (r == -1)
		throw new EOFException();
	    boolean b = false;
	    for (int x=0; x<tms.length; x++)
		{
		    if (tms[x] == r)
			{
			    b=true;
			    break;
			}
		}
	    if (b)
		break;
	    tmp.append(r);
	}
      if (ir != -1)
	  i.unread((int) r);
      return tmp.toString();
    }
    
}

class EOFException extends Exception
{
}



/*
 *$Log: RawMessage.java,v $
 *Revision 1.12  2000/02/29 06:46:56  blanu
 *Stray debugging println removed.
 *
 *Revision 1.11  2000/02/29 06:35:36  blanu
 *Handshaking is now in a message-like format. It actually uses RawMessage to construct the handshake.
 *RawMessage now supports the EndMessage field to end a message instead of EOF.
 *This is useful for handshaking and also for future Keepalive connections.
 *
 *Revision 1.10  2000/02/17 14:12:54  hobbex
 *catch unknown message types
 *
 *Revision 1.9  2000/02/17 12:52:23  hobbex
 *removed runtime exception on bad connection (annoying)
 *
 *Revision 1.8  2000/02/12 13:59:57  hobbex
 *fixed tunneling on inserts for large files. Britney never sounded so good!
 *
 *Revision 1.7  2000/02/11 20:12:01  hobbex
 *Fixed tunneling to use only one thread. Changed Rawmessage to write to output streams rather than return inputstreams. Fixed releasing of DataStream when a message is dropped.
 *
 *Revision 1.6  2000/02/09 23:56:39  hobbex
 *Preliminary tunneling of DataReplies and DataInserts (for now)
 *
 *Revision 1.5  2000/01/27 15:45:39  sanity
 *Some bug fixes, added localAddress field to Node.java which is used by
 *DataRequest and DataReply messages to set dataSource field.  Also removed
 *untidy debugging stuff from RawMessage.
 *
 *Revision 1.4  2000/01/11 06:32:05  blanu
 *fixed problem with hex uniqueid and name=value field format
 *
 *Revision 1.3  2000/01/11 02:02:51  blanu
 *changed from -: to = field format
 *
 *Revision 1.2  2000/01/03 16:32:07  hobbex
 *Changed various println to Logger - think I got them all
 *
 *Revision 1.1.1.1  1999/12/31 11:53:02  sanity
 *Initial import from OpenProjects
 *
 *Revision 1.4  1999/12/29 07:14:06  michael
 *Additional LF after CR of trailing field - bug fixed
 *
 *
 */
