#!/usr/bin/perl -w
# Liberator v 0.03
# Freenet client by Oskar Sandberg
#
# see http://freenet.sourceforge.net for more info.
#
# This code is released under the GNU Public License. I hate legal babel.
# Warranty my ass.
#
# version history
#
# 991228 v 0.001
# 2K0102 v 0.002 Removed fork (thanks Steven Hazel) and fixed 
#                handling of FailedRequest (Michael fixed them in 
#                Freenet).
# 2K0116 v 0.03  Now asks socket for local address, can save/read from
#                a file, supports freenet protocol changes, checks
#                for correct id (and server) on reply, added verbous, strict, 
#                dh, and dl options, and cleaned up syntax.
#
# 2K0304 v 0.04  Change for new handshake protocol using message format
#                Update FREENETPROTO for this new protocol
#                Use SHA hash of key
#                Include DataLength field in insert messages
#                Change default port to be consistent with current distribution
#                (by Hal Finney)
# 2K0305 v 0.05  Change to use native SHA-1 rather than external module
# 2K0306 v 0.06  Change to use new two-message Insert protocol
# 2K0309 v 0.07  Use new two-message request/reply Handshake protocol
# 2K0320 v 0.08  Add EndMessage field; handle QueryRestarted message
#

use strict;
use Socket;
use Getopt::Long;

my ($remote, $port, $iaddr, $paddr, $paddr2, $proto, $line, $myhost, $myport, $mysockaddr, $sendport, $message, $id, $depth, $handshake, $pid);

my $FREENETPROTO = "1.1";
my $DEFAULTNODE = "localhost";
my $DEFAULTPORT = 19114;
my $DEFAULTHOPS = 10;
my $DEFAULTDL = 5;
my $DEFAULTDH = 30;

sub usage {
  print(" " x 10 . "Liberator v 0.08, Freenet Client by Oskar Sandberg\n",
	"The supported protocol is $FREENETPROTO",
	"THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY! USE AT YOUR OWN RISK!\n",
	"usage: liberator -i|--insert -k|--key=KEY [Options]\n",
	"   or: liberator -r|--request -k|--key=KEY [Options]\n",
	"Options:\n",
	"-n, --node=ADDRESS ($DEFAULTNODE\:$DEFAULTPORT)   Address of first Freenet node\n",
	"-h, --hops=HOPSTOLIVE ($DEFAULTHOPS)   Messages hops to live on Freenet\n",
	"-f, --file=FILENAME (stdout/stdin)   Filename to write to or read from\n",
	"-dl, --depthlow=LOWDEPTH ($DEFAULTDL)\n",
	"-dh,--depthhigh=HIGHDEPTH ($DEFAULTDH)\n   The original depth of the message is set to random number between\n   depthhigh and depthlow. The former must be smaller or equal (duh)\n",
	"--verbose   Print sent message to stdout\n",
	"--strict   Use strict security\n",
        "--help   Print this screen\n",
	"\nFor more info see The Freenet Project at http:\\freenet.sourceforge.net\n");
  exit 0;
}

my ($insert, $request, $node, $key, $hop, $dl, $dh, $verbose, $filename, $strict, $help);
GetOptions("insert|i" => \$insert,
	   "request|r" => \$request,
	   "node|n=s" => \$node,
	   "key|k=s" => \$key,
	   "hop|h=i" => \$hop,
	   "dl|depthlow=i" => \$dl,
	   "dh|depthhigh=i" => \$dh,
	   "verbose" => \$verbose,
	   "file|f=s" => \$filename,
	   "help" => \$help,
	   "strict" => \$strict) || usage;

# Check that all the paras are set correct.

$strict && ($dh && $dl || die("STRICT: depthhigh and depthlow must be set (never trust a machine)\n"));

(($insert xor $request) && $key && ($hop || ($hop = $DEFAULTHOPS)) && ($dl || ($dl = $DEFAULTDL)) && ($dh || ($dh = $DEFAULTDH)) && ($dh >= $dl) && !$help)  || usage;

($remote, $port) = split(/\:/,$node) if $node;

($port || ($port = $DEFAULTPORT)) && ($remote || ($remote = $DEFAULTNODE));

if ($port =~ /\D/) { $port = getservbyname($port, 'tcp') }
usage unless $port;
$iaddr   = inet_aton($remote) || die "no host: $remote";
$paddr   = pack_sockaddr_in($port, $iaddr);
$proto   = getprotobyname('tcp');

# Open a socket that listens for answers

socket(Server, PF_INET, SOCK_STREAM, $proto) || die "socket: $!";
setsockopt(Server, SOL_SOCKET, SO_REUSEADDR, pack("l", 1)) || die "setsockopt: $!";
until (bind(Server, sockaddr_in(( $myport = int(rand(32768-1024)) + 1025), INADDR_ANY))) {;}
# as far as I know there are 2**16 ports available, but the Freenet node uses a signed short for it (maybe we should fix that some day)

listen(Server,SOMAXCONN) || die "listen: $!";



# now open the server socket to send the DataInsert or DataRequest
socket(SOCK, PF_INET, SOCK_STREAM, $proto)  || die "socket: $!";
connect(SOCK, $paddr) || die "connect: $!";

# get the local host from the socket
$mysockaddr = getsockname(SOCK);
($sendport, $myhost) = unpack_sockaddr_in($mysockaddr);
$myhost = gethostbyaddr($myhost, AF_INET); 

# use /dev/random if possible
if (open(RAND,"/dev/random")) {
  read RAND, $id, 8;
  $id = pack("a16",unpack("h16",$id));
} else {
  $id = '';
  for (1..16) {$id .= sprintf("%x",int(rand(16)))}
}
substr($id,0,1) = '1' if substr($id,0,1) =~ "0";

# assemble handshake message

$message = "HandshakeRequest\n";
$message .= "Source=tcp/$myhost:$myport\n";
$message .= "UniqueID=$id\n";
$message .= "Depth=1\n";
$message .= "HopsToLive=1\n";
$message .= "EndMessage\n";

print SOCK $message;
print $message if $verbose;
close (SOCK) || die "close: $!";

# Handle handshake reply
&handlereply;


# now open the server socket to send the DataInsert or DataRequest
socket(SOCK, PF_INET, SOCK_STREAM, $proto)  || die "socket: $!";
connect(SOCK, $paddr) || die "connect: $!";

# assemble next message

if ($insert) {
  $message = "InsertRequest\n";
} elsif ($request) {
  $message = "DataRequest\n";
}

$message .= "Source=tcp/$myhost:$myport\n";

$key = sha1hex($key);
$message .= "SearchKey=$key\n";


# Change $id to be different
if(substr($id,0,1) =~ "1") {
  substr($id,0,1) = '2';
} else {
  substr($id,0,1) = '1';
}
$message .= "UniqueID=$id\n";
$message .= "Depth=" . (int(rand($dh - $dl + 1)) + $dl) . "\n";
$message .= "HopsToLive=$hop\n";
$message .= "EndMessage\n";


print SOCK $message;
print $message if $verbose;
close (SOCK) || die "close: $!";

# Print the reply
&handlereply;

close Server;


# Get second message for insert
if ($insert) {
  my ($infile, @indata, $count);

  $message = "DataInsert\n";
  $message .= "DataSource=tcp/$remote:$port\n"; # lie about datasource
  $message .= "Source=tcp/$myhost:$myport\n";
  $message .= "SearchKey=$key\n";
  $message .= "UniqueID=$id\n";
  $message .= "Depth=" . (int(rand($dh - $dl + 1)) + $dl) . "\n";
  $message .= "HopsToLive=$hop\n";

  socket(SOCK, PF_INET, SOCK_STREAM, $proto)  || die "socket: $!";
  connect(SOCK, $paddr) || die "connect: $!";

  print SOCK $message;
  print $message if $verbose;

  if ($filename) {
    open(INFILE, $filename) || die("Could not open $filename");
    $infile = \*INFILE;
  } else {
    $infile = \*STDIN;
  }
#  read all of data in so we can get its length
  @indata = <$infile>;
  $count = 0;
  foreach $line ( 0 .. $#indata ) {
      $count += length($indata[$line]);
  }
  print SOCK "DataLength=$count\n";
  print "DataLength=$count\n" if $verbose;
  print SOCK "Data\n";
  print "Data\n" if $verbose;
  print SOCK @indata;
  close SOCK;
}


exit;

# Check for valid handshake and version
sub handlereply {
  for ( ; $paddr2 = accept(Client,Server); close Client) {
    
    if ($strict) {
      my($sourceport,$sourceaddress) = unpack_sockaddr_in($paddr2);
      if ($iaddr ne $sourceaddress) {
        print "STRICT: Got reply, but from ",inet_ntoa($sourceaddress), ", sent to " , inet_ntoa($iaddr) , ". Will not reply.\n";
        next;
      }
    }
    
    print "_________ \n Received:\n" if $verbose;
    my $messagetype = <Client>;
    print ("$messagetype\n") if $verbose;
    my(%fields, $field);      # read field values
    while (defined($field = <Client>) && ($field  =~ /\=/)) {
      
      print $field if $verbose;
      my($fieldname, $fieldvalue) = split('=',$field);
      chop $fieldvalue;      #dump \n at the end of every field
      $fields{$fieldname} = $fieldvalue;
    };
    print "\n" if $verbose;
  
    if ($fields{'UniqueID'} ne $id  && !($messagetype =~ /HandshakeRequest/) ) {
      print "Recieved message with wrong id!\n";
      print "Recieved message '$fields{'UniqueID'}' expected '$id'\n" if ($verbose);
      next;
    }
    if ($messagetype =~ /HandshakeRequest/) {
      my( $haddr, $hprotoname, $hremote, $hport, $hiaddr, $hpaddr, $hproto );
      ($hprotoname, $haddr) = split(/\//,$fields{'Source'});
      ($hremote, $hport) = split(/\:/,$haddr);
      $hiaddr   = inet_aton($hremote) || die "no host: $hremote";
      $hpaddr   = pack_sockaddr_in($hport, $hiaddr);
      $hproto   = getprotobyname($hprotoname);
      socket(HSOCK, PF_INET, SOCK_STREAM, $hproto)  || die "socket: $!";
      connect(HSOCK, $hpaddr) || die "connect: $!";
      $message = "HandshakeReply\n";
      $message .= "Depth=$fields{'Depth'}\n";
      $message .= "HopsToLive=$fields{'HopsToLive'}\n";
      $message .= "UniqueID=$fields{'UniqueID'}\n";
      $message .= "Source=tcp/$myhost:$myport\n";
      $message .= "Version=$FREENETPROTO\n";
      $message .= "EndMessage\n";
      print HSOCK $message;
      close HSOCK;
      print "Sending handshake reply message:\n" if $verbose;
      print $message if $verbose;
    } elsif ($messagetype =~ /DataReply/) {
      if ($request) { 
  	my $outfile; 
  	if ($filename) {
  	  die("File '$filename' already exists\n") if (-f $filename);
  	  open(OUTFILE, ">$filename") || die("Could not open $filename");
  	  $outfile = \*OUTFILE;
  	} else {
  	  $outfile = \*STDOUT;
  	}
  	while (defined(my $line = <Client>)) {
  	  print $outfile $line;
  	}
      } elsif($insert) {
  	print "You insert collided with previous data with the same key ($fields{'SearchKey'}). Please insert it again with a new key and discourage anyone from attempting to access you data with this key. We recommend you attempt to request the key before inserting to make sure it is not used already";
      }
      last;
    } elsif ($messagetype =~ /HandshakeReply/) {
      die "Server running version $fields{'Version'}, we support $FREENETPROTO"
           unless( $fields{'Version'} == "$FREENETPROTO" );
      last;
    } elsif ($messagetype =~ /InsertReply/) {
      last;
    } elsif ($messagetype =~ /QueryRestarted/) {
      ;
    } elsif ($messagetype =~ /RequestFailed/) {
      print "Sorry, your Requested key was not found\n";
      last;
    } elsif ($messagetype =~ /TimedOut/) {
      if ($request) {
  	print "Your request timed out, try raising the Hops To Live (-h)\n";
      } elsif ($insert) {
  	print "You insert was successful\n";
      }
      last;
    } elsif ($messagetype =~ /InsertFailed/) {
      print "Your insert reached every possible node before it timed out (is this good or bad?)\n";
      last;
    } else {
      print "Recieved unknown reply. Maybe an upgrade is in order";
    }
  }
}

# SHA-1 in Perl
sub sha1hex {
  my ( $instr ) = @_;
  my ( @A, @K, @F, @W, $total, $bytes, $padded, $t, $a, $b, $c, $d, $e, $r, $r20 );

  # Initial values
  @A = ( 0x67452301, 0xefcdab89, 0x98badcfe, 0x10325476, 0xc3d2e1f0 );
  # Constants used in algorithm
  @K = ( 0x5a827999, 0x6ed9eba1, 0x8f1bbcdc, 0xca62c1d6 );
  # Four different nonlinear functions used in different iterations
  @F = (sub{$b&($c^$d)^$d},sub{$b^$c^$d},sub{($b|$c)&$d|$b&$c},sub{$b^$c^$d});
  # Fix overflowing adds to be mod 2^32
  sub addfix{
   my( $val ) = @_;
   my( $denom );
   $denom = 1 + ~0;
   return $val - ($denom * int( $val/$denom ));
   };
  # Rotate left the specified amount
  sub lrotate{
   my( $val, $rot ) = @_;
   my( $mask );
   $mask = 2**$rot - 1;
   return ($val<<$rot) | (($val>>(32-$rot)) & $mask);
   }

  # Loop for each 512 bit block of input
  do {
   # Get next 512 bits of input
   $_ = substr($instr, 0, 64);
   substr($instr, 0, 64) = "";
   $total += $bytes = length $_;

   # Add byte of 0x80 after last byte of input
   if( $bytes < 64 && !$padded ) {
     $bytes++;
     $_ .= "\x80";
     $padded = 1;
   }

   # Convert 64 bytes of input into 16 words
   @W = unpack "N16", $_ . "\0"x64;

   # If last block, set total bit length in last word.
   if ($bytes <= 56) {
     $W[15] = $total*8;
   }

   # Fill rest of word array.  Note that lrotate distinguishes SHA-1 from SHA.
   for $r (16..79){
    $W[$r] = lrotate($W[$r-3]^$W[$r-8]^$W[$r-14]^$W[$r-16], 1)
   }

   # Get state variables $a - $e.
   ($a,$b,$c,$d,$e) = @A;

   # Loop for 80 rounds
   for $r (0..79){
    # Do one round of SHA transformation
    $r20 = $r / 20;
    $t = addfix( lrotate($a, 5) + &{$F[$r20]} + $e + $W[$r] + $K[$r20] );
    $e = $d;
    $d = $c;
    $c = lrotate($b, 30);
    $b = $a;
    $a = $t
   }

   # Add back into initial state variables to finish processing this block
   $A[0] = addfix($A[0]+$a);
   $A[1] = addfix($A[1]+$b);
   $A[2] = addfix($A[2]+$c);
   $A[3] = addfix($A[3]+$d);
   $A[4] = addfix($A[4]+$e);
  } while ($bytes > 56);

  # Done, return in Freenet format, 8 upper case chars followed by space.
  return sprintf( "%.8X"x5, @A );
}
