package Freenet.message;
import java.io.*;
import java.math.*;
import Freenet.*;
import Freenet.support.*;

/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

  Explanation of Code Versions: 
    0.0.0      = Initial Description
    0.0.1      = API Specified
    0.x (x>0)  = Partial Implementation
    x.0 (x>0)  = Operational
		
  Requires Classes: Node (1.0)
                    Address (1.0)
		    Message (1.0)
 */

/**
 * This is the DataReply message
 *
 * @see Node
 * @see Address
 * @author Brandon Wiley (blanu@uts.cc.utexas.edu)
 * @author Ian Clarke (I.Clarke@strs.co.uk)
 **/

public abstract class DataSend extends Message
{
  public InputStream in=null;
  public BigInteger length=null;
  public Address dataSource=null;

  public DataSend(long idnum, long htl, long depth, Address dsrc)
    {
      super(idnum, htl, depth);
      this.dataSource = dsrc;
    }

    public DataSend(long idnum, long htl, long depth, Address dsrc, Data data) throws DataNotReadyException, IOException {
	super(idnum, htl, depth);
	this.dataSource = dsrc;
	length=data.getLength();
	in=data.toInputStream();
    }

  public DataSend(RawMessage raw) throws InvalidMessageException
  {
    super(raw);
    dataSource = new Address(raw.readField("DataSource"));
    if(raw.trailingFieldLength!=null)
      length=raw.trailingFieldLength;
    else
      throw new InvalidMessageException("Data sending message requires the trailing field length to be specified");
    if(raw.trailingFieldName!=null)
	in=new BufferedInputStream(raw.trailingFieldStream);
    if(in==null) throw new InvalidMessageException("Data sending message requires a trailing field called Data");
  }

  public RawMessage toRawMessage()
  {
    RawMessage raw=super.toRawMessage();
    raw.setField("DataSource", dataSource.toString());
    if(in!=null)
    {
	raw.trailingFieldStream=in;
        raw.trailingFieldLength=length;
    }
    else
      return raw;
    raw.trailingFieldName="Data";
    return raw;
  }

  /**
   * Called by a node after it receives this message
   * @param n The node that called this message.  This should be used
   *          to make any nescessary modifications to the DataStore etc.
   * @param sb Null unless this node has been seen before.  If non-null it
   *           is the Object returned by the received method of the
   *           last message seen with this ID.
   * @return The object to be passed to any messages received in the near
   *         future with the same ID as this one.  Null if no object should
   *         be passed.
   **/
  public MessageMemory pReceived(Node n, MessageMemory sb)
  {
      Logger.log("message/DataReply.java","Got a DataSend!",Logger.MINOR);
    if(sb==null || !(sb instanceof KeyedMM)) {// No one requested this, or I forgot about it
	try{in.close();} catch(IOException e) {}
	return null;
    }
    else // Forward it to whoever requested it.
    {
	// Cancel any timer waiting for this message
	Node.timer.cancel(id);

	SplitOutputStream datatunnel = new SplitOutputStream();
	KeyedMM kmm = (KeyedMM)sb;
	if (n.ds.searchData(kmm.searchKey) == null) { 
	    // if I got some data between request and reply, don't recache
	    if(!Node.tunneling) {
		Data mydata = new Data(in);
		n.ds.put(kmm.searchKey, dataSource, mydata); // Store the data
		// recreate InputStream from file
		try {
		    in = mydata.toInputStream();
		} catch (Exception e) {
		    Logger.log("message/DataInsert","Failed to restore data from store!",Logger.ERROR);
		}
	    } else {
		n.ds.put(kmm.searchKey, dataSource, new Data(datatunnel)); // Cache the data
	    }
	}

	if (kmm.dataref == null) { // So weeeeee've come, to the eeeeeend of the road...
	    if (Node.tunneling) 
		(new Conduit(in, datatunnel)).asyncFeed(datatunnel);
	} else { // still got places to go
	    try {
		n.sendMessage(this, kmm.dataref, datatunnel);
	    } catch(SendFailedException sfe) {
		Logger.log("message/DataSend.java","Couldn't proxy reply to " + sfe.peer,Logger.NORMAL);
	    } 
	}

	return null; // My job is done. Forget about this.
    }
  }

    public void sending(Node n, Address peer, Address myAddress) throws SendFailedException
    {
	if (myAddress.equals(peer))
	    throw new SendFailedException(peer); // don't send to yourself
	source = myAddress;
    }

      // Protected/Private Methods

    protected MessageMemory timeOut(Node n, MessageMemory sb)
    {
	if (sb == null || !(sb instanceof KeyedMM)) {
	    try{in.close();} catch(IOException e) {}
	    return null;
	}

	KeyedMM kmm = (KeyedMM) sb;

	Data mydata = new Data(in);
	n.ds.put(kmm.searchKey, dataSource, mydata);

	return null;
    }
}


