package Freenet.support;

import java.io.Serializable;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

  Explanation of Code Versions: 
    0.0.0      = Initial Description
    0.0.1      = API Specified
    0.x (x>0)  = Partial Implementation
    x.0 (x>0)  = Operational
		
  Requires Classes: Class (version)
                    Class (version)
                    ...
 */

/**
 * This class represents an array which is of finite length
 * and can have information added to it in such a way that
 * if the array fills to capacity, the last information added
 * will be deleted.
 *
 * @version 1.0
 * @author Ian Clarke (I.Clarke@strs.co.uk)
 **/

public class CyclicArray implements Serializable
{
  /**
   * The array that contains the objects
   **/
  Object[] ar;
  /**
   * A pointer to the next item to be deleted, moves backwards
   * through array
   **/
  int ptr;

  public CyclicArray(int size)
    {
      ar = new Object[size];
      ptr = size-1;
    }

  /**
   * Obtain the length of this CyclicArray
   * @return Length of array
   **/
  public int length()
    {
      return ar.length;
    }

  /**
   * Return the object at a given position in the CyclicArray
   * @param objectPos The position of the object to return (0 is the most
   *                  recently added
   * @return The object at that position (or null if no object)
   **/
  public Object get(int objectPos)
    {
      if (objectPos > length()-1)
	return null;
      else
	return ar[getPos(objectPos)];
    }

  /**
   * Return the position of the object in the internal array
   * @return Position of object
   **/
  protected int getPos(int objectPos)
    {
      return (ptr+objectPos+1) % length();
    }

  /**
   * Adds an object to the top of the cyclic array.  This may result in an
   * old object being deleted from the array.
   * @param newObj The object to add to the CyclicArray
   * @return Any object deleted from the array as a result of this call (or
   *         null if no object deleted)
   **/
  public Object put(Object newObj)
    {
      Object del = ar[ptr];
      ar[ptr] = newObj;
      ptr--;
      if (ptr == -1)
	ptr = ar.length-1;
      return del;
    }

  /**
   * Removes an object from the specified position in the CyclicArray
   * @param objectPos The position of the object to remove (0 is most
   *                  recently added)
   **/
  public void remove(int objectPos)
    {
      int apos = getPos(objectPos);
      if (apos < ptr)
	{
	  System.arraycopy(ar, apos+1, ar, apos, ptr-apos);
	}
      else if (apos > ptr)
	{
	  System.arraycopy(ar, apos+1, ar, apos, (ar.length-apos)-1);
	  ar[ar.length-1] = ar[0];
	  System.arraycopy(ar, 1, ar, 0, ptr);
	}
      ar[ptr] = null;
    }
}


