
/***************
**
**  sortpwd.c
**  last revised: april 14, 1992
**
**  Sortpwd sorts a password-file on a selected password-field
**  in the selected order.
**
**  Written for DESPERATE password-cracker with HADES encryption engine by
**  Remote.
**
**  Copyright (C)1992, Zabkar
**
**  root@waves.hacktic.nl (Zabkar)
**  root@room101.hacktic.nl (Remote)
**
*/

#include <stdio.h>
#include <string.h>
#include <alloc.h>
#include "pwd.h"


#define NAME       0
#define PASSWORD   1
#define UID        2
#define GID        3
#define GECOS      4
#define HOMEDIR    5
#define SHELL      6


FILE *infile;
FILE *outfile;
extern FILE *_pw_file;


struct entries
{
    struct passwd password;
    struct entries *next;
};



/***************
 haltusage()
 prints correct usage and exits
****************/

void haltusage()
{
  fprintf(stderr,
  "Sortpwd version 1.00 alpha, Copyright (C)1992 Zabkar\n"\
  "DESPERATE password-cracker 1.0 alpha using HADES engine by Remote.\n\n"\
  "Usage: sortpwd [-u|-p|-i|-g|-n|-h|-s] [-r] [-f passwordfile] "\
  "[-o outfile]\n\n"\
  "\t-u: sort by username\n"\
  "\t-p: sort by password (salt)\n"\
  "\t-i: sort by user-id\n"\
  "\t-g: sort by group-id\n"\
  "\t-n: sort by real name (gecos-field)\n"\
  "\t-h: sort by home-directory\n"\
  "\t-s: sort by shell\n"\
  "\t-r: reverse sorting order (default is ascending)\n"\
  "\t-f: read from 'passwordfile' instead of stdin\n"\
  "\t-o: write to 'outfile' instead of stdout\n\n"\
  "default field to be sorted on is user-id\n"\
  "no -f specified: dictfile read from stdin\n"\
  "no -o specified: output written to stdout\n");
  exit(0);
}


/***************
 freebuf()
 freebuf erases all memory occupied by linked-list begin.
***************/

void free_buf(struct entries *begin)
{
    struct entries *p;

    if (begin)
    {
      p = begin;
      while (p != NULL)
      {
        p = p->next;
        free(begin);
        begin = p;
      }
    }
}



/***************
 haltmemory()
 displays out-of-memory message and stops execution.
***************/

void haltmemory(struct entries *begin)
{
    free_buf(begin);

    fprintf(stderr, "Not enaugh memory to sort entire file\n");
    exit(0);
}
    


/***************
 getfield()
 returns the field that we are going to sort on in dest. numeric fields
 are presentated right-aligned in a string[6], so that they can be
 strcmp'd too.
***************/

char *get_field(char *dest, struct passwd *p, int field)
{
    switch(field)
    {
        case NAME     : strcpy(dest,p->pw_name); break;
        case PASSWORD : strcpy(dest,p->pw_passwd); break;
        case UID      : sprintf(dest, "%6d", p->pw_uid); break;
        case GID      : sprintf(dest, "%6d", p->pw_gid); break;
        case GECOS    : strcpy(dest,p->pw_gecos); break;
        case HOMEDIR  : strcpy(dest,p->pw_dir); break;
        case SHELL    : strcpy(dest,p->pw_shell); break;
    }
    return (dest);
}



/***************
 sort()
 sorts passwordfile `infile' by field `field' to linked-list begin.
 if `asc'==0 then passwordfile is sorted in descending order, otherwise
 in ascending order.
***************/

void *sort(FILE *infile, struct entries **begin, char field, char asc)
{
    struct entries *p;
    struct entries *q;
    struct entries *r;
    struct passwd *temp;
    char dest1[256], dest2[256];

    _pw_file = infile;

    if (!(*begin = malloc(sizeof(struct entries))))
       haltmemory(*begin);

    memset(&((*begin)->password), 0, sizeof(struct passwd));
    (*begin)->next = NULL;

    while ((temp = getpwent()) != NULL)
    {
        if (!(q = malloc(sizeof(struct entries))))
           haltmemory(*begin);

        p = *begin;

        get_field(dest1, temp, field);

        if (asc)
          while ((p->next != NULL) &&
            (strcmp(get_field(dest2, &(p->next->password), field), dest1) < 0))
             p = p->next;
        else
          while ((p->next != NULL) &&
            (strcmp(get_field(dest2, &(p->next->password), field), dest1) > 0))
             p = p->next;

        r = p->next;
        p->next = q;

        /* We have to copy the entire structure since getpwent uses a
           static structure that is overwritten each call */
        memcpy(&(q->password), temp, sizeof(struct passwd));

        q->next = r;
    }
}



/***************
 to_file()
 writes contents of linked-list begin to file of.
***************/

void to_file(FILE *of, struct entries *begin)
{
    struct entries *p;

    p = begin->next;
    while (p != NULL)
    {
        fprintf(of, "%s:%s:%d:%d:%s:%s:%s\n",
        p->password.pw_name, p->password.pw_passwd, p->password.pw_uid,
        p->password.pw_gid, p->password.pw_gecos, p->password.pw_dir,
        p->password.pw_shell);
        p = p->next;
    }
}



/***************
 do_sortpwd()
 sorts passwordfile `inf' to file `of' by field. If asc==0 sorting is
 done descending, otherwise ascending.
***************/

void do_sortpwd(FILE *inf, FILE *of, char field, char asc)
{
  struct entries *begin;

  sort(inf, &begin, field, asc);
  to_file(of, begin);
  free_buf(begin);
}


/***************
 main()
 main function of password-sorter.
***************/

main (char argc, char **argv)
{
  char fname[80], oname[80];
  char ascending=1;
  char field=UID;
  char switch_defined=0;
  int i;

  strcpy(fname, "");
  strcpy(oname, "");

  if (argc > 1)
  {
    for (i=1; i<argc; i++)
    {
      switch(argv[i][0])
      {
      case '-': switch(toupper(argv[i][1]))
		  {
            case 'F' : if (strlen(argv[i]) > 2)
                          strcpy(fname, &argv[i][2]);
                        else if (argc < (i+2))
                          haltusage();
                        else
                          strcpy(fname, argv[++i]);
                        break;
            case 'O' : if (strlen(argv[i]) > 2)
                          strcpy(oname, &argv[i][2]);
                        else if (argc < (i+2))
                           haltusage();
                        else
                           strcpy(oname, argv[++i]);
                        break;
            case 'U' : if (switch_defined) haltusage();
                       field=NAME;
                       break;
            case 'P' : if (switch_defined) haltusage();
                       field=PASSWORD;
                       break;
            case 'I' : if (switch_defined) haltusage();
                       field=UID;
                       break;
            case 'G' : if (switch_defined) haltusage();
                       field=GID;
                       break;
            case 'N' : if (switch_defined) haltusage();
                       field=GECOS;
                       break;
            case 'H' : if (switch_defined) haltusage();
                       field=HOMEDIR;
                       break;
            case 'S' : if (switch_defined) haltusage();
                       field=SHELL;
                       break;
            case 'R' : ascending = 0; break;
            default  : haltusage();
		  }
          break;
      default : haltusage();
      }
    }
  }

  if (strcmp(fname,""))
    infile = fopen(fname, "rt");
  else
    infile = stdin;

  if (!infile)
    {
    fprintf(stderr, "sortpwd: %s: couldn't open file\n", fname);
    exit(0);
    }

  if (strcmp(oname, ""))
    outfile = fopen(oname, "wt");
  else
    outfile = stdout;

  if (!outfile)
    {
    fprintf(stderr, "sortpwd: %s: could't create file\n", oname);
    exit(0);
    }

  do_sortpwd(infile, outfile, field, ascending);

  fclose(infile);
  fclose(outfile);

  }

/* EOF SORTPWD.C */
 
