#include <conio.h>
#include <dos.h>
#include <graph.h>
#include <afxcoll.h>
#include "colors.h"
#include "osw.h"
#include "designat.h"
#include "freqasg.h"
#include "trunksys.h"

Scanner::Scanner () : lp (0) 
{
	myPort = 0;
	volume = 130;	// reasonable mid point
	squelch = 80;	// another reasonable level
}

void
Scanner::setComPort (ComPort *comport, double baudrate)
{
	myPort = comport;
	myPort->setBaud (baudrate);
}

void
Scanner::topIs (frequencyAssignment *fp)
{
	if (fp != lp) 
	{
		if (lp) 
			QuickOut (lp->y, lp->x + PRTYSTART - 1, LIGHTGREEN, " ");

		if (fp && (fp->scancmd [0] || fp->scancmdlen))
		{
			QuickOut (fp->y, fp->x + PRTYSTART - 1, LIGHTGREEN, "*");

			if (fp->scancmdlen)
				putStringN (fp->scancmdlen, fp->scancmd);
			else
				putString (fp->scancmd);
		}

		lp = fp;
	}
}

PCR1000::PCR1000 (ComPort *comport, double baudrate) : Scanner ()
{
	Scanner::setComPort (comport, baudrate);

	comport->setBaud (9600.0);
	putString ("H101\n");	// set power on

	if (baudrate == 300.0)
		putString ("G100\n");
	else if (baudrate == 1200.0)
		putString ("G101\n");
	else if (baudrate == 4800.0)
		putString ("G102\n");
	else if (baudrate == 9600.0)
		putString ("G103\n");
	else if (baudrate == 19200.0)
		putString ("G104\n");
	else if (baudrate == 38400.0)
		putString ("G105\n");
	else
	{
		baudrate = 38400.0;
		putString ("G105\n");
	}

	comport->setBaud (baudrate);
	putString ("G300\n");	// set return data off
	putString ("J4501\n");	// set AGC on
	putString ("J4600\n");	// set NB off
	putString ("J4700\n");	// set attenuator off
	putString ("J5100\n");	// set tone squelch off

	setVolume (volume);
	setSquelch (squelch);
}

void
PCR1000::shutDown (void)
{
	putString ("H100\n");	// turn power off
}

void
PCR1000::setFreqString (frequencyAssignment *f) 
{
	if (strstr (f->freqdisp, ".")) 
	{
		f->scancmd [0]	= 'K';				// K
		f->scancmd [1]	= '0';				// 0
		f->scancmd [2]	= '0';				// 0
		f->scancmd [3]	= f->freqdisp [3];	// 8
		f->scancmd [4]	= f->freqdisp [4];	// x
		f->scancmd [5]	= f->freqdisp [5];	// x
		f->scancmd [6]	= f->freqdisp [7];	// f
		f->scancmd [7]	= f->freqdisp [8];	// f 
		f->scancmd [8]	= f->freqdisp [9];	// f 
		f->scancmd [9]	= f->freqdisp [10]; // f
		f->scancmd [10] = '0';				// 0
		f->scancmd [11] = '0';				// 0
		f->scancmd [12] = '0';				// 0
		f->scancmd [13] = '5';				// 5
		f->scancmd [14] = '0';				// 0
		f->scancmd [15] = '2';				// 2
		f->scancmd [16] = '0';				// 0
		f->scancmd [17] = '0';				// 0
		f->scancmd [18] = '\n';				// 0
		f->scancmd [19] = '\0';
	} 
	else 
	{
		f->scancmd [0] = 0; // unknown mapping
	}
}

void
PCR1000::setVolume (int volume)
{
	char volumeCmd [32];

	sprintf (volumeCmd, "J40%02x\n", volume & 0xff);
	putString (volumeCmd);
}

void
PCR1000::setSquelch (int squelch)
{
	char squelchCmd [32];

	sprintf (squelchCmd, "J41%02x\n", squelch & 0xff);
	putString (squelchCmd);
}

R10::R10 (ComPort *comport, double baudrate) : Scanner ()
{
	char Init [] = {0xfe, 0xfe, 0x52, 0xe0, 0x06, 0x05, 0x02, 0xfd};

	Scanner::setComPort (comport, baudrate);

	putStringN (8, Init);
}

void
R10::setFreqString (frequencyAssignment *f) 
{
	if (strstr (f->freqdisp, ".")) 
	{
		f->scancmd [0]	= 0xfe;						// Preamble
		f->scancmd [1]	= 0xfe;						// Preamble
		f->scancmd [2]	= 0x52;						// Radio type (R10)
		f->scancmd [3]	= 0xe0;						// Transmit address (controller)
		f->scancmd [4]	= 0x05;						// Write operating frequency
		f->scancmd [5]	= 0x00;						// 10Hz/1Hz
		f->scancmd [6]	= ((f->freqdisp [9] & 0x0f) << 4) | (f->freqdisp [10] & 0x0f); // 1Khz/100Hz
		f->scancmd [7]	= ((f->freqdisp [7] & 0x0f) << 4) | (f->freqdisp [8] & 0x0f);  // 100Khz / 10Khz
		f->scancmd [8]	= ((f->freqdisp [4] & 0x0f) << 4) | (f->freqdisp [5] & 0x0f);  // 10Mhz/1Mhz
		f->scancmd [9]	= f->freqdisp [3] & 0x0f;	// 100's Mhz
		f->scancmd [10] = 0xfd;						// Terminator

		//
		//  Setting this to a non-zero means treated it as a counted string
		//
		f->scancmdlen = 11;
	} 
	else
	{
		f->scancmd [0] = 0; // unknown mapping
	}
}

R8500::R8500 (ComPort *comport, double baudrate) : Scanner ()
{
	char Init [] = {0xfe, 0xfe, 0x4a, 0xe0, 0x06, 0x05, 0x02, 0xfd};

	Scanner::setComPort (comport, baudrate);

	putStringN (8, Init);
}

void
R8500::setFreqString (frequencyAssignment *f) 
{
	if (strstr (f->freqdisp, ".")) 
	{
		f->scancmd [0]	= 0xfe;						// Preamble
		f->scancmd [1]	= 0xfe;						// Preamble
		f->scancmd [2]	= 0x4a;						// Radio type (R8500)
		f->scancmd [3]	= 0xe0;						// Transmit address (controller)
		f->scancmd [4]	= 0x05;						// Write operating frequency
		f->scancmd [5]	= 0x00;						// 10Hz/1Hz
		f->scancmd [6]	= ((f->freqdisp [9] & 0x0f) << 4) | (f->freqdisp [10] & 0x0f); // 1Khz/100Hz
		f->scancmd [7]	= ((f->freqdisp [7] & 0x0f) << 4) | (f->freqdisp [8] & 0x0f);  // 100Khz / 10Khz
		f->scancmd [8]	= ((f->freqdisp [4] & 0x0f) << 4) | (f->freqdisp [5] & 0x0f);  // 10Mhz/1Mhz
		f->scancmd [9]	= f->freqdisp [3] & 0x0f;	// 100's Mhz
		f->scancmd [10] = 0xfd;						// Terminator

		//
		//  Setting this to a non-zero means treated it as a counted string
		//
		f->scancmdlen = 11;
	} 
	else
	{
		f->scancmd [0] = 0; // unknown mapping
	}
}

R7100::R7100 (ComPort *comport, double baudrate) : Scanner ()
{
	char Init [] = {0xfe, 0xfe, 0x34, 0xe0, 0x06, 0x05, 0x02, 0xfd};

	Scanner::setComPort (comport, baudrate);

	putStringN (8, Init);
}

void
R7100::setFreqString (frequencyAssignment *f) 
{
	if (strstr (f->freqdisp, ".")) 
	{
		f->scancmd [0]	= 0xfe;						// Preamble
		f->scancmd [1]	= 0xfe;						// Preamble
		f->scancmd [2]	= 0x34;						// Radio type (R7100)
		f->scancmd [3]	= 0xe0;						// Transmit address (controller)
		f->scancmd [4]	= 0x05;						// Write operating frequency
		f->scancmd [5]	= 0x00;						// 10Hz/1Hz
		f->scancmd [6]	= ((f->freqdisp [9] & 0x0f) << 4) | (f->freqdisp [10] & 0x0f); // 1Khz/100Hz
		f->scancmd [7]	= ((f->freqdisp [7] & 0x0f) << 4) | (f->freqdisp [8] & 0x0f);  // 100Khz / 10Khz
		f->scancmd [8]	= ((f->freqdisp [4] & 0x0f) << 4) | (f->freqdisp [5] & 0x0f);  // 10Mhz/1Mhz
		f->scancmd [9]	= f->freqdisp [3] & 0x0f;	// 100's Mhz
		f->scancmd [10] = 0xfd;						// Terminator

		//
		//  Setting this to a non-zero means treated it as a counted string
		//
		f->scancmdlen = 11;
	} 
	else
	{
		f->scancmd [0] = 0; // unknown mapping
	}
}

R7000::R7000 (ComPort *comport, double baudrate) : Scanner ()
{
	char Init [] = {0xfe, 0xfe, 0x08, 0xe0, 0x06, 0x05, 0x02, 0xfd};

	Scanner::setComPort (comport, baudrate);

	putStringN (8, Init);
}

void
R7000::setFreqString (frequencyAssignment *f) 
{
	if (strstr (f->freqdisp, ".")) 
	{
		f->scancmd [0]	= 0xfe;						// Preamble
		f->scancmd [1]	= 0xfe;						// Preamble
		f->scancmd [2]	= 0x08;						// Radio type (R7000)
		f->scancmd [3]	= 0xe0;						// Transmit address (controller)
		f->scancmd [4]	= 0x05;						// Write operating frequency
		f->scancmd [5]	= 0x00;						// 10Hz/1Hz
		f->scancmd [6]	= ((f->freqdisp [9] & 0x0f) << 4) | (f->freqdisp [10] & 0x0f); // 1Khz/100Hz
		f->scancmd [7]	= ((f->freqdisp [7] & 0x0f) << 4) | (f->freqdisp [8] & 0x0f);  // 100Khz / 10Khz
		f->scancmd [8]	= ((f->freqdisp [4] & 0x0f) << 4) | (f->freqdisp [5] & 0x0f);  // 10Mhz/1Mhz
		f->scancmd [9]	= f->freqdisp [3] & 0x0f;	// 100's Mhz
		f->scancmd [10] = 0xfd;						// Terminator

		//
		//  Setting this to a non-zero means treated it as a counted string
		//
		f->scancmdlen = 11;
	} 
	else
	{
		f->scancmd [0] = 0; // unknown mapping
	}
}

AR8000::AR8000 (ComPort *comport, double baudrate) : Scanner () 
{
	Scanner::setComPort (comport, baudrate);

	putString ("DD\r");			// establish vfo operation
	putString ("MD1\r");		// establish NFM mode
	putString ("AT0\r");		// attenuator off
	putString ("ST012500\r");	// step 12.5 no offset
}

void
AR8000::setFreqString (frequencyAssignment *f) 
{
	if (strstr (f->freqdisp, ".")) 
	{
		f->scancmd [0]	= 'R';				// R
		f->scancmd [1]	= 'F';				// F
		f->scancmd [2]	= '0';				// 0
		f->scancmd [3]	= f->freqdisp [3];	// 8
		f->scancmd [4]	= f->freqdisp [4];	// x
		f->scancmd [5]	= f->freqdisp [5];	// x
		f->scancmd [6]	= f->freqdisp [7];	// f
		f->scancmd [7]	= f->freqdisp [8];	// f 
		f->scancmd [8]	= f->freqdisp [9];	// f 
		f->scancmd [9]	= f->freqdisp [10]; // f
		f->scancmd [10] = '0';				// 0
		f->scancmd [11] = '0';				// 0
		f->scancmd [12] = '\r';
		f->scancmd [13] = '\0';
	} 
	else 
	{
		f->scancmd [0] = 0; // unknown mapping
	}
}

AR3000::AR3000 (ComPort *comport, double baudrate) : Scanner ()
{
	Scanner::setComPort (comport, baudrate);

	putString ("12.5S");		// establish vfo operation
}

void
AR3000::setFreqString (frequencyAssignment *f) 
{
	if (strstr (f->freqdisp, "."))
	{
		sprintf (f->scancmd, "%sN", &f->freqdisp [3]);
	}
	else 
	{
		f->scancmd [0] = 0; // unknown mapping
	}
}

AR3000A::AR3000A (ComPort *comport, double baudrate) : Scanner ()
{
	Scanner::setComPort (comport, baudrate);

	putString ("12.5S");		// establish vfo operation
}

void
AR3000A::setFreqString (frequencyAssignment *f) 
{
	if (strstr (f->freqdisp,".")) 
	{
		sprintf (f->scancmd, "%sN\r", &f->freqdisp [3]);
	} 
	else 
	{
		f->scancmd [0] = 0; // unknown mapping
	}

}

AR2700::AR2700 (ComPort *comport, double baudrate) : Scanner ()
{
	Scanner::setComPort (comport, baudrate);

	putString ("AU0\r"); // establish vfo operation
	putString ("MD1\r"); // establish NFM mode
	putString ("AT0\r"); // attenuator off
}

void
AR2700::setFreqString (frequencyAssignment *f)
{
	if (strstr (f->freqdisp, ".")) 
	{
		f->scancmd [0]	= 'R';				// R
		f->scancmd [1]	= 'F';				// F
		f->scancmd [2]	= '0';				// 0
		f->scancmd [3]	= f->freqdisp [3];	// 8
		f->scancmd [4]	= f->freqdisp [4];	// x
		f->scancmd [5]	= f->freqdisp [5];	// x
		f->scancmd [6]	= f->freqdisp [6];	// .
		f->scancmd [7]	= f->freqdisp [7];	// f
		f->scancmd [8]	= f->freqdisp [8];	// f 
		f->scancmd [9]	= f->freqdisp [9];	// f 
		f->scancmd [10] = f->freqdisp [10]; // f
		f->scancmd [11] = '0';				// 0
		f->scancmd [12] = '\r';
		f->scancmd [13] = '\0';
	} 
	else 
	{
		f->scancmd [0] = 0; // unknown mapping
	}
}

Kenwood::Kenwood (ComPort *comport, double baudrate) : Scanner ()
{
	Scanner::setComPort (comport, baudrate);

	putString ("MD4;"); // establish NFM mode
}

void
Kenwood::setFreqString (frequencyAssignment *f)
{
	if (strstr (f->freqdisp, ".")) 
	{
		f->scancmd [0]	= 'F';				// F
		f->scancmd [1]	= 'A';				// A
		f->scancmd [2]	= '0';				// 0
		f->scancmd [3]	= '0';				// 0
		f->scancmd [4]	= f->freqdisp [3];	// 8
		f->scancmd [5]	= f->freqdisp [4];	// x
		f->scancmd [6]	= f->freqdisp [5];	// x
		f->scancmd [7]	= f->freqdisp [7];	// f
		f->scancmd [8]	= f->freqdisp [8];	// f 
		f->scancmd [9]	= f->freqdisp [9];	// f 
		f->scancmd [10] = f->freqdisp [10]; // f
		f->scancmd [11] = '0';				// 0
		f->scancmd [12] = '0';				// 0
		f->scancmd [13] = ';';				// ;
		f->scancmd [14] = '\0';
	} 
	else 
	{
		f->scancmd [0] = 0; // unknown mapping
	}
}

BC895::BC895(ComPort *comport, double baudrate) : Scanner ()
{
	Scanner::setComPort (comport, baudrate);

	putString ("KEY01\r"); // stop scanning
}

void
BC895::setFreqString (frequencyAssignment *f)
{
	if (strstr (f->freqdisp, ".")) 
	{
		f->scancmd [0]	= 'R';				// R
		f->scancmd [1]	= 'F';				// F
		f->scancmd [2]	= '0';				// 0
		f->scancmd [3]	= f->freqdisp [3];	// 8
		f->scancmd [4]	= f->freqdisp [4];	// x
		f->scancmd [5]	= f->freqdisp [5];	// x
		f->scancmd [6]	= f->freqdisp [7];	// f
		f->scancmd [7]	= f->freqdisp [8];	// f 
		f->scancmd [8]	= f->freqdisp [9];	// f 
		f->scancmd [9]	= f->freqdisp [10]; // f
		f->scancmd [10] = '\r';
		f->scancmd [11] = '\0';
	} 
	else 
	{
		f->scancmd [0] = 0; // unknown mapping
	}
}

NOSCANNER::NOSCANNER ()
{
}

void 
NOSCANNER::setFreqString (frequencyAssignment *f) 
{
	f->scancmd [0] = 0; // unknown mapping
}
