/*
 *  $Id$
 *
 *  Linux pingd sourcefile
 *  ping.c - main sourcefile
 *  Copyright (c) 1998 by daemon9|route <route@infonexus.com>
 *  
 *
 *  $Log$
 */

#include "pingd.h"

int
main(int argc, char **argv)
{
    int sock_fd, c;
    u_short csum;
    struct icmp_packet i_pack;
    struct passwd *pwd_p;

    /*
     *  Make sure we have UID 0.
     */
    if (geteuid() || getuid())
    {
        fprintf(stderr, "Inadequate privledges\n");
        exit(1);
    }

    /*
     *  Open a raw ICMP socket and set IP_HDRINCL.
     */
    if ((sock_fd = open_raw_sock(IPPROTO_ICMP)) == -1)
    {
        perror("socket allocation");
        exit(1);
    }

    /*
     *  Now that we have the raw socket, we no longer need root privledges
     *  so we drop our UID to nobody.
     */
    if (!(pwd_p = getpwnam(NOBODY))) 
    {
        fprintf(stderr, "Can't get pwnam info on nobody");
        exit(1);
    }
    else if (setuid(pwd_p->pw_uid) == -1)
    {
        perror("Can't drop privledges");
        exit(1);
    }

    while((c = getopt(argc, argv, "dm:uv
#ifdef  SKEW_RTT
                                          s:
#endif  /* SKEW_RTT */
                                            ")) != EOF)
    {
        switch (c)
        {
            case 'd':
                d = 1;
                break;

            case 'm':
                max_packet = atoi(optarg);
                if (max_packet > MAX_PACKET || max_packet < MIN_PACKET)
                {
                    fprintf(stderr,
                        "Nono, maximum packet size must be between %d and %d\n",
                        MIN_PACKET, MAX_PACKET);
                    exit(0);
                }

            case 'u':
                u = 1;
                break;

            case 'v':
                version_info(argv[0]);
                exit(0);

#ifdef  SKEW_RTT
            case 's':
                skew_vector = atoi(optarg);
                break;
#endif  /* SKEW_RTT */

            default:
                usage(argv[0]);
        }
    }

    if (!d) daemon();
    if (d) fprintf(stderr, "Max packetsize of %d bytes\n", 
            max_packet - sizeof(struct ip));

#ifdef  LOG
    openlog("pingd", 0, 0);
    syslog(LOG_DAEMON|LOG_INFO, "started: %d", getpid());
#endif  /* LOG */
    /*
     *  We're powered up.  From here on out, everything should run swimmingly.
     */
    for (;;)
    {
        bzero(&i_pack, sizeof(i_pack));
        c = recv(sock_fd, (struct icmp_packet *)&i_pack, sizeof(i_pack), 0);
        if (c == -1)
        {
            if (d) fprintf(stderr, "truncated read: %s", strerror(errno));
            continue;
        }

        /*
         *  Make sure we don't spin into an infinite loop grabbing our own
         *  packets.
         */
        if (is_local(&i_pack)) continue;

        /*
         *  If the packet is too big or too small, we don't even care about 
         *  it's contents.  If we are so confirgured, send an
         *  ICMP_HOST_UNREACH.
         */
        if (c < HEADER_MATERIAL || c > max_packet)
        {
#ifdef  LOG
            syslog(
                    LOG_DAEMON|LOG_INFO,
                    "bad packet size (%d bytes from %s)",
                    ntohs(i_pack.iph.ip_len) - sizeof(i_pack.iph),
                    host_lookup(i_pack.iph.ip_src.s_addr));
#endif  /* LOG */
            if (d) fprintf(stderr,
                    "bad packet size (%d bytes from %s)\n",
                    ntohs(i_pack.iph.ip_len) - sizeof(i_pack.iph),
                    host_lookup(i_pack.iph.ip_src.s_addr));
            if (u) icmp_reflect(ICMP_DEST_UNREACH, &i_pack, sock_fd);
            continue;
        }

        /*
         *  We only want ICMP_ECHO packets.
         */
        if (i_pack.icmph.type != ICMP_ECHO) continue;
        else if (d)
                fprintf(stderr,
                "%d byte ICMP_ECHO from %s\n",
                ntohs(i_pack.iph.ip_len) - sizeof(i_pack.iph),
                host_lookup(i_pack.iph.ip_src.s_addr));

#ifdef  VERIFY_CHECKSUM
        /*
         *  Verify the ICMP header checksum.
         */
        csum = i_pack.icmph.checksum;
        i_pack.icmph.checksum = 0;
        if (ip_check((u_short *)&i_pack.icmph, ntohs(i_pack.iph.ip_len) -
            sizeof(struct ip)) != csum)
        {
#ifdef  LOG
            syslog(
                    LOG_DAEMON|LOG_INFO,
                    "bad packet checksum (%d bytes from %s)",
                    ntohs(i_pack.iph.ip_len) - sizeof(i_pack.iph),
                    host_lookup(i_pack.iph.ip_src.s_addr));
#endif  /* LOG */
            if (d) fprintf(stderr,
                    "bad packet size (%d bytes from %s)\n",
                    ntohs(i_pack.iph.ip_len) - sizeof(i_pack.iph),
                    host_lookup(i_pack.iph.ip_src.s_addr));
             continue;
        }
#endif  /* VERIFY_CHECKSUM */

        /*
         *  Pass packet to the access control mechanism.
         */
#ifdef  WRAP
        if (!verify(&i_pack))
        {
#ifdef  LOG
            syslog(
                    LOG_DAEMON|LOG_INFO,
                    "ICMP_ECHO denied by wrapper (%d bytes from %s)",
                    ntohs(i_pack.iph.ip_len) - sizeof(i_pack.iph),
                    host_lookup(i_pack.iph.ip_src.s_addr));
#endif  /* LOG */
        }
        else 
        {
#ifdef  LOG
            syslog(
                    LOG_DAEMON|LOG_INFO,
                    "ICMP_ECHO allowed by wrapper (%d bytes from %s)",
                    ntohs(i_pack.iph.ip_len) - sizeof(i_pack.iph),
                    host_lookup(i_pack.iph.ip_src.s_addr));
#endif  /* LOG */
#endif  /* WRAP*/
            icmp_reflect(ICMP_ECHOREPLY, &i_pack, sock_fd);
#ifdef  WRAP
        }
#endif  /* WRAP */
    }
}


void
icmp_reflect(int type, struct icmp_packet *p_ptr, int sock_fd)
{
    int c;
    u_long tmp;
    struct sockaddr_in sin;

    bzero((struct sockaddr_in *)&sin, sizeof(sin));

    tmp = p_ptr->iph.ip_dst.s_addr;

    if (type == ICMP_DEST_UNREACH)
    {
        p_ptr->icmph.code = ICMP_HOST_UNREACH;
        /*
         *  ICMP_UNREACH wants the IP header and 8 bytes of data from the
         *  offending packet (in this case it's the ICMP_ECHO header).
         */
        bcopy(&p_ptr->iph, p_ptr->payload, UNREACH_AMT);
        p_ptr->iph.ip_len =
                        htons(sizeof(struct ip) + sizeof(struct icmphdr) +
                        UNREACH_AMT);
        p_ptr->icmph.un.gateway = 0;
    }
#ifdef  SKEW_RTT
    else
    {
        skew_rtt(p_ptr);
    }
#endif  /* SKEW_RTT */
    p_ptr->iph.ip_dst.s_addr = p_ptr->iph.ip_src.s_addr;
    p_ptr->iph.ip_src.s_addr = tmp;
    p_ptr->icmph.type        = type;

    p_ptr->icmph.checksum    = 0;
    p_ptr->icmph.checksum    =
                        ip_check((u_short *)&p_ptr->icmph,
                        ntohs(p_ptr->iph.ip_len) - sizeof(struct ip));
    sin.sin_family      = AF_INET;
    sin.sin_addr.s_addr = p_ptr->iph.ip_dst.s_addr;

    c = sendto(sock_fd,
            (struct icmp_packet *)p_ptr,
            ntohs(p_ptr->iph.ip_len),
            0,
            (struct sockaddr *) &sin, sizeof(sin));

    if (c != ntohs(p_ptr->iph.ip_len))
    {
        if (d) perror("truncated write");
        return;
    }
    else if (d)
    {
        if (type == ICMP_DEST_UNREACH)
        {
            fprintf(stderr, "ICMP_DEST_UNREACH sent\n");
        }
        else
        {
            fprintf(stderr, "ICMP_ECHOREPLY sent\n");
        }
    }
}


#ifdef  WRAP
int
verify(struct icmp_packet *p_ptr)
{
    if (!hosts_ctl("ping", 
                    host_lookup(p_ptr->iph.ip_src.s_addr),
                    host_lookup(p_ptr->iph.ip_src.s_addr),
                    STRING_UNKNOWN))
        return (0);

    else return (1);
}
#endif  /* WRAP */


void
usage(char *argv0)
{
    fprintf(stderr,
            "usage: %s [-duv] [-m maxpacketsize]"
#ifdef  SKEW_RTT
            " [-s skewvector]"
#endif  /* SKEW_RTT */
            "\n", argv0);
    exit(0);
}


void
version_info(char *argv0)
{
    int i;

    fprintf(stderr,
            "%s version %s built on %s %s with the following options:\n",
            argv0, VERSION, __DATE__, __TIME__);

    /*
     *  Conditional options.
     */
    for (i = 0; options[i]; i++)
    {
        fprintf(stderr, "%s", options[i]);
    }
}

#ifdef  SKEW_RTT
void
skew_rtt(struct icmp_packet *p_ptr)
{
    struct timeval *t_ptr;
    t_ptr = (struct timeval *)p_ptr->payload;
    t_ptr->tv_usec += (-skew_vector);
}
#endif  /* SKEW_RTT */

int
is_local(struct icmp_packet *p_ptr)
{
    if (p_ptr->iph.ip_src.s_addr == p_ptr->iph.ip_dst.s_addr)
    {
        if (p_ptr->icmph.type != ICMP_ECHO) return (1);
    }
    return (0);
}


/* EOF */
