; A low-level monitor for the 8051 family for the A51 assembler
; Copyright 1989 by Steve Kemplin
; This program is provided for personal, private use only.
; Commercial use in whole or in part without express written permission of
; the author is prohibited.
; No warranty is made as to the suitability of the program for any
; particular use.
;
; Based on a monitor written and copyrighted by Ron Stubbers
; [Ver 900626]

        .ORG    0x40FC          ;SET UP LABEL ADDRESS FOR PATCH
        .EQU    PATCH,*         ;MUST BE RAM, BOTH PROG & DATA SPACE

        .EQU    CR,0x0D         ;CHARACTER EQUATES
        .EQU    LF,0x0A
        .EQU    ESC,0x1B

        .EQU    VECTAB,0x2000         ;ADDRESS OF BASIC VECTORS
        .EQU    RSTVEC,VECTAB         ;JUMPS HERE AT POWER UP
        .EQU    SERINVEC,VECTAB+3     ;JUMPS HERE TO READ A CHARACTER
        .EQU    SEROUTVEC,VECTAB+6    ;JUMPS HERE TO SEND A CHARACTER
        .EQU    INT1VEC,VECTAB+0x10   ;JUMPS HERE ON INT1
        .EQU    SERVEC,VECTAB+0x18    ;JUMPS HERE ON SERIAL INTERRUPT
        .EQU    TMR0VEC,VECTAB+0x20   ;JUMPS HERE ON TIMER 0 INTERRUPT
        .EQU    TMR1VEC,VECTAB+0x28   ;JUMPS HERE ON TIMER 1 INTERRUPT
        .EQU    TMR2VEC,VECTAB+0x30   ;JUMPS HERE ON TIMER 2 INTERRUPT

        .EQU    RAMPAGE,0x3F    ;BASE ADDRESS OF MONITOR RAM AREA

        .EQU    BUFFER,0x80     ;BUFFER AREA FOR PROM LOAD

        .EQU    CNTLO,0x70      ;LOW BYTE OF COUNT
        .EQU    CNTHI,CNTLO+1   ;HIGH BYTE OF COUNT
        .EQU    ADHI,CNTLO+2    ;HIGH ADDRESS
        .EQU    ADLO,CNTLO+3    ;LOW ADDRESS
        .EQU    DATA,CNTLO+4    ;TEMPORARY BYTE STORAGE LOCATION
        .EQU    CKSUM,CNTLO+5   ;CHECKSUM FOR PROM LOAD
        .EQU    TEMP1,CNTLO+6   ;TEMPORARY STORAGE LOCATIONS
        .EQU    TEMP2,CNTLO+7
        .EQU    TEMP3,CNTLO+8

        .EQU    ASAVE,0x7A      ;REGISTER IMAGES
        .EQU    BSAVE,ASAVE+1
        .EQU    PSWSAVE,ASAVE+2
        .EQU    DPHSAVE,ASAVE+3
        .EQU    DPLSAVE,ASAVE+4

        .EQU    BRKPT1,0xD0
        .EQU    BRKPT2,BRKPT1+5
        .EQU    BRKPT3,BRKPT1+10
        .EQU    BRKPT4,BRKPT1+15

        .EQU    BKFLAG,0x7F

        .EQU    BK1MASK,0x01
        .EQU    BK2MASK,0x02
        .EQU    BK3MASK,0x04
        .EQU    BK4MASK,0x08
        .EQU    ALLMASK,BK1MASK|BK2MASK|BK3MASK|BK4MASK

        .EQU    BPHLR_HI,((BPHLR>>8)&0xFF)
        .EQU    BPHLR_LO,(BPHLR&0xFF)

        .eject

        .ORG    0
        LJMP    INIT            ;GO TO INIT AT POWER UP

        .ORG    0x03
        .EQU    INT0,*          ;VECTOR FOR EXTERNAL INTERRUPT 0
        LJMP    MONENT

        .ORG    0x0B
        .EQU    TMR0,*          ;VECTOR FOR TIMER 0
        LJMP    TMR0VEC

        .ORG    0x13
        .EQU    INT1,*          ;VECTOR FOR EXTERNAL INTERRUPT 1
        LJMP    INT1VEC

        .ORG    0x1B
        .EQU    TMR1,*          ;VECTOR FOR TIMER 1
        LJMP    TMR1VEC

        .ORG    0x23
        .EQU    SERINT,*        ;VECTOR FOR SERIAL I/O INTERRUPT
        LJMP    SERVEC

        .ORG    0x2B
        .EQU    TMR2,*          ;VECTOR FOR TIMER 2
        .EQU    INT2,*          ;ALSO FOR EXTERNAL INTERRUPT 2
        LJMP    TMR2VEC

        .eject

;START OF MONITOR PROGRAM
;
        .ORG    0x30            ;START OF MONITOR CODE

ENTLST: LJMP    INIT            ;Entry point to the monitor
                                ;
        LJMP    DELAY           ;Pause for 1 msec times value in ACC
                                ;
        LJMP    SERIN           ;Serial Input Primitive
                                ;
        LJMP    GETC            ;Read char from serial port,
                                ;Echo char back to serial port
                                ;
        LJMP    PUTC            ;Send char in ACC to serial port
                                ;
        LJMP    PUTS            ;Sends text string to serial port,
                                ;DPTR points to start, null ends
                                ;
        LJMP    CRLF            ;Sends CR,LF to serial port
                                ;
        LJMP    ASC2HEX         ;Converts ASCII char in ACC to HEX equivalent,
                                ;Returns value in lower nibble, upper nibble zeroes
                                ;
        LJMP    HEX2ASC         ;Convert lower nibble in ACC to ASCII char,
                                ;returned in ACC
                                ;
        LJMP    RDHEX           ;Reads 2 char from serial port,
                                ;converts to byte returned in ACC
                                ;
        LJMP    WRBYTE          ;Writes byte in ACC to serial port,
                                ;as 2 ASCII char
                                ;

        .eject

        .ORG    0x0080
INIT:   MOV     SP,#0x30        ;SET THE STACK POINTER SOMEWHERE SAFE
        MOV     SCON,#0x52
        MOV     TMOD,#0x20
        MOV     PCON,#0X80      ;SET FOR DOUBLE BAUD RATE
        MOV     A, #0xCC        ;1200 BAUD, #0xF3=>1200 BAUD (12 MHz)
        MOV     TH1,A
        SETB    TR1             ;START TIMER FOR SERIAL PORT
        SETB    P3.2            ;THIS TURNS OFF INT0
        MOV     A,#10           ;SHORT DELAY TO MAKE SURE IT GOT DONE
        DJNZ    ACC,*
        ORL     IE,#0x81        ;ENABLE INT0
        ORL     IP,#0x01        ;AND MAKE IT HIGH PRIORITY
        MOV     BKFLAG,#0       ;CLEAR ALL BREAKPOINTS
        JB      P1.6,GETCMD     ;IF PIN P1.6 NOT GROUNDED,CONTINUE
        LJMP    RSTVEC          ;ELSE JUMP TO USER PROGRAM

SAVEREG: MOV    PSWSAVE,PSW     ;SAVE EVERYTHING
        MOV     ASAVE,ACC
        MOV     BSAVE,B
        MOV     DPHSAVE,DPH
        MOV     DPLSAVE,DPL
        ACALL   RSTALL          ;FIX UP THE CODE FROM BREAKPOINTS
        RET

CALLRET: ACALL  SAVEREG         ;SAVE THE REGISTERS,FIXUP THE CODE
        SJMP    GETCMD          ;AND GO GET A COMMAND

RESTORE: MOV    PSW,PSWSAVE     ;RESTORE EVERYTHING EXCEPT PC
        MOV     ACC,ASAVE
        MOV     B,BSAVE
        MOV     DPH,DPHSAVE
        MOV     DPL,DPLSAVE
        RET

MONENT: CLR     EX0             ;DISABLE THE INTERRUPT
        ACALL   SAVEREG         ;SAVE THE REGISTERS,FIXUP THE CODE
        POP     ADHI            ;GET THE PROGRAM COUNTER
        POP     ADLO
        PUSH    ADLO
        PUSH    ADHI
        MOV     A,ADHI          ;DISPLAY THE ADDRESS
        ACALL   WRBYTE
        MOV     A,ADLO
        ACALL   WRBYTE          ;FALL THROUGH INTO GETCMD

        .eject

GETCMD: ACALL   CRLF            ;START OFF ON A NEW LINE
        MOV     A,#'#'          ;PROMPT FOR A COMMAND
        ACALL   PUTC
        ACALL   GETC            ;GET THE COMMAND
I_TEST: CJNE    A,#'I',M_TEST   ;IF NOT I, THEN CHECK M
        ACALL   INTRAM          ;I, SO RUN INTERNAL
M_TEST: CJNE    A,#'M',L_TEST   ;IF NOT M, THEN CHECK L
        ACALL   MEMORY          ;M, SO RUN MEMORY MODIFY
L_TEST: CJNE    A,#'L',G_TEST   ;IF NOT L, THEN CHECK G
        ACALL   LOAD            ;L, SO LOAD HEX FILE
G_TEST: CJNE    A,#'G',S_TEST   ;IF NOT G, THEN CHECK S
        AJMP    GO              ;G, SO GO TO ADDRESS
S_TEST: CJNE    A,#'S',C_TEST   ;IF NOT S, THEN CHECK C
        AJMP    STEP            ;S, SO SINGLE STEP
C_TEST: CJNE    A,#'C',D_TEST   ;IF NOT C, THEN CHECK D
        AJMP    UCALL           ;C, SO CALL
D_TEST: CJNE    A,#'D',P_TEST   ;IF NOT D, THEN CHECK P
        ACALL   DUMP            ;D, SO DUMP RAM
P_TEST: CJNE    A,#'P',F_TEST   ;IF NOT P, THEN CHECK F
        ACALL   PRMLD
F_TEST: CJNE    A,#'F',V_TEST   ;IF NOT F, THEN CHECK V
        ACALL   FILL            ;F, SO FILL RAM
V_TEST: CJNE    A,#'V',B_TEST   ;IF NOT V THEN NO MATCH SO TRY AGAIN
        ACALL   MOVE
B_TEST: CJNE    A,#'B',R_TEST   ;IF NOT B THEN CHECK R
        ACALL   BPDISP          ;CALL BREAKPOINT DISPATCHER
R_TEST: CJNE    A,#'R',NOMATCH  ;IF NOT R THEN NO MATCH SO TRY AGAIN
        ACALL   REGSTR          ;CALL REGISTER HANDLING ROUTINE
NOMATCH: MOV    A,#0xFF         ;WAIT A WHILE
        ACALL   DELAY
        SJMP    GETCMD          ;NOT FOUND, SO BACK TO GETCMD

        .eject

INTRAM: PUSH    DPH             ;SAVE DPTR
        PUSH    DPL
        ACALL   SPACE           ;OUTPUT A SPACE
        ACALL   RDHEX           ;GET ADDRESS OF DIRECT BYTE TO READ
RAMENT: PUSH    ACC             ;SAVE IT
        MOV     DPTR,#PATCH     ;MOV INSTRUCTION WILL BE ASSEMBLED AT PATCH
        MOV     A,#0xE5         ;0xE5 IS A MOV A,DIRECT
        MOVX    @DPTR,A         ;PUT THE INSTRUCTION AT PATCH
        POP     ACC             ;RETRIEVE THE ADDRESS OF THE DIRECT BYTE
        INC     DPTR            ;STORE IT TO THE NEXT LOCATION
        MOVX    @DPTR,A
        MOV     A,#0x22         ;0x22 IS RETURN INSTRUCTION
        INC     DPTR            ;PUT RET INSTRUCTION IN PATCH
        MOVX    @DPTR,A
        LCALL   PATCH           ;EXECUTE PATCH
        ACALL   SPACE           ;OUTPUT A SPACE
        ACALL   WRBYTE          ;SHOW BYTE RETRIEVED BY PATCH
        ACALL   SPACE           ;OUTPUT A SPACE
INTRAM1: ACALL  GETC            ;GET THE NEXT CHARACTER
        CJNE    A,#' ',INTRAM2  ;ITS NOT A SPACE, SO KEEP CHECKING
        SJMP    INTRAM1         ;A SPACE DOESN'T MEAN ANYTHING, SO LOOP
INTRAM2: CJNE   A,#CR,NEWINT    ;ITS NOT A RETURN EITHER
        SJMP    INTRAMX         ;ITS A RETURN SO WE'RE DONE
NEWINT: ACALL   RDHX1           ;PRESUME ITS A NEW VALUE
        PUSH    ACC             ;SAVE THE NEW VALUE
        MOV     DPTR,#PATCH     ;MOV INSTRUCTION WILL BE ASSEMBLED AT PATCH
        MOV     A,#0xF5         ;0xF5 IS A MOV DIRECT,A
        MOVX    @DPTR,A         ;PUT THE INSTRUCTION AT PATCH
        POP     ACC             ;RECOVER THE NEW VALUE
        LCALL   PATCH           ;DATA IN A, INSTRs AT PATCH, SO EXECUTE
INTRAMX: ACALL  CRLF            ;WE'RE DONE
        POP     DPL             ;RESTORE DPTR
        POP     DPH
        RET                     ;BACK FOR NEXT COMMAND

        .eject

MEMORY: PUSH    ACC             ;SAVE THE ACCUMULATOR
        PUSH    DPH             ;AND DPTR
        PUSH    DPL
        ACALL   RDHEX           ;GET THE UPPER ADDRESS BYTE
        MOV     DPH,A
        ACALL   RDHEX           ;GET THE LOWER ADDRESS BYTE
        MOV     DPL,A
MEM1:   ACALL   CRLF            ;GO TO A NEW LINE
        MOV     A,DPH           ;SHOW THE ADDRESS
        ACALL   WRBYTE
        MOV     A,DPL
        ACALL   WRBYTE
        ACALL   SPACE           ;SPACE BETWEEN ADDRESS AND DATA
        MOVX    A,@DPTR         ;SHOW THE CURRENT DATA
        ACALL   WRBYTE
        ACALL   SPACE           ;OUTPUT A SPACE
        ACALL   GETC            ;GET THE NEXT CHARACTER
        CJNE    A,#' ',MEM2     ;ITS NOT A SPACE SO KEEP CHECKING
        INC     DPTR            ;SPACE, SO SHOW NEXT LOCATION
        SJMP    MEM1
MEM2:   CJNE    A,#'^',MEM4     ;ITS NOT A ^ SO KEEP CHECKING
        XCH     A,DPL           ;THIS ROUTINE DECREMENTS DPTR
        JNZ     MEM3            ;DPL NOT ZERO, SO DON'T CHANGE DPH
        DEC     DPH
MEM3:   DEC     A
        XCH     A,DPL
        SJMP    MEM1            ;SO WE CAN DISPLAY PREVIOUS
MEM4:   CJNE    A,#CR,NEWMEM    ;PRESUME ITS A NEW VALUE
        SJMP    MEMX            ;ITS A RETURN SO EXIT
NEWMEM: ACALL   RDHX1           ;GET THE NEW VALUE
        MOVX    @DPTR,A         ;MOVE THE NEW VALUE TO MEMORY
        INC     DPTR            ;POINT TO NEXT LOCATION
        SJMP    MEM1            ;AND CONTINUE
MEMX:   POP     DPL             ;RESTORE DPTR
        POP     DPH
        POP     ACC             ;AND THE ACCUMULATOR
        RET                     ;BACK FOR NEXT COMMAND

        .eject

LOAD:   PUSH    DPH             ;SAVE DPTR
        PUSH    DPL
        PUSH    CNTLO           ;SAVE CNTLO
        PUSH    ADHI            ;SAVE ADHI AND ADLO
        PUSH    ADLO
        MOV     ADHI,#0         ;INITIALIZE ADHI AND ADLO TO 0
        MOV     ADLO,#0
LOADX:  ACALL   GETC            ;GET THE NEXT CHARACTER
        CJNE    A,#' ',LOAD1    ;NOT A SPACE SO KEEP CHECKING
        SJMP    LOADX           ;SPACE DOESN'T MEAN ANYTHING SO LOOP
LOAD1:  CJNE    A,#CR,LOAD2     ;NOT A RETURN SO PRESUME A NEW ADDRESS
        SJMP    LOAD3           ;RETURN, SO LOAD FROM OLD ADDRESS
LOAD2:  ACALL   RDHX1           ;PRESUME A NEW ADDRESS
        MOV     ADHI,A
        ACALL   RDHEX
        MOV     ADLO,A
LOAD3:  ACALL   CRLF
        ACALL   CRLF
WAITBGN: ACALL  GETC            ;WAIT UNTIL ':' IS RECEIVED (START OF LINE)
        CJNE    A,#':',WAITBGN
        ACALL   RDHEX           ;GET THE BYTE COUNT
        JZ      ALLRD           ;LAST RECORD HAS 0 BYTE COUNT
        MOV     CNTLO,A         ;LOAD BYTE COUNT FOR LINE INTO CNTLO
        ACALL   RDHEX           ;READ HIGH BYTE OF START ADDRESS
        MOV     DPH,A
        ACALL   RDHEX           ;READ LOW BYTE OF START ADDRESS
        ADD     A,ADLO          ;ADD OFFSET ADDRESS
        MOV     DPL,A
        MOV     A,DPH
        ADDC    A,ADHI
        MOV     DPH,A
        ACALL   RDHEX           ;READ RECORD TYPE 00 = DATA, 01 = END
RDLP:   ACALL   RDHEX           ;GET DATA BYTE
        MOVX    @DPTR,A         ;STORE IT
        INC     DPTR            ;POINT TO NEXT BYTE ADDRESS
        DJNZ    CNTLO,RDLP      ;DECR BYTE COUNTER, LOOP IF NOT FINISHED
        ACALL   RDHEX           ;SKIP THE CHECKSUM
        ACALL   CRLF            ;GO TO A NEW LINE
        SJMP    WAITBGN         ;IGNORE CHECKSUM, LINE FEED, CARRIAGE RETURN
ALLRD:  ACALL   RDHEX           ;READ ADDRESS
        ACALL   RDHEX           ;READ ADDRESS
        ACALL   RDHEX           ;READ RECORD TYPE
        ACALL   RDHEX           ;READ LAST CHECKSUM
        ACALL   GETC            ;READ LAST CR
        ACALL   CRLF            ;GO TO A NEW LINE
        POP     ADLO            ;RESTORE ADHI AND ADLO
        POP     ADHI
        POP     CNTLO           ;RESTORE CNTLO
        POP     DPL             ;RESTORE DPTR
        POP     DPH
        RET                     ;DONE, BACK FOR NEXT COMMAND

        .eject

GO:     POP     DPH             ;GET THE OLD PROGRAM COUNTER
        POP     DPL             ;USE DPTR SINCE IT WILL BE RESTORED
        ACALL   GETC            ;GET THE NEXT CHARACTER
        CJNE    A,#' ',GO1      ;NOT A SPACE SO KEEP CHECKING
        SJMP    GO              ;SPACE DOESN'T MEAN ANYTHING SO LOOP
GO1:    CJNE    A,#CR,GO2       ;NOT A RETURN SO PRESUME A NEW ADDRESS
        SJMP    GO3             ;RETURN, SO GO FROM OLD ADDRESS
GO2:    ACALL   RDHX1           ;PRESUME A NEW ADDRESS
        MOV     DPH,A
        ACALL   RDHEX
        MOV     DPL,A
GO3:    PUSH    DPL             ;PC IS NOW OK SO RESTORE IT
        PUSH    DPH
        ACALL   RESTORE         ;RESTORE EVERYTHING ELSE
        ACALL   SETALL          ;SET ALL ACTIVE BREAKPOINTS
        RET                     ;RETURN IS ACTUALLY A GO!!

STEP:   POP     DPH             ;GET THE OLD PROGRAM COUNTER
        POP     DPL             ;USE DPTR SINCE IT WILL BE RESTORED
        ACALL   GETC            ;GET THE NEXT CHARACTER
        CJNE    A,#' ',STEP1    ;NOT A SPACE SO KEEP CHECKING
        SJMP    STEP            ;SPACE DOESN'T MEAN ANYTHING SO LOOP
STEP1:  CJNE    A,#CR,STEP2     ;NOT A RETURN SO PRESUME A NEW ADDRESS
        SJMP    STEP3           ;RETURN, SO STEP FROM OLD ADDRESS
STEP2:  ACALL   RDHX1           ;PRESUME A NEW ADDRESS
        MOV     DPH,A
        ACALL   RDHEX
        MOV     DPL,A
STEP3:  ACALL   CRLF            ;GO TO A NEW LINE TO MAKE IT PRETTY
        PUSH    DPL             ;PC IS NOW OK SO RESTORE
        PUSH    DPH
        ACALL   RESTORE         ;RESTORE EVERYTHING ELSE
        PUSH    ACC             ;SAVE THE ACCUMULATOR
        SETB    P3.2            ;MAKE SURE INT0 IS OFF
        MOV     A,#10           ;SHORT DELAY TO MAKE SURE IT GOT DONE
        DJNZ    ACC,*
        ORL     IE,#0x81        ;ENABLE INT0
        ORL     IP,#0x01        ;AND MAKE IT HIGH PRIORITY
        MOV     A,#10           ;SHORT DELAY TO MAKE SURE IT GOT DONE
        DJNZ    ACC,*
        POP     ACC             ;RESTORE THE ACCUMULATOR
        ACALL   SETALL          ;SET ALL BREAKPOINTS
        CLR     P3.2            ;CAUSE INT0 AFTER THE NEXT INSTRUCTION
        RETI                    ;RETURN IS ACTUALLY A GO!!

        .eject

UCALL:  POP     ADHI            ;GET THE OLD PROGRAM COUNTER
        POP     ADLO
        ACALL   GETC            ;GET THE NEXT CHARACTER
        CJNE    A,#' ',CALL1    ;NOT A SPACE SO KEEP CHECKING
        SJMP    UCALL           ;SPACE DOESN'T MEAN ANYTHING SO LOOP
CALL1:  CJNE    A,#CR,CALL2     ;NOT A RETURN SO PRESUME A NEW ADDRESS
        SJMP    CALL3           ;RETURN, SO GO FROM OLD ADDRESS
CALL2:  ACALL   RDHX1           ;PRESUME A NEW ADDRESS
        MOV     ADHI,A
        ACALL   RDHEX
        MOV     ADLO,A
CALL3:  MOV     DPTR,#CALLRET   ;PUSH THE ADDRESS OF CALLRET SO THAT
        PUSH    DPL             ;THE RETURN OF THE CALLED ROUTINE
        PUSH    DPH             ;WILL GO THERE
        ACALL   RESTORE         ;RESTORE EVERYTHING EXCEPT PC
        PUSH    ADLO            ;PC IS NOW OK SO RESTORE
        PUSH    ADHI
        ACALL   SETALL          ;SET ALL BREAKPOINTS
        RET                     ;RETURN IS ACTUALLY A GO!!

        .eject

DUMP:   PUSH    ACC             ;SAVE THE ACCUMULATOR
        PUSH    DPH             ;AND THE DPTR
        PUSH    DPL
        PUSH    TEMP1           ;AND THE WORK AREAS
        PUSH    CNTHI
        PUSH    CNTLO
        ACALL   RDHEX           ;GET THE UPPER BYTE OF THE START ADDRESS
        MOV     DPH,A
        ACALL   RDHEX           ;GET THE LOWER BYTE OF THE START ADDRESS
        ANL     A,#0xF0         ;START ON 16-BYTE BOUNDARY
        MOV     DPL,A
        ACALL   RDHEX           ;GET THE HIGH BYTE OF THE COUNT
        MOV     CNTHI,A
        ACALL   RDHEX           ;AND THE LOW BYTE OF THE COUNT
        MOV     CNTLO,A
        ACALL   CRLF            ;START ON A NEW LINE
LOUTLP: MOV     TEMP1,#0x10     ;SET UP BYTE COUNTER
        MOV     A,DPH           ;AT START OF LINE OUTPUT ADDR
        ACALL   WRBYTE
        MOV     A,DPL
        ACALL   WRBYTE
        ACALL   SPACE           ;AND SPACE TO MAKE IT TIDY
        PUSH    DPH             ;SAVE DPTR FOR LATER
        PUSH    DPL
BOUTLP: MOVX    A,@DPTR         ;OUTPUT HEX ONE BYTE AT A TIME
        ACALL   WRBYTE
        ACALL   SPACE           ;A SPACE TO MAKE IT NEAT
        INC     DPTR
        DJNZ    TEMP1,BOUTLP    ;IF NOT DONE, LOOP
        ACALL   SPACE           ;A SPACE TO MAKE IT NEAT
        POP     DPL             ;RESTORE THE DPTR
        POP     DPH
        MOV     TEMP1,#0x10     ;SET UP BYTE COUNTER
AOUTLP: MOVX    A,@DPTR         ;OUTPUT ASCII ONE BYTE AT A TIME
        CJNE    A,#127,AOUT1    ;SEE IF ITS >127
AOUT1:  JNC     AOUT3           ;YES, SO OUTPUT '.'
        CJNE    A,#' ',AOUT2    ;SEE IF ITS <32
AOUT2:  JC      AOUT3           ;YES, SO OUTPUT '.'
        SJMP    AOUT4           ;ITS PRINTABLE, SO PRINT IT
AOUT3:  MOV     A,#'.'          ;NOT PRINTABLE, SO MAKE IT '.'
AOUT4:  ACALL   PUTC            ;PRINT IT
        INC     DPTR
        DJNZ    TEMP1,AOUTLP    ;IF NOT DONE, LOOP
        ACALL   CRLF            ;GO TO NEXT LINE
        CLR     C               ;SET UP FOR SUBTRACT
        MOV     A,CNTLO         ;SUBTRACT 16 FROM THE COUNT
        SUBB    A,#0x10
        MOV     CNTLO,A
        MOV     A,CNTHI
        SUBB    A,#0
        MOV     CNTHI,A
        ORL     A,CNTLO         ;SEE IF WE'VE REACHED ZERO YET
        JZ      DMPSKP          ;IF SO, WE'RE DONE
        JNC     LOUTLP          ;NO CARRY, SO NOT DONE YET
DMPSKP: POP     CNTLO           ;RESTORE THE WORK AREAS
        POP     CNTHI
        POP     TEMP1
        POP     DPL             ;AND THE DPTR
        POP     DPH
        POP     ACC             ;AND THE ACCUMULATOR
        RET                     ;BACK FOR NEXT COMMAND

        .eject

PRMLD:  MOV     BKFLAG,#0       ;CLEAR ALL BREAKPOINTS
        MOV     A,R0            ;SAVE R0
        PUSH    ACC
        PUSH    DPH             ;SAVE DPTR
        PUSH    DPL
        PUSH    ADHI            ;SAVE ADHI AND ADLO
        PUSH    ADLO
        PUSH    TEMP1           ;SAVE TEMP1 THRU TEMP3
        PUSH    TEMP2
        PUSH    TEMP3
        PUSH    P2              ;SAVE P2
        MOV     P2,#RAMPAGE     ;POINT TO MONITOR RAM AREA
        MOV     TEMP2,#0        ;SET UP ZERO OFFSET
        MOV     TEMP3,#0
PRM:    ACALL   GETC            ;GET THE NEXT CHARACTER
        CJNE    A,#' ',PRM1     ;NOT A SPACE SO KEEP CHECKING
        SJMP    PRM             ;SPACE DOESN'T MEAN ANYTHING SO LOOP
PRM1:   CJNE    A,#CR,PRM2      ;NOT A RETURN SO PRESUME A NEW ADDRESS
        SJMP    PRMWAIT         ;RETURN, SO GO FROM ZERO
PRM2:   ACALL   RDHX1           ;PRESUME AN OFFSET
        MOV     TEMP3,A
        ACALL   RDHEX
        MOV     TEMP2,A
        ACALL   CRLF
PRMLIN: ACALL  CRLF
PRMWAIT: ACALL  GETC            ;WAIT UNTIL ':' IS RECEIVED (START OF LINE)
        CJNE    A,#':',PRMWAIT
        MOV     CKSUM,#0        ;INITIALIZE THE CHECKSUM
        MOV     R0,#BUFFER      ;POINT R0 TO THE BUFFER AREA
        ACALL   RDHEX           ;GET THE BYTE COUNT
        MOV     CNTLO,A         ;INITIALIZE THE COUNT
        ADD     A,CKSUM         ;UPDATE THE CHECKSUM
        MOV     CKSUM,A
        JZ      PRMDONE         ;LAST RECORD HAS 0 BYTE COUNT
        MOV     TEMP1,CNTLO     ;LOAD BYTE COUNT FOR LINE INTO TEMP1
        ACALL   RDHEX           ;READ HIGH BYTE OF START ADDRESS
        MOV     ADHI,A
        ADD     A,CKSUM         ;UPDATE THE CHECKSUM
        MOV     CKSUM,A
        ACALL   RDHEX           ;READ LOW BYTE OF START ADDRESS
        MOV     ADLO,A
        ADD     A,CKSUM         ;UPDATE THE CHECKSUM
        MOV     CKSUM,A
        ACALL   RDHEX           ;READ RECORD TYPE 00 = DATA, 01 = END
        ADD     A,CKSUM         ;UPDATE THE CHECKSUM
        MOV     CKSUM,A

        .eject

PRMRDLP: ACALL  RDHEX           ;GET DATA BYTE
        MOVX    @R0,A           ;STORE IT IN THE BUFFER
        INC     R0              ;BUMP THE POINTER
        ADD     A,CKSUM         ;UPDATE THE CHECKSUM
        MOV     CKSUM,A
        DJNZ    TEMP1,PRMRDLP   ;DECR BYTE COUNTER, LOOP IF NOT FINISHED
        ACALL   RDHEX           ;GET THE CHECKSUM
        ADD     A,CKSUM         ;ADD IT IN
        JZ      PRMPRG          ;IF CKSUM = 0, THEN EVERYTHING'S OK
PRMER1: MOV     A,#'?'          ;ELSE SEND ERROR CODE
        ACALL   PUTC
        SJMP    PRMLIN          ;GO BACK TO TRY AGAIN
PRMPRG: MOV     TEMP1,CNTLO     ;LOAD BYTE COUNT FOR LINE INTO TEMP1
        MOV     DPH,ADHI        ;GET HIGH BYTE OF START ADDRESS
        MOV     DPL,ADLO        ;GET LOW BYTE OF START ADDRESS
        MOV     A,DPL           ;ADD IN THE OFFSET
        ADD     A,TEMP2
        MOV     DPL,A
        MOV     A,DPH
        ADDC    A,TEMP3
        MOV     DPH,A
        MOV     R0,#BUFFER      ;POINT TO THE BEGINNING OF THE BUFFER
PRGLP:  MOVX    A,@R0           ;GET THE DATA BYTE
        MOV     DATA,A          ;SAVE IT
        MOVX    @DPTR,A         ;LOAD IT INTO THE PROM
        MOV     A,#10           ;WAIT 10 MILLISECONDS FOR THE PROM
        ACALL   DELAY
        MOVX    A,@DPTR         ;READ THE BYTE BACK FROM THE PROM
        CJNE    A,DATA,PRMER2   ;IF IT DIDN'T PROGRAM RIGHT, GIVE ERROR MESSAGE
        SJMP    PRGCONT         ;ELSE CONTINUE
PRMER2: MOV     A,#'!'          ;SEND ERROR MESSAGE
        ACALL   PUTC
        AJMP    PRMLIN          ;GO BACK TO TRY AGAIN
PRGCONT: INC    R0              ;POINT TO NEXT BYTE
        INC     DPTR            ;POINT TO NEXT PROM ADDRESS
        DJNZ    TEMP1,PRGLP     ;DECR BYTE COUNTER, LOOP IF NOT FINISHED
        MOV     A,#'$'          ;SEND OK MESSAGE
        ACALL   PUTC
        AJMP    PRMLIN          ;GO BACK FOR MORE
        .eject

PRMDONE: ACALL  RDHEX           ;READ HIGH ADDRESS
        ACALL   RDHEX           ;READ LOW ADDRESS
        ACALL   RDHEX           ;READ RECORD TYPE
        ACALL   RDHEX           ;READ LAST CHECKSUM
        MOV     A,#'$'          ;SEND OK CODE
        ACALL   PUTC
PRMEX:  ACALL   GETC            ;READ LAST CR
        POP     P2              ;RESTORE P2
        POP     TEMP3           ;RESTORE TEMP1 THRU TEMP3
        POP     TEMP2
        POP     TEMP1
        POP     ADLO            ;RESTORE ADLO AND ADHI
        POP     ADHI
        POP     DPL             ;RESTORE DPTR
        POP     DPH
        POP     ACC             ;RESTORE R0
        MOV     R0,A
        RET                     ;DONE, BACK FOR NEXT COMMAND

        .eject

FILL:   PUSH    ADLO            ;SAVE ALL USED REGISTERS
        PUSH    ADHI
        PUSH    CNTLO
        PUSH    CNTHI
        PUSH    DPH
        PUSH    DPL
        PUSH    ACC
        ACALL   RDHEX           ;GET HIGH BYTE OF ADDRESS
        MOV     ADHI,A          ;SAVE IT
        ACALL   RDHEX           ;GET LOW BYTE OF ADDRESS
        MOV     ADLO,A          ;SAVE IT
        ACALL   RDHEX           ;GET HIGH BYTE OF COUNT
        INC     A               ;FIX DJNZ WIERDITY
        MOV     CNTHI,A         ;SAVE IT
        ACALL   RDHEX           ;GET LOW BYTE OF COUNT
        INC     A               ;FIX DJNZ WIERDITY
        MOV     CNTLO,A         ;SAVE IT
        ACALL   RDHEX           ;BYTE TO FILL WITH
        MOV     DPH,ADHI        ;HIGH BYTE OF ADDRESS
        MOV     DPL,ADLO        ;LOW BYTE OF ADDRESS
FLP:    MOVX    @DPTR,A         ;STORE THE BYTE
        INC     DPTR            ;POINT TO THE NEXT LOCATION
        DJNZ    CNTLO,FLP       ;LOOP IF NOT DONE
        DJNZ    CNTHI,FLP       ;LOOP IF NOT DONE
        POP     ACC             ;RESTORE ALL USED REGISTERS
        POP     DPL
        POP     DPH
        POP     CNTHI
        POP     CNTLO
        POP     ADHI
        POP     ADLO
        RET                     ;BACK FOR MORE

        .eject

MOVE:   PUSH    ADLO            ;SAVE ALL USED REGISTERS
        PUSH    ADHI
        PUSH    TEMP1
        PUSH    TEMP2
        PUSH    CNTLO
        PUSH    CNTHI
        PUSH    DPH
        PUSH    DPL
        PUSH    ACC
        ACALL   RDHEX           ;GET HIGH BYTE OF SOURCE ADDRESS
        MOV     ADHI,A          ;SAVE IT
        ACALL   RDHEX           ;GET LOW BYTE OF SOURCE ADDRESS
        MOV     ADLO,A          ;SAVE IT
        ACALL   RDHEX           ;GET HIGH BYTE OF DEST ADDRESS
        MOV     TEMP2,A         ;SAVE IT
        ACALL   RDHEX           ;GET LOW BYTE OF DEST ADDRESS
        MOV     TEMP1,A         ;SAVE IT
        ACALL   RDHEX           ;GET HIGH BYTE OF COUNT
        INC     A               ;FIX DJNZ WIERDITY
        MOV     CNTHI,A         ;SAVE IT
        ACALL   RDHEX           ;GET LOW BYTE OF COUNT
        INC     A               ;FIX DJNZ WIERDITY
        MOV     CNTLO,A         ;SAVE IT
MVLP:   MOV     DPH,ADHI        ;HIGH BYTE OF SOURCE ADDRESS
        MOV     DPL,ADLO        ;LOW BYTE OF SOURCE ADDRESS
        MOVX    A,@DPTR         ;FETCH THE DATA BYTE
        INC     DPTR            ;POINT TO THE NEXT LOCATION
        MOV     ADHI,DPH        ;SAVE THE HIGH BYTE
        MOV     ADLO,DPL        ;SAVE THE LOW BYTE
        MOV     DPH,TEMP2       ;HIGH BYTE OF DEST ADDRESS
        MOV     DPL,TEMP1       ;LOW BYTE OF DEST ADDRESS
        MOVX    @DPTR,A         ;STORE THE BYTE
        INC     DPTR            ;POINT TO THE NEXT LOCATION
        MOV     TEMP2,DPH       ;SAVE THE HIGH BYTE
        MOV     TEMP1,DPL       ;SAVE THE LOW BYTE
        DJNZ    CNTLO,MVLP      ;LOOP IF NOT DONE
        DJNZ    CNTHI,MVLP      ;LOOP IF NOT DONE
        POP     ACC             ;RESTORE ALL USED REGISTERS
        POP     DPL
        POP     DPH
        POP     CNTHI
        POP     CNTLO
        POP     TEMP2
        POP     TEMP1
        POP     ADHI
        POP     ADLO
        RET                     ;BACK FOR MORE

        .eject

RSTBP:  PUSH    P2              ;SAVE P2
        MOV     P2,#RAMPAGE     ;POINT TO MONITOR RAM AREA
        MOVX    A,@R0           ;RESTORE A BRKPT
        MOV     DPH,A           ;R0 POINTS TO BP DATA
        INC     R0
        MOVX    A,@R0
        MOV     DPL,A
        INC     R0
        MOVX    A,@R0           ;RESTORE THE CODE
        MOVX    @DPTR,A
        INC     DPTR
        INC     R0
        MOVX    A,@R0
        MOVX    @DPTR,A
        INC     DPTR
        INC     R0
        MOVX    A,@R0
        MOVX    @DPTR,A
        POP     P2              ;RESTORE P2
        RET

RSTALL: PUSH    DPH             ;SAVE THE REGISTERS
        PUSH    DPL
        PUSH    ACC
        MOV     A,R0
        PUSH    ACC
        MOV     A,BKFLAG
        ANL     A,#BK1MASK      ;IF BK1 NOT SET,SKIP
        JZ      RSTB2
        MOV     R0,#BRKPT1
        ACALL   RSTBP
RSTB2:  MOV     A,BKFLAG
        ANL     A,#BK2MASK      ;IF BK2 NOT SET, SKIP
        JZ      RSTB3
        MOV     R0,#BRKPT2
        ACALL   RSTBP
RSTB3:  MOV     A,BKFLAG
        ANL     A,#BK3MASK      ;IF BK3 NOT SET, SKIP
        JZ      RSTB4
        MOV     R0,#BRKPT3
        ACALL   RSTBP
RSTB4:  MOV     A,BKFLAG
        ANL     A,#BK4MASK      ;IF BK4 NOT SET, SKIP
        JZ      RSTX
        MOV     R0,#BRKPT4
        ACALL   RSTBP
RSTX:   POP     ACC             ;RESTORE THE REGISTERS
        MOV     R0,A
        POP     ACC
        POP     DPL
        POP     DPH
        RET

        .eject

SETBP:  PUSH    ACC             ;SAVE THE ACCUMULATOR
        PUSH    P2              ;SAVE P2
        MOV     P2,#RAMPAGE     ;POINT TO MONITOR RAM AREA
        MOVX    A,@R0           ;(RE)SET A BREAKPOINT
        MOV     DPH,A
        INC     R0              ;R0 POINTS TO THE BREAKPOINT RECORD
        MOVX    A,@R0
        MOV     DPL,A
        INC     R0
        PUSH    DPH             ;SAVE DPTR FOR LATER
        PUSH    DPL
        MOVX    A,@DPTR         ;GET FIRST CODE BYTE
        MOVX    @R0,A
        INC     DPTR
        INC     R0
        MOVX    A,@DPTR         ;GET SECOND CODE BYTE
        MOVX    @R0,A
        INC     DPTR
        INC     R0
        MOVX    A,@DPTR         ;GET THIRD CODE BYTE
        MOVX    @R0,A
        POP     DPL             ;RETRIEVE DPTR
        POP     DPH
SETCODE:                        ;SETUP JUMP TO BP HANDLER
        MOV     A,#0x12         ;LCALL OPCODE
        MOVX    @DPTR,A
        INC     DPTR
        MOV     A,#BPHLR_HI     ;HIGH BYTE OF BPT HANDLER ADDRESS
        MOVX    @DPTR,A
        INC     DPTR
        MOV     A,#BPHLR_LO     ;LOW BYTE OF BREAKPOINT HANDLER
        MOVX    @DPTR,A
        POP     P2              ;RESTORE P2
        POP     ACC             ;RESTORE ACCUMULATOR
        RET

        .eject
        
SETALL: PUSH    DPH             ;SAVE THE REGISTERS
        PUSH    DPL
        PUSH    ACC
        MOV     A,R0
        PUSH    ACC
        MOV     A,BKFLAG
        ANL     A,#BK1MASK      ;IF BK1 NOT SET,SKIP
        JZ      SETB2
        MOV     R0,#BRKPT1
        ACALL   SETBP
SETB2:  MOV     A,BKFLAG
        ANL     A,#BK2MASK      ;IF BK2 NOT SET, SKIP
        JZ      SETB3
        MOV     R0,#BRKPT2
        ACALL   SETBP
SETB3:  MOV     A,BKFLAG
        ANL     A,#BK3MASK      ;IF BK3 NOT SET, SKIP
        JZ      SETB4
        MOV     R0,#BRKPT3
        ACALL   SETBP
SETB4:  MOV     A,BKFLAG
        ANL     A,#BK4MASK      ;IF BK4 NOT SET, SKIP
        JZ      SETX
        MOV     R0,#BRKPT4
        ACALL   SETBP
SETX:   POP     ACC             ;RESTORE THE REGISTERS
        MOV     R0,A
        POP     ACC
        POP     DPL
        POP     DPH
        RET

BPHLR:  ACALL   SAVEREG         ;SAVE THE REGISTERS,RESTORE THE CODE
        ACALL   CRLF            ;PUT IT ON A NEW LINE
        MOV     A,#'B'          ;SEND "B=" MESSAGE
        ACALL   PUTC
        MOV     A,#'='
        ACALL   PUTC
        POP     ADHI            ;GET THE PROGRAM COUNTER
        POP     ACC
        CLR     C
        SUBB    A,#0x03         ;SUBTRACT 3 TO MAKE IT RIGHT
        MOV     ADLO,A
        MOV     A,ADHI
        SUBB    A,#0x00
        MOV     ADHI,A
        ACALL   WRBYTE          ;DISPLAY THE ADDRESS
        MOV     A,ADLO
        ACALL   WRBYTE
        AJMP    GETCMD

        .eject

BPDISP: PUSH    DPH             ;SAVE USED REGISTERS
        PUSH    DPL
        PUSH    TEMP1
        PUSH    ACC
        MOV     A,R0            ;INCLUDING R0
        PUSH    ACC
BPDISS: ACALL   GETC            ;GET THE NEXT CHARACTER
        CJNE    A,#' ',BPDIS1   ;NOT A SPACE SO CONTINUE
        SJMP    BPDISS          ;SPACE DOESN'T MEAN ANYTHING SO LOOP
BPDIS1: CJNE    A,#'1',BPDIS2   ;IF ITS NOT BP #1, CONTINUE
        MOV     TEMP1,#BK1MASK  ;SET UP BIT MASK FOR CLEAR
        MOV     R0,#BRKPT1      ;POINT TO FIRST BREAKPOINT
        SJMP    BPCON           ;CONTINUE
BPDIS2: CJNE    A,#'2',BPDIS3   ;IF ITS NOT BP #2, CONTINUE
        MOV     TEMP1,#BK2MASK  ;SET UP BIT MASK FOR CLEAR
        MOV     R0,#BRKPT2      ;POINT TO SECOND BREAKPOINT
        SJMP    BPCON           ;CONTINUE
BPDIS3: CJNE    A,#'3',BPDIS4   ;IF ITS NOT BP #3, CONTINUE
        MOV     TEMP1,#BK3MASK  ;SET UP BIT MASK FOR CLEAR
        MOV     R0,#BRKPT3      ;POINT TO THIRD BREAKPOINT
        SJMP    BPCON           ;CONTINUE
BPDIS4: CJNE    A,#'4',BPEX     ;IF ITS NOT BP #4, EXIT
        MOV     TEMP1,#BK4MASK  ;SET UP BIT MASK FOR CLEAR
        MOV     R0,#BRKPT4      ;POINT TO FOURTH BREAKPOINT
BPCON:  ACALL   GETC            ;GET THE NEXT CHARACTER
        CJNE    A,#' ',BPCON2   ;NOT A SPACE SO CONTINUE
        SJMP    BPCON           ;SPACE DOESN'T MEAN ANYTHING SO LOOP
BPCON2: CJNE    A,#'-',BPSET    ;IF ITS A '-' THEN CLEAR A BREAKPOINT
        ACALL   RSTBP           ;RESTORE THE CODE
        MOV     A,BKFLAG        ;SET UP TO CLEAR THE APPROPRIATE BIT
        XRL     TEMP1,#0xFF     ;FLIP THE BITS IN TEMP1
        ANL     A,TEMP1         ;TEMP1 CONTAINS THE MASK
        MOV     BKFLAG,A
        SJMP    BPEX            ;AND EXIT
BPSET:  ACALL   RDHX1           ;ELSE ASSUME ITS A BREAKPOINT TO SET
        MOV     DPH,A           ;AND GET THE ADDRESS
        PUSH    P2              ;SAVE P2
        MOV     P2,#RAMPAGE     ;POINT TO MONITOR RAM AREA
        MOVX    @R0,A
        INC     R0
        ACALL   RDHEX
        MOV     DPL,A
        MOVX    @R0,A
        MOV     A,BKFLAG        ;GET READY TO ENABLE THE BREAKPOINT
        ORL     A,TEMP1
        MOV     BKFLAG,A        ;AND ENABLE IT
        POP     P2              ;RESTORE P2
BPEX:   POP     ACC             ;RESTORE R0
        MOV     R0,A
        POP     ACC             ;AND THE OTHER REGISTERS
        POP     TEMP1
        POP     DPL
        POP     DPH
        RET

        .eject

REGSTR: PUSH    DPH             ;SAVE DPTR (FOR WHEN WE CALL INTRAM LATER)
        PUSH    DPL
        ACALL   SPACE           ;OUTPUT A SPACE
REGST1: ACALL   GETC            ;GET THE CHARACTER
        CJNE    A,#' ',REGST2   ;SPACE DOESN'T MEAN ANYTHING
        SJMP    REGST1          ;SO LOOP
REGST2: CJNE    A,#'A',REGST3   ;IF ITS NOT 'A', CONTINUE
        MOV     A,#ASAVE        ;POINT TO ACCUMULATOR SAVE AREA
        AJMP    RAMENT          ;USE INTRAM CODE
REGST3: CJNE    A,#'B',REGST4   ;IF ITS NOT 'B', CONTINUE
        MOV     A,#BSAVE        ;POINT TO B REGISTER SAVE AREA
        AJMP    RAMENT          ;USE INTRAM CODE
REGST4: CJNE    A,#'H',REGST5   ;IF ITS NOT 'H', CONTINUE
        MOV     A,#DPHSAVE      ;POINT TO DPH SAVE AREA
        AJMP    RAMENT          ;USE INTRAM CODE
REGST5: CJNE    A,#'L',REGST6   ;IF ITS NOT 'L', CONTINUE
        MOV     A,#DPLSAVE      ;POINT TO DPL SAVE AREA
        AJMP    RAMENT          ;USE INTRAM CODE
REGST6: CJNE    A,#'S',REGST7   ;IF ITS NOT 'S', CONTINUE
        MOV     A,#PSWSAVE      ;POINT TO PSW SAVE AREA
        AJMP    RAMENT          ;USE INTRAM CODE
REGST7: ACALL   ASC2HEX         ;ASSUME ITS R0-R7, CONVERT R NUMBER
        ANL     A,#0x07         ;MAKE SURE ITS 0-7
        PUSH    B               ;SAVE THE B REGISTER
        MOV     B,A             ;SAVE THE R NUMBER IN B
        MOV     A,PSWSAVE       ;GET THE BITS INDICATING REGISTER SET
        ANL     A,#0x18         ;MASK OFF EVERYTHING ELSE
        ORL     A,B             ;BUILD THE POINTER VALUE
        POP     B               ;RESTORE REGISTER B
        AJMP    RAMENT          ;USE THE INTRAM CODE

        .eject

; READS TWO ASCII CHARACTERS FROM THE SERIAL PORT AND
; CONVERTS THEM INTO A BYTE RETURNED IN A.
;
RDHEX:  ACALL   GETC            ;GET THE FIRST CHARACTER
        CJNE    A,#' ',RDHX1    ;SKIP LEADING SPACES
        SJMP    RDHEX
RDHX1:  PUSH    TEMP1           ;SAVE TEMP1
        ACALL   ASC2HEX         ;CONVERT TO HEX NIBBLE
        SWAP    A               ;HIGH ORDER NIBBLE IS FIRST ONE
        MOV     TEMP1,A         ;SAVE THE HIGH ORDER NIBBLE
RDHX2:  ACALL   GETC            ;GET LOW ORDER CHARACTER
        CJNE    A,#' ',RDHX3    ;SKIP LEADING SPACES
        SJMP    RDHX2
RDHX3:  ACALL   ASC2HEX         ;CHANGE TO HEX NIBBLE
        ORL     A,TEMP1         ;PUT NIBBLES TOGETHER IN A
        POP     TEMP1           ;RESTORE TEMP1
        RET

; WRITES BYTE IN A TO THE SERIAL PORT AS TWO ASCII CHARACTERS.
;
WRBYTE: PUSH    TEMP1           ;SAVE TEMP1
        MOV     TEMP1,A
        SWAP    A
        ACALL   HEX2ASC
        ACALL   PUTC
        MOV     A,TEMP1
        ACALL   HEX2ASC
        ACALL   PUTC
        POP     TEMP1           ;RESTORE TEMP1
        RET

; CONVERTS AN ASCII CHARACTER IN A TO ITS HEX EQUIVALENT.
; RETURNS VALUE IN LOWER NIBBLE, UPPER NIBBLE ZEROS
;
ASC2HEX: PUSH   B
        CLR     C
        SUBB    A,#'0'
        MOV     B,A
        SUBB    A,#0x0A
        JC      A2LT10
        MOV     A,B
        SUBB    A,#0x07
        MOV     B,A
A2LT10: MOV     A,B
        POP     B
        RET

        .eject

; CONVERTS THE LOWER NIBBLE IN A TO AN ASCII CHARACTER RETURNED IN A
;
HEX2ASC: PUSH   B
        ANL     A,#0x0F
        MOV     B,A
        CLR     C
        SUBB    A,#0x0A
        MOV     A,B
        JC      H2LT10
        ADD     A,#0x07
H2LT10: ADD     A,#0x30
        POP     B
        RET

; PUTS SENDS THE TEXT STRING POINTED TO BY DPTR TO THE SERIAL PORT.
; THE TEXT STRING MUST END WITH A NULL CHARACTER
;
PUTS:   PUSH    ACC             ;SAVE THE ACCUMULATOR
PUTS1:  CLR     A               ;(ZERO OFFSET)
        MOVC    A,@A+DPTR       ;FETCH NEXT CHAR IN STRING
        JNZ     PUTS2           ;IF NOT NULL, SEND IT
        POP     ACC             ;ELSE RESTORE THE ACCUMULATOR
        RET                     ;AND RETURN
PUTS2:  ACALL   PUTC            ;SEND THE CHARACTER
        INC     DPTR            ;BUMP POINTER
        SJMP    PUTS1           ;BACK FOR MORE

; SPACE OUTPUTS A SPACE TO THE SERIAL PORT
;
SPACE:  PUSH    ACC
        MOV     A,#' '
        ACALL   PUTC
        POP     ACC
        RET

; CRLF SENDS A CARRIAGE RETURN AND LINE FEED OUT THE SERIAL PORT
;
CRLF:   PUSH    ACC
        MOV     A,#CR
        ACALL   PUTC
        MOV     A,#LF
        ACALL   PUTC
        POP     ACC
        RET

        .eject

; GETC READS A CHARACTER INTO A FROM THE SERIAL PORT AND
; ECHOS THE CHARACTER BACK TO THE SERIAL PORT
;
GETC:   ACALL   SERIN
        ANL     A,#0x7F
        CJNE    A,#ESC,GETC2    ;IF ESC, ABORT CURRENT COMMAND
        AJMP    GETCMD
GETC2:  CJNE    A,#'a',GETC3    ;IF <'a', DON'T CHANGE IT
GETC3:  JC      GETC5           ;DITTO
        CJNE    A,#'z',GETC4    ;IF >'z', DON'T CHANGE IT
GETC4:  JNC     GETC5           ;DITTO
        ANL     A,#0xDF         ;ELSE MAKE IT UPPER CASE
GETC5:  ACALL   PUTC            ;ECHO IT
        RET

; SERIN READS THE SERIAL PORT AND RETURNS THE VALUE IN A
;
SERIN:  JNB     RI,*
        CLR     RI
        MOV     A,SBUF
        RET

; PUTC SENDS THE CHARACTER IN A OUT THE SERIAL PORT(MUST BE AN ASCII CHARACTER)
;
PUTC:   JNB     TI,*
        CLR     TI
        MOV     SBUF,A
        RET

; DELAY PAUSES FOR 1 MILLISECOND TIMES THE VALUE IN A.
;
DELAY:  PUSH    B
        MOV     B,#0xF0         ;12 MHz, CHANGE TO 0xC9 FOR 11.0592
        DJNZ    B,*
        DJNZ    B,*             ;NO, THIS ISN'T A MISTAKE
        POP     B
        DJNZ    ACC,DELAY
        RET

