
/************************************************************************
*									*
*			1988 - Jim Schimpf				*
*									*
*			EMULATOR MAIN PROGRAM				*
*									*
*	Module:mterm.c							*
*									*
*	Version	   Date		Person  	Description		*
*-----------------------------------------------------------------------*
*       0.1     28-Nov-88       J.Schimpf       Initial Version         *
*	0.2	 7-Dec-88	J.Schimpf	Terminal emulator for	*
*						HC11 board EVB		*
*	0.3	23-Jan-89	J.Schimpf	Modify to quit with no	*
*						check on exit		*
*	0.4	20-Jul-89	J.Schimpf	Add parity,bits to setup*
*	0.5	21-Jul-89	J.Schimpf	Add int driven output &	*
*						show flags on default	*
*	0.6	20-Nov-89	J.Schimpf	Change parity to ODD	*
*						EVEN and NONE not 0,+-1	*
*	0.7	 6-Jun-90	J.Schimpf	Convert to ANSI		*
*	0.8	 4-Sep-90	J.Schimpf	Add LF/CR switch	*
*	0.9	 1-Feb-91	J.Schimpf	Add no echo switch	*
*	1.0	29-May-91	J.Schimpf	Change to support new	*
*						SERCOM			*
*									*
*	DESCRIPTION:							*
*		This program is a terminal emulator for the EVB system	*
*	as such it comes up at 9600 baud connected to a selected port	*
*									*
*	SYNTAX: mterm [-p 1/2] [-s [9600....300]] [-lf] [-ne]		*
*									*
*		-p <Port #>	1,2 for COM1 or COM2	DEF = COM1	*
*		-s <Baud rate>	9600 to 300 baud	DEF = 9600	*
*		-b <# Bits>	# Bits in data word	DEF = 8		*
*		-pa [ODD NONE EVEN]			DEF = NONE	*
*		-lf		Turn CR -> LFCR		DEF = NO	*
*		-ne		Turn off ECHO		DEF = ON
*									*
*		EXAMPLE:						*
*			mterm -p 2 -s 4800	Use COM2 @ 4800 baud	*
*									*
*	` is the HOT KEY to switch to command mode, hit it to get	*
*	command menu							*
*									*
*	Procedures:							*
*									*
*	Written+		Name	Description 			*
*-----------------------------------------------------------------------*
*									*
************************************************************************/

/******* INCLUDE FILES ******/

#include <stdio.h>
#include <process.h>
#include "base.h"
#include "menu.h"
#include "mterm.h"
#include "sercom.h"

/*  GLOBAL DATA STRUCTURES */

extern int errno;
BOOLEAN lfcr = FALSE;
BOOLEAN no_echo = FALSE;

COM *c;

int dos_action();
int log_action();
int xfer_action();
int quit_action();

FILE *out_file;
FILE *log_file;

MENU	abort_menu[] = { "DOS",dos_action,
			 "Open/Close Log File",log_action,
			 "Open Xfer file",xfer_action,
			 "Quit Program",quit_action, };
		 
/*  FORWARD DEFS OF FUNCTIONS */

void initialization();

/*************************************************************************
*
*	MAIN ROUTINE
*
**************************************************************************/

main(int argc,char **argv)
{
	int i;
	char ch;
	long j;
	float x;

	/*  1) Init I/O channels & error if problem */
	
	initialization(argc,argv);
	
	/*  2) Communication loop, any data from serial put on
		screen and any keys hit check for ABORT
	*/

	i = 0;
	while( TRUE )
	{

	/*  (2a) If we have and open output file then send stuff from
		there to the serial output
	*/
		
		if( out_file != NULL && !feof(out_file) )
		{
			ch = fgetc(out_file);
			if( ch == 0x0a )	/* Change LF's -> CR's */
				ch = 0x0d;
			putser(ch,c);

			/*  Handle EOF here */
			
			if( feof( out_file ) )
			{
				fclose(out_file );
				out_file = NULL;
			}
		}
		
	/*  (2b) Check for serial data comming back */
		
		if( serhit(c))
		{
			i++;
			ch = getser(c);
			if( out_file != NULL  && no_echo )
			{	/* Show that we are sending */
				if( (i%256) == 0 )
					putchar('.');
			}
			else
				putchar(ch);
	
	/*  (2c) If we have a log file then send this to it */
			
			if( log_file )
				fputc(ch,log_file);
			if( lfcr && (ch&0x7f) == 0x0d )
			{
				if( !no_echo )
					putchar( 0x0a );
				if( log_file )
					fputc(0x0a,log_file);
			}
		}
		
	/*  (2d) Check for user output */
		
		if( kbhit() )
		{
			ch = getch();

			if( ch == ABORT_CH )
			{
				if( !abort_action() )
					break;
			}
			else
				putser(ch,c);
		}
		
	}

	/*  3) All done return to user */

	printf("\n");
	print_center("*** END RUN ***");
	ser_close(c);
}

/**************************************************************************
*
*  INITIALIZATION(argc,argv)
*
*	INPUT:	argc,argv	- Argument list
*
*	OUTPUT:	Init serial channel
*		ERROR exit if bad
*
**************************************************************************/
	
void initialization(int argc,char **argv)
{
	unsigned int result;
	int data,port;
	char out[64];
	unsigned int speed;
	char *rtn;
	int bit;
	int parity,i;
	BOOLEAN def_flag;
	
	/*  Init File channels */
	
	out_file = NULL;
	log_file = NULL;

	/*  Tell the user what he's got */

	printf("\n");
	sprintf(out,"RS-232 Terminal Emulator  MOT Version %s",VERSION);
	print_center(out);

	/*  Get the serial port value and init the port */
	
	if( flag_srch(argc,argv,PORT_FLAG,TRUE,&rtn) )
	{
		sscanf(rtn,"%d",&port);
		switch( port ) {
			case 1:	port = COM1;
				print_center("*** USING COM1 ***");
				break;
				
			case 2:	port = COM2;
				print_center("*** USING COM2 ***");
				break;
				
			default: port = COM1;
				print_center("*** USING COM1 ***");
				break;
		}
	}
	else
	{
		port = COM1;
		print_center("*** USING COM1 (-p 1/2) ***");
	}
	
	/*  Set speed */

	if( flag_srch(argc,argv,SPEED_FLAG,TRUE,&rtn) )
	{
		sscanf(rtn,"%d",&speed);
		def_flag = FALSE;
	}
	else
	{
		speed = DEF_SPEED;
		def_flag = TRUE;
	}
	
	switch( speed ) {
		
	case 9600:	speed = BAUD_9600;
			if(def_flag)
print_center("** SPEED = 9600 (-s 9600/4800/2400/1200/600/300/110) **");
			else
				print_center("** SPEED = 9600 **");
			break;
			
	case 4800:	speed = BAUD_4800;
			print_center("** SPEED = 4800 **");
			break;
	
	case 2400:	speed = BAUD_2400;
			print_center("** SPEED = 2400 **");
			break;

	case 1200:	speed = BAUD_1200;
			print_center("** SPEED = 1200 **");
			break;
			
	case 600:	speed = BAUD_600;
			print_center("** SPEED = 600 **");
			break;
			
	case 300:	speed = BAUD_300;
			print_center("** SPEED = 300 **");
			break;
			
	case 110:	speed = BAUD_110;
			print_center("** SPEED = 110 **");
			break;
	
	default:	speed = BAUD_9600;
			print_center("** BAD VALUE SPEED = 9600 **");
			break;
	}
	
	/*  Define BITS */

	if( flag_srch(argc,argv,BIT_FLAG,TRUE,&rtn) )
	{
		sscanf(rtn,"%d",&bit);
		def_flag = FALSE;
	}
	else
	{
		bit = DEF_BIT;
		def_flag = TRUE;
	}
	
	switch( bit ) {
		
	case 7:		bit = _COM_CHR7;
			print_center("** BITS = 7 **");
			break;
			
	case 8:		bit = _COM_CHR8;
			if( def_flag )
				print_center("** BITS = 8 (-b 7/8) **");
			else
				print_center("** BITS = 8 **");
			break;
	
	default:	bit = _COM_CHR8;
			print_center("** BAD VALUE BITS = 8 **");
			break;
	}
	

	/*  Define Parity */

	if( flag_srch(argc,argv,PARITY_FLAG,TRUE,&rtn) )
	{
		parity = toupper( rtn[0] );
		def_flag = FALSE;
	}
	else
	{
		parity = DEF_PARITY;
		def_flag = TRUE;
	}
	
	switch( parity ) {
		
	case 'E':	parity = _COM_EVENPARITY;
			print_center("** PARITY EVEN **");
			break;
			
	case 'N':	parity = _COM_NOPARITY;
			if( def_flag )
			  print_center("* PARITY NONE (-pa EVEN/NONE/ODD) *");
			else
				print_center("** PARITY NONE **");
			break;

	case 'O':	parity = _COM_ODDPARITY;
			print_center("** PARITY ODD **");
			break;

	default:	parity = _COM_NOPARITY;
			print_center("** DEFAULT PARITY NONE **");
			break;

	}

	c = ser_init( port,speed,bit,parity,4096,4096 );
	if( c == NULL )
	{
		print_center("**** SER PORT NOT OPENED ***");
		exit(-1);
	}

	if( flag_srch(argc,argv,LFCR_FLAG,FALSE,&rtn) )
		lfcr = TRUE;
	else
		lfcr = FALSE;
	
	if( lfcr )
		print_center("** CR -> LFCR **");
	else
		print_center("** CR NORMAL (-lf for CR -> LFCR) **");
	
	if( flag_srch( argc,argv,NOECHO_FLAG,FALSE,&rtn) )
		no_echo = TRUE;
	else
		no_echo = FALSE;

	if( no_echo )
		print_center("** NO ECHO ON FILE SEND **");
	else
		print_center("** ECHO NORMAL (-ne for ECHO SUPPRESSION) **");

	/*  If error then exit with a problem */
			
	if( !result )
	{
		print_center("??? PORT PROBLEM ???");
		exit(-1);
	}
}

/***********************************************************************
*
*  ABORT_ACTION()
*
*	INPUT:	User hit <ABORT KEY>
*
*	OUTPUT:	TRUE if user wants to keep running
*		FALSE if not
*
*	ACTION:	Ask if user wants to:
*		1:	Down load a file
*		2:	Capture data
*
************************************************************************/

int abort_action(void )
{
	int ret;
	
	/*  Tell the user what we have for this run */
	
	printf("\n");
	printf("MTERM Actions\n");
	ret = pgm_menu(abort_menu,MENU_CNT(abort_menu) );
	
	return( ret );
}

/***********************************************************************
*  		MENU_ACTIONS
***********************************************************************/
/***********************************************************************
*
*  DOS_ACTION()	- Run a command interperter
*
*       INPUT:  NONE
*
*       OUTPUT: TRUE
*		Run command interperter and return on EXIT
*
***********************************************************************/

int dos_action(void)
{
	int ret;
	
	printf("Type EXIT to return\n");
	ret = spawnlp(P_WAIT,"command.com","command",NULL);
	
	return( TRUE );
}

/***********************************************************************
*
*  LOG_ACTION()	- Open/close log file
*
*       INPUT:  NONE
*
*       OUTPUT: TRUE
*		If log file open close it, if not open one
*
***********************************************************************/

int log_action(void)
{
	static char log_name[64];
	
	/*  (1) Check for log file open or closed */
	
	if( log_file != NULL )
	{
	/*  (2) If open close it */
		
		printf("Closing LOG FILE %s\n",log_name);
		fclose( log_file );
		log_file = NULL;
	}
	else
	{
	/* (3) If closed then open a new one if user wants to */
		
		printf("Open a log file: ");
		gets(log_name);
		if( strlen(log_name) == 0 )
			return( TRUE );
		log_file = fopen( log_name,"w");
		if( log_file == NULL )
			printf("*** COULDN'T OPEN %s as LOG FILE\n",log_name);
	}
	return( TRUE );
}

/***********************************************************************
*
*  XFER_ACTION()	- Open/close transfer file
*
*       INPUT:  NONE
*
*       OUTPUT: TRUE
*		Open a transfer file
*
***********************************************************************/

int xfer_action(void)
{
	static char out_name[64];
	
	/*  (1) Check for out file open or closed */
	
	if( out_file != NULL )
	{
	/*  (2) If open close it */
		
		printf("Closing OUT FILE %s\n",out_name);
		fclose( out_file );
		out_file = NULL;
	}
	else
	{
	/* (3) If closed then open a new one if user wants to */
		
		printf("Open a Transfer file: ");
		gets(out_name);
		if( strlen(out_name) == 0 )
			return( TRUE );
		out_file = fopen( out_name,"r");
		if( out_file == NULL )
			printf("*** COULDN'T OPEN %s as OUT FILE\n",out_name);
	}
	return( TRUE );
}

/***********************************************************************
*
*  QUIT_ACTION()	- Quit if user requests it
*
*       INPUT:  NONE
*
*       OUTPUT: TRUE, if user doesn't want to quit
*		FALSE if quit is desired
*		CHANGE Just return FALSE as we really want to quit
*
***********************************************************************/

int quit_action(void)
{
	char in[20];
	
	/*printf("Quit program: (Y to quit) ");
	gets(in);
	if( in[0] == 'Y' || in[0] == 'y' )
		return( FALSE );
	else
		return( TRUE );*/
	return( FALSE );
}

/***********************************************************************
*               SUPPORT ROUTINES
************************************************************************/
/***********************************************************************
*
*  PRINT_CENTER( str )  - Print this string in the center of the screen
*
*       INPUT:  str     - String to be printed
*
*       OUTPUT: Clear and print the string in the center of the screen
*
*       ASSUME: 24 Lines on screen
*               80 Characters across
*
***********************************************************************/

void print_center( char *str )
{
	int len,down,left;
	int i;

	/*  1) Calculate the # of spaces to move in for this string */

	len = strlen( str );
	left = (80-len)/2;

	/*  2) Print this number of spaces & then the string */

	for( i=0; i<left; i++)
		printf(" ");

	printf("%s\n",str);

}

/***********************************************************************
*
*  ERROR_EXIT( str )  - Print this string & error out
*
*       INPUT:  str     - String to be printed
*
*       OUTPUT: Clear and print the string in the center of the screen
*               Halt and wait for user to hit a key then EXIT
*
*       ASSUME: 80 Characters across
*
***********************************************************************/

void error_exit( char *str )
{
	/* 1) Print the string & output close the serial output */

	print_center( str );
	ser_close(c);

	/* 2) Prompt the user and wait for him to hit a key */

	printf("*** HIT ANY KEY TO EXIT PROGRAM ***\n");

	while( TRUE )
	{
		if( kbhit() )
		{
			getch();
			exit( -1 );
		}
	}
}

/*********************************************************************
*
*  PGM_MENU(menu,n)
*
*	INPUT:	menu	- menu of programs to be run
*		n		- # Programs in menu
*
*	OUTPUT:	Program selected from menu, run as a subroutine
*			Passes it the value form the program menu
*			Returns integer output of program
*
**********************************************************************/

int pgm_menu(MENU *menu,int n)
{
	int i,pgm,result;
	char in[4];
	
	/*  1) Output header then menu */
	
	while( TRUE )
	{
		for( i=0; i<n; i++ )
			printf("%2d - %s\n",i+1,menu[i].label);
		printf("Select 1-%d:\n",n);
		gets(in);
		
	/*  2) Get program value & run it */

		if( strlen(in) != 0 )
		{
			sscanf(in,"%d",&pgm);
			pgm--;
		}
			
		if( pgm >= 0 && pgm < n )
			break;
		else
			return( TRUE );
	}
	
	/*  3) Now run the routine and return */
	
	result = (*menu[pgm].pgm)();
	return( result );
}
	
/***********************************************************************
*
*  FLAG_SRCH(argc,argv,flag,get_value,&rtn )
*
*	INPUT:	argc,argv	- Command line parameters
*		flag		- String (flag) to be found
*		get_value	- TRUE return pointer to next item
*				  FALSE no pointer returned
*		rtn		- Return pointer (NULL if not used or
*				  found)
*
*	OUTPUT:	TRUE if flag found
*		FALSE if not
*		Scan argument list for flag
*		If found then TRUE is returned and if get_value TRUE
*		return rtn pointing to string in argv[] following 
*		the flag
*		NOTE only exact matches will be found
*
**********************************************************************/

int flag_srch(int argc,char **argv,char *flag,int get_value,char **rtn )
{
	int i;
	char *ptr;
	
	/*  Scan through the argv's and look for a matching flag */
	
	for( i=0; i<argc; i++ )
	{
		ptr = argv[i];
		if( strcmp(ptr,flag) == 0 )
		{
		
		/*  Match found, return pointer to the following
			(if requested or NULL if not)
		*/
			i++;
			if( get_value && i < argc )
				*rtn = argv[i];
			else
				*rtn = NULL;
			return( TRUE );
		}
	}
	
	/*  Failure exit here, so return to user with FALSE */
	
	return( FALSE );
}
