#define PPTP_MAGIC 0x1A2B3C4D /* Magic cookie for PPTP datagrams */
#define PPTP_PORT  1723       /* PPTP TCP port number            */
#define PPTP_PROTO 47         /* PPTP IP protocol number         */

#define PPTP_MESSAGE_CONTROL            1
#define PPTP_MESSAGE_MANAGE             2

#define PPTP_VERSION_STRING     "1.00"
#define PPTP_VERSION            0x100
#define PPTP_FIRMWARE_STRING    "0.01"
#define PPTP_FIRMWARE_VERSION   0x001

/* (Control Connection Management) */
#define PPTP_START_CTRL_CONN_RQST       1
#define PPTP_START_CTRL_CONN_RPLY       2
#define PPTP_STOP_CTRL_CONN_RQST        3
#define PPTP_STOP_CTRL_CONN_RPLY        4
#define PPTP_ECHO_RQST                  5
#define PPTP_ECHO_RPLY                  6

/* (Call Management) */
#define PPTP_OUT_CALL_RQST              7
#define PPTP_OUT_CALL_RPLY              8
#define PPTP_IN_CALL_RQST               9
#define PPTP_IN_CALL_RPLY               10
#define PPTP_IN_CALL_CONNECT            11
#define PPTP_CALL_CLEAR_RQST            12
#define PPTP_CALL_CLEAR_NTFY            13

/* (Error Reporting) */
#define PPTP_WAN_ERR_NTFY               14

/* (PPP Session Control) */
#define PPTP_SET_LINK_INFO              15

/* (Framing capabilities for msg sender) */
#define PPTP_FRAME_ASYNC        1
#define PPTP_FRAME_SYNC         2
#define PPTP_FRAME_ANY          3

/* (Bearer capabilities for msg sender) */
#define PPTP_BEARER_ANALOG      1
#define PPTP_BEARER_DIGITAL     2
#define PPTP_BEARER_ANY         3

struct pptp_header {
  u_int16_t length;       /* message length in octets, including header */
  u_int16_t pptp_type;    /* PPTP message type. 1 for control message.  */
  u_int32_t magic;        /* this should be PPTP_MAGIC.                 */
  u_int16_t ctrl_type;    /* Control message type (0-15)                */
  u_int16_t reserved0;    /* reserved.  MUST BE ZERO.                   */
};

struct pptp_start_ctrl_conn { /* for control message types 1 and 2 */
  struct pptp_header header;

  u_int16_t version;      /* PPTP protocol version.  = PPTP_VERSION     */
  u_int8_t  result_code;  /* these two fields should be zero on rqst msg*/
  u_int8_t  error_code;   /* 0 unless result_code==2 (General Error)    */
  u_int32_t framing_cap;  /* Framing capabilities                       */
  u_int32_t bearer_cap;   /* Bearer Capabilities                        */
  u_int16_t max_channels; /* Maximum Channels (=0 for PNS, PAC ignores) */
  u_int16_t firmware_rev; /* Firmware or Software Revision              */
  u_int8_t  hostname[64]; /* Host Name (64 octets, zero terminated)     */
  u_int8_t  vendor[64];   /* Vendor string (64 octets, zero term.)      */
  /* MS says that end of hostname/vendor fields should be filled with   */
  /* octets of value 0, but Win95 PPTP driver doesn't do this.          */
};

struct pptp_out_call_rqst { /* for control message type 7 */
  struct pptp_header header;
  u_int16_t call_id;      /* Call ID (unique id used to multiplex data)  */
  u_int16_t call_sernum;  /* Call Serial Number (used for logging)       */
  u_int32_t bps_min;      /* Minimum BPS (lowest acceptable line speed)  */
  u_int32_t bps_max;      /* Maximum BPS (highest acceptable line speed) */
  u_int32_t bearer;       /* Bearer type                                 */
  u_int32_t framing;      /* Framing type                                */
  u_int16_t recv_size;    /* Recv. Window Size (no. of buffered packets) */
  u_int16_t delay;        /* Packet Processing Delay (in 1/10 sec)       */
  u_int16_t phone_len;    /* Phone Number Length (num. of valid digits)  */
  u_int16_t reserved1;    /* MUST BE ZERO                                */
  u_int8_t  phone_num[64]; /* Phone Number (64 octets, null term.)       */
  u_int8_t subaddress[64]; /* Subaddress (64 octets, null term.)         */
};

struct pptp_out_call_rply { /* for control message type 8 */
  struct pptp_header header;
  u_int16_t call_id;      /* Call ID (used to multiplex data over tunnel)*/
  u_int16_t call_id_peer; /* Peer's Call ID (call_id of pptp_out_call_rqst)*/
  u_int8_t  result_code;  /* Result Code (1 is no errors)                */
  u_int8_t  error_code;   /* Error Code (=0 unless result_code==2)       */
  u_int16_t cause_code;   /* Cause Code (addt'l failure information)     */
  u_int32_t speed;        /* Connect Speed (in BPS)                      */
  u_int16_t recv_size;    /* Recv. Window Size (no. of buffered packets) */
  u_int16_t delay;        /* Packet Processing Delay (in 1/10 sec)       */
  u_int32_t channel;      /* Physical Channel ID (for logging)           */
};

struct pptp_set_link_info {   /* for control message type 15 */
  struct pptp_header header;
  u_int16_t call_id_peer; /* Peer's Call ID (call_id of pptp_out_call_rqst) */
  u_int16_t reserved1;    /* MUST BE ZERO                                   */
  u_int32_t send_accm;    /* Send ACCM (for PPP packets; default 0xFFFFFFFF)*/
  u_int32_t recv_accm;    /* Receive ACCM (for PPP pack.;default 0xFFFFFFFF)*/
};

#define PPTP_GRE_PROTO  0x880B
#define PPTP_GRE_VER    0x1

#define PPTP_GRE_FLAG_C 0x80
#define PPTP_GRE_FLAG_R 0x40
#define PPTP_GRE_FLAG_K 0x20
#define PPTP_GRE_FLAG_S 0x10
#define PPTP_GRE_FLAG_A 0x80

#define PPTP_GRE_IS_C(f) ((f)&PPTP_GRE_FLAG_C)
#define PPTP_GRE_IS_R(f) ((f)&PPTP_GRE_FLAG_R)
#define PPTP_GRE_IS_K(f) ((f)&PPTP_GRE_FLAG_K)
#define PPTP_GRE_IS_S(f) ((f)&PPTP_GRE_FLAG_S)
#define PPTP_GRE_IS_A(f) ((f)&PPTP_GRE_FLAG_A)

struct pptp_gre_header {
  u_int8_t flags;               /* bitfield */
  u_int8_t ver;                 /* should be PPTP_GRE_VER (enhanced GRE) */
  u_int16_t protocol;           /* should be PPTP_GRE_PROTO (ppp-encaps) */
  u_int16_t payload_len;        /* size of ppp payload, not inc. gre header */
  u_int16_t call_id;            /* peer's call_id for this session */
  u_int32_t seq;                /* sequence number.  Present if S==1 */
  u_int32_t ack;                /* seq number of highest packet recieved by */
                                /*  sender in this session */
};

#define PACKET_MAX 8196

#define PPP_ADDRESS                     0xFF
#define PPP_CONTROL                     0x03

/* PPP Protocols */
#define PPP_PROTO_LCP                   0xc021
#define PPP_PROTO_CHAP                  0xc223

/* LCP Codes */
#define PPP_LCP_CODE_CONF_RQST          1
#define PPP_LCP_CODE_CONF_ACK           2
#define PPP_LCP_CODE_IDENT              12

/* LCP Config Options */
#define PPP_LCP_CONFIG_OPT_AUTH         3
#define PPP_LCP_CONFIG_OPT_MAGIC        5
#define PPP_LCP_CONFIG_OPT_PFC          7
#define PPP_LCP_CONFIG_OPT_ACFC         8

/* Auth Algorithms */
#define PPP_LCP_AUTH_CHAP_ALGO_MSCHAP   0x80

/* CHAP Codes */
#define PPP_CHAP_CODE_CHALLENGE                 1
#define PPP_CHAP_CODE_RESPONCE                  2
#define PPP_CHAP_CODE_SUCESS                    3
#define PPP_CHAP_CODE_FAILURE                   4
#define PPP_CHAP_CODE_MSCHAP_PASSWORD_V1        5
#define PPP_CHAP_CODE_MSCHAP_PASSWORD_V2        6

#define PPP_CHAP_CHALLENGE_SIZE         8
#define PPP_CHAP_RESPONCE_SIZE          49

#define MSCHAP_ERROR    "E=648 R=0"

struct ppp_header {
    u_int8_t address;
    u_int8_t control;
    u_int16_t proto;
};

struct ppp_lcp_chap_header {
  u_int8_t code;
  u_int8_t ident;
  u_int16_t length;
};

struct ppp_lcp_packet {
  struct ppp_header ppp;
  struct ppp_lcp_chap_header lcp;
};

struct ppp_lcp_chap_auth_option {
  u_int8_t type;
  u_int8_t length;
  u_int16_t auth_proto;
  u_int8_t algorithm;
};

struct ppp_lcp_magic_option {
  u_int8_t type;
  u_int8_t length;
  u_int32_t magic;
};

struct ppp_lcp_pfc_option {
  u_int8_t type;
  u_int8_t length;
};

struct ppp_lcp_acfc_option {
  u_int8_t type;
  u_int8_t length;
};

struct ppp_chap_challenge {
  u_int8_t size;
  union {
    unsigned char challenge[8];
    struct {
      unsigned char lanman[24];
      unsigned char nt[24];
      u_int8_t flag;
    } responce;
  } value;
  /* name */
};

struct ppp_mschap_change_password {
  char old_lanman[16];
  char new_lanman[16];
  char old_nt[16];
  char new_nt[16];
  u_int16_t pass_length;
  u_int16_t flags;
};

#define ppp_chap_responce       ppp_chap_challenge


