/*
 * chntpw.c - Offline Password Edit Utility for NT 3.51 & 4.0 SAM database.
 * 
 * Copyright (c) 1997 Petter Nordahl-Hagen.
 * Freely distributable in source or binary for noncommercial purposes.
 * 
 * Uses (links with) DES SSL Library by Eric Young (eay@mincom.oz.au)
 * and MD4 routines by RSA.
 * 
 * Part of some routines, information and ideas taken from
 * pwdump by Jeremy Allison.
 *
 * Some stuff (like proto.h) from NTCrack by Jonathan Wilkins.
 * 
 * Please see the COPYING file for more details on
 * copyrights & credits.
 *  
 * THIS SOFTWARE IS PROVIDED BY PETTER NORDAHL-HAGEN `AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */ 

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

/* These are from the DESlib & MD packages, see Makefile & README */
#include <des.h>
#include <mddriver/global.h>
#include <md4/md4.h>

#include "proto.h"

const char version[] = "chntpw version PRE-1 970611, (c) Petter N Hagen";

int f;        /* Our main file, usually winnt/system32/config/sam */
off_t sz = 0; /* File (and buffer) size */
char *buf;    /* Our main buffer */
int dirty = 0; /* True if buffer dirty */



/* stdin input */

int fmyinput(char *prmpt, char *ibuf, int maxlen)
{
   
   printf("%s",prmpt);
   
   fgets(ibuf,maxlen+1,stdin);
   
   ibuf[strlen(ibuf)-1] = 0;
   
   return(strlen(ibuf));
}

/* Open & read the file into our workbuffer */

void openfile(char *fnam)
{
   char errb[100];
   struct stat sbuf;
   int got = 0;
   
   if ( (f = open(fnam,O_RDWR)) == -1) {
      sprintf(errb,"chntpw: open(): %s",fnam);
      perror(errb);
      exit(1);
   }
   if ( fstat(f,&sbuf) ) {
      sprintf(errb,"chntpw: stat(): %s",fnam);
      perror("stat()");
      exit(1);
   }
   
   sz = sbuf.st_size;
   printf("opened %s, size = %u [%0x]\n",fnam,sz,sz);

   buf = (char *)malloc(sz+64);
   if (!buf) {
      perror("chntpw: malloc of buffer");
      exit(1);
   }
   
   got = read(f,buf,sz);
   if (got < 0) {
      perror("chntpw: file read");
      exit(1);
   }
   if (got != sz) {
      fprintf(stderr,"file read: expected %d bytes, got %d bytes. HELP!\n",sz,got);
      exit(1);
   }

   printf("Read ok.\n");
 
}
   
/* Write the buffer back to the file */

void writefile(void)
{
   int wl;
   
   if ( lseek(f,0,SEEK_SET) < 0 ) {
      perror("fseek to start of file (for write) failed");
      exit(1);
   }
   
   wl = write(f,buf,sz);
   if (wl < 0) {
      perror("chntpw: write of buf to file failed");
      exit(1);
   }
   
   printf("Wrote %d [%0x] bytes.\n",wl,wl);
   
   if (wl < sz) {
      fprintf(stderr,"ERROR: Only wrote %d out of %d bytes!\n",wl,sz);
      exit(1);
   }
   
   close(f);
   
   printf("File closed, all OK!\n");
}
   
/* Print len number of hexbytes */

void hexprnt(char *s, unsigned char *bytes, int len)
{
int i;

   printf("%s",s);
   for (i = 0; i < len; i++) {
      printf("%02x ",bytes[i]);
   }
   printf("\n");
}


/* HexDump all or a part of some buffer */

void hexdump(char *hbuf, int start, int stop)
{
   char c;
   int diff,i;
   
   while (start < stop ) {
      
      diff = stop - start;
      if (diff > 16) diff = 16;
      
      printf("%05X: ",start);

      for (i = 0; i < diff; i++) {
	 printf("%02X ",(unsigned char)*(hbuf+start+i));
      }
      for (i = diff; i < 16; i++) printf("   ");
      for (i = 0; i < diff; i++) {
	 c = *(hbuf+start+i);
	 printf("%c", isprint(c) ? c : '.');
      }
      printf("\n");
      start += 16;
   }
}


/* General search routine, find something in something else */
int find_in_buf(char *what, int len, int start)
{
   int i;
   
   for (; start < sz; start++) {
      for (i = 0; i < len; i++) {
	if (*(buf+start+i) != *(what+i)) break;
      }
      if (i == len) return(start);
   }
   return(0);
}

/* Get INTEGER from memory. This is probably low-endian specific? */
int get_int( char *array )
{
	return ((array[0]&0xff) + ((array[1]<<8)&0xff00) +
		   ((array[2]<<16)&0xff0000) +
		   ((array[3]<<24)&0xff000000));
}


/* Quick and dirty UNICODE to std. ascii */

void cheap_uni2ascii(char *src, char *dest, int l)
{
   
   for (; l > 0; l -=2) {
      *dest = *src;
      dest++; src +=2;
   }
   *dest = 0;
}


/* Quick and dirty ascii to unicode */

void cheap_ascii2uni(char *src, char *dest, int l)
{
   for (; l > 0; l--) {
      *dest++ = *src++;
      *dest++ = 0;

   }
}

/* Zero out string for lanman passwd, then uppercase
 * the supplied password and put it in here */

void make_lanmpw(char *p, char *lm, int len)
{
   int i;
   
   for (i=0; i < 15; i++) lm[i] = 0;
   for (i=0; i < len; i++) lm[i] = toupper(p[i]);
}

/*
 * Convert a 7 byte array into an 8 byte des key with odd parity.
 */

void str_to_key(unsigned char *str,unsigned char *key)
{
	void des_set_odd_parity(des_cblock *);
	int i;

	key[0] = str[0]>>1;
	key[1] = ((str[0]&0x01)<<6) | (str[1]>>2);
	key[2] = ((str[1]&0x03)<<5) | (str[2]>>3);
	key[3] = ((str[2]&0x07)<<4) | (str[3]>>4);
	key[4] = ((str[3]&0x0F)<<3) | (str[4]>>5);
	key[5] = ((str[4]&0x1F)<<2) | (str[5]>>6);
	key[6] = ((str[5]&0x3F)<<1) | (str[6]>>7);
	key[7] = str[6]&0x7F;
	for (i=0;i<8;i++) {
		key[i] = (key[i]<<1);
	}
	des_set_odd_parity((des_cblock *)key);
}

/*
 * Function to convert the RID to the first decrypt key.
 */

void sid_to_key1(unsigned long sid,unsigned char deskey[8])
{
	unsigned char s[7];

	s[0] = (unsigned char)(sid & 0xFF);
	s[1] = (unsigned char)((sid>>8) & 0xFF);
	s[2] = (unsigned char)((sid>>16) & 0xFF);
	s[3] = (unsigned char)((sid>>24) & 0xFF);
	s[4] = s[0];
	s[5] = s[1];
	s[6] = s[2];

	str_to_key(s,deskey);
}

/*
 * Function to convert the RID to the second decrypt key.
 */

void sid_to_key2(unsigned long sid,unsigned char deskey[8])
{
	unsigned char s[7];
	
	s[0] = (unsigned char)((sid>>24) & 0xFF);
	s[1] = (unsigned char)(sid & 0xFF);
	s[2] = (unsigned char)((sid>>8) & 0xFF);
	s[3] = (unsigned char)((sid>>16) & 0xFF);
	s[4] = s[0];
	s[5] = s[1];
	s[6] = s[2];

	str_to_key(s,deskey);
}

/* DES encrypt, for LANMAN */

void E1(uchar *k, uchar *d, uchar *out)
{
  des_key_schedule ks;
  des_cblock deskey;

  str_to_key(k,(uchar *)deskey);
#ifdef __FreeBSD__
  des_set_key(&deskey,ks);
#else /* __FreeBsd__ */
  des_set_key((des_cblock *)deskey,ks);
#endif /* __FreeBsd__ */
  des_ecb_encrypt((des_cblock *)d,(des_cblock *)out, ks, DES_ENCRYPT);
}

/* Decode the V-struct, and change the password
 * vofs - offset into SAM buffer, start of V struct
 * rid - the users RID, required for the DES decrypt stage
 *       If rid==0 it will try to extract the rid out of the V struct
 *       else the supplied one will be used
 *
 * Some of this is ripped & modified from pwdump by Jeremy Allison
 * 
 */
char *seek_n_destroy(int vofs, int rid, int vlen, int stat)
{
   
   uchar x1[] = {0x4B,0x47,0x53,0x21,0x40,0x23,0x24,0x25};
   char magic[] = {0x24, 0, 0x44 };
   int pl,ridofs,vend;
   char *vp;
   static char username[128],fullname[128];
   char comment[128],homedir[128],md4[32],lanman[32];
   char newunipw[34], newp[20], despw[20], newlanpw[16], newlandes[20];
   char yn[4];
   int username_offset,username_len;
   int fullname_offset,fullname_len;
   int comment_offset,comment_len;
   int homedir_offset,homedir_len;
   int pw_offset,i;
   des_key_schedule ks1, ks2;
   des_cblock deskey1, deskey2;

   MD4_CTX context;
   unsigned char digest[16];
/*   
   printf("V record found at offset: %05x\n",vofs);

   hexdump(buf, vofs, vofs+0x305);
   printf("\n");
*/ 
   vp = buf + vofs;
   username_offset = get_int(vp + 0xC);
   username_len = get_int(vp + 0x10); 
   fullname_offset = get_int(vp + 0x18);
   fullname_len = get_int(vp + 0x1c);
   comment_offset = get_int(vp + 0x24);
   comment_len = get_int(vp + 0x28);
   homedir_offset = get_int(vp + 0x48);
   homedir_len = get_int(vp + 0x4c);
   pw_offset = get_int(vp + 0x9c);

   *username = 0;
   *fullname = 0;
   *comment = 0;
   *homedir = 0;
   
   if(username_len <= 0 || username_len > vlen ||
      username_offset <= 0 || username_offset >= vlen ||
      comment_len < 0 || comment_len > vlen   ||
      fullname_len < 0 || fullname_len > vlen ||
      homedir_offset < 0 || homedir_offset >= vlen ||
      comment_offset < 0 || comment_offset >= vlen ||
      pw_offset < 0 || pw_offset >= vlen)
     {
	if (stat != 1) printf("Not a legal struct? (negative lengths)\n");
	return(NULL);
     }

   username_offset += 0xCC;
   fullname_offset += 0xCC;
   comment_offset += 0xCC;
   homedir_offset += 0xCC;
   pw_offset += 0xCC;
   
   cheap_uni2ascii(vp + username_offset,username,username_len);
   cheap_uni2ascii(vp + fullname_offset,fullname,fullname_len);
   cheap_uni2ascii(vp + comment_offset,comment,comment_len);
   cheap_uni2ascii(vp + homedir_offset,homedir,homedir_len);
   
   if (!rid) {
      ridofs = find_in_buf(magic,3, vofs);
      if (!ridofs || ridofs > (vofs+vlen)) {
	 printf("No RID found in struct..\n");
	 rid = 0;
      } else {
	 rid = get_int(buf+ridofs+0x1e);
      }
   } else {
     printf("WARNING: Overriding RID detection with supplied RID [%04x]\n",rid);
   }
   if (stat) {
      printf("Offset: %05x, RID: %04x, Username: <%s>%s\n",
	     vofs, rid, username,  ( !rid ? ", account locked?" : (pw_offset+0x20 >= vlen) ? ", *BLANK password*" : "")  );
      return(username);
   }

   printf("[file offset: %05x]\n",vofs);
   printf("RID     : %04d [%04x]\n",rid,rid);
   printf("Username: %s\n",username);
   printf("fullname: %s\n",fullname);
   printf("comment : %s\n",comment);
   printf("homedir : %s\n\n",homedir);
   
   if (pw_offset+0x20 >= vlen) {
      printf("** This user has a BLANK password! (Unable to change it, let NT do it instead :)\n\n");
      return(0);
   }
   
   if (!rid) {
      printf("No RID (lower part of the userid) was found for this user,\n");
      printf("(it's in there somewhere, but I don't know how to find it yet)\n");
      printf("This value is required to encrypt the password properly!\n");
      printf("The lack of the RID in this struct usually means that there\n"
	     "is no password for this user,\n");
      printf("which again usually implies a locked account. Which I cannot unlock.\n");
      
      pl = fmyinput("Please give a RID (decimal value) or nothing to quit : ",newp,16);
      if (!pl) return(0);
      rid = atoi(newp);
      printf("RID = %4d [%04x]\n\n",rid,rid);
   }
   
   hexprnt("Crypted NT pw: ",(vp+pw_offset+16),16);
   hexprnt("Crypted LM pw: ",(vp+pw_offset),16);

   /* Get the two decrpt keys. */
   sid_to_key1(rid,(unsigned char *)deskey1);
   des_set_key((des_cblock *)deskey1,ks1);
   sid_to_key2(rid,(unsigned char *)deskey2);
   des_set_key((des_cblock *)deskey2,ks2);
   
   /* Decrypt the NT md4 password hash as two 8 byte blocks. */
   des_ecb_encrypt((des_cblock *)(vp+pw_offset+16 ),
		   (des_cblock *)md4, ks1, DES_DECRYPT);
   des_ecb_encrypt((des_cblock *)(vp+pw_offset+16 + 8),
		   (des_cblock *)&md4[8], ks2, DES_DECRYPT);

   /* Decrypt the lanman password hash as two 8 byte blocks. */
   des_ecb_encrypt((des_cblock *)(vp+pw_offset),
		   (des_cblock *)lanman, ks1, DES_DECRYPT);
   des_ecb_encrypt((des_cblock *)(vp+pw_offset + 8),
		   (des_cblock *)&lanman[8], ks2, DES_DECRYPT);
   
   
   hexprnt("MD4 hash     : ",md4,16);
   hexprnt("LANMAN hash  : ",lanman,16);
   
   pl = fmyinput("\nPlease enter new password\nor nothing to leave unchanged: ",newp,16);
   
   printf("password: [%s] have length %d\n",newp,pl);

   if (!pl) { printf("Nothing changed.\n"); return(0); }
   
   cheap_ascii2uni(newp,newunipw,pl);
   
   make_lanmpw(newp,newlanpw,pl);

   printf("Ucase Lanman: %s\n",newlanpw);
   
   MD4Init (&context);
   MD4Update (&context, newunipw, pl<<1);
   MD4Final (digest, &context);
   
   hexprnt("\nNEW MD4 hash    : ",digest,16);
   
   E1(newlanpw,   x1, lanman);
   E1(newlanpw+7, x1, lanman+8);
   
   hexprnt("NEW LANMAN hash : ",lanman,16);
   
   /* Encrypt the NT md4 password hash as two 8 byte blocks. */
   des_ecb_encrypt((des_cblock *)digest,
		   (des_cblock *)despw, ks1, DES_ENCRYPT);
   des_ecb_encrypt((des_cblock *)(digest+8),
		   (des_cblock *)&despw[8], ks2, DES_ENCRYPT);

   des_ecb_encrypt((des_cblock *)lanman,
		   (des_cblock *)newlandes, ks1, DES_ENCRYPT);
   des_ecb_encrypt((des_cblock *)(lanman+8),
		   (des_cblock *)&newlandes[8], ks2, DES_ENCRYPT);

      
   hexprnt("NEW DES crypt   : ",despw,16);
   hexprnt("NEW LANMAN crypt: ",newlandes,16);
  
   fmyinput("\nDo you really wish to change it? (y/n) [n] ",yn,2);
   
   if (*yn == 'y') {
      for (i = 0; i < 16; i++) {
	 (unsigned char)*(vp+pw_offset+16+i) = despw[i];
	 (unsigned char)*(vp+pw_offset+i) = newlandes[i];
      }
      hexprnt("Pw in buffer: ",(vp+pw_offset+16),16);
      hexprnt("Lm in buffer: ",(vp+pw_offset),16);
      dirty = 1;
      printf("\n");
   } else {
      printf("Nothing changed.\n");
   }
   
   printf("\n");
   return(username);
}

void skipspace(char **c)
{
   while( **c == ' ' ) (*c)++;
}

int gethex(char **c)
{
   int value;
   
   skipspace(c);
   
   if (!(**c)) return(0);

   sscanf(*c,"%x",&value);

   while( **c != ' ' && (**c)) (*c)++;

   return(value);
}
   
/* Get a string of HEX bytes (space separated),
 * or if first char is ' get an ASCII string instead.
 */

int gethexorstr(char **c, char *wb)
{
   int l = 0;
   
   skipspace(c);
   
   if ( **c == '\'') {
      (*c)++;
      while ( **c ) {
	 *(wb++) = *((*c)++);
	 l++;
      }
   } else {
      do {
	 *(wb++) = gethex(c);
	 l++;
	 skipspace(c);
      } while ( **c );
   }
   return(l);
}

/* Simple buffer debugger */

void debugit(void)
{
   
   char inbuf[100],whatbuf[100],*bp;

   int to,from,l,i,j,wlen,cofs = 0;
   
   printf("Buffer debugger. '?' for help.\n");
   
   while (1) {
      l = fmyinput(".",inbuf,90);
      bp = inbuf;

      skipspace(&bp);

      if (l > 0 && *bp) {
	 switch(*bp) {
	  case 'd' :
	    bp++;
	    if (*bp) {
	       from = gethex(&bp);
	       to   = gethex(&bp);
	    } else {
	       from = cofs; to = 0;
	    }
	    if (to == 0) to = from + 0x100;
	    if (to > sz) to = sz;
	    hexdump(buf,from,to);
	    cofs = to;
	    break;
	  case 'q':
	    dirty = 0;
	    return;
	    break;
	  case 's':
	    if (!dirty) printf("Buffer has not changed, no need to write..\n");
	    return;
	    break;
	  case 'h':
	    bp++;
	    if (*bp == 'a') {
	       from = 0;
	       to = sz;
	       bp++;
	    } else {
	       from = gethex(&bp);
	       to   = gethex(&bp);
	    }
	    wlen = gethexorstr(&bp,whatbuf);
	    if (to > sz) to = sz;
	    printf("from: %x, to: %x, wlen: %d\n",from,to,wlen);
	    for (i = from; i < to; i++) {
	       for (j = 0; j < wlen; j++) {
		  if ( *(buf+i+j) != *(whatbuf+j)) break;
	       }
	       if (j == wlen) printf("%06x ",i);
	    }
	    printf("\n");
	    break;
	  case ':':
	    bp++;
	    if (!*bp) break;
	    from = gethex(&bp);
	    wlen = gethexorstr(&bp,whatbuf);
	    
	    printf("from: %x, wlen: %d\n",from,wlen);

	    memcpy(buf+from,whatbuf,wlen);
	    dirty = 1;
	    break;
	  case 'p':
	    j = 0;
	    if (*(++bp) != 0) {
	       from = gethex(&bp);
	    }
	    if (*(++bp) != 0) {
	       j = gethex(&bp);
	    }
	    printf("from: %x, rid: %x\n",from,j);
	    seek_n_destroy(from,j,500,0);
	    break;
	  case '?':
	    printf("d [<from>] [<to>] - dump buffer within range (given as offset)\n");
	    printf(": <offset> <hexbyte> [<hexbyte> ...] - change bytes\n");
	    printf("h <from> <to> <hexbyte> [<hexbyte> ...] - hunt (search) for bytes\n");
	    printf("ha <hexbyte> [<hexbyte] - Hunt all (whole buffer)\n");
	    printf("p [<offset>] [<rid/sid>] - Change password in V struct at this offset\n");
	    printf("s - save & quit\n");
	    printf("q - quit (no save)\n");
	    printf("  instead of <hexbyte> etc. you may give 'string to enter/search a string\n");
	    break;
	  default:
	    printf("?\n");
	    break;
	 }
      }
   }
}

/* This one tries to find the start of the next chain.
 * A chain seems to start with B2 B2 xx xx xx xx
 * where x is a negative value which when turned positive
 * will describe the lenght of the next segment.
 * The first 32 bytes part of each "page" (4k)
 * is the page header, so we skip that.
 * Also seek past lengths that are impossiple (max len ~ page len)
 * (yup, this is a hack..)
 */

int find_vorhbin(int f)
{
   int v, h;
   do {
      if (f >= sz-4) return(0);
      if ( (f & 0xff) < 0x20) f = ((f & 0x7fffff00) | 0x20);
      v = get_int(buf+f);
      f += 4;
   } while ( (-v > 0xfe0 || -v <= 1) );
   return(f-4);
}
  

/* 
 * Walk through the registry (with _some_ guessing..)
 *  and list out all or seek a specific username
 * 
 * The first page contains only info on the file/hive,
 * so we skip that.
 * The data is organized in "chains", with a 32bit length
 * first (made negative), then the data, then a new length
 * indicator for the next chunk. A lenght of (-)1 (FF FF FF FF)
 * indicates end of chain. The problem is, I don't know
 * how to find the start of the next chain in a reliable
 * way (except when there's no more space in the current "page")
 * .. see above..
 * If the chunk is above a certain size, we simply try it,
 * and see if the data it contains matches the rules of
 * the password/userinfo structure.
 * (very hackish..)
 */

void seek_vs(int verb,char *who)
{
   int vofs = 0x1020, seglen, prevofs, tot=0;
   char *uname;

   while ( (vofs = find_vorhbin(vofs)) ) {
      if (verb) printf("Possible chain found at %0x\n",vofs);
      seglen = -get_int(buf+vofs);
      while(seglen > 0 && seglen < 0x0fe0) {
	 while ( (seglen & 3) ) seglen++;
	 tot += seglen;
	 if (verb) printf("Segment has length: %d [%0x]\n",seglen,seglen);
	 if (verb) hexdump(buf,vofs,vofs+seglen+4);
	 if (seglen < 0x150) {
	    if (verb) printf("Too short to bee passwd struct!\n");
	 } else {
	    if ( (uname = seek_n_destroy(vofs+4,0,seglen,verb+1)) ) {
	       if (who && !strcasecmp(uname,who)) {
		  seek_n_destroy(vofs+4,0,seglen,0);   /* Do the password change */
		  return;
	       }
	    }
	 }
	 vofs += seglen;
	 seglen = -get_int(buf+vofs);
      }
      if (verb) printf("End of chain. Looking for next chain.\n");
      prevofs = vofs;
   }
   if (verb) printf("Trace ended at offset: %x\n",prevofs);
   if (verb) printf("Total length covered: %d [%x]\n",tot,tot);
}

void usage(void) {
   printf("chntpw: change password of a user in a NT SAM file\n"
	  "chntpw [OPTIONS] <samfile>\n"
	  " -h          This message\n"
	  " -u <user>   Username to change, Administrator is default\n"
	  " -l          (try to) list all users in SAM file\n"
	  " -i          Interactive. List users (as -l) then ask for username to change\n"
	  " -d          Enter buffer debugger instead (hex editor)\n"
          " -t          Trace. Show hexdump of structs/segments. (debug function)\n"
          "See readme file on how to extract/read/write the NT's SAM file\n"
	  "if it's on an NTFS partition!\n"
          "Source/binary freely distributable. See README/COPYING for details\n"
	  "(Contains DESlib code (c) Eric Young)\n"
          "NOTE: This is a VERY preliminary release! You are on your own!\n"
	  );
}

int main(int argc, char **argv)
{
   
   int vofs;
   int dodebug = 0, list = 1, inter = 0,il;
   extern int opterr,optind;
   extern char* optarg;
   char *filename,c;
   char *who = "Administrator";
   char iwho[20];
   
   char *options = "idhltu:";
   
   printf("%s\n",version);
   while((c=getopt(argc,argv,options)) > 0) {
      switch(c) {
       case 'd': dodebug = 1; break;
       case 'l': list = 1; who = 0; break;
       case 't': list = 2; who = 0; break;
       case 'i': list = 1; who = 0; inter = 1; break;
       case 'u': who = optarg; list = 1; break;
       case 'h': usage(); exit(0); break;
       default: usage(); exit(1); break;
      }
   }
   filename=argv[optind];
   if (!filename || !*filename) {
      usage(); exit(1);
   }
     
   openfile(filename);
   
   if (dodebug) debugit();
   else {
      seek_vs(list-1,who);
   }

   while (inter) {
      il = fmyinput("\nUsername to change (! to quit): [administrator] ",iwho,14);
      if (il == 1 && *iwho == '!') { inter = 0; break; }
      if (il == 0) strcpy(iwho,"administrator");
      seek_vs(list-1,iwho);
   }
   if (dirty) writefile();
   
   return(dirty ? 2 : 0);

}

