/*
 *  $Id: listener.c,v 1.4 1998/12/05 18:24:52 route Exp $
 *
 *  Firewalk
 *  listener implementation
 *  listener.c - Packet reading back-end
 *
 *  Copyright (c) 1998 Mike D. Schiffman <mds@es2.net>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include "./firewalk.h"
#include "./packet.h"
#include "./signal.h"
#include "./listener.h"
#include "./p_cap.h"


int
listener(struct firepack *fp)
{
    struct ip *ip_hdr;
    struct pcap_pkthdr pc_hdr;
    int result;

    /*
     *  We timed out waiting a response.  This hopefully means the packet was
     *  dropped by the gateway and our program works as expected.  Of course,
     *  a packet could just be dropped due to the connectionless nature of IP.
     *  Either way, we got nothing.
     */
    if (sigsetjmp(env, 1))
    {
        if (!fp->quiet)
        {
            fprintf(stdout, " *");
            fflush(stdout);
            if (fp->red_cnt > 1) fp->red_flag = 1;
        }
        return (NO_REPLY);
    }

    /*
     *  Setup our signal handler to handle a response timeout.
     */
    if (catch_sig(SIGALRM, expired_timer) == -1)
    {
        perror("udpwalk: cannot catch SIGALRM");
        destroy_listener(fp);
        exit(1);
    }
    alarm(fp->pcap_timeout);

    result = 0;
    /*
     *  Read packets from the pcap device.
     */
    for (;packet = GET_RAW(fp->pd, &pc_hdr);)
    {
        /*
         *  Align the IP header beyond the link header.
         */
        ip_hdr = (struct ip *)(packet + fp->packet_offset);

        /*
         *  Make sure this is a transport protocol we are interested in.
         */
        switch (ip_hdr->ip_p)
        {
            case IPPROTO_ICMP:
                switch (result = icmp_verify(packet, fp))
                {
                    case UNREACH_REPLY:
                    case EXPIRED_REPLY:
                    case UNREACH_DEST_REPLY:
                    case EXPIRED_DEST_REPLY:
                    case UNREACH_GW_REPLY:
                    case EXPIRED_GW_REPLY:
                        alarm(0);
                        return (result);
                    default:
                        break;
                }
            default:
                continue;
        }
    }
    destroy_listener(fp);
    exit(0);
}


void
set_up_listener(struct firepack *fp, u_char *device)
{
    fp->pd = init_pcap_machine(LOW_SNAPLEN, PROMISCUITY_DECISION, 500, device);
    fp->packet_offset = get_datalink_offset(fp->pd);
/*    set_pcap_filter(fp->pd, device, FILTER); */
}


void
destroy_listener(struct firepack *fp)
{
    pcap_close(fp->pd);
}


/* EOF */
