/*
 *  $Id: main.c,v 1.3 1998/11/08 01:37:34 route Exp $
 *
 *  Firewalk
 *  firewalk client
 *  main.c - main control logic
 *
 *  Copyright (c) 1998 Mike D. Schiffman <mds@es2.net>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include "./main.h"
#include "./firewalk.h"
#include "./firepack.h"
#include "./listener.h"
#include "./ifaddrlist.h"
#include "./version.h"


int
main(int argc, char *argv[])
{
    int c, i, fd = -2;
#if (SOLARIS)
    /*
     *  We have no interface selection code under Solaris.  So, we just
     *  hardcode this relatively common device name in here.
     */
    u_char *device = "le0";
#else
    u_char *device = NULL;
#endif
    u_char *w_prots[] = {"UDP", "TCP", "ICMP", 0};
#if (__DEBUG)
    struct protoent *p_ent;
#endif

    if (geteuid() && getuid())
    {
        fprintf(stderr, "root access required\n");
        exit(1);
    }
    /*
     *  Get some memory for our firepack.
     */
    fp = (struct firepack *)malloc(sizeof(struct firepack));
    if (!fp)
    {
        perror("main: malloc");
        exit(1);
    }

    /*
     *  Initialize the firepack structure with the program defaults.
     */
    init_firepack();
    /*
     *  Process commandline arguments.
     */
    while ((c = getopt(argc, argv, "hI:i:no:P:p:qr:S:s:T:t:v")) != EOF)
    {
        switch (c)
        {
            case 'h':   /* Help */
                usage(argv[0]);
                break;
            case 'I':   /* The initial port to use for TTL ramping */
                fp->init_probe_port = atoi(optarg);
                if (fp->init_probe_port > 65535 || fp->init_probe_port < 1)
                {
                    fprintf(stderr, "Invalid probe port : %d\n", fp->init_probe_port);
                    usage(argv[0]);
                }
                break;
            case 'i':   /* Select interface */
                device = optarg;
                break;
            case 'n':   /* Do not use names */
                fp->use_name = 0;
                break;
            case 'o':   /* Output to a file */
                if ((fd = open(optarg, O_WRONLY|O_CREAT|O_TRUNC, 0666)) == -1)
                {
                    perror("main: cannot create file");
                    exit(1);
                }
                fprintf(stdout, "Writing all output to %s\n", optarg);
                if (dup2(fd, STDOUT_FILENO) == -1)
                {
                    perror("main: dup");
                    exit(1);
                }
                break;
            case 'P':   /* Time to pause in between sends */
                fp->write_pause = atoi(optarg);
                if (fp->write_pause > 2000 || fp->write_pause < 1)
                {
                    fprintf(stderr, "Invalid pause : %d\n", fp->write_pause);
                    usage(argv[0]);
                }
                break;
            case 'p':   /* Select walking protocol */
                for (i = 0; w_prots[i]; i++)
                {
                    if ((!strcasecmp(w_prots[i], optarg)))
                    {
                        switch (i)
                        {
                            case 0:
                                fp->protocol = IPPROTO_UDP;
                                break;
                            case 1:
                                fp->protocol = IPPROTO_TCP;
                                break;
                        }
                        break;
                    }
                }
                /*
                 *  Did we hit the end of the protocol array?
                 */
                if (!w_prots[i])
                {
                    fprintf(stderr, "\"%s\" unknown.  Supported : ", optarg);
                    for (i = 0; w_prots[i]; i++)
                    {
                        fprintf(stderr, "%s ", w_prots[i]);
                    }
                    puts(".");
                    exit(1);
                }
                break;
            case 'q':   /* Quiet mode */
                fp->quiet = 1;
                break;
            case 'r':   /* Redundancy count */
                fp->red_cnt = atoi(optarg);
                if (fp->red_cnt > 10 || fp->red_cnt < 1)
                {
                    fprintf(stderr, "Invalid redundancy count : %d\n", fp->red_cnt);
                    usage(argv[0]);
                }
                break;
            case 'S':   /* Scan these ports */
                if (parse_port_list(optarg) == -1)
                {
                    fprintf(stderr, "Could not build port list\n");
                    exit(1);
                }
                break;
            case 's':   /* Source port */
                fp->sport = atoi(optarg);
                if (fp->sport > 65535 || fp->sport < 1)
                {
                    fprintf(stderr, "Invalid source port : %d\n", fp->sport);
                    usage(argv[0]);
                }
                break;
            case 'T':   /* Time to wait for packets from other end */
                fp->pcap_timeout = atoi(optarg);
                if (fp->pcap_timeout > 1000 || fp->pcap_timeout < 1)
                {
                    fprintf(stderr,
                            "Invalid pcap_timeout : %d\n",
                            fp->pcap_timeout);
                    usage(argv[0]);
                }
                break;
            case 't':   /* Set initial IP TTL */
                fp->ttl = atoi(optarg);
                if (fp->ttl > MAX_HOP || fp->ttl < 1)
                {
                    fprintf(stderr,
                            "Invalid initial TTL : %d\n",
                            fp->ttl);
                    usage(argv[0]);
                }
                break;
            case 'v':   /* Version */
                fprintf(stderr, BANNER "Firewalk version : %s\n", version);
                exit(1);
            default:
                usage(argv[0]);
        }
    }

    /*
     *  Set up the network interface and determine our outgoing IP address.
     *  If this is Solaris, sorry, no interface detection code.
     */
#if (!SOLARIS)
    device = set_up_interface(device, &fp->sin);
#endif

    /*
     *  Do we have a destination host and a gateway host?
     */
    if (argc <= optind + 1)
    {
        fprintf(stderr, "\nFire, walk with me where?\n\n");
        usage(argv[0]);
        exit(1);
    }
    else if ((fp->gateway = name_resolve(argv[optind], 1)) == -1)
    {
        fprintf(stderr, "Cannot resolve gateway IP address\n");
        exit(1);
    }
    else if ((fp->destination = name_resolve(argv[optind + 1], 1)) == -1)
    {
        fprintf(stderr, "Cannot resolve destination IP address\n");
        exit(1);
    }
    else if (fp->gateway == fp->destination)
    {
        fprintf(stderr, "Gateway and destination are the same?  Can't do that.\n");
        exit(1);
    }

    /*
     *  Open the raw IP socket and set IPHDR_INCL.
     */
    fp->sock = open_raw_sock(IPPROTO_RAW);
    if (fp->sock == -1)
    {
        perror("No socket");
        exit(1);
    }

    /*
     *  Make sure we have a port list to scan.
     */
    if (!fp->plist) parse_port_list(strdup(DEFAULT_PORT_LIST));

#if (__DEBUG)
    _DEBUG("Debugging enabled\n");
    p_ent = getprotobynumber(fp->protocol);
    _DEBUG("%s firewalk\n", p_ent->p_name);
    if (fp->protocol == IPPROTO_UDP || fp->protocol == IPPROTO_TCP)
    {
        _DEBUG("Source port: %d\n", fp->sport);
        _DEBUG("Scanning ports: ");
        dump_port_list(fp->plist);
    }
    _DEBUG("Response wait timeout set to %d\n", fp->pcap_timeout);
    _DEBUG("Initial probe IP TTL at %d\n", fp->ttl);
#endif

    if (!fp->quiet)
    {
        fprintf(stdout,
            "Firewalking through %s (towards %s) with a maximum of %d hops.\n",
            host_lookup(fp->gateway, fp->use_name),
            host_lookup(fp->destination, fp->use_name),
            MAX_HOP);
    }

    /*  
     *  Magic time!
     */
    firewalk(device);

    if (close(fp->sock) == -1)
    {
        perror("Cannot close the socket\n");
        exit(1);
    }
    free_port_list(fp->plist);
    free(fp);

    return (0);
}


void
usage(u_char *argv0)
{
    fprintf(stderr, "\nUsage : %s [options] gateway destination\n"
                    "\t\t   [-h] program help\n"
                    "\t\t   [-I] initial port to use for TTL ramping (1 - 65535)\n"
                    "\t\t   [-i] specify alternative interface\n"
                    "\t\t   [-n] do not resolve IP addresses into hostnames\n"
                    "\t\t   [-o] write output to a file\n"
                    "\t\t   [-P] network writing pause (1 - 2000)\n"
                    "\t\t   [-p] firewalking protocol (TCP, UDP)\n"
                    "\t\t   [-q] enable quiet mode\n"
                    "\t\t   [-r] redundancy count (1 - 10)\n"
                    "\t\t   [-S] port range to scan (x - y, z)\n"
                    "\t\t   [-s] source port (1 - 65535)\n"
                    "\t\t   [-T] packet reading timeout in ms (1 - 1000)\n"
                    "\t\t   [-t] IP time to live (1 - %d)\n"
                    "\t\t   [-v] program version\n"
                    "\n",
                    argv0,
                    MAX_HOP);
    exit(0);
}

/* EOF */
