/*
 *  $Id: port_list.c,v 1.2 1998/10/19 23:36:39 route Exp $
 *
 *  Firewalk
 *  port_list implementation
 *  port_list.c - Port list parsing code.  To efficiently store multiple
 *  port ranges we allocate a linked list with eash node holding the begining
 *  and ending port for that range.  Neat.  Props to Aaron Bornstein and
 *  Joel Boutros for the initial idea.
 *
 *  Copyright (c) 1998 Mike D. Schiffman <mds@es2.net>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */ 

#include "./main.h"
#include "./port_list.h"
#include "./firepack.h"


int
parse_port_list(char *plist)
{
    struct port_list *tmp;
    char legal_tokens[] = "0123456789,- ";
    char *tok;
    int i, j, valid_token, cur_node;

    /*
     *  Make sure we have legal tokens.
     */
    for (i = 0; plist[i]; i++)
    {
        for (j = 0, valid_token = 0; legal_tokens[j]; j++)
        {
            if (legal_tokens[j] == plist[i])
            {
                valid_token = 1;
                break;
            }
        }
        if (!valid_token)
        {
            fprintf(stderr,
                    "parse_port_list: illegal token # %d (%c)\n",
                    i + 1,
                    plist[i]);
            return (-1);
        }
    }

    /*
     *  Head node.
     */
    fp->plist = (struct port_list *)malloc(sizeof(struct port_list));
    if (!fp->plist)
    {
        perror("parse_port_list: malloc");
        return(-1);
    }
    tmp = fp->plist;

    tmp->node = cur_node = 0;
    tmp->next = NULL;
    /*
     *  Using strtok successively proved problematic.  We solve this by 
     *  calling it once, then manually extracting the elements from the token.
     *  The case of bport > eport is ignored for now.
     */
    for (i = 0; (tok = strtok(!i ? plist : NULL, ",")); i = 1, cur_node++)
    {
        /*
         *  The first iteration we will have a head node allocated so we don't
         *  need to malloc().
         */
        if (i)
        {
            tmp->next = (struct port_list *)malloc(sizeof(struct port_list));
            if (!tmp)
            {
                perror("parse_port_list: malloc");
                return(-1);
            }
            tmp = tmp->next;
            tmp->node = cur_node;
            tmp->next  = NULL;
        }
        tmp->bport = atoi(tok);

        /*
         *  Step past this port number.
         */
        j = 0;
        while (isdigit(tok[j])) j++;

        /*
         *  If we have a delimiting dash and are NOT at the end of the token
         *  array, we can assume it's the end port, otherwise if we just have
         *  a dash, we consider it shorthand for `inclusive of all ports up to
         *  65535.  Finally, if we have no dash, we assume this token is a
         *  single port only.
         */
        if (tok[j] == '-')
        {
            tmp->eport = (++j != strlen(tok)) ? atoi(&tok[j]) : 65535;
        }
        else
        {
            tmp->eport = tmp->bport;
        }
    }
    /*
     *  The nead node needs to hold the total node count.
     */
    fp->plist->node = cur_node;
    return (1);
}


void
dump_port_list(struct port_list *p)
{
    for (; p; p = p->next)
    {
        if (p->bport == p->eport)
        {
            fprintf(stdout, "%d ", p->bport);
        }
        else
        {
            fprintf(stdout, "%d-%d ", p->bport, p->eport);
        }
    }
    fprintf(stdout, "\n");
}


void
free_port_list(struct port_list *p)
{
    u_short i;
    struct port_list *tmp;

    for (i = fp->plist->node; i; i--)
    {
        tmp = p;
        p = p->next;
        free(tmp);
    }
}


int
get_next_port_pair(struct port_list *p, u_short *bport, u_short *eport)
{
    static u_short node_cnt;
    u_short tmp_cnt;

    if (node_cnt == p->node) return (0);

    for (tmp_cnt = node_cnt; tmp_cnt; tmp_cnt--, p = p->next) ;
    *bport = p->bport;
    *eport = p->eport;
    node_cnt++;
    return (1);
}

/* EOF */
