/*
 *  $Id: udptcpwalk.c,v 1.3 1998/12/05 18:24:52 route Exp $
 *
 *  Firewalk
 *  udptcpwalk implementation
 *  udptcpwalk.c - UDP and TCP firewalking module
 *
 *  Copyright (c) 1998 Mike D. Schiffman <mds@es2.net>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include "./firewalk.h"
#include "./firepack.h"
#include "./listener.h"
#include "./signal.h"
#include "./packet.h"


void
udptcpwalk(struct firepack *fp)
{
    u_char *pak_buf;
    u_short bport,          /* begining port in the range we are scanning */
            eport,          /* ending port in the range we are scanning */
            cport;          /* current port we are scanning */
    int c, i, j, t_size;
    int done;               /* true when we are done scanning */

    t_size = (fp->protocol == IPPROTO_TCP ? TCP_H : UDP_H);
    /*
     *  Packet header memory.
     */
    pak_buf = malloc(IP_H + t_size);
    if (!pak_buf)
    {
        perror("No memory for packet header");
        exit(1);
    }

    /*
     *  As long as we have a port pair, we should be scanning.  If we get here
     *  this will always happen at least once.
     */
    while (get_next_port_pair(fp->plist, &bport, &eport))
    {
        /*
         *  Keep going until we have exhausted this port range.  Make sure we
         *  don't wrap around at MAX_PORT (2^16 -1).
         */
        while (!(bport > eport) && bport != 0)
        {
            /*
             *  Check to see if we are bound.  If so, we don't need to ramp up
             *  TTLs again to get there again.  TTL ramping should only occur
             *  once.
             */
            if (fp->bound)
            {
                /*
                 *  Once we are bound, grab the current port and restrict the
                 *  hop count to one beyond the gateway.
                 */
                cport = bport++;
                fp->ttl = fp->gway_hop_cnt + 1;
            }
            else
            {
                /*
                 *  We need to ramp up TTLs until we get one beyond the
                 *  gateway.  Then we are bound and can simply send packets
                 *  with that binding TTL and look for the ICMP expired in
                 *  transit messages.
                 */
                if (!fp->quiet)
                {
                    fprintf (stdout, "Ramping up hopcounts to binding host...\n");
                }
                cport = fp->init_probe_port;
            }
            done = 0;

            /*
             *  TTL ramping occurs here.
             */
            for (i = 0; fp->ttl < MAX_HOP && !done; i++, fp->ttl++)
            {
                /*
                 *  Build the UDP packet.  We increment the packet ID so we can
                 *  differentiate between probes.  The TTL is bumped up so we
                 *  can determine how many hops away the gateway is.
                 */
                build_ip(t_size,
                        0,                      /* regular service */
                        ++fp->id,
                        0,                      /* No fragmentation bits set */
                        fp->ttl,
                        fp->protocol,
                        fp->sin.sin_addr.s_addr,
                        fp->destination,
                        NULL,
                        0,
                        pak_buf);

                if (fp->protocol == IPPROTO_UDP)
                {
                    build_udp(fp->sport, cport, NULL, 0, IP_H + pak_buf);
                }
                else
                {
                    build_tcp(fp->sport,
                            cport,
                            get_prand(PRu32),
                            0L,                 /* No ACK number */
                            TH_SYN,
                            1024,
                            0,                  /* No urgent data */
                            NULL,
                            0,
                            pak_buf + IP_H);
                }
                do_checksum(pak_buf, fp->protocol, t_size);

                for (j = 0, fp->red_flag = 0; j < fp->red_cnt; j++)
                {
                    /*
                     *  Write the packet to the network.
                     */
                    usleep(fp->write_pause);
                    c = write_ip(fp->sock, pak_buf, IP_H + t_size);
                    if (c < IP_H + t_size)
                    {
                        fprintf(stderr,
                            "udptcpwalk: error writing probe %d (only %d bytes)\n",
                            i + 1, c);
                        continue;
                    }
                    fp->p_sent++;

                    /*
                     *  Only print the ramping template while we are ramping,
                     *  and not for redundant packets.
                     */
                    if (!fp->bound && !fp->red_flag && !fp->quiet)
                    {
                        fprintf(stdout, "probe: %2d  TTL: %2d  port %3d:  ",
                                    i + 1, fp->ttl, cport);
                    }
                    else if (!fp->red_flag && !fp->quiet)
                    {
                        fprintf(stdout, "\nport %3d: ", cport);
                    }
                    /*
                     *  Call the listener to recieve the response (if any).
                     *  Then process what it gives us.
                     */
                    if (process_response(&cport, listener(fp)) == -1)
                    {
                        /*
                         *  Either we got a TTL expired from the gateway or
                         *  the gateway host was not found.
                         */
                        goto clean_up;
                    }
                    if (fp->bound) done = 1;
                    if (j + 1 == fp->red_cnt && !fp->quiet)
                    {
                        fprintf(stdout, "\n");
                    }
                }   /* Innermost sending loop */
            }   /* Third TTL ramping loop */
            if (fp->ttl == MAX_HOP)
            {
                fprintf(stdout, "Hop count exceeded\n");
                goto clean_up;
            }
        }   /* Second port incrementing loop */
    }   /* Outer port pair grabbing loop */
    clean_up:
    free(pak_buf);
    p_stats(fp);
}

/* EOF */
