/*
 *  $Id: arp_whohas.c,v 1.2 2004/10/20 19:03:21 mike Exp $
 *
 *  Hummingbird - Asynchronous scanning engine
 *  Originally based off of sift.c from the book 
 *  Building Open Source Network Security Tools
 *  Copyright (c) 2002 - 2005 Mike D. Schiffman <stolencreditcard@gmail.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
 
#include "./hummingbird.h"

void *
arp_whohas_user_create_injector(char *errbuf)
{
    struct arp_whohas_injector *arp_whohas_injector;
    
    arp_whohas_injector = malloc(sizeof (struct arp_whohas_injector));
    if (arp_whohas_injector == NULL)
    {
        snprintf(errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (NULL);
    }
    return ((void *)arp_whohas_injector);
}

int
arp_whohas_user_init_injector(injector_t *injector)
{
    struct arp_whohas_injector *arp_whohas_injector;

    arp_whohas_injector = (struct arp_whohas_injector *)injector->user;

    injector->libnet_mode = LIBNET_LINK;
    arp_whohas_injector->arp = LIBNET_PTAG_INITIALIZER;
    arp_whohas_injector->eth = LIBNET_PTAG_INITIALIZER;

    return (1);
}

void *
arp_whohas_user_create_listener(char *errbuf)
{
    struct arp_whohas_listener *arp_whohas_listener;
    
    arp_whohas_listener = malloc(sizeof (struct arp_whohas_listener));
    if (arp_whohas_listener == NULL)
    {
        snprintf(errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (NULL);
    }

    return ((void *)arp_whohas_listener);
}

int
arp_whohas_user_init_listener(listener_t *listener)
{
    int c, n;
    libnet_t *l;
    char filter[64];
    struct libnet_ether_addr *e;
    struct bpf_program filter_code;
    bpf_u_int32 local_net, netmask;
    struct arp_whohas_listener *arp_whohas_listener;

    arp_whohas_listener = (struct arp_whohas_listener *)listener->user;

    arp_whohas_listener->total_responses = 0;

    /**
     * Build the ARP filter. We have to cheat here since we need the address of 
     * the interface we're using to create the filter since arp doesn't have 
     * enough state for us to create a filter to capture responses to just our 
     * probes. Since the injector isn't initialized, we'll create a libnet 
     * context real fast like and dump it when we're done.
     */
    l = libnet_init(LIBNET_LINK, listener->device, listener->errbuf);
    if (l == NULL)
    {
        return (-1);
    }
    e = libnet_get_hwaddr(l);
    if (e == NULL)
    {
        /** they're the same size, everytime! I promise! */
        strcpy(listener->errbuf, l->err_buf);
        return (-1);
    }
    memset(filter, 0, sizeof (filter));
    sprintf(filter, "%s", ARP_WHOHAS_FILTER);
    for (n = strlen(ARP_WHOHAS_FILTER), c = 0; c < 6; c++)
    {
        sprintf(&filter[n], "%2.2x", e->ether_addr_octet[c]);
        n += 2;
        if (c != 5)
        {
            sprintf(&filter[n], ":");
            n++;
        }
    }
    libnet_destroy(l);
    free(e);

    arp_whohas_listener->pcap_filter = filter;

    /** get the subnet mask of the interface */
    if (pcap_lookupnet(listener->device, &local_net, &netmask,
            listener->errbuf) == -1)
    {
        return (-1);
    }

    /** compile the BPF filter code */
    if (pcap_compile(listener->p, &filter_code,
            arp_whohas_listener->pcap_filter, 1, netmask) == -1)
    {
        snprintf(listener->errbuf, LIBNET_ERRBUF_SIZE,
                "pcap_compile(): %s (%s)", pcap_geterr(listener->p),
                arp_whohas_listener->pcap_filter);
        return (-1);
    }

    /** apply the filter to the interface */
    if (pcap_setfilter(listener->p, &filter_code) == -1)
    {
        snprintf(listener->errbuf, LIBNET_ERRBUF_SIZE,
                "pcap_setfilter(): %s", pcap_geterr(listener->p));
        return (-1);
    }

    return (1);
}

int
arp_whohas_builder(void *arg)
{
    injector_t *injector;
    struct arp_whohas_injector *user;
    u_int8_t broadcast[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

    injector = (injector_t *)arg;
    user     = (struct arp_whohas_injector *)injector->user;

    user->arp = libnet_build_arp(ARPHRD_ETHER, ETHERTYPE_IP, 6, 4, 
            ARPOP_REQUEST, (u_int8_t *)injector->src_mac->ether_addr_octet,
            (u_int8_t *)&injector->src_ip, broadcast, 
            (u_int8_t *)&injector->dst_ip, NULL, 0, injector->l, user->arp);
    if (user->arp == -1)
    {
        sprintf(injector->errbuf, "Can't build ARP header: %s\n",
                libnet_geterror(injector->l));
        return (-1);
    }
    user->eth = libnet_build_ethernet(broadcast,
            (u_int8_t *)injector->src_mac->ether_addr_octet, ETHERTYPE_ARP,
            NULL, 0, injector->l, user->eth); 
    if (user->eth == -1)
    {
        sprintf(injector->errbuf, "Can't build ethernet header: %s\n",
                libnet_geterror(injector->l));
        return (-1);
    }
    return (1);
}

void
arp_whohas_decoder(u_char *arg, const struct pcap_pkthdr *pcap_hdr,
const u_char *packet)
{
    u_int16_t c;
    listener_t *listener;
    struct libnet_ethernet_hdr *eth;
    struct libnet_arp_hdr *arp;
    struct arp_whohas_listener *user;

    listener = (listener_t *)arg;
    user     = (struct arp_whohas_listener *)listener->user;

    if (packet == NULL)
    {
        /**
         *  We have to be careful here as pcap_next() can return NULL
         *  if the timer expires with no data in the packet buffer or
         *  under some special circumstances under linux.
         */
        return;
    }

    eth = (struct libnet_ethernet_hdr *)(packet);
    arp = (struct libnet_arp_hdr *)(packet + LIBNET_ETH_H);

    fprintf(listener->out, "%d.%d.%d.%d is-at ", 
            packet[28] & 255, packet[29] & 255, 
            packet[30] & 255, packet[31] & 255);
    for (c = 0; c < 6; c++)
    {
        fprintf(listener->out, "%2.2x", eth->ether_shost[c]);
        if (c != 5)
        {
            fprintf(listener->out, ":");
        }
    }
    fprintf(listener->out, "\n");
    user->total_responses++;
}

void
arp_whohas_reporter(listener_t *listener, injector_t *injector, int mode)
{
    struct timeval r, e;
    struct arp_whohas_listener *user;

    user = (struct arp_whohas_listener *)listener->user;

    switch (mode)
    {
        case HB_PREAMBLE:
            fprintf(listener->out,
                "----------------------------------------------\n"
                "Hummingbird asynchronous scanning engine\n"
                "----------------------------------------------\n"
                "starting ARP whohas scan\n"
                "----------------------------------------------\n");
            break;
        case HB_DENOUEMENT:
            gettimeofday(&e, NULL);
            PTIMERSUB(&e, &(listener->timer), &r);
            fprintf(listener->out,
                "----------------------------------------------\n"
                "running time:\t\t\t%6d seconds\n"
                "----------------------------------------------\n"
                "total probes sent:\t\t\t%6d\n"
                "total responses received:\t\t%6d (%.02f%% of probes sent)\n"
                "total timeouts or drops:\t\t%6d (%.02f%% of probes sent)\n"
                "----------------------------------------------\n",
                (u_int32_t)r.tv_sec, injector->sent_probes,
                user->total_responses, ((float)user->total_responses /
                (float)injector->sent_probes) * 100,
                injector->sent_probes - user->total_responses,
                ((float)(injector->sent_probes - user->total_responses) /
                (float)injector->sent_probes) * 100);
            break;
    }   
}

int
arp_whohas_parser(listener_t *listener, injector_t *injector, char *config)
{
    xmlDocPtr doc = NULL;
    xmlNodePtr option;
    xmlChar *keyword, *attribute;
    struct arp_whohas_injector *user;

    user = (struct arp_whohas_injector *)injector->user;

    /** boilerplate to open config file, do basic checks, and return a ptr */
    option = setup_xml_parser(config, doc, "arp_whohas", injector->errbuf);
    if (option == NULL)
    {
        /** error msg already set */
        return (-1);
    }

    /** we silently ignore empty configuration sections */
    for (; option; option = option->next)
    {
         /** run through each option, pull out the attribute and apply */
         if ((xmlStrcmp(option->name, (const xmlChar *)"option")) == 0)
         {
             keyword = xmlNodeListGetString(doc, option->children, 1);
             attribute = xmlGetProp(option, (const xmlChar *)"type");
             if (attribute == NULL)
             {
                 xmlFree(keyword);
                 continue;
             }
             if (strcmp((char *)attribute, "throttle") == 0)
             {
                 injector->throttle = atoi((char *)keyword);
                 xmlFree(keyword);
                 xmlFree(attribute);
                 continue;
             }
             if (strcmp((char *)attribute, "snapshot") == 0)
             {
                 listener->pcap_snaplen = atoi((char *)keyword);
                 xmlFree(keyword);
                 xmlFree(attribute);
                 continue;
             }
             fprintf(stderr, "parser(): ignoring unknown option: %s\n",
                     attribute);
             xmlFree(keyword);
             xmlFree(attribute);
         }
    }
    xmlFreeDoc(doc);
    return (1);
}

/** EOF */
