/*
 *  $Id: dns_chaos.c,v 1.6 2004/10/20 08:41:07 mike Exp $
 *
 *  Hummingbird - Asynchronous scanning engine
 *  Originally based off of sift.c from the book 
 *  Building Open Source Network Security Tools
 *  Copyright (c) 2002 - 2005 Mike D. Schiffman <stolencreditcard@gmail.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
 
#include "./hummingbird.h"

void *
dns_chaos_user_create_injector(char *errbuf)
{
    struct dns_chaos_injector *dns_chaos_injector;
    
    dns_chaos_injector = malloc(sizeof (struct dns_chaos_injector));
    if (dns_chaos_injector == NULL)
    {
        snprintf(errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (NULL);
    }

    return ((void *)dns_chaos_injector);
}

int
dns_chaos_user_init_injector(injector_t *injector)
{
    struct dns_chaos_injector *dns_chaos_injector;

    dns_chaos_injector = (struct dns_chaos_injector *)injector->user;
    
    dns_chaos_injector->dns = LIBNET_PTAG_INITIALIZER;
    dns_chaos_injector->udp = LIBNET_PTAG_INITIALIZER;
    dns_chaos_injector->ip  = LIBNET_PTAG_INITIALIZER;

    return (1);
}

void *
dns_chaos_user_create_listener(char *errbuf)
{
    struct dns_chaos_listener *dns_chaos_listener;
    
    dns_chaos_listener = malloc(sizeof (struct dns_chaos_listener));
    if (dns_chaos_listener == NULL)
    {
        snprintf(errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (NULL);
    }

    return ((void *)dns_chaos_listener);
}

int
dns_chaos_user_init_listener(listener_t *listener)
{
    char filter[64];
    struct bpf_program filter_code;
    bpf_u_int32 local_net, netmask;
    struct dns_chaos_listener *dns_chaos_listener;

    dns_chaos_listener = (struct dns_chaos_listener *)listener->user;

    dns_chaos_listener->total_responses = 0;
    dns_chaos_listener->valid_responses = 0;
    dns_chaos_listener->not_implemented = 0;
    dns_chaos_listener->server_failed   = 0;
    dns_chaos_listener->format_error    = 0;

    /** build the filter */
    memset(filter, 0, sizeof (filter));
    sprintf(filter, "%s%d", DNS_CHAOS_FILTER, listener->id);

    dns_chaos_listener->pcap_filter = filter;

    /** get the subnet mask of the interface */
    if (pcap_lookupnet(listener->device, &local_net, &netmask,
            listener->errbuf) == -1)
    {
        return (-1);
    }

    /** compile the BPF filter code */
    if (pcap_compile(listener->p, &filter_code,
            dns_chaos_listener->pcap_filter, 1, netmask) == -1)
    {
        snprintf(listener->errbuf, LIBNET_ERRBUF_SIZE,
                "pcap_compile(): %s (%s)", pcap_geterr(listener->p),
                dns_chaos_listener->pcap_filter);
        return (-1);
    }

    /** apply the filter to the interface */
    if (pcap_setfilter(listener->p, &filter_code) == -1)
    {
        snprintf(listener->errbuf, LIBNET_ERRBUF_SIZE,
                "pcap_setfilter(): %s", pcap_geterr(listener->p));
        return (-1);
    }

    return (1);
}

int
dns_chaos_builder(void *arg)
{
    u_int32_t packet_size;
    injector_t *injector;
    struct dns_chaos_injector *user;

    /**
     *  The chaos class query resource record:
     *  07 'V' 'E' 'R' 'S' 'I' 'O' 'N' 07 'B' 'I' 'N' 'D' 00 16 00 03
     */
    u_int8_t chaos_query[] = {0x07, 0x76, 0x65, 0x72, 0x73, 0x69,
                              0x6f, 0x6e, 0x04, 0x62, 0x69, 0x6e,
                              0x64, 0x00, 0x00, 0x10, 0x00, 0x03};

    injector = (injector_t *)arg;
    user     = (struct dns_chaos_injector *)injector->user;

    packet_size = LIBNET_IPV4_H + LIBNET_UDP_H + LIBNET_UDP_DNSV4_H +
            CHAOS_QUERY_S;

    /**
     *  Build a dns chaos class query request packet. We save the ptag after
     *  the first usage so future calls will modify this packet header template
     *  rather than build a new one.
     */
    user->dns = libnet_build_dnsv4(LIBNET_UDP_DNSV4_H, injector->id, 0x0100,   
        1, 0, 0, 0, chaos_query, CHAOS_QUERY_S, injector->l, user->dns); 
    if (user->dns == -1)
    {
        sprintf(injector->errbuf, "Can't build DNS header: %s\n",
                libnet_geterror(injector->l));
        return (-1);
    }

    /**
     *  The UDP header only has to be built once.  Checksums will have
     *  to be recomputed everytime since the DNS header is changing
     *  but we don't need to modify the header explicitly after it's
     *  built.
     */
    if (user->udp == LIBNET_PTAG_INITIALIZER)
    {
        user->udp = libnet_build_udp(SOURCE_PORT, 53,
            LIBNET_UDP_H + LIBNET_UDP_DNSV4_H + CHAOS_QUERY_S, 0, NULL, 0, 
            injector->l, user->udp);
        if (user->udp == -1)
        {
            sprintf(injector->errbuf, "Can't build UDP header: %s\n",
                    libnet_geterror(injector->l));
            return (-1);
        }
    }

    /**
     *  After building it, we'll need to update the IP header every time
     *  with the new address.
     */
    user->ip = libnet_build_ipv4(packet_size, 0, 242, 0, 64, IPPROTO_UDP, 0, 
        injector->src_ip, injector->dst_ip, NULL, 0, injector->l, user->ip);
    if (user->ip == -1)
    {
        sprintf(injector->errbuf, "Can't build IP header: %s\n",
                libnet_geterror(injector->l));
        return (-1);
    }

    return (1);
}

void
dns_chaos_decoder(u_char *arg, const struct pcap_pkthdr *pcap_hdr,
const u_char *packet)
{
    int j, l, m;
    u_char *payload;
    char version[256];
    u_int16_t ip_hl, count;
    listener_t *listener;
    struct libnet_ipv4_hdr *ip;
    struct libnet_dnsv4udp_hdr *dns;
    struct dns_chaos_listener *user;

    listener = (listener_t *)arg;
    user     = (struct dns_chaos_listener *)listener->user;

    if (packet == NULL)
    {
        /**
         *  We have to be careful here as pcap_next() can return NULL
         *  if the timer expires with no data in the packet buffer or
         *  under some special circumstances under linux.
         */
        return;
    }

    ip = (struct libnet_ipv4_hdr *)(packet + LIBNET_ETH_H);

    ip_hl = ip->ip_hl << 2;
    dns = (struct libnet_dnsv4udp_hdr *)(packet + LIBNET_ETH_H + ip_hl +
            LIBNET_UDP_H);

    fprintf(listener->out, "%s:\t", libnet_addr2name4(ip->ip_src.s_addr,
            LIBNET_DONT_RESOLVE));

    /** check to see if the CHAOS class is implemented */
    if ((ntohs(dns->flags) & DNS_NOTIMPL))
    {
        fprintf(listener->out, "not implemented\n");
        user->total_responses++;
        user->not_implemented++;
        return;
    }
    /** check to see if the server failed */
    if ((ntohs(dns->flags) & DNS_SERVFAILED))
    {
        fprintf(listener->out, "server failed\n");
        user->total_responses++;
        user->server_failed++;
        return;
    }
    /** check to see if there was a format error */
    if ((ntohs(dns->flags) & DNS_FORMATERR))
    {
        fprintf(listener->out, "format error\n");
        user->total_responses++;
        user->format_error++;
        return;
    }
    /**
     *  Every response to our chaos class query should have our
     *  safely point payload past that query rr directly to
     *  the answer rr which is what we want to parse.
     */
    payload = (u_char *)(packet + LIBNET_ETH_H + ip_hl +
            LIBNET_UDP_H + LIBNET_UDP_DNSV4_H + CHAOS_QUERY_S);

    /**
     *  Some DNS servers will be smart and compress their
     *  response to our query.  We check for that case here.
     */
    if (payload[0] & 0xc0)
    {
        /**
         *  When the two high-order bits are set (values
         *  192 - 255) it indicates the response is compressed.
         *  Shave off the low-order 14 bits to determine the
         *  offset.  It's pretty bitwise code but unfortunately
         *  we have no use for it in this version.
         */
        /** offset = (payload[0] << 0x08 | payload[1]) & 0x3fff; */
        /**
         *  The 11th and 12th bytes will contain the count
         *  (number of bytes) of the answer.
         */
        count = payload[10] << 0x08 | payload[11];
        j = 13;
    }
    else
    {
        /**
         *  If we're not compressed step over the 24 bytes of
         *  answer stuff we don't care about.
         */
        count = payload[22] << 0x08 | payload[23];
        j = 25;
    }

    /**
     *  Our buffer to hold the version info is only 256 bytes and
     *  we need to account for the terminating NULL.
     */
    count > 255 ? count = 255 : count ;
    memset(version, 0, 256);

    /**
     *  Run through the payload pulling out only the printable
     *  ASCII characters which are between 0x20 ( ) and 0x7e (~).
     */
    for (l = 0, m = 0; l < count - 1; l++)
    {
        if (payload[j + l] >= 0x20 && payload[j + l] <= 0x7e)
        {
            version[m] = payload[j + l];
            m++;
        }
    }

    /** finally, dump the version */
    fprintf(listener->out, "%s\n", version);
    user->valid_responses++;
    user->total_responses++;
}

void
dns_chaos_reporter(listener_t *listener, injector_t *injector, int mode)
{
    struct timeval r, e;
    struct dns_chaos_listener *user;

    user = (struct dns_chaos_listener *)listener->user;

    switch (mode)
    {
        case HB_PREAMBLE:
            fprintf(listener->out,
                "----------------------------------------------\n"
                "Hummingbird asynchronous scanning engine\n"
                "----------------------------------------------\n"
                "starting DNS chaos class query scan\n"
                "----------------------------------------------\n");
            break;
        case HB_DENOUEMENT:
            gettimeofday(&e, NULL);
            PTIMERSUB(&e, &(listener->timer), &r);
            fprintf(listener->out,
                "----------------------------------------------\n"
                "running time:\t\t\t%6d seconds\n"
                "----------------------------------------------\n"
                "responses containing version:\t\t%6d\n"
                "not implemented:\t\t\t%6d\n"
                "server failed:\t\t\t\t%6d\n"
                "format errors:\t\t\t\t%6d\n"
                "----------------------------------------------\n"
                "total probes sent:\t\t\t%6d\n"
                "total responses received:\t\t%6d (%.02f%% of probes sent)\n"
                "total timeouts or drops:\t\t%6d (%.02f%% of probes sent)\n"
                "----------------------------------------------\n",
                (u_int32_t)r.tv_sec, user->valid_responses,
                user->not_implemented, user->server_failed, user->format_error,                 injector->sent_probes, user->total_responses,
                ((float)user->total_responses /
                (float)injector->sent_probes) * 100,
                injector->sent_probes - user->total_responses,
                ((float)(injector->sent_probes - user->total_responses) /
                (float)injector->sent_probes) * 100);
            break;
    }
}

int
dns_chaos_parser(listener_t *listener, injector_t *injector, char *config)
{
    xmlDocPtr doc = NULL;
    xmlNodePtr option;
    xmlChar *keyword, *attribute;

    /** boilerplate to open config file, do basic checks, and return a ptr */
    option = setup_xml_parser(config, doc, "dns_chaos", injector->errbuf);
    if (option == NULL)
    {
        /** error msg already set */
        return (-1);
    }

    /** we silently ignore empty configuration sections */
    for (; option; option = option->next)
    {
         /** run through each option, pull out the attribute and apply */
         if ((xmlStrcmp(option->name, (const xmlChar *)"option")) == 0)
         {
             keyword = xmlNodeListGetString(doc, option->children, 1);
             attribute = xmlGetProp(option, (const xmlChar *)"type");
             if (attribute == NULL)
             {
                 xmlFree(keyword);
                 continue;
             }
             if (strcmp((char *)attribute, "throttle") == 0)
             {
                 injector->throttle = atoi((char *)keyword);
                 xmlFree(keyword);
                 xmlFree(attribute);
                 continue;
             }
             if (strcmp((char *)attribute, "snapshot") == 0)
             {
                 listener->pcap_snaplen = atoi((char *)keyword);
                 xmlFree(keyword);
                 xmlFree(attribute);
                 continue;
             }
             fprintf(stderr, "parser(): ignoring unknown option: %s\n",
                     attribute);
             xmlFree(keyword);
             xmlFree(attribute);
         }
    }
    xmlFreeDoc(doc);
    return (1);
}

/** EOF */
