/*
 *  $Id: icmp_echo.c,v 1.4 2004/10/20 08:41:07 mike Exp $
 *
 *  Hummingbird - Asynchronous scanning engine
 *  Originally based off of sift.c from the book 
 *  Building Open Source Network Security Tools
 *  Copyright (c) 2002 - 2005 Mike D. Schiffman <stolencreditcard@gmail.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
 
#include "./hummingbird.h"

void *
icmp_echo_user_create_injector(char *errbuf)
{
    struct icmp_echo_injector *icmp_echo_injector;
    
    icmp_echo_injector = malloc(sizeof (struct icmp_echo_injector));
    if (icmp_echo_injector == NULL)
    {
        snprintf(errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (NULL);
    }

    return ((void *)icmp_echo_injector);
}

int
icmp_echo_user_init_injector(injector_t *injector)
{
    struct icmp_echo_injector *icmp_echo_injector;

    icmp_echo_injector = (struct icmp_echo_injector *)injector->user;

    icmp_echo_injector->icmp = LIBNET_PTAG_INITIALIZER;
    icmp_echo_injector->ip   = LIBNET_PTAG_INITIALIZER;
    icmp_echo_injector->seq  = 0;

    return (1);
}

void *
icmp_echo_user_create_listener(char *errbuf)
{
    struct icmp_echo_listener *icmp_echo_listener;
    
    icmp_echo_listener = malloc(sizeof (struct icmp_echo_listener));
    if (icmp_echo_listener == NULL)
    {
        snprintf(errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (NULL);
    }

    return ((void *)icmp_echo_listener);
}

int
icmp_echo_user_init_listener(listener_t *listener)
{
    char filter[64];
    struct bpf_program filter_code;
    bpf_u_int32 local_net, netmask;
    struct icmp_echo_listener *icmp_echo_listener;

    icmp_echo_listener = (struct icmp_echo_listener *)listener->user;

    icmp_echo_listener->total_responses = 0;

    /** build the filter */
    memset(filter, 0, sizeof (filter));
    sprintf(filter, "%s%d", ICMP_ECHO_FILTER, listener->id);

    icmp_echo_listener->pcap_filter = filter;

    /** get the subnet mask of the interface */
    if (pcap_lookupnet(listener->device, &local_net, &netmask,
            listener->errbuf) == -1)
    {
        return (-1);
    }

    /** compile the BPF filter code */
    if (pcap_compile(listener->p, &filter_code,
            icmp_echo_listener->pcap_filter, 1, netmask) == -1)
    {
        snprintf(listener->errbuf, LIBNET_ERRBUF_SIZE,
                "pcap_compile(): %s (%s)", pcap_geterr(listener->p),
                icmp_echo_listener->pcap_filter);
        return (-1);
    }

    /** apply the filter to the interface */
    if (pcap_setfilter(listener->p, &filter_code) == -1)
    {
        snprintf(listener->errbuf, LIBNET_ERRBUF_SIZE,
                "pcap_setfilter(): %s", pcap_geterr(listener->p));
        return (-1);
    }

    return (1);
}

int
icmp_echo_builder(void *arg)
{
    u_int32_t packet_size;
    injector_t *injector;
    struct icmp_echo_injector *user;

    injector = (injector_t *)arg;
    user     = (struct icmp_echo_injector *)injector->user;

    packet_size = LIBNET_IPV4_H + LIBNET_ICMPV4_ECHO_H;

    user->icmp = libnet_build_icmpv4_echo(
            ICMP_ECHO,                            /* type */
            0,                                    /* code */
            0,                                    /* checksum */
            injector->id,                         /* id */
            user->seq++,                          /* sequence number */
            NULL,                                 /* payload */
            0,                                    /* payload size */
            injector->l,                          /* libnet contect */
            user->icmp);                          /* ptag */
    if (user->icmp == -1)
    {
        sprintf(injector->errbuf, "Can't build ICMP_ECHO header: %s\n",
                libnet_geterror(injector->l));
        return (-1);
    }

    /*
     *  After building it, we'll need to update the IP header every time
     *  with the new address.
     */
    user->ip = libnet_build_ipv4(
        packet_size,                    /* total packet size */
        0,                              /* type of service */
        242,                            /* identification */
        0,                              /* fragmentation */
        64,                             /* time to live */
        IPPROTO_ICMP,                   /* protocol */
        0,                              /* checksum */
        injector->src_ip,               /* source */
        injector->dst_ip,               /* destination */
        NULL,                           /* payload */
        0,                              /* payload size */
        injector->l,                    /* libnet context */
        user->ip);                      /* ptag */
    if (user->ip == -1)
    {
        sprintf(injector->errbuf, "Can't build IP header: %s\n",
                libnet_geterror(injector->l));
        return (-1);
    }

    return (1);
}

void
icmp_echo_decoder(u_char *arg, const struct pcap_pkthdr *pcap_hdr,
const u_char *packet)
{
    u_int16_t ip_hl;
    listener_t *listener;
    struct libnet_ipv4_hdr *ip;
    struct libnet_icmpv4_hdr *icmp;
    struct icmp_echo_listener *user;

    listener = (listener_t *)arg;
    user     = (struct icmp_echo_listener *)listener->user;

    if (packet == NULL)
    {
        /*
         *  We have to be careful here as pcap_next() can return NULL
         *  if the timer expires with no data in the packet buffer or
         *  under some special circumstances under linux.
         */
        return;
    }

    ip = (struct libnet_ipv4_hdr *)(packet + LIBNET_ETH_H);

    ip_hl = ip->ip_hl << 2;
    icmp  = (struct libnet_icmpv4_hdr *)(packet + LIBNET_ETH_H + ip_hl);

    fprintf(listener->out, "%s:\t\t", libnet_addr2name4(ip->ip_src.s_addr,
            LIBNET_DONT_RESOLVE));

    fprintf(listener->out, "%d bytes icmp_seq %5d ttl %3d\n", 
            ntohs(ip->ip_len), icmp->icmp_seq, ip->ip_ttl);

    user->total_responses++;
}

void
icmp_echo_reporter(listener_t *listener, injector_t *injector, int mode)
{
    struct timeval r, e;
    struct icmp_echo_listener *user;

    user = (struct icmp_echo_listener *)listener->user;

    switch (mode)
    {
        case HB_PREAMBLE:
            fprintf(listener->out,
                "----------------------------------------------\n"
                "Hummingbird asynchronous scanning engine\n"
                "----------------------------------------------\n"
                "starting ICMP echo scan\n"
                "----------------------------------------------\n");
            break;
        case HB_DENOUEMENT:
            gettimeofday(&e, NULL);
            PTIMERSUB(&e, &(listener->timer), &r);
            fprintf(listener->out,
                "----------------------------------------------\n"
                "running time:\t\t\t%6d seconds\n"
                "----------------------------------------------\n"
                "total probes sent:\t\t\t%6d\n"
                "total responses received:\t\t%6d (%.02f%% of probes sent)\n"
                "total timeouts or drops:\t\t%6d (%.02f%% of probes sent)\n"
                "----------------------------------------------\n",
                (u_int32_t)r.tv_sec, injector->sent_probes,
                user->total_responses, ((float)user->total_responses /
                (float)injector->sent_probes) * 100,
                injector->sent_probes - user->total_responses,
                ((float)(injector->sent_probes - user->total_responses) /
                (float)injector->sent_probes) * 100);
            break;
     }
}

int
icmp_echo_parser(listener_t *listener, injector_t *injector, char *config)
{
    xmlDocPtr doc = NULL;
    xmlNodePtr option;
    xmlChar *keyword, *attribute;

    /* boilerplate to open config file, do basic checks, and return a ptr */
    option = setup_xml_parser(config, doc, "icmp_echo", injector->errbuf);
    if (option == NULL)
    {
        /* error msg already set */
        return (-1);
    }

    /* we silently ignore empty configuration sections */
    for (; option; option = option->next)
    {
         /* run through each option, pull out the attribute and apply */
         if ((xmlStrcmp(option->name, (const xmlChar *)"option")) == 0)
         {
             keyword = xmlNodeListGetString(doc, option->children, 1);
             attribute = xmlGetProp(option, (const xmlChar *)"type");
             if (attribute == NULL)
             {
                 xmlFree(keyword);
                 continue;
             }
             if (strcmp((char *)attribute, "throttle") == 0)
             {
                 injector->throttle = atoi((char *)keyword);
                 xmlFree(keyword);
                 xmlFree(attribute);
                 continue;
             }
             if (strcmp((char *)attribute, "snapshot") == 0)
             {
                 listener->pcap_snaplen = atoi((char *)keyword);
                 xmlFree(keyword);
                 xmlFree(attribute);
                 continue;
             }
             fprintf(stderr, "parser(): ignoring unknown option: %s\n",
                     attribute);
             xmlFree(keyword);
             xmlFree(attribute);
         }
    }
    xmlFreeDoc(doc);
    return (1);
}

/* EOF */
