/*
 *  $Id: injector.c,v 1.9 2004/10/20 19:03:21 mike Exp $
 *
 *  Hummingbird - Asynchronous scanning engine
 *  Originally based off of sift.c from the book 
 *  Building Open Source Network Security Tools
 *  Copyright (c) 2002 - 2005 Mike D. Schiffman <stolencreditcard@gmail.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
 
#include "./hummingbird.h"

injector_t *
create_injector(char *errbuf, void *(user_create)())
{
    injector_t *injector;

    injector = malloc(sizeof (injector_t));
    if (injector == NULL)
    {
        snprintf(errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (NULL);
    }
    memset(injector, 0, sizeof (injector_t));

    /** set some defaults which can be overridden prior to initialization */
    injector->libnet_mode = LIBNET_RAW4;
    injector->targettype  = TT_IPV4;

    /** create the user defined data structure */
    injector->user = user_create(errbuf);
    if (injector->user == NULL)
    {
        return (NULL);
    }

    return (injector);
}

int
initialize_injector(injector_t *injector, char *device, u_int16_t id, 
u_int16_t throttle, char *in, int (*builder)(), int (*user_init)())
{
    /** call the user spcified initiailize routine */
    if (user_init(injector) == -1)
    {
        /** err message already set */
        return (-1);
    }

    if (device == NULL)
    {
        device = pcap_lookupdev(injector->errbuf);
        if (device == NULL)
        {
            return (-1);
        }
    }

    injector->id          = id;
    injector->throttle    = throttle;
    injector->sent_probes = 0;
    injector->build       = builder;
    injector->l           = libnet_init(injector->libnet_mode, device,
                                    injector->errbuf);
    if (injector->l == NULL)
    {
        return (-1);
    }

    injector->src_ip = libnet_get_ipaddr4(injector->l);
    if (injector->libnet_mode == LIBNET_LINK)
    {
        /** should check for libnet advanced link mode too */
        injector->src_mac = libnet_get_hwaddr(injector->l);
        if (injector->src_mac == NULL)
        {
            /** they're the same size, everytime! I promise! */
            strcpy(injector->errbuf, injector->l->err_buf);
            return (-1);
        }
    }

    /** open the host list */
    injector->in = fopen(in, "r");
    if (injector->in == NULL)
    {
        snprintf(injector->errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (-1);
    }

    printf("injector initialized\n");
    return (1);
}
 
void *
run_injector(void *arg)
{
    u_int32_t c;
    char host[128];
    injector_t *injector;

    injector = (injector_t *)arg;
    /** pull entries from the host list and send queries */
    while (fgets(host, sizeof (host) - 1, injector->in))
    {
        if (host[0] == '#' || host[0] == '\n')
        {
            /** ignore comments and lines that begin with a '\n' */
            continue;
        }

        switch (injector->targettype)
        {
            case TT_ETH:
                break;
            case TT_IPV4:
                /** remove newline and resolve */
                host[strlen(host) - 1] = 0;
                if ((injector->dst_ip = libnet_name2addr4(injector->l, host,
                        LIBNET_RESOLVE)) == -1)
                {
                    fprintf(stderr, "\ninjector: %s (%s)\n",
                        libnet_geterror(injector->l), host);
                    continue;
                }
                break;
            case TT_IPV6:
                break;
            default:
                return (NULL);
        }

        if (injector->build((void *)injector) == -1)
        {
           fprintf(stderr, "\ninjector: build(): %s",
                   injector->errbuf);
           /** insert pause to make readable to user? threshold to quit? */
           continue;
        }

        c = libnet_write(injector->l);
        if (c == -1)
        {
            fprintf(stderr, "\ninjector: libnet_write(): %s\n",
                    libnet_geterror(injector->l));
           /** insert pause to make readable to user? threshold to quit? */
            continue;
        }

        injector->sent_probes++;
        fprintf(stderr, "injector: [%05d]\r", injector->sent_probes);

        /** slow things down */
        if (injector->throttle)
        {
            usleep(injector->throttle);
        }
    }
    /** make pretty when we exit */
    printf("\n");
    return (NULL);
}

void
destroy_injector(injector_t *injector)
{
    if (injector)
    {
        if (injector->l)
        {
            libnet_destroy(injector->l);
            if (injector->src_mac)
            {
                free (injector->src_mac);
            }
            if (injector->dst_mac)
            {
                free (injector->dst_mac);
            }
        }
        if (injector->in)
        {
            fclose(injector->in);
        }
        if (injector->user)
        {
            free(injector->user);
        }
        free(injector);
    }
}
 
/** EOF */
