/*
 *  $Id: listener.c,v 1.8 2004/10/20 08:41:07 mike Exp $
 *
 *  Hummingbird - Asynchronous scanning engine
 *  Originally based off of sift.c from the book 
 *  Building Open Source Network Security Tools
 *  Copyright (c) 2002 - 2005 Mike D. Schiffman <stolencreditcard@gmail.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
 
#include "./hummingbird.h"

listener_t *
create_listener(char *errbuf, void *(user_create)())
{
    listener_t *listener;

    listener = malloc(sizeof (listener_t));
    if (listener == NULL)
    {
        snprintf(errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (NULL);
    }
    memset(listener, 0, sizeof (listener_t));

    /** set some defaults which be overridden later prior to initialization */
    listener->pcap_snaplen = HB_SNAPLEN;
    listener->pcap_promisc = HB_PROMISC;
    listener->pcap_to_ms   = HB_TIMEOUT;
    listener->outname      = "hb.";

    /** create the user defined data structure */
    listener->user = user_create(errbuf);
    if (listener->user == NULL)
    {
        return (NULL);
    }

    return (listener);
}

int
initialize_listener(listener_t *listener, char *device, u_int16_t id,
void (*decoder)(), int (*user_init)())
{
    char filename[16];

    if (device == NULL)
    {
        device = pcap_lookupdev(listener->errbuf);
        if (device == NULL)
        {
            return (-1);
        }
    }

    listener->device = device;
    listener->id     = id;
    listener->decode = decoder;
    listener->p      = pcap_open_live(device, listener->pcap_snaplen,
                                      listener->pcap_promisc,
                                      listener->pcap_to_ms,
                                      listener->errbuf);
    if (listener->p == NULL)
    {
        return (-1);
    }

    if (pcap_datalink(listener->p) != DLT_EN10MB)
    {
        sprintf(listener->errbuf, "Hummingbird only works with ethernet.\n");
        return (-1);
    }

    /** call the user specified initiailize routine */
    if (user_init(listener) == -1)
    {
        /** err message already set */
        return (-1);
    }

    /** construct the output filename and open the descriptor */
    sprintf(filename, listener->outname);
    snprintf(filename + strlen(filename), sizeof(filename), "%d", id);

    listener->out = fopen(filename, "w");
    if (listener->out == NULL)
    {
        snprintf(listener->errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (-1);
    }

    /** unbuffer output to file */
    if (setvbuf(listener->out, NULL, _IONBF, 0) == EOF)
    {
        snprintf(listener->errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (-1);
    }

    /** start the timer */
    if (gettimeofday(&(listener->timer), NULL) == -1)
    {
        snprintf(listener->errbuf, LIBNET_ERRBUF_SIZE,
                "gettimeofday(): %s", strerror(errno));
        return (-1);
    }

    printf("listener initialized, output file will be written to hb.%d\n", id);
    return (1);
}

void *
run_listener(void *arg)
{
    listener_t *listener;

    listener = (listener_t *)arg;
    for (;;)
    {
        pcap_loop(listener->p, 1, listener->decode, (u_char *)listener);
    }
    return (NULL);
}

void
destroy_listener(listener_t *listener)
{
    if (listener)
    {
        if (listener->p)
        {
            pcap_close(listener->p);
        }
        if (listener->out)
        {
            fclose(listener->out);
        }
        if (listener->user)
        {
            free(listener->user);
        }
        free(listener);
    }
} 

/* EOF */
