/*
 *  $Id: main.c,v 1.11 2004/11/21 17:33:10 mike Exp $
 *
 *  Hummingbird - Asynchronous scanning engine
 *  Originally based off of sift.c from the book 
 *  Building Open Source Network Security Tools
 *  Copyright (c) 2002 - 2005 Mike D. Schiffman <stolencreditcard@gmail.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
 
#include "./hummingbird.h"
#include "./scantypes.h"

int
main(int argc, char **argv)
{
    int c, module_index;
    u_int16_t throttle, wait;
    char errbuf[LIBNET_ERRBUF_SIZE];
    listener_t *listener;
    injector_t *injector;
    pthread_t injector_thread, listener_thread;
    char *device, *scantype, *config, outputtype;
 
    throttle    = 0;
    config      = "hummingbird.xml";
    device      = NULL;
    wait        = 2;
    scantype    = NULL;
    outputtype  = 'h';
    while ((c = getopt(argc, argv, "c:hi:o:s:t:vw:")) != EOF)
    {
        switch (c)
        {
            case 'c':
                config = optarg;
                break;
            case 'h':
                usage(argv[0]);
                return (EXIT_SUCCESS);
                break;
            case 'i':
                device = optarg;
                break;
            case 'o':
                outputtype = (char)optarg[0];
                if (outputtype == 'c' || outputtype == 'h')
                {
                    break;
                }
                fprintf(stderr, "unknown output type \"%c\"\n", outputtype);
                usage(argv[0]);
                return (EXIT_FAILURE);
            case 's':
                if (strcmp(optarg, "showall") == 0)
                {
                    dump_scantypes();
                    return (EXIT_SUCCESS);
                }
                scantype = optarg;
                break;
            case 't':
                throttle = atoi(optarg);
                break;
            case 'v':
                version();
                return (EXIT_SUCCESS);
            case 'w':
                wait = atoi(optarg);
                break;
            default:
                usage(argv[0]);
                return (EXIT_FAILURE);
        }
    }

    if (argc - optind != 1)
    {
        fprintf(stderr, "specify a file of targets\n");
        usage(argv[0]);
        return (EXIT_FAILURE);
    }
    if (scantype == NULL)
    {
        fprintf(stderr, "specify a scantype\n");
        usage(argv[0]);
        return (EXIT_FAILURE);
    }

    /** 
     * First we figure out how to index the scanning module so we can call
     * the proper functions laster on.
     */
    module_index = load_module_index(scantype);
    if (module_index == -1)
    {
        fprintf(stderr, "unknown scantype: %s\n", scantype);
        return (EXIT_FAILURE);
    }

    printf("Hummingbird asynchronous scanning engine\n");
    printf("loading %s scan module\n", scantype);

    /** create listener control context */
    listener = create_listener(errbuf, user_create_listener[module_index]);
    if (listener == NULL)
    {
        fprintf(stderr, "create_listener(): %s\n", errbuf);
        return (EXIT_FAILURE);
    }

    /** create injector control context */
    injector = create_injector(errbuf, user_create_injector[module_index]);
    if (injector == NULL)
    {
        fprintf(stderr, "create_injector(): %s\n", errbuf);
        return (EXIT_FAILURE);
    }

    /** parse the configuration file */
    if (parser[module_index](listener, injector, config) == -1)
    {
        fprintf(stderr, "error parsing config file: %s\n", injector->errbuf);
        shutitdown(listener, injector);
        return (EXIT_FAILURE);
    }
    printf("parsed configuration file\n");

    /** initialize listener monolithic control context */
    c = initialize_listener(listener, device, getpid(), decoder[module_index],
            user_init_listener[module_index]);
    if (c == -1)
    {
        fprintf(stderr, "initialize_listener(): %s\n", listener->errbuf);
        shutitdown(listener, injector);
        return (EXIT_FAILURE);
    }

    /** initialize injector monolithic control context */
    c = initialize_injector(injector, device, getpid(), throttle, argv[optind],
            builder[module_index], user_init_injector[module_index]);
    if (c == -1)
    {
        fprintf(stderr, "initialize_injector(): %s\n", injector->errbuf);
        shutitdown(listener, injector);
        return (EXIT_FAILURE);
    }

    /** prepend header to output file */
    reporter[module_index](listener, injector, HB_PREAMBLE);

    /** spin listener thread */
    c = pthread_create(&listener_thread, NULL, run_listener, listener);
    if (c != 0) 
    {
        fprintf(stderr, "pthread_create() run_listener: %s\n", strerror(errno));
        shutitdown(listener, injector);
        return (EXIT_FAILURE);
    }
    printf("listener thread spun\n");

    /** spin injector thread */
    c = pthread_create(&injector_thread, NULL, run_injector, injector);
    if (c != 0)
    {
        fprintf(stderr, "pthread_create() run_injector: %s\n", strerror(errno));
        shutitdown(listener, injector);
        return (EXIT_FAILURE);
    }
    printf("injector thread spun\n");

    /** wait for injector thread to finish */
    c = pthread_join(injector_thread, NULL);
    if (c != 0)
    {
        fprintf(stderr, "pthread_join(): %s\n", strerror(errno));
        return (EXIT_FAILURE);
    }
    printf("injector: completed, waiting %d seconds for responses\n", wait);
    sleep(wait);

    /** report summary of results to output file */
    reporter[module_index](listener, injector, HB_DENOUEMENT);

    /** shutdown */
    shutitdown(listener, injector);
    printf("all clear\n");

    return (EXIT_SUCCESS);
}

void
dump_scantypes()
{
    int c;

    for (c = 0; scantype_table[c].name; c++)
    {
        printf("%s version %s\n", scantype_table[c].name,
                scantype_table[c].version);
    }
}

int
load_module_index(char *scantype)
{
    int c;

    for (c = 0; scantype_table[c].name; c++)
    {
        if (strcmp(scantype, scantype_table[c].name) == 0)
        {
            return (c);
        }
    }
    return (-1);
}

void
shutitdown(listener_t *listener, injector_t *injector)
{
    destroy_listener(listener);
    destroy_injector(injector);
}

void
version()
{
    printf("Hummingbird asynchronous scanning engine version %s ", VERSION);
    printf("(Mike Schiffman <stolencreditcard@gmail.com>)\n");
    printf("libraries used:\n");
    printf("%s\n", libnet_version());
#if (HAVE_PCAP_LIB_VERSION)
    printf("%s\n", pcap_lib_version());
#else
    printf("can't determine libpcap version\n");
#endif
    printf("libxml2 version %s\n", LIBXML_DOTTED_VERSION);
    printf("modules loaded:\n");
    dump_scantypes();
}

void
usage(char *name)
{
    printf("usage %s [options] -s scantype targetlist\n"
    "-c config\tspecify configuration file\n"
    "-i device\tspecify network interface\n"
/*"-o outputtype\tset type of output, 'h' for human readable, 'c' for CSV\n"*/
    "-s scantype\tset scantype; use -s showall to see supported types\n"
    "-t ms\t\tmicroseconds to usleep after every packet sent by injector\n"
    "-w seconds\tafter injector finishes, time to wait for late responses\n"
    "-v\t\tversion\n", name);
}

/** EOF */
