/*
 *  $Id: tcp_port.c,v 1.6 2004/10/22 20:16:43 mike Exp $
 *
 *  Hummingbird - Asynchronous scanning engine
 *  Originally based off of sift.c from the book 
 *  Building Open Source Network Security Tools
 *  Copyright (c) 2002 - 2005 Mike D. Schiffman <stolencreditcard@gmail.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
 
#include "./hummingbird.h"

void *
tcp_port_user_create_injector(char *errbuf)
{
    struct tcp_port_injector *tcp_port_injector;
    
    tcp_port_injector = malloc(sizeof (struct tcp_port_injector));
    if (tcp_port_injector == NULL)
    {
        snprintf(errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (NULL);
    }
    return ((void *)tcp_port_injector);
}

int
tcp_port_user_init_injector(injector_t *injector)
{
    struct tcp_port_injector *tcp_port_injector;

    tcp_port_injector = (struct tcp_port_injector *)injector->user;

    tcp_port_injector->tcp = LIBNET_PTAG_INITIALIZER;
    tcp_port_injector->ip  = LIBNET_PTAG_INITIALIZER;
    //tcp_port_injector->dst_port = 80;

    return (1);
}

void *
tcp_port_user_create_listener(char *errbuf)
{
    struct tcp_port_listener *tcp_port_listener;
    
    tcp_port_listener = malloc(sizeof (struct tcp_port_listener));
    if (tcp_port_listener == NULL)
    {
        snprintf(errbuf, LIBNET_ERRBUF_SIZE, strerror(errno));
        return (NULL);
    }

    return ((void *)tcp_port_listener);
}

int
tcp_port_user_init_listener(listener_t *listener)
{
    struct bpf_program filter_code;
    bpf_u_int32 local_net, netmask;
    struct tcp_port_listener *tcp_port_listener;
    char buf[BUFSIZ];

    tcp_port_listener = (struct tcp_port_listener *)listener->user;

    tcp_port_listener->total_responses = 0;
    tcp_port_listener->synacks         = 0;
    tcp_port_listener->rsts            = 0;
    memset(buf, 0, sizeof (buf));
    sprintf(buf, TCP_PORT_FILTER, tcp_port_listener->dst_port, TCP_SOURCE_PORT);
    tcp_port_listener->pcap_filter     = buf;
    /* get the subnet mask of the interface */
    if (pcap_lookupnet(listener->device, &local_net, &netmask,
            listener->errbuf) == -1)
    {
        return (-1);
    }

    /* compile the BPF filter code */
    if (pcap_compile(listener->p, &filter_code,
            tcp_port_listener->pcap_filter, 1, netmask) == -1)
    {
        snprintf(listener->errbuf, LIBNET_ERRBUF_SIZE,
                "pcap_compile(): %s (%s)", pcap_geterr(listener->p),
                tcp_port_listener->pcap_filter);
        return (-1);
    }

    /* apply the filter to the interface */
    if (pcap_setfilter(listener->p, &filter_code) == -1)
    {
        snprintf(listener->errbuf, LIBNET_ERRBUF_SIZE,
                "pcap_setfilter(): %s", pcap_geterr(listener->p));
        return (-1);
    }

    return (1);
}

int
tcp_port_builder(void *arg)
{
    u_int32_t packet_size;
    injector_t *injector;
    struct tcp_port_injector *user;

    injector = (injector_t *)arg;
    user     = (struct tcp_port_injector *)injector->user;

    packet_size = LIBNET_IPV4_H + LIBNET_TCP_H + user->message_s;

    /*
     *  Build a tcp packet. We save the ptag after the first usage so future
     *  calls will modify this packet header template rather than build a new
     *  one.
     */
    user->tcp = libnet_build_tcp(TCP_SOURCE_PORT, user->dst_port, injector->id,
        0, TH_SYN, 32767, 0, 0, LIBNET_TCP_H + user->message_s, 
        (u_int8_t *)user->message, user->message_s, injector->l, user->tcp);
    if (user->tcp == -1)
    {
        sprintf(injector->errbuf, "Can't build TCP header: %s\n",
                libnet_geterror(injector->l));
        return (-1);
    }

    user->ip = libnet_build_ipv4(packet_size, 0, 242, 0, 64, IPPROTO_TCP,
        0, injector->src_ip, injector->dst_ip, NULL, 0, injector->l, user->ip);
    if (user->ip == -1)
    {
        sprintf(injector->errbuf, "Can't build IP header: %s\n",
                libnet_geterror(injector->l));
        return (-1);
    }
    return (1);
}

void
tcp_port_decoder(u_char *arg, const struct pcap_pkthdr *pcap_hdr,
const u_char *packet)
{
    u_int16_t ip_hl;
    listener_t *listener;
    struct libnet_ipv4_hdr *ip;
    struct libnet_tcp_hdr *tcp;
    struct tcp_port_listener *user;

    listener = (listener_t *)arg;
    user     = (struct tcp_port_listener *)listener->user;

    if (packet == NULL)
    {
        /*
         *  We have to be careful here as pcap_next() can return NULL
         *  if the timer expires with no data in the packet buffer or
         *  under some special circumstances under linux.
         */
        return;
    }

    ip = (struct libnet_ipv4_hdr *)(packet + LIBNET_ETH_H);
    ip_hl = ip->ip_hl << 2;
    tcp = (struct libnet_tcp_hdr *)(packet + LIBNET_ETH_H + ip_hl);

    /* 
     * Not sure why some hosts set the ACK to be one larger than the initial
     * sequence number and some set it to be one larger than the total initial
     * packet size. It must depend on the target host TCP stack's handling of
     * TCP SYNs with data. If they accept the data, it's ISN + data + 1, if not
     * it's ISN + 1.
     */
    if (ntohl(tcp->th_ack) == listener->id + 1 ||
        ntohl(tcp->th_ack) == listener->id + user->message_s + 1)
    {
        fprintf(listener->out, "%s,%d:\t\t",
                libnet_addr2name4(ip->ip_src.s_addr, LIBNET_DONT_RESOLVE),
                ntohs(tcp->th_sport));

        if ((tcp->th_flags & TH_SYN) && (tcp->th_flags & TH_ACK))
        {
            fprintf(listener->out, "open\n");
            user->total_responses++;
            user->synacks++;
        }
        else if ((tcp->th_flags) & TH_RST)
        {
            fprintf(listener->out, "closed\n");
            user->total_responses++;
            user->rsts++;
        }
    }
}

void
tcp_port_reporter(listener_t *listener, injector_t *injector, int mode)
{
    struct timeval r, e;
    struct tcp_port_injector *user_i;
    struct tcp_port_listener *user_l;

    user_l = (struct tcp_port_listener *)listener->user;
    user_i = (struct tcp_port_injector *)injector->user;

    switch (mode)
    {
        case HB_PREAMBLE:
            fprintf(listener->out,
                "----------------------------------------------\n"
                "Hummingbird asynchronous scanning engine\n"
                "----------------------------------------------\n"
                "starting TCP port scan to port %d\n"
                "----------------------------------------------\n",
                user_i->dst_port);
            break;
        case HB_DENOUEMENT:
            gettimeofday(&e, NULL);
            PTIMERSUB(&e, &(listener->timer), &r);
            fprintf(listener->out,
                "----------------------------------------------\n"
                "running time:\t\t\t%6d seconds\n"
                "----------------------------------------------\n"
                "ports open:\t\t\t\t%6d\n"
                "ports closed:\t\t\t\t%6d\n"
                "----------------------------------------------\n"
                "total probes sent:\t\t\t%6d\n"
                "total responses received:\t\t%6d (%.02f%% of probes sent)\n"
                "total timeouts or drops:\t\t%6d (%.02f%% of probes sent)\n"
                "----------------------------------------------\n",
                (u_int32_t)r.tv_sec, user_l->synacks, user_l->rsts,
                injector->sent_probes, user_l->total_responses,
                ((float)user_l->total_responses /
                (float)injector->sent_probes) * 100,
                injector->sent_probes - user_l->total_responses,
                ((float)(injector->sent_probes - user_l->total_responses) /
                (float)injector->sent_probes) * 100);
            if (user_i->message)
            {
                /* don't have a place for this, do we need user_destroy()? */
                free(user_i->message);
            }
            break;
    }   
}

int
tcp_port_parser(listener_t *listener, injector_t *injector, char *config)
{
    xmlDocPtr doc = NULL;
    xmlNodePtr option;
    xmlChar *keyword, *attribute;
    struct tcp_port_injector *user_i;
    struct tcp_port_listener *user_l;

    user_i = (struct tcp_port_injector *)injector->user;
    user_l = (struct tcp_port_listener *)listener->user;

    /* boilerplate to open config file, do basic checks, and return a ptr */
    option = setup_xml_parser(config, doc, "tcp_port", injector->errbuf);
    if (option == NULL)
    {
        /* error msg already set */
        return (-1);
    }

    /* we silently ignore empty configuration sections */
    for (; option; option = option->next)
    {
         /* run through each option, pull out the attribute and apply */
         if ((xmlStrcmp(option->name, (const xmlChar *)"option")) == 0)
         {
             keyword = xmlNodeListGetString(doc, option->children, 1);
             attribute = xmlGetProp(option, (const xmlChar *)"type");
             if (attribute == NULL)
             {
                 xmlFree(keyword);
                 continue;
             }
             if (strcmp((char *)attribute, "throttle") == 0)
             {
                 injector->throttle = atoi((char *)keyword);
                 xmlFree(keyword);
                 xmlFree(attribute);
                 continue;
             }
             if (strcmp((char *)attribute, "snapshot") == 0)
             {
                 listener->pcap_snaplen = atoi((char *)keyword);
                 xmlFree(keyword);
                 xmlFree(attribute);
                 continue;
             }
             if (strcmp((char *)attribute, "dst_port") == 0)
             {
                 user_i->dst_port = atoi((char *)keyword);
                 user_l->dst_port = atoi((char *)keyword);
                 xmlFree(keyword);
                 xmlFree(attribute);
                 continue;
             }
             if (strcmp((char *)attribute, "message") == 0)
             {
                 user_i->message   = strdup((char *)keyword);
                 user_i->message_s = strlen((char *)keyword);
                 user_l->message_s = strlen((char *)keyword);
                 xmlFree(keyword);
                 xmlFree(attribute);
                 continue;
             }
             fprintf(stderr, "parser(): ignoring unknown option: %s\n",
                     attribute);
             xmlFree(keyword);
             xmlFree(attribute);
         }
    }
    xmlFreeDoc(doc);
    return (1);
}

/* EOF */
