/*
 *  $Id$
 *
 *  libnet.h    -   Network routine library
 *
 *  Copyright (c) 1998 route|daemon9 <route@infonexus.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef __LIBNET_H
#define __LIBNET_H

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <ctype.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#if !(__linux__)
#include <netinet/ip_var.h>
#endif
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <netdb.h>
#include <errno.h>
#include <assert.h>


#if (__DEBUG)
#define DEBUG(fmt, arg...) fprintf(stderr, fmt, ##arg)
#else
#define DEBUG(fmt, arg...)
#endif

/*
 *  Linux specific issues.
 */

#if (__linux__)
#define uh_sport source 
#define uh_dport dest
#define uh_ulen  len
#define uh_sum   check
#define icmp     icmphdr
#define IP_MAXPACKET 65535


/*
 *  Linux has a radically different IP options structure from BSD.
 */

struct ipoption {
    struct  in_addr ipopt_dst;          /* first-hop dst if source routed */
    char ipopt_list[MAX_IPOPTLEN];      /* options proper */
};
#endif


/*
 *  TCP options structure..
 */

struct tcpoption
{
    u_char tcpopt_list[MAX_IPOPTLEN];
};


/*
 *  Some BSD variants have this endianess problem.
 */

#if (BSD_BYTE_SWAP)
#define FIX(n)    (n)
#define UNFIX(n)  (n)
#else
#define FIX(n)    htons(n)
#define UNFIX(n)  ntohs(n)
#endif


/* 
 *  Standard (IPv4) header sizes in bytes.  OH WAIT.  THESE ARE NETWORK STRUCTURES.
 *  I'M SORRY.  OCTETS.
 */

#define ICMP_H  0x8
#define UDP_H   0x8
#define TCP_H   0x14
#define IP_H    0x14
#define P_H     0xc


/*
 *  prand constants
 */

#define PR2     0
#define PR8     1
#define PR16    2
#define PRu16   3
#define PR32    4
#define PRu32   5


/*
 *  The psuedoheader for TCP/UDP checksums.
 */

struct psuedohdr
{
    u_long ip_src;      /* source IP address */
    u_long ip_dst;      /* destination IP address */
    u_char null;        /* padding octect */
    u_char protocol;    /* TCP or UDP */
    u_short len;        /* packet size */
};





/*
 *  Seeds the pseudorandom number generator with gettimeofday.
 */

int
seed_prand();


/*
 *  Returns a psuedorandom positive integer.
 */

u_long
get_prand(
    int                 /* One of the PR* constants */
    );


/*
 *  Calculates IPv4 family checksum on packet headers.
 */

int                     /* 1 on success, -1 on failure */
do_checksum(
    u_char *,           /* Pointer to the packet buffer */
    int,                /* Protocol */
    int                 /* Packet size */
    );


/*
 *  Network byte order into IP address
 *  Previous versions had a memory leak (returned a strdup'd pointer -- strdup
 *  has an implicit malloc which wasn't getting freed).  This static var hack
 *  thingy was used to preserve existing code without having to change much.
 *  You can simply use the return value of the function directly allowing you
 *  to write tighter, more obvious code (rather then having to do allocate an
 *  additional buffer for the output).
 *  Thanks to Red for the idea.
 */

u_char *                /* Pointer to hostname or dotted decimal IP address */
host_lookup(
    u_long,             /* Network byte ordered (big endian) IP address */
    u_short             /* Use domain names or no */
    );


/*
 *  Network byte order into IP address
 *  Threadsafe version.
 */

void
host_lookup_r(
    u_long,         /* Network byte ordered (big endian) IP address */
    u_short,        /* Use domain names or no */
    u_char *        /* Pointer to hostname or dotted decimal IP address */
    );


/*
 *  IP address into network byte order
 */

u_long                  /* Network byte ordered IP address or -1 on error */
name_resolve(
    u_char *,           /* Pointer the hostname or dotted decimal IP address */
    u_short             /* Use domain names or no */
    );


/*
 *  Fast x86 TCP checksum.  This function builds it's own psuedoheader.
 */
 
u_short                 /* Standard TCP checksum of header and data */
tcp_check(
    struct tcphdr *,    /* TCP header pointer */
    int,                /* Packet length */
    u_long,             /* Source IP address */
    u_long              /* Destination IP address */
    );


/*
 *  Fast x86 IP checksum.
 */

u_short                 /* Standard IP checksum of header and data */
ip_check(
    u_short *,          /* Pointer to the buffer to be summed */
    int len             /* Packet length */
    );


/*
 *  Opens a socket for writing raw IP datagrams to.  Set IP_HDRINCL to let the 
 *  kernel know we've got it all under control.
 */

int                     /* Opened file desciptor, or -1 on error */
open_raw_sock(
    int                 /* Protocol of raw socket (from /etc/protocols) */
    );


/*
 *  TCP packet assembler.
 */

void
build_tcp(
    u_short,        /* Source port */
    u_short,        /* Destination port */
    u_long,         /* Sequence Number */
    u_long,         /* Acknowledgement Number */
    u_char,         /* Control bits */
    u_short,        /* Advertised Window Size */
    u_short,        /* Urgent Pointer */
    const u_char *, /* Pointer to packet data (or NULL) */
    int,            /* Packet payload size */
    u_char *        /* Pointer to packet header memory */
    );


/*
 * UDP packet assembler.
 */

void
build_udp(
    u_short,        /* Source port */
    u_short,        /* Destination port */
    const u_char *, /* Pointer to packet data (or NULL) */
    int,            /* Packet payload size */
    u_char *        /* Pointer to packet header memory */
    );


/*
 *  IP packet assembler.
 */

void
build_ip(
    u_short,        /* Length of packet data */
    u_short,        /* IP ID */
    u_short,        /* Fragmentation flags and offset */
    u_char,         /* TTL */
    u_char,         /* Protocol */
    u_long,         /* Source IP Address */
    u_long,         /* Destination IP Address */
    const u_char *, /* Pointer to packet data (or NULL) */
    int,            /* Packet payload size */
    u_char *        /* Pointer to packet header memory */
    );


/*
 *  Insert IP options to a prebuilt IP packet.
 */

int                 /* 1 on success, -1 on failure */
insert_ipo(
    struct ipoption *, /* Pointer to the ip options structure */ 
    u_char,         /* IP option list size */
    u_char *        /* Pointer to packet buf */
    );

/*
 *  Insert TCP options to a prebuilt IP packet.
 */

int                 /* 1 on success, -1 on failure */
insert_tcpo(
    struct tcpoption *, /* Pointer to the tcp options structure */ 
    u_char,         /* TCP option list size */
    u_char *        /* Pointer to packet buf */
    );


int                 /* 1 if successful, -1 on error */
write_ip(
    int sock,       /* Previously opened raw socket */
    const u_char *, /* Pointer a complete IP datagram */
    int             /* Packet size */
    );


#endif  /* __LIBNET_H */

/* EOF */
