/*
 *  $Id: write_ip.c,v 1.1.1.1 1998/10/16 17:53:09 route Exp $
 *
 *  libnet
 *  write_ip implementation
 *  write_ip.c - writes a prebuilt IP packet to the network
 *
 *  Copyright (c) 1998 route|daemon9 <route@infonexus.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include "../include/libnet.h"

int
write_ip(int sock, const u_char *buf, int len)
{
    int c, s;
    struct sockaddr_in sin;
    struct ip *ip_hdr;
    struct udphdr *udp_hdr;
    struct tcphdr *tcp_hdr;

    ip_hdr = (struct ip *)(buf);

    memset(&sin, 0, sizeof(struct sockaddr_in));
    sin.sin_family      = AF_INET;
    sin.sin_addr.s_addr = ip_hdr->ip_dst.s_addr;


    /*
     *  Figure out which transport (if any) we are using and fill in the 
     *  socket address port information accordingly.  The relative complexity
     *  here is the price we pay for the simple interface.
     */
    switch (ip_hdr->ip_p)
    {
        case IPPROTO_TCP:
            tcp_hdr = (struct tcphdr *)(buf + IP_H);
            sin.sin_port = tcp_hdr->th_sport; 
            break;
        case IPPROTO_UDP:
            udp_hdr = (struct udphdr *)(buf + IP_H);
            sin.sin_port = udp_hdr->uh_sport; 
            break;
        case IPPROTO_ICMP:
        case IPPROTO_IP:
            sin.sin_port = 0;   /* Unused in this case */
            break;
        default:
            fprintf(stderr, "Unknown transport\n");
            return (-1);
    }

    s = sizeof(struct sockaddr);
    c = sendto(sock, buf, len, 0, (struct sockaddr *)&sin, s);
    if (c != len)
    {
        /*
         *  Up to the user to check for error.
         */
        DEBUG("write_ip: error: %d bytes written (%s)\n", c, strerror(errno));
    }
    else
    {
        DEBUG("write_ip: wrote %d bytes\n", c);
    }
    return (c);
}

/* EOF */
