#define PRISM2_PCI

/* Host AP driver's support for Intersil Prism2.5 PCI cards is based on
 * driver patches from Reyk Floeter <reyk@vantronix.net> and
 * Andy Warner <andyw@pobox.com> */

#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/wireless.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>

#include <linux/ioport.h>
#include <linux/pci.h>

#include "prism2_config.h"


static char *version =
"hostap_pci.c " PRISM2_VERSION " (SSH Communications Security Corp, "
"Jouni Malinen)";
static char *dev_info = "hostap_pci";


MODULE_AUTHOR("SSH Communications Security Corp, Jouni Malinen");
MODULE_DESCRIPTION("Support for Intersil Prism2.5-based 802.11 wireless LAN "
		   "PCI cards.");
MODULE_SUPPORTED_DEVICE("Intersil Prism2.5-based WLAN PCI cards");
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif


#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
/* PCI initialization uses Linux 2.4.x version and older kernels do not support
 * this */
#error Prism2.5 PCI version requires at least Linux kernel version 2.4.0
#endif /* kernel < 2.4.0 */


#include "prism2_wlan.h"


/* FIX: do we need mb/wmb/rmb with memory operations? */


static struct pci_device_id prism2_pci_id_table[] __devinitdata = {
	/* Intersil Prism2.5 ISL3874 11Mb/s WLAN Controller */
	{ 0x1260, 0x3873, PCI_ANY_ID, PCI_ANY_ID },
	{ 0 }
};


/* FIX: This might change at some point.. */
#include "prism2.c"

/* FIX: can jiffies be used this way from every place cor_sreset could be
 * called? (mainly, spin_lock_irq? hw interrupt handler?) */

static void prism2_pci_cor_sreset(local_info_t *local)
{
	struct net_device *dev = local->dev;

	/* linux-wlan-ng uses extremely long hold and settle times for
	 * COR sreset. A comment in the driver code mentions that the long
	 * delays appear to be necessary. However, at least IBM 22P6901 seems
	 * to work fine with shorter delays.
	 *
	 * Longer delays can be configured by uncommenting following line: */
/* #define PRISM2_PCI_USE_LONG_DELAYS */

#ifdef PRISM2_PCI_USE_LONG_DELAYS
	int timeout;

	HFA384X_OUTW(0x0080, HFA384X_PCICOR_OFF);
	timeout = jiffies + HZ / 4;
	while (time_before(jiffies, timeout))
		udelay(5);

	HFA384X_OUTW(0x0, HFA384X_PCICOR_OFF);
	timeout = jiffies + HZ / 2;
	while (time_before(jiffies, timeout))
		udelay(5);

	/* Wait for f/w to complete initialization (CMD:BUSY == 0) */
	timeout = jiffies + 2 * HZ;
	while ((HFA384X_INW(HFA384X_CMD_OFF) & HFA384X_CMD_BUSY) &&
	       time_before(jiffies, timeout))
		udelay(10);

#else /* PRISM2_PCI_USE_LONG_DELAYS */

	HFA384X_OUTW(0x0080, HFA384X_PCICOR_OFF);
	mdelay(1);
	HFA384X_OUTW(0x0, HFA384X_PCICOR_OFF);
	mdelay(1);

#endif /* PRISM2_PCI_USE_LONG_DELAYS */

	if (HFA384X_INW(HFA384X_CMD_OFF) & HFA384X_CMD_BUSY) {
		printk(KERN_DEBUG "%s: COR sreset timeout\n", dev->name);
	}
}


static struct prism2_helper_functions prism2_pci_funcs =
{
	card_present: NULL,
	cor_sreset: prism2_pci_cor_sreset
};


static int prism2_pci_probe(struct pci_dev *pdev,
			    const struct pci_device_id *id)
{
	unsigned long phymem;
	unsigned long mem = 0;
	local_info_t *local = NULL;
	struct net_device *dev;

	if (pci_enable_device(pdev))
		return -EIO;

	phymem = pci_resource_start(pdev, 0);

	if (!request_mem_region(phymem, pci_resource_len(pdev, 0), "Prism2")) {
		printk(KERN_ERR "prism2: Cannot reserve PCI memory region\n");
		goto err_out_disable;
	}

	mem = (unsigned long) ioremap(phymem, pci_resource_len(pdev, 0));
	if (!mem) {
		printk(KERN_ERR "prism2: Cannot remap PCI memory region\n") ;
		goto fail;
	}

	local = prism2_init_local_data(&prism2_pci_funcs);
	if (local == NULL)
		goto fail;

	dev = local->dev;

        dev->irq = pdev->irq;
        dev->mem_start = mem;
        dev->mem_end = mem + pci_resource_len(pdev, 0);

	if (prism2_init_dev(local))
		goto fail;

	prism2_pci_cor_sreset(local);

	if (prism2_hw_config(dev, 1)) {
		printk(KERN_DEBUG "%s: hardware initialization failed\n",
		       dev_info);
		goto fail;
	}
	prism2_check_sta_fw_version(local);

	pci_set_drvdata(pdev, dev);

	if (request_irq(dev->irq, prism2_interrupt, SA_SHIRQ, dev->name,
			dev)) {
		printk(KERN_WARNING "%s: request_irq failed\n", dev->name);
		goto fail;
	}

	printk(KERN_INFO "%s: Intersil Prism2.5 PCI: "
	       "mem=0x%lx, irq=%d\n", dev->name, phymem, dev->irq);

	return 0;

 fail:
	prism2_free_local_data(local);

	if (mem)
		iounmap((void *) mem);

	release_mem_region(phymem, pci_resource_len(pdev, 0));

 err_out_disable:
	pci_disable_device(pdev);

	return -ENODEV;
}


static void prism2_pci_remove(struct pci_dev *pdev)
{
	struct net_device *dev = pci_get_drvdata(pdev);
	local_info_t *local = (local_info_t *) dev->priv;
	unsigned long mem_start;

	/* Reset the hardware, and ensure interrupts are disabled. */
	prism2_pci_cor_sreset(local);
	hfa384x_disable_interrupts(dev);

	if (dev->irq)
		free_irq(dev->irq, dev);

	mem_start = dev->mem_start;
	prism2_free_local_data(local);

	iounmap((void *) mem_start);

	release_mem_region(pci_resource_start(pdev, 0),
			   pci_resource_len(pdev, 0));

	pci_disable_device(pdev);
}


MODULE_DEVICE_TABLE(pci, prism2_pci_id_table);

static struct pci_driver prism2_pci_drv_id = {
	name: "prism2_pci",
	id_table: prism2_pci_id_table,
	probe: prism2_pci_probe,
	remove: prism2_pci_remove,
	save_state: NULL,
	suspend: NULL,
	resume: NULL,
	enable_wake: NULL
};


static int __init init_prism2_pci(void)
{
	printk(KERN_INFO "%s: %s\n"
	       "%s: (c) SSH Communications Security Corp <jkm@ssh.com>\n",
	       dev_info, version, dev_info);

	if (pci_register_driver(&prism2_pci_drv_id) <= 0) {
		printk("prism2_pci: No devices found, driver not "
		       "installed.\n");
		pci_unregister_driver(&prism2_pci_drv_id);
		return -ENODEV;
	}

	return 0;
}


static void __exit exit_prism2_pci(void)
{
	pci_unregister_driver(&prism2_pci_drv_id);
	prism2_cleanup_proc();
	printk(KERN_INFO "%s: Driver unloaded\n", dev_info);
}


module_init(init_prism2_pci);
module_exit(exit_prism2_pci);
