/* ioctl() (mostly Linux Wireless Extensions) routines for Host AP driver */

#ifdef WIRELESS_EXT

static struct iw_statistics *prism2_get_wireless_stats(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;

	local->wstats.status = 0;
	local->wstats.discard.code =
		local->comm_tallies.rx_discards_wep_undecryptable;
	local->wstats.discard.misc =
		local->comm_tallies.rx_fcs_errors +
		local->comm_tallies.rx_discards_no_buffer +
		local->comm_tallies.tx_discards_wrong_sa;

	if (local->iw_mode != IW_MODE_MASTER) {
		struct hfa384x_comms_quality sq;
		if (hfa384x_get_rid(local->dev,
					   HFA384X_RID_COMMSQUALITY,
					   &sq, sizeof(sq), 1)) {
			local->wstats.qual.qual = le16_to_cpu(sq.comm_qual);
			local->wstats.qual.level = HFA384X_LEVEL_TO_dBm(
				le16_to_cpu(sq.signal_level));
			local->wstats.qual.noise = HFA384X_LEVEL_TO_dBm(
				le16_to_cpu(sq.noise_level));
			local->wstats.qual.updated = 7;
		}
	}

	return &local->wstats;
}


#if WIRELESS_EXT > 8
static int prism2_get_key_idx(struct net_device *dev, int idx)
{
	u16 val;

	if (idx < 1 || idx > 4) {
		if (hfa384x_get_rid(dev,
				    HFA384X_RID_CNFWEPDEFAULTKEYID,
				    &val, 2, 1) < 0)
			return -1;
		le16_to_cpus(&val);
		if (val > 3) {
			printk("Invalid CNFWEPDEFAULT_KEYID %d\n", val);
			return -1;
		}

		return val;
	}

	return idx - 1;
}


static int prism2_ioctl_siwencode(struct net_device *dev, struct iw_point *erq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int i;

	i = prism2_get_key_idx(dev, erq->flags & IW_ENCODE_INDEX);
	if (i < 0 || i >= WEP_KEYS)
		return -EINVAL;

	if (erq->pointer != (caddr_t) 0) {
		char keybuf[WEP_KEY_LEN + 1];

		if (erq->length > WEP_KEY_LEN)
			return -EINVAL;

		memset(keybuf, 0, sizeof(keybuf));
		if (copy_from_user(keybuf, erq->pointer, erq->length))
			return -EFAULT;
		memcpy(local->wep_keys[i], keybuf, WEP_KEY_LEN + 1);
		local->wep_key_len[i] = erq->length <= 5 ? 5 : 13;
	} else
		local->wep_tx_key = i;

	if (erq->flags & IW_ENCODE_OPEN)
		local->open_wep = 1;
	else
		local->open_wep = 0;

	if ((erq->pointer != (caddr_t) 0 && erq->length == 0) ||
	    erq->flags & IW_ENCODE_DISABLED)
		local->use_wep = 0;
	else
		local->use_wep = 1;

	if (prism2_set_encryption(local) || prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}


static int prism2_ioctl_giwencode(struct net_device *dev, struct iw_point *erq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int i;
	u16 val;
	char keybuf[14];
	int keylen;

	if (erq->pointer == (caddr_t) 0)
		return -EINVAL;

	if (local->iw_mode == IW_MODE_MASTER)
		i = local->wep_tx_key;
	else
		i = prism2_get_key_idx(dev, erq->flags & IW_ENCODE_INDEX);
	if (i < 0 || i >= WEP_KEYS)
		return -EINVAL;

	erq->flags = i + 1;

	/* Reads from HFA384X_RID_CNFDEFAULTKEY* return bogus values, so show
	 * the keys from driver buffer */
	keylen = hfa384x_get_rid(dev, HFA384X_RID_CNFDEFAULTKEY0 + i,
				 keybuf, sizeof(keybuf), 0);
	if (copy_to_user(erq->pointer, local->wep_keys[i],
			 local->wep_key_len[i]))
		return -EFAULT;
	else
		erq->length = local->wep_key_len[i];

	if (hfa384x_get_rid(dev, HFA384X_RID_CNFWEPFLAGS, &val, 2, 1) < 0) {
		printk("CNFWEPFLAGS reading failed\n");
		return -EOPNOTSUPP;
	}
	le16_to_cpus(&val);
	if (val & HFA384X_WEPFLAGS_PRIVACYINVOKED)
		erq->flags |= IW_ENCODE_ENABLED;
	else
		erq->flags |= IW_ENCODE_DISABLED;
	if (val & HFA384X_WEPFLAGS_EXCLUDEUNENCRYPTED)
		erq->flags |= IW_ENCODE_RESTRICTED;
	else
		erq->flags |= IW_ENCODE_OPEN;

	return 0;
}


static int prism2_ioctl_giwspy(struct net_device *dev, struct iw_point *srq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct sockaddr addr[IW_MAX_SPY];
	struct iw_quality qual[IW_MAX_SPY];

	if (srq->pointer == NULL) {
		printk(KERN_DEBUG "%s: SIOCGIWSPY: pointer == NULL\n",
		       dev->name);
		srq->length = 0;
		return -EINVAL;
	}

	if (local->iw_mode != IW_MODE_MASTER) {
		printk("SIOCGIWSPY is currently only supported in Host AP "
		       "mode\n");
		srq->length = 0;
		return -EOPNOTSUPP;
	}

	srq->length = prism2_ap_get_sta_qual(local, addr, qual, IW_MAX_SPY, 0);

	if (copy_to_user(srq->pointer, &addr, sizeof(addr[0]) * srq->length))
		return -EFAULT;
	if (copy_to_user(srq->pointer + sizeof(addr[0]) * srq->length, &qual,
			 sizeof(qual[0]) * srq->length))
		return -EFAULT;

	return 0;
}


static int prism2_ioctl_siwrate(struct net_device *dev, struct iw_param *rrq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int ret = 0;

	if (rrq->fixed) {
		switch (rrq->value) {
		case 11000000:
			local->tx_rate_control = HFA384X_RATES_11MBPS;
			break;
		case 5500000:
			local->tx_rate_control = HFA384X_RATES_5MBPS;
			break;
		case 2000000:
			local->tx_rate_control = HFA384X_RATES_2MBPS;
			break;
		case 1000000:
			local->tx_rate_control = HFA384X_RATES_1MBPS;
			break;
		default:
			local->tx_rate_control = HFA384X_RATES_1MBPS |
				HFA384X_RATES_2MBPS | HFA384X_RATES_5MBPS |
				HFA384X_RATES_11MBPS;
			break;
		}
	} else {
		switch (rrq->value) {
		case 11000000:
			local->tx_rate_control = HFA384X_RATES_1MBPS |
				HFA384X_RATES_2MBPS | HFA384X_RATES_5MBPS |
				HFA384X_RATES_11MBPS;
			break;
		case 5500000:
			local->tx_rate_control = HFA384X_RATES_1MBPS |
				HFA384X_RATES_2MBPS | HFA384X_RATES_5MBPS;
			break;
		case 2000000:
			local->tx_rate_control = HFA384X_RATES_1MBPS |
				HFA384X_RATES_2MBPS;
			break;
		case 1000000:
			local->tx_rate_control = HFA384X_RATES_1MBPS;
			break;
		default:
			local->tx_rate_control = HFA384X_RATES_1MBPS |
				HFA384X_RATES_2MBPS | HFA384X_RATES_5MBPS |
				HFA384X_RATES_11MBPS;
			break;
		}
	}

	ret = (prism2_set_word(dev, HFA384X_RID_TXRATECONTROL,
			       local->tx_rate_control) ||
	       prism2_reset_port(dev));
		
	if (ret) {
		printk("%s: TXRateControl setting to 0x%x failed\n",
		       dev->name, local->tx_rate_control);
	}
	return ret;
}

static int prism2_ioctl_giwrate(struct net_device *dev, struct iw_param *rrq)
{
	u16 val;
	int ret = 0;

	if (hfa384x_get_rid(dev, HFA384X_RID_TXRATECONTROL, &val, 2, 1) < 0)
		return -EINVAL;

	if ((val & 0x1) && (val > 1))
		rrq->fixed = 0;
	else
		rrq->fixed = 1;

	if (hfa384x_get_rid(dev, HFA384X_RID_CURRENTTXRATE, &val, 2, 1) < 0)
		return -EINVAL;

	switch (val) {
	case HFA384X_RATES_1MBPS:
		rrq->value = 1000000;
		break;
	case HFA384X_RATES_2MBPS:
		rrq->value = 2000000;
		break;
	case HFA384X_RATES_5MBPS:
		rrq->value = 5500000;
		break;
	case HFA384X_RATES_11MBPS:
		rrq->value = 11000000;
		break;
	default:
		/* should not happen */
		rrq->value = 11000000;
		ret = -EINVAL;
		break;
	}

	return ret;
}


static int prism2_ioctl_siwsens(struct net_device *dev, struct iw_param *sens)
{
	/* Set the desired AP density */
	if (sens->value < 1 || sens->value > 3)
		return -EINVAL;

	if (prism2_set_word(dev, HFA384X_RID_CNFSYSTEMSCALE, sens->value) ||
	    prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}

static int prism2_ioctl_giwsens(struct net_device *dev, struct iw_param *sens)
{
	u16 val;

	/* Get the current AP density */
	if (hfa384x_get_rid(dev, HFA384X_RID_CNFSYSTEMSCALE, &val, 2, 1) < 0)
		return -EINVAL;

	sens->value = __le16_to_cpu(val);
	sens->fixed = 1;

	return 0;
}


static int prism2_ioctl_giwaplist(struct net_device *dev,
				  struct iw_point *data)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct sockaddr addr[IW_MAX_AP];
	struct iw_quality qual[IW_MAX_AP];

	if (data->pointer == NULL) {
		printk(KERN_DEBUG "%s: SIOCGIWAPLIST: pointer == NULL\n",
		       dev->name);
		data->length = 0;
		return -EINVAL;
	}

	if (local->iw_mode != IW_MODE_MASTER) {
		printk(KERN_DEBUG "SIOCGIWAPLIST is currently only supported "
		       "in Host AP mode\n");
		data->length = 0;
		return -EOPNOTSUPP;
	}

	data->length = prism2_ap_get_sta_qual(local, addr, qual, IW_MAX_AP, 1);

	if (copy_to_user(data->pointer, &addr, sizeof(addr[0]) * data->length))
		return -EFAULT;
	data->flags = 1; /* has quality information */
	if (copy_to_user(data->pointer + sizeof(addr[0]) * data->length, &qual,
			 sizeof(qual[0]) * data->length))
		return -EFAULT;

	return 0;
}


static int prism2_ioctl_siwrts(struct net_device *dev, struct iw_param *rts)
{
	u16 val;

	if (rts->disabled)
		val = __constant_cpu_to_le16(2347);
	else if (rts->value < 0 || rts->value > 2347)
		return -EINVAL;
	else
		val = __cpu_to_le16(rts->value);

	if (hfa384x_set_rid(dev, HFA384X_RID_RTSTHRESHOLD, &val, 2) ||
	    prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}

static int prism2_ioctl_giwrts(struct net_device *dev, struct iw_param *rts)
{
	u16 val;

	if (hfa384x_get_rid(dev, HFA384X_RID_RTSTHRESHOLD, &val, 2, 1) < 0)
		return -EINVAL;

	rts->value = __le16_to_cpu(val);
	rts->disabled = (rts->value == 2347);
	rts->fixed = 1;

	return 0;
}


static int prism2_ioctl_siwfrag(struct net_device *dev, struct iw_param *rts)
{
	u16 val;

	if (rts->disabled)
		val = __constant_cpu_to_le16(2346);
	else if (rts->value < 256 || rts->value > 2346)
		return -EINVAL;
	else
		val = __cpu_to_le16(rts->value & ~0x1); /* even numbers only */

	if (hfa384x_set_rid(dev, HFA384X_RID_FRAGMENTATIONTHRESHOLD, &val, 2)
	    || prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}

static int prism2_ioctl_giwfrag(struct net_device *dev, struct iw_param *rts)
{
	u16 val;

	if (hfa384x_get_rid(dev, HFA384X_RID_FRAGMENTATIONTHRESHOLD,
			    &val, 2, 1) < 0)
		return -EINVAL;

	rts->value = __le16_to_cpu(val);
	rts->disabled = (rts->value == 2346);
	rts->fixed = 1;

	return 0;
}
#endif /* WIRELESS_EXT > 8 */


static int prism2_ioctl_giwap(struct net_device *dev, struct sockaddr *ap_addr)
{
	local_info_t *local = (local_info_t *) dev->priv;

	if (hfa384x_get_rid(dev, HFA384X_RID_CURRENTBSSID, &ap_addr->sa_data,
			    ETH_ALEN, ETH_ALEN) < 0)
		return -EOPNOTSUPP;

	/* FIX: local->bssid should be updated in a better place;
	 * maybe in info frame LinkStatus reception? */
	memcpy(local->bssid, &ap_addr->sa_data, ETH_ALEN);
	ap_addr->sa_family = ARPHRD_ETHER;

	return 0;
}


#if WIRELESS_EXT > 8
static int prism2_ioctl_siwnickn(struct net_device *dev, struct iw_point *data)
{
	local_info_t *local = (local_info_t *) dev->priv;

	if (!data->pointer)
		return -EINVAL;

	if (data->length > MAX_NAME_LEN + 1)
		return -E2BIG;

	copy_from_user(local->name, data->pointer, data->length);
	local->name[MAX_NAME_LEN] = '\0';
	local->name_set = 1;

	if (prism2_set_string(dev, HFA384X_RID_CNFOWNNAME, local->name) ||
	    prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}

static int prism2_ioctl_giwnickn(struct net_device *dev, struct iw_point *data)
{
	int len;
	char name[MAX_NAME_LEN + 3];
	u16 val;

	if (!data->pointer)
		return -EINVAL;

	len = hfa384x_get_rid(dev, HFA384X_RID_CNFOWNNAME,
			      &name, MAX_NAME_LEN + 2, 0);
	val = __le16_to_cpu(*(u16 *) name);
	if (len > MAX_NAME_LEN + 2 || len < 0 || val > MAX_NAME_LEN)
		return -EOPNOTSUPP;

	name[val + 2] = '\0';
	data->length = val + 1;
	copy_to_user(data->pointer, name + 2, val);

	return 0;
}
#endif /* WIRELESS_EXT > 8 */


static int prism2_ioctl_siwfreq(struct net_device *dev, struct iw_freq *freq)
{
	local_info_t *local = (local_info_t *) dev->priv;

	/* freq => chan. */
	if (freq->e == 1 &&
	    freq->m / 100000 >= freq_list[0] &&
	    freq->m / 100000 <= freq_list[FREQ_COUNT - 1]) {
		int ch;
		int fr = freq->m / 100000;
		for (ch = 0; ch < FREQ_COUNT; ch++) {
			if (fr == freq_list[ch]) {
				freq->e = 0;
				freq->m = ch + 1;
				break;
			}
		}
	}

	if (freq->e != 0 || freq->m < 1 || freq->m > FREQ_COUNT ||
	    !(local->channel_mask & (1 << (freq->m - 1))))
		return -EINVAL;

	local->channel = freq->m; /* channel is used in prism2_setup_rids() */
	if (prism2_set_word(dev, HFA384X_RID_CNFOWNCHANNEL, local->channel) ||
	    prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}

static int prism2_ioctl_giwfreq(struct net_device *dev, struct iw_freq *freq)
{
	u16 val;

	if (hfa384x_get_rid(dev, HFA384X_RID_CURRENTCHANNEL, &val, 2, 1) < 0)
		return -EINVAL;

	le16_to_cpus(&val);
	if (val < 1 || val > FREQ_COUNT)
		return -EINVAL;

	freq->m = freq_list[val - 1] * 100000;
	freq->e = 1;

	return 0;
}


#if WIRELESS_EXT > 8
static int prism2_ioctl_siwessid(struct net_device *dev, struct iw_point *data)
{
	local_info_t *local = (local_info_t *) dev->priv;
	char ssid[MAX_SSID_LEN + 1];

	if (!data->pointer)
		return -EINVAL;

	if (data->flags == 0) {
		ssid[0] = '\0'; /* ANY */
	} else {
		if (data->length > MAX_SSID_LEN + 1)
			return -E2BIG;

		if (copy_from_user(ssid, data->pointer, data->length))
			return -EFAULT;
		ssid[MAX_SSID_LEN] = '\0';
	}

	if (local->iw_mode == IW_MODE_MASTER && ssid[0] == '\0') {
		/* Setting SSID to empty string seems to kill the card in
		 * Host AP mode */
		printk(KERN_DEBUG "%s: Host AP mode does not support "
		       "'Any' essid\n", dev->name);
		return -EINVAL;
	}

	memcpy(local->essid, ssid, sizeof(local->essid));

	if (prism2_set_string(dev, HFA384X_RID_CNFDESIREDSSID, local->essid) ||
	    prism2_set_string(dev, HFA384X_RID_CNFOWNSSID, local->essid) ||
	    prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}

static int prism2_ioctl_giwessid(struct net_device *dev, struct iw_point *data)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 val;

	if (!data->pointer)
		return -EINVAL;

	data->flags = 1; /* active */
	if (local->iw_mode == IW_MODE_MASTER) {
		data->length = strlen(local->essid) + 1;
		copy_to_user(data->pointer, local->essid,
			     sizeof(local->essid));
	} else {
		int len;
		char ssid[MAX_SSID_LEN + 3];
		len = hfa384x_get_rid(dev, HFA384X_RID_CURRENTSSID,
				      &ssid, MAX_SSID_LEN + 2, 0);
		val = __le16_to_cpu(*(u16 *) ssid);
		if (len > MAX_SSID_LEN + 2 || len < 0 || val > MAX_SSID_LEN) {
			return -EOPNOTSUPP;
		}
		ssid[val + 2] = '\0';
		data->length = val + 1;
		copy_to_user(data->pointer, ssid + 2, val);
	}

	return 0;
}


static int prism2_ioctl_giwrange(struct net_device *dev, struct iw_point *data)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct iw_range range;
	u16 val;
	int i;

	if (verify_area(VERIFY_WRITE, data->pointer, sizeof(range)))
		return -EINVAL;

	data->length = sizeof(range);
	memset(&range, 0, sizeof(range));

#if WIRELESS_EXT > 9
	/* TODO: could fill num_txpower and txpower array with
	 * something; however, there are 128 different values.. */

	range.txpower_capa = IW_TXPOW_DBM;
#endif /* WIRELESS_EXT > 9 */

#if WIRELESS_EXT > 10
	range.we_version_compiled = WIRELESS_EXT;
	range.we_version_source = 11;
#endif /* WIRELESS_EXT > 10 */

	range.min_nwid = range.max_nwid = 0;

	range.num_channels = FREQ_COUNT;

	val = 0;
	for (i = 0; i < FREQ_COUNT; i++) {
		if (local->channel_mask & (1 << i)) {
			range.freq[val].i = i + 1;
			range.freq[val].m = freq_list[i] * 100000;
			range.freq[val].e = 1;
			val++;
		}
		if (val == IW_MAX_FREQUENCIES)
			break;
	}
	range.num_frequency = val;

	range.max_qual.qual = 92; /* 0 .. 92 */
	range.max_qual.level = 154; /* 27 .. 154 */
	range.max_qual.noise = 154; /* 27 .. 154 */
	range.sensitivity = 3;

	if (copy_to_user(data->pointer, &range, sizeof(range)))
		return -EFAULT;

	return 0;
}


static int prism2_ioctl_siwmode(struct net_device *dev, __u32 mode)
{
	local_info_t *local = (local_info_t *) dev->priv;

	if (mode != IW_MODE_ADHOC && mode != IW_MODE_INFRA &&
	    mode != IW_MODE_MASTER)
		return -EOPNOTSUPP;

	if (mode == local->iw_mode)
		return 0;

	printk(KERN_DEBUG "prism2: %s: operating mode changed "
	       "%d -> %d\n", dev->name, local->iw_mode, mode);
	local->iw_mode = mode;

	if (prism2_set_word(dev, HFA384X_RID_CNFPORTTYPE,
			    prism2_get_porttype(local)))
		return -EOPNOTSUPP;

	if (prism2_reset_port(dev))
		return -EINVAL; 

	return 0;
}


static int prism2_ioctl_siwpower(struct net_device *dev, struct iw_param *wrq)
{
	int ret = 0;

	if (wrq->disabled)
		return prism2_set_word(dev, HFA384X_RID_CNFPMENABLED, 0);

	switch (wrq->flags & IW_POWER_MODE) {
	case IW_POWER_UNICAST_R:
		ret = prism2_set_word(dev, HFA384X_RID_CNFMULTICASTRECEIVE, 0);
		if (ret)
			return ret;
		ret = prism2_set_word(dev, HFA384X_RID_CNFPMENABLED, 1);
		if (ret)
			return ret;
		break;
	case IW_POWER_ALL_R:
		ret = prism2_set_word(dev, HFA384X_RID_CNFMULTICASTRECEIVE, 1);
		if (ret)
			return ret;
		ret = prism2_set_word(dev, HFA384X_RID_CNFPMENABLED, 1);
		if (ret)
			return ret;
		break;
	case IW_POWER_ON:
		break;
	default:
		return -EINVAL;
	}
		
	if (wrq->flags & IW_POWER_TIMEOUT) {
		ret = prism2_set_word(dev, HFA384X_RID_CNFPMENABLED, 1);
		if (ret)
			return ret;
		ret = prism2_set_word(dev, HFA384X_RID_CNFPMHOLDOVERDURATION,
				      wrq->value / 1024);
		if (ret)
			return ret;
	}
	if (wrq->flags & IW_POWER_PERIOD) {
		ret = prism2_set_word(dev, HFA384X_RID_CNFPMENABLED, 1);
		if (ret)
			return ret;
		ret = prism2_set_word(dev, HFA384X_RID_CNFMAXSLEEPDURATION,
				      wrq->value / 1024);
		if (ret)
			return ret;
	}

	return ret;
}


static int prism2_ioctl_giwpower(struct net_device *dev, struct iw_param *rrq)
{
	u16 enable, mcast;

	if (hfa384x_get_rid(dev, HFA384X_RID_CNFPMENABLED, &enable, 2, 1) < 0)
		return -EINVAL;

	if (!__le16_to_cpu(enable)) {
		rrq->disabled = 1;
		return 0;
	}

	rrq->disabled = 0;

	if ((rrq->flags & IW_POWER_TYPE) == IW_POWER_TIMEOUT) {
		u16 timeout;
		if (hfa384x_get_rid(dev, HFA384X_RID_CNFPMHOLDOVERDURATION,
				    &timeout, 2, 1) < 0)
			return -EINVAL;

		rrq->flags = IW_POWER_TIMEOUT;
		rrq->value = __le16_to_cpu(timeout) * 1024;
	} else {
		u16 period;
		if (hfa384x_get_rid(dev, HFA384X_RID_CNFMAXSLEEPDURATION,
				    &period, 2, 1) < 0)
			return -EINVAL;

		rrq->flags = IW_POWER_PERIOD;
		rrq->value = __le16_to_cpu(period) * 1024;
	}

	if (hfa384x_get_rid(dev, HFA384X_RID_CNFMULTICASTRECEIVE, &mcast,
			    2, 1) < 0)
		return -EINVAL;

	if (__le16_to_cpu(mcast))
		rrq->flags |= IW_POWER_ALL_R;
	else
		rrq->flags |= IW_POWER_UNICAST_R;

	return 0;
}
#endif /* WIRELESS_EXT > 8 */


#if WIRELESS_EXT > 10
static int prism2_ioctl_siwretry(struct net_device *dev, struct iw_param *rrq)
{
	local_info_t *local = (local_info_t *) dev->priv;

	if (rrq->disabled)
		return -EINVAL;

	/* setting retry limits is not supported with the current station
	 * firmware code; simulate this with alternative retry count for now */
	if (rrq->flags == IW_RETRY_LIMIT) {
		if (rrq->value < 0) {
			/* disable manual retry count setting and use firmware
			 * defaults */
			local->manual_retry_count = -1;
			local->tx_control &= ~HFA384X_TX_CTRL_ALT_RTRY;
		} else {
			if (prism2_set_word(dev, HFA384X_RID_CNFALTRETRYCOUNT,
					    rrq->value)) {
				printk(KERN_DEBUG "%s: Alternate retry count "
				       "setting to %d failed\n",
				       dev->name, rrq->value);
				return -EOPNOTSUPP;
			}

			local->manual_retry_count = rrq->value;
			local->tx_control |= HFA384X_TX_CTRL_ALT_RTRY;
		}
		return 0;
	}

	return -EOPNOTSUPP;

#if 0
	/* what could be done, if firmware would support this.. */

	if (rrq->flags & IW_RETRY_LIMIT) {
		if (rrq->flags & IW_RETRY_MAX)
			HFA384X_RID_LONGRETRYLIMIT = rrq->value;
		else if (rrq->flags & IW_RETRY_MIN)
			HFA384X_RID_SHORTRETRYLIMIT = rrq->value;
		else {
			HFA384X_RID_LONGRETRYLIMIT = rrq->value;
			HFA384X_RID_SHORTRETRYLIMIT = rrq->value;
		}

	}

	if (rrq->flags & IW_RETRY_LIFETIME) {
		HFA384X_RID_MAXTRANSMITLIFETIME = rrq->value / 1024;
	}

	return 0;
#endif /* 0 */
}

static int prism2_ioctl_giwretry(struct net_device *dev, struct iw_param *rrq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 shortretry, longretry, lifetime;

	if (hfa384x_get_rid(dev, HFA384X_RID_SHORTRETRYLIMIT, &shortretry, 2,
			    1) < 0 ||
	    hfa384x_get_rid(dev, HFA384X_RID_LONGRETRYLIMIT, &longretry, 2, 1)
	    < 0 ||
	    hfa384x_get_rid(dev, HFA384X_RID_MAXTRANSMITLIFETIME, &lifetime, 2,
			    1) < 0)
		return -EINVAL;

	le16_to_cpus(&shortretry);
	le16_to_cpus(&longretry);
	le16_to_cpus(&lifetime);

	rrq->disabled = 0;

	if ((rrq->flags & IW_RETRY_TYPE) == IW_RETRY_LIFETIME) {
		rrq->flags = IW_RETRY_LIFETIME;
		rrq->value = lifetime * 1024;
	} else {
		if (local->manual_retry_count >= 0) {
			rrq->flags = IW_RETRY_LIMIT;
			rrq->value = local->manual_retry_count;
		} else if ((rrq->flags & IW_RETRY_MAX)) {
			rrq->flags = IW_RETRY_LIMIT | IW_RETRY_MAX;
			rrq->value = longretry;
		} else {
			rrq->flags = IW_RETRY_LIMIT;
			rrq->value = shortretry;
			if (shortretry != longretry)
				rrq->flags |= IW_RETRY_MIN;
		}
	}
	return 0;
}
#endif /* WIRELESS_EXT > 10 */


#if WIRELESS_EXT > 9
/* Map HFA386x's CR31 to and from dBm with some sort of ad hoc mapping..
 * This version assumes following mapping:
 * CR31 is 7-bit value with -64 to +63 range.
 * -64 is mapped into +20dBm and +63 into -43dBm.
 * This is certainly not an exact mapping for every card, but at least
 * increasing dBm value should correspond to increasing TX power.
 */

static int prism2_txpower_hfa386x_to_dBm(u16 val)
{
	signed char tmp;

	if (val > 255)
		val = 255;

	tmp = val;
	tmp >>= 2;

	return -12 - tmp;
}

static u16 prism2_txpower_dBm_to_hfa386x(int val)
{
	signed char tmp;

	if (val > 20)
		return 128;
	else if (val < -43)
		return 127;

	tmp = val;
	tmp = -12 - tmp;
	tmp <<= 2;
	
	return (unsigned char) tmp;
}


static int prism2_ioctl_siwtxpow(struct net_device *dev, struct iw_param *rrq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	char *tmp;
	u16 val;
	int ret = 0;

	if (rrq->disabled) {
		if (local->txpower_type != PRISM2_TXPOWER_OFF) {
			val = 0xff; /* use all standby and sleep modes */
			ret = hfa384x_cmd(dev, HFA384X_CMDCODE_WRITEMIF,
					  HFA386X_CR_A_D_TEST_MODES2,
					  &val, NULL);
			printk(KERN_DEBUG "%s: Turning radio off: %s\n",
			       dev->name, ret ? "failed" : "OK");
			local->txpower_type = PRISM2_TXPOWER_OFF;
		}
		return (ret ? -EOPNOTSUPP : 0);
	}

	if (local->txpower_type == PRISM2_TXPOWER_OFF) {
		val = 0; /* disable all standby and sleep modes */
		ret = hfa384x_cmd(dev, HFA384X_CMDCODE_WRITEMIF,
				  HFA386X_CR_A_D_TEST_MODES2, &val, NULL);
		printk(KERN_DEBUG "%s: Turning radio on: %s\n",
		       dev->name, ret ? "failed" : "OK");
		local->txpower_type = PRISM2_TXPOWER_UNKNOWN;
	}

	if (!rrq->fixed && local->txpower_type != PRISM2_TXPOWER_AUTO) {
		printk(KERN_DEBUG "Setting ALC on\n");
		val = HFA384X_TEST_CFG_BIT_ALC;
		hfa384x_cmd(dev, HFA384X_CMDCODE_TEST |
			    (HFA384X_TEST_CFG_BITS << 8), 1, &val, NULL);
		local->txpower_type = PRISM2_TXPOWER_AUTO;
		return 0;
	}

	if (local->txpower_type != PRISM2_TXPOWER_FIXED) {
		printk(KERN_DEBUG "Setting ALC off\n");
		val = HFA384X_TEST_CFG_BIT_ALC;
		hfa384x_cmd(dev, HFA384X_CMDCODE_TEST |
			    (HFA384X_TEST_CFG_BITS << 8), 0, &val, NULL);
			local->txpower_type = PRISM2_TXPOWER_FIXED;
	}

	if (rrq->flags == IW_TXPOW_DBM)
		tmp = "dBm";
	else if (rrq->flags == IW_TXPOW_MWATT)
		tmp = "mW";
	else
		tmp = "UNKNOWN";
	printk(KERN_DEBUG "Setting TX power to %d %s\n", rrq->value, tmp);

	if (rrq->flags != IW_TXPOW_DBM) {
		printk("SIOCSIWTXPOW with mW is not supported; use dBm\n");
		return -EOPNOTSUPP;
	}

	local->txpower = rrq->value;
	val = prism2_txpower_dBm_to_hfa386x(local->txpower);
	if (hfa384x_cmd(dev, HFA384X_CMDCODE_WRITEMIF,
			HFA386X_CR_MANUAL_TX_POWER, &val, NULL))
		ret = -EOPNOTSUPP;

	return ret;
}

static int prism2_ioctl_giwtxpow(struct net_device *dev, struct iw_param *rrq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 resp0;

	rrq->flags = IW_TXPOW_DBM;
	rrq->disabled = 0;
	rrq->fixed = 0;

	if (local->txpower_type == PRISM2_TXPOWER_AUTO) {
		if (hfa384x_cmd(dev, HFA384X_CMDCODE_READMIF,
				HFA386X_CR_MANUAL_TX_POWER,
				NULL, &resp0) == 0) {
			rrq->value = prism2_txpower_hfa386x_to_dBm(resp0);
		} else {
			/* Could not get real txpower; guess 15 dBm */
			rrq->value = 15;
		}
	} else if (local->txpower_type == PRISM2_TXPOWER_OFF) {
		rrq->value = 0;
		rrq->disabled = 1;
	} else if (local->txpower_type == PRISM2_TXPOWER_FIXED) {
		rrq->value = local->txpower;
		rrq->fixed = 1;
	} else {
		printk("SIOCGIWTXPOW - unknown txpower_type=%d\n",
		       local->txpower_type);
	}
	return 0;
}
#endif /* WIRELESS_EXT > 9 */


#if WIRELESS_EXT > 8
static int prism2_ioctl_giwpriv(struct net_device *dev, struct iw_point *data)
{
	struct iw_priv_args priv[] = {
#ifdef PRISM2_MONITOR
		{ PRISM2_IOCTL_MONITOR,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "monitor" },
#endif /* PRISM2_MONITOR */
		{ PRISM2_IOCTL_PRISM2_PARAM,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 2, 0,
		  "prism2_param" },
		{ PRISM2_IOCTL_READMIF,
		  IW_PRIV_TYPE_BYTE | IW_PRIV_SIZE_FIXED | 1,
		  IW_PRIV_TYPE_BYTE | IW_PRIV_SIZE_FIXED | 1, "readmif" },
		{ PRISM2_IOCTL_WRITEMIF,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 2, 0, "writemif" },
		{ PRISM2_IOCTL_RESET,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "reset" },
		{ PRISM2_IOCTL_INQUIRE,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "inquire" },
		{ PRISM2_IOCTL_WDS_ADD,
		  IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 18, 0, "wds_add" },
		{ PRISM2_IOCTL_WDS_DEL,
		  IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 18, 0, "wds_del" }
	};

	if (!data->pointer ||
	    verify_area(VERIFY_WRITE, data->pointer, sizeof(priv)))
		return -EINVAL;

	data->length = sizeof(priv) / sizeof(priv[0]);
	copy_to_user(data->pointer, priv, sizeof(priv));
	return 0;
}
#endif /* WIRELESS_EXT > 8 */
#endif /* WIRELESS_EXT */


static int prism2_ioctl_priv_inquire(struct net_device *dev, int *i)
{
	if (hfa384x_cmd(dev, HFA384X_CMDCODE_INQUIRE, *i, NULL, NULL))
		return -EOPNOTSUPP;

	return 0;
}


static int prism2_ioctl_priv_prism2_param(struct net_device *dev, int *i)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int param = *i;
	int value = *(i + 1);
	int ret = 0;
	u16 val;

	switch (param) {
	case PRISM2_PARAM_PTYPE:
		if (prism2_set_word(dev, HFA384X_RID_CNFPORTTYPE, value)) {
			ret = -EOPNOTSUPP;
			break;
		}

		if (prism2_reset_port(dev))
			ret = -EINVAL;
		break;

	case PRISM2_PARAM_TXRATECTRL:
		local->fw_tx_rate_control = value;
		break;

	case PRISM2_PARAM_BEACON_INT:
		if (prism2_set_word(dev, HFA384X_RID_CNFBEACONINT, value) ||
		    prism2_reset_port(dev))
			ret = -EINVAL;
		else
			local->beacon_int = value;
		break;

	case PRISM2_PARAM_PSEUDO_IBSS:
		if (value == local->pseudo_adhoc)
			break;

		if (value != 0 && value != 1) {
			ret = -EINVAL;
			break;
		}

		printk(KERN_DEBUG "prism2: %s: pseudo IBSS change %d -> %d\n",
		       dev->name, local->pseudo_adhoc, value);
		local->pseudo_adhoc = value;
		if (local->iw_mode != IW_MODE_ADHOC)
			break;

		if (prism2_set_word(dev, HFA384X_RID_CNFPORTTYPE,
				    prism2_get_porttype(local))) {
			ret = -EOPNOTSUPP;
			break;
		}

		if (prism2_reset_port(dev))
			ret = -EINVAL;
		break;

	case PRISM2_PARAM_ALC:
		printk(KERN_DEBUG "%s: %s ALC\n", dev->name,
		       value == 0 ? "Disabling" : "Enabling");
		val = HFA384X_TEST_CFG_BIT_ALC;
		hfa384x_cmd(dev, HFA384X_CMDCODE_TEST |
			    (HFA384X_TEST_CFG_BITS << 8),
			    value == 0 ? 0 : 1, &val, NULL);
		break;

	case PRISM2_PARAM_TXPOWER:
		val = value;
		if (hfa384x_cmd(dev, HFA384X_CMDCODE_WRITEMIF,
				HFA386X_CR_MANUAL_TX_POWER, &val, NULL))
			ret = -EOPNOTSUPP;
		break;

	case PRISM2_PARAM_DUMP:
		local->frame_dump = value;
		break;

	case PRISM2_PARAM_OTHER_AP_POLICY:
		if (value < 0 || value > 3) {
			ret = -EINVAL;
			break;
		}
		if (local->ap != NULL)
			local->ap->ap_policy = value;
		break;

	case PRISM2_PARAM_AP_MAX_INACTIVITY:
		if (value < 0 || value > 7 * 24 * 60 * 60) {
			ret = -EINVAL;
			break;
		}
		if (local->ap != NULL)
			local->ap->max_inactivity = value * HZ;
		break;

	case PRISM2_PARAM_AP_BRIDGE_PACKETS:
		if (local->ap != NULL)
			local->ap->bridge_packets = value;
		break;

	case PRISM2_PARAM_DTIM_PERIOD:
		if (value < 0 || value > 65535) {
			ret = -EINVAL;
			break;
		}
		if (prism2_set_word(dev, HFA384X_RID_CNFOWNDTIMPERIOD, value)
		    || prism2_reset_port(dev))
			ret = -EINVAL;
		else
			local->dtim_period = value;
		break;

	case PRISM2_PARAM_AP_NULLFUNC_ACK:
		if (local->ap != NULL)
			local->ap->nullfunc_ack = value;
		break;

	case PRISM2_PARAM_MAX_WDS:
		local->wds_max_connections = value;
		break;

	case PRISM2_PARAM_AP_AUTOM_AP_WDS:
		if (local->ap != NULL)
			local->ap->autom_ap_wds = value;
		break;

	case PRISM2_PARAM_AP_AUTH_ALGS:
		if (local->ap != NULL)
			local->ap->auth_algs = value;
		break;

#ifdef PRISM2_MONITOR
	case PRISM2_PARAM_MONITOR_ALLOW_FCSERR:
		local->monitor_allow_fcserr = value;
		break;
#endif /* PRISM2_MONITOR */

	default:
		printk(KERN_DEBUG "%s: prism2_param: unknown param %d\n",
		       dev->name, param);
		ret = -EOPNOTSUPP;
		break;
	}

	return ret;
}


static int prism2_ioctl_priv_readmif(struct net_device *dev, struct iwreq *wrq)
{
	u16 val, resp0;

	val = *((int *) wrq->u.name);
	if (hfa384x_cmd(dev, HFA384X_CMDCODE_READMIF, val, NULL, &resp0))
		return -EOPNOTSUPP;
	else
		*(wrq->u.name) = resp0;

	return 0;
}


static int prism2_ioctl_priv_writemif(struct net_device *dev, int *i)
{
	u16 cr, val;

	cr = *i;
	val = *(i + 1);
	if (hfa384x_cmd(dev, HFA384X_CMDCODE_WRITEMIF, cr, &val, NULL))
		return -EOPNOTSUPP;

	return 0;
}


#ifdef PRISM2_MONITOR
static int prism2_ioctl_priv_monitor(struct net_device *dev, int *i)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int ret = 0;

	if (*i == 0) {
		struct sock *nl = local->nl_monitor;
		printk(KERN_DEBUG "Disabling monitor mode\n");
		dev->type = ARPHRD_ETHER;
		dev->hard_header_parse = local->saved_eth_header_parse;
		local->monitor_type = PRISM2_MONITOR_OFF;
		local->nl_monitor = NULL;
		if (nl != NULL) {
			if (nl->socket == NULL)
				printk("nl->socket == NULL\n");

			/* FIX: this seems to crash the kernel, but
			 * something needs to be released.. */
			/* sock_release(nl->socket); */
		}
		hfa384x_cmd(dev, HFA384X_CMDCODE_TEST |
			    (HFA384X_TEST_STOP << 8),
			    0, NULL, NULL);
		ret = prism2_set_word(dev, HFA384X_RID_CNFPORTTYPE,
				      prism2_get_porttype(local));
		prism2_reset_port(dev);
	} else if (*i == 1 || *i == 2) {
		printk(KERN_DEBUG "Enabling monitor mode(%i)\n", *i);
		if (*i == 1) {
			dev->type = ARPHRD_ETHER;
			dev->hard_header_parse = local->saved_eth_header_parse;
			local->monitor_type = PRISM2_MONITOR_NL;
		} else {
			dev->type = ARPHRD_IEEE80211;
			dev->hard_header_parse = prism2_80211_header_parse;
			local->monitor_type = PRISM2_MONITOR_DEV;
		}
		if (local->monitor_type == PRISM2_MONITOR_NL &&
		    local->nl_monitor == NULL) {
			local->nl_monitor =
				netlink_kernel_create(NETLINK_USERSOCK,
						      monitor_tx);
			if (local->nl_monitor == NULL)
				printk(KERN_WARNING
				       "netlink_kernel_create failed\n");
		}
#ifdef PRISM2_MONITOR_PACKET_INJECT
		ret = prism2_set_word(dev, HFA384X_RID_CNFPORTTYPE,
				      HFA384X_PORTTYPE_HOSTAP);
#else /* PRISM2_MONITOR_PACKET_INJECT */
		ret = prism2_set_word(dev, HFA384X_RID_CNFPORTTYPE,
				      HFA384X_PORTTYPE_PSEUDO_IBSS);
#endif /* PRISM2_MONITOR_PACKET_INJECT */
		if (ret) {
			printk(KERN_DEBUG "Port type setting for monitor mode "
			       "failed\n");
			return -EOPNOTSUPP;
		}
		/* Host decrypt is needed to get the IV and ICV fields;
		 * however, monitor mode seems to remove WEP flag from frame
		 * control field */
		ret = prism2_set_word(dev, HFA384X_RID_CNFWEPFLAGS,
				      HFA384X_WEPFLAGS_HOSTENCRYPT |
				      HFA384X_WEPFLAGS_HOSTDECRYPT);
		if (ret) {
			printk(KERN_DEBUG "WEP flags setting failed\n");
			return -EOPNOTSUPP;
		}
#if 0
		ret = prism2_set_word(dev, HFA384X_RID_PROMISCUOUSMODE, 1);
		if (ret) {
			printk(KERN_DEBUG "Promiscuous mode setting failed\n");
			return -EOPNOTSUPP;
		}
#endif /* 0 */
		prism2_reset_port(dev);
#ifndef PRISM2_MONITOR_PACKET_INJECT
		hfa384x_cmd(dev, HFA384X_CMDCODE_TEST |
			    (HFA384X_TEST_MONITOR << 8),
			    0, NULL, NULL);
#endif /* !PRISM2_MONITOR_PACKET_INJECT */
	} else
		ret = -EINVAL;

	return ret;
}
#endif /* PRISM2_MONITOR */


static int prism2_ioctl_priv_reset(struct net_device *dev, int *i)
{
	printk(KERN_DEBUG "%s: manual reset request(%d)\n", dev->name, *i);
	switch (*i) {
	case 0:
		/* Disable and enable card */
		prism2_hw_shutdown(dev, 1);
		prism2_hw_config(dev, 0);
		break;

	case 1:
		/* COR sreset */
		prism2_hw_reset(dev);
		break;

	case 2:
		/* Disable and enable port 0 */
		prism2_reset_port(dev);
		break;

	default:
		printk(KERN_DEBUG "Unknown reset request %d\n", *i);
		return -EOPNOTSUPP;
	}

	return 0;
}


static inline int hex2int(char c)
{
	if (c >= '0' && c <= '9')
		return (c - '0');
	if (c >= 'a' && c <= 'f')
		return (c - 'a' + 10);
	if (c >= 'A' && c <= 'F')
		return (c - 'A' + 10);
	return -1;
}

static int macstr2addr(char *macstr, u8 *addr)
{
	int i, val, val2;
	char *pos = macstr;

	for (i = 0; i < 6; i++) {
		val = hex2int(*pos++);
		if (val < 0)
			return -1;
		val2 = hex2int(*pos++);
		if (val2 < 0)
			return -1;
		addr[i] = (val * 16 + val2) & 0xff;

		if (i < 5 && *pos++ != ':')
			return -1;
	}

	return 0;
}


static int prism2_ioctl_priv_wds(struct net_device *dev, int add, char *macstr)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u8 addr[6];

	if (macstr2addr(macstr, addr)) {
		printk(KERN_DEBUG "Invalid MAC address\n");
		return -EINVAL;
	}

	if (add)
		return prism2_wds_add(local, addr, 1);
	else
		return prism2_wds_del(local, addr, 1, 0);
}


static int prism2_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{
#ifdef WIRELESS_EXT
	struct iwreq *wrq = (struct iwreq *) ifr;
#endif
#if WIRELESS_EXT > 8
	local_info_t *local = (local_info_t *) dev->priv;
#endif /* WIRELESS_EXT > 8 */
	int ret = 0;

	switch (cmd) {

#ifdef WIRELESS_EXT
	case SIOCGIWNAME:
		strcpy(wrq->u.name, "IEEE 802.11-DS");
		break;

	case SIOCSIWFREQ:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwfreq(dev, &wrq->u.freq);
		break;
	case SIOCGIWFREQ:
		ret = prism2_ioctl_giwfreq(dev, &wrq->u.freq);
		break;

#if WIRELESS_EXT > 8
	case SIOCSIWESSID:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwessid(dev, &wrq->u.data);
		break;
	case SIOCGIWESSID:
		ret = prism2_ioctl_giwessid(dev, &wrq->u.data);
		break;

	case SIOCSIWRATE:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwrate(dev, &wrq->u.bitrate);
		break;
	case SIOCGIWRATE:
		ret = prism2_ioctl_giwrate(dev, &wrq->u.bitrate);
		break;

	case SIOCSIWRTS:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwrts(dev, &wrq->u.rts);
		break;
	case SIOCGIWRTS:
		ret = prism2_ioctl_giwrts(dev, &wrq->u.rts);
		break;

	case SIOCSIWFRAG:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwfrag(dev, &wrq->u.rts);
		break;
	case SIOCGIWFRAG:
		ret = prism2_ioctl_giwfrag(dev, &wrq->u.rts);
		break;

	case SIOCSIWENCODE:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwencode(dev, &wrq->u.encoding);
		break;
	case SIOCGIWENCODE:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_giwencode(dev, &wrq->u.encoding);
		break;
#endif /* WIRELESS_EXT > 8 */

	case SIOCGIWAP:
		ret = prism2_ioctl_giwap(dev, &wrq->u.ap_addr);
		break;

#if WIRELESS_EXT > 8
	case SIOCSIWNICKN:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwnickn(dev, &wrq->u.data);
		break;
	case SIOCGIWNICKN:
		ret = prism2_ioctl_giwnickn(dev, &wrq->u.data);
		break;

	case SIOCGIWSPY:
		ret = prism2_ioctl_giwspy(dev, &wrq->u.data);
		break;

	case SIOCGIWRANGE:
		ret = prism2_ioctl_giwrange(dev, &wrq->u.data);
		break;

	case SIOCSIWSENS:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwsens(dev, &wrq->u.sens);
		break;
	case SIOCGIWSENS:
		ret = prism2_ioctl_giwsens(dev, &wrq->u.sens);
		break;

	case SIOCGIWAPLIST:
		ret = prism2_ioctl_giwaplist(dev, &wrq->u.data);
		break;

	case SIOCSIWMODE:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwmode(dev, wrq->u.mode);
		break;
	case SIOCGIWMODE:
		wrq->u.mode = local->iw_mode;
		break;

	case SIOCSIWPOWER:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwpower(dev, &wrq->u.power);
		break;
	case SIOCGIWPOWER:
		ret = prism2_ioctl_giwpower(dev, &wrq->u.power);
		break;
#endif /* WIRELESS_EXT > 8 */

#if WIRELESS_EXT > 9
	case SIOCSIWTXPOW:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwtxpow(dev, &wrq->u.txpower);
		break;
	case SIOCGIWTXPOW:
		ret = prism2_ioctl_giwtxpow(dev, &wrq->u.txpower);
		break;
#endif /* WIRELESS_EXT > 9 */

#if WIRELESS_EXT > 10
	case SIOCSIWRETRY:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwretry(dev, &wrq->u.retry);
		break;
	case SIOCGIWRETRY:
		ret = prism2_ioctl_giwretry(dev, &wrq->u.retry);
		break;
#endif /* WIRELESS_EXT > 10 */

#if WIRELESS_EXT > 8
	case SIOCGIWPRIV:
		ret = prism2_ioctl_giwpriv(dev, &wrq->u.data);
		break;
#endif /* WIRELESS_EXT > 8 */

	case PRISM2_IOCTL_INQUIRE:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_inquire(dev, (int *) wrq->u.name);
		break;

	case PRISM2_IOCTL_PRISM2_PARAM:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_prism2_param(dev,
							  (int *) wrq->u.name);
		break;

	case PRISM2_IOCTL_READMIF:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_readmif(dev, wrq);
		break;

#ifdef PRISM2_MONITOR

	case PRISM2_IOCTL_MONITOR:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_monitor(dev, (int *) wrq->u.name);
		break;
#endif /* PRISM2_MONITOR */

	case PRISM2_IOCTL_RESET:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_reset(dev, (int *) wrq->u.name);
		break;

	case PRISM2_IOCTL_WRITEMIF:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_writemif(dev,
						      (int *) wrq->u.name);
		break;

	case PRISM2_IOCTL_WDS_ADD:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_wds(dev, 1,
						 (char *) wrq->u.data.pointer);
		break;

	case PRISM2_IOCTL_WDS_DEL:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_wds(dev, 0,
						 (char *) wrq->u.data.pointer);
		break;


	/* not supported wireless extensions */
#ifdef SIOCSIWNAME
	case SIOCSIWNAME:
#endif
	case SIOCSIWNWID:
	case SIOCGIWNWID:
	case SIOCSIWPRIV:
		ret = -EOPNOTSUPP;
		break;

	/* FIX: add support for (at least some of) these: */
	case SIOCSIWRANGE:
	case SIOCSIWSPY:
	case SIOCSIWAP:
		printk(KERN_DEBUG "%s: %s unsupported WIRELESS_EXT "
		       "ioctl(0x%04x)\n", dev_info, dev->name, cmd);
		ret = -EOPNOTSUPP;
		break;

#endif /* WIRELESS_EXT */

	default:
		printk(KERN_DEBUG "%s: %s unsupported ioctl(0x%04x)\n",
		       dev_info, dev->name, cmd);
		ret = -EOPNOTSUPP;
		break;
	}

	return ret;
}
