/* /proc routines for Host AP driver */

static struct proc_dir_entry *prism2_proc = NULL;


static int prism2_debug_proc_read(char *page, char **start, off_t off,
				  int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	int i;
	unsigned long flags;

	if (off != 0) {
		*eof = 1;
		return 0;
	}

	spin_lock_irqsave(&local->txfidlock, flags);
	p += sprintf(p, "next_txfid=%d next_alloc=%d\n",
		     local->next_txfid, local->next_alloc);
	for (i = 0; i < PRISM2_TXFID_COUNT; i++)
		p += sprintf(p, "FID: tx=%04X intransmit=%04X\n",
			     local->txfid[i], local->intransmitfid[i]);
	spin_unlock_irqrestore(&local->txfidlock, flags);
	p += sprintf(p, "FW TX rate control: %d\n", local->fw_tx_rate_control);
	p += sprintf(p, "beacon_int=%d\n", local->beacon_int);
	p += sprintf(p, "dtim_period=%d\n", local->dtim_period);
	p += sprintf(p, "wds_max_connections=%d\n",
		     local->wds_max_connections);

	return (p - page);
}


static int prism2_stats_proc_read(char *page, char **start, off_t off,
				  int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	struct comm_tallies_sums *sums = (struct comm_tallies_sums *)
		&local->comm_tallies;

	if (off != 0) {
		*eof = 1;
		return 0;
	}

	p += sprintf(p, "TxUnicastFrames=%u\n", sums->tx_unicast_frames);
	p += sprintf(p, "TxMulticastframes=%u\n", sums->tx_multicast_frames);
	p += sprintf(p, "TxFragments=%u\n", sums->tx_fragments);
	p += sprintf(p, "TxUnicastOctets=%u\n", sums->tx_unicast_octets);
	p += sprintf(p, "TxMulticastOctets=%u\n", sums->tx_multicast_octets);
	p += sprintf(p, "TxDeferredTransmissions=%u\n",
		     sums->tx_deferred_transmissions);
	p += sprintf(p, "TxSingleRetryFrames=%u\n",
		     sums->tx_single_retry_frames);
	p += sprintf(p, "TxMultipleRetryFrames=%u\n",
		     sums->tx_multiple_retry_frames);
	p += sprintf(p, "TxRetryLimitExceeded=%u\n",
		     sums->tx_retry_limit_exceeded);
	p += sprintf(p, "TxDiscards=%u\n", sums->tx_discards);
	p += sprintf(p, "RxUnicastFrames=%u\n", sums->rx_unicast_frames);
	p += sprintf(p, "RxMulticastFrames=%u\n", sums->rx_multicast_frames);
	p += sprintf(p, "RxFragments=%u\n", sums->rx_fragments);
	p += sprintf(p, "RxUnicastOctets=%u\n", sums->rx_unicast_octets);
	p += sprintf(p, "RxMulticastOctets=%u\n", sums->rx_multicast_octets);
	p += sprintf(p, "RxFCSErrors=%u\n", sums->rx_fcs_errors);
	p += sprintf(p, "RxDiscardsNoBuffer=%u\n",
		     sums->rx_discards_no_buffer);
	p += sprintf(p, "TxDiscardsWrongSA=%u\n", sums->tx_discards_wrong_sa);
	p += sprintf(p, "RxDiscardsWEPUndecryptable=%u\n",
		     sums->rx_discards_wep_undecryptable);
	p += sprintf(p, "RxMessageInMsgFragments=%u\n",
		     sums->rx_message_in_msg_fragments);
	p += sprintf(p, "RxMessageInBadMsgFragments=%u\n",
		     sums->rx_message_in_bad_msg_fragments);
	/* FIX: this may grow too long for one page(?) */

	return (p - page);
}


static int prism2_registers_proc_read(char *page, char **start, off_t off,
				      int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;

	if (off != 0) {
		*eof = 1;
		return 0;
	}

#define SHOW_REG(n) \
p += sprintf(p, "%s=%04x\n", #n, \
hfa384x_read_reg(local->dev, HFA384X_##n##_OFF))

	SHOW_REG(CMD);
	SHOW_REG(PARAM0);
	SHOW_REG(PARAM1);
	SHOW_REG(PARAM2);
	SHOW_REG(STATUS);
	SHOW_REG(RESP0);
	SHOW_REG(RESP1);
	SHOW_REG(RESP2);
	SHOW_REG(INFOFID);
	SHOW_REG(CONTROL);
	SHOW_REG(SELECT0);
	SHOW_REG(SELECT1);
	SHOW_REG(OFFSET0);
	SHOW_REG(OFFSET1);
	SHOW_REG(RXFID);
	SHOW_REG(ALLOCFID);
	SHOW_REG(TXCOMPLFID);
	SHOW_REG(SWSUPPORT0);
	SHOW_REG(SWSUPPORT1);
	SHOW_REG(SWSUPPORT2);
	SHOW_REG(EVSTAT);
	SHOW_REG(INTEN);
	SHOW_REG(EVACK);
	/* Do not read data registers, because they change the state of the
	 * MAC (offset += 2) */
	/* SHOW_REG(DATA0); */
	/* SHOW_REG(DATA1); */
	SHOW_REG(AUXPAGE);
	SHOW_REG(AUXOFFSET);
	/* SHOW_REG(AUXDATA); */
#ifdef PRISM2_PCI
	SHOW_REG(PCICOR);
	SHOW_REG(PCIHCR);
	SHOW_REG(PCI_M0_ADDRH);
	SHOW_REG(PCI_M0_ADDRL);
	SHOW_REG(PCI_M0_LEN);
	SHOW_REG(PCI_M0_CTL);
	SHOW_REG(PCI_STATUS);
	SHOW_REG(PCI_M1_ADDRH);
	SHOW_REG(PCI_M1_ADDRL);
	SHOW_REG(PCI_M1_LEN);
	SHOW_REG(PCI_M1_CTL);
#endif /* PRISM2_PCI */

	return (p - page);
}

#define RID(n,t) { HFA384X_RID_##n, #n, t }
enum { RID_HEXDUMP, RID_WORD, RID_HWADDR, RID_STRING, RID_COMPID,
       RID_SUPRANGE };

static struct {
	u16 rid;
	char *name;
	int type;
} rid_table[] = {
	RID(CNFPORTTYPE, RID_WORD),
	RID(CNFOWNMACADDR, RID_HWADDR),
	RID(CNFDESIREDSSID, RID_STRING),
	RID(CNFOWNCHANNEL, RID_WORD),
	RID(CNFOWNSSID, RID_STRING),
	RID(CNFOWNATIMWINDOW, RID_WORD),
	RID(CNFSYSTEMSCALE, RID_WORD),
	RID(CNFMAXDATALEN, RID_WORD),
	RID(CNFWDSADDRESS, RID_HWADDR),
	RID(CNFPMENABLED, RID_WORD),
	RID(CNFPMEPS, RID_WORD),
	RID(CNFMULTICASTRECEIVE, RID_WORD),
	RID(CNFMAXSLEEPDURATION, RID_WORD),
	RID(CNFPMHOLDOVERDURATION, RID_WORD),
	RID(CNFOWNNAME, RID_STRING),
	RID(CNFOWNDTIMPERIOD, RID_WORD),
	RID(CNFWDSADDRESS1, RID_HWADDR),
	RID(CNFWDSADDRESS2, RID_HWADDR),
	RID(CNFWDSADDRESS3, RID_HWADDR),
	RID(CNFWDSADDRESS4, RID_HWADDR),
	RID(CNFWDSADDRESS5, RID_HWADDR),
	RID(CNFWDSADDRESS6, RID_HWADDR),
	RID(CNFMULTICASTPMBUFFERING, RID_WORD),
	RID(UNKNOWN1, RID_WORD),
	RID(UNKNOWN2, RID_WORD),
	RID(CNFWEPDEFAULTKEYID, RID_WORD),
	RID(CNFDEFAULTKEY0, RID_HEXDUMP),
	RID(CNFDEFAULTKEY1, RID_HEXDUMP),
	RID(CNFDEFAULTKEY2, RID_HEXDUMP),
	RID(CNFDEFAULTKEY3, RID_HEXDUMP),
	RID(CNFWEPFLAGS, RID_HEXDUMP),
	RID(CNFWEPKEYMAPPINGTABLE, RID_HEXDUMP),
	RID(CNFAUTHENTICATION, RID_WORD),
	RID(CNFMAXASSOCSTA, RID_WORD),
	RID(CNFTXCONTROL, RID_WORD),
	RID(CNFROAMINGMODE, RID_WORD),
	RID(CNFHOSTAUTHENTICATION, RID_WORD),
	RID(CNFRCVCRCERROR, RID_WORD),
	RID(CNFMMLIFE, RID_WORD),
	RID(CNFALTRETRYCOUNT, RID_WORD),
	RID(CNFBEACONINT, RID_WORD),
	RID(CNFAPPCFINFO, RID_HEXDUMP),
	RID(CNFSTAPCFINFO, RID_HEXDUMP),
	RID(CNFPRIORITYQUSAGE, RID_HEXDUMP),
	RID(CNFTIMCTRL, RID_WORD),
	RID(UNKNOWN3, RID_HEXDUMP),
	RID(CNFTHIRTY2TALLY, RID_WORD),
	RID(CNFENHSECURITY, RID_WORD),
	RID(CNFDBMADJUST, RID_WORD),
	RID(GROUPADDRESSES, RID_HEXDUMP),
	RID(CREATEIBSS, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD, RID_WORD),
	RID(RTSTHRESHOLD, RID_WORD),
	RID(TXRATECONTROL, RID_WORD),
	RID(PROMISCUOUSMODE, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD0, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD1, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD2, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD3, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD4, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD5, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD6, RID_WORD),
	RID(RTSTHRESHOLD0, RID_WORD),
	RID(RTSTHRESHOLD1, RID_WORD),
	RID(RTSTHRESHOLD2, RID_WORD),
	RID(RTSTHRESHOLD3, RID_WORD),
	RID(RTSTHRESHOLD4, RID_WORD),
	RID(RTSTHRESHOLD5, RID_WORD),
	RID(RTSTHRESHOLD6, RID_WORD),
	RID(TXRATECONTROL0, RID_WORD),
	RID(TXRATECONTROL1, RID_WORD),
	RID(TXRATECONTROL2, RID_WORD),
	RID(TXRATECONTROL3, RID_WORD),
	RID(TXRATECONTROL4, RID_WORD),
	RID(TXRATECONTROL5, RID_WORD),
	RID(TXRATECONTROL6, RID_WORD),
	RID(CNFSHORTPREAMBLE, RID_WORD),
	RID(CNFEXCLUDELONGPREAMBLE, RID_WORD),
	RID(CNFAUTHENTICATIONRSPTO, RID_WORD),
	RID(CNFBASICRATES, RID_HEXDUMP),
	RID(CNFSUPPORTEDRATES, RID_HEXDUMP),
	RID(UNKNOWN5, RID_WORD),
	RID(WEPKEYDISABLE, RID_WORD),
	RID(TICKTIME, RID_WORD),
	RID(SCANREQUEST, RID_HEXDUMP),
	RID(JOINREQUEST, RID_HEXDUMP),
	RID(AUTHENTICATESTATION, RID_HEXDUMP),
	RID(CHANNELINFOREQUEST, RID_HEXDUMP),
	RID(HOSTSCAN, RID_HEXDUMP),

	RID(MAXLOADTIME, RID_WORD),
	RID(DOWNLOADBUFFER, RID_HEXDUMP),
	RID(PRIID, RID_COMPID),
	RID(PRISUPRANGE, RID_SUPRANGE),
	RID(CFIACTRANGES, RID_SUPRANGE),
	RID(NICSERNUM, RID_STRING),
	RID(NICID, RID_COMPID),
	RID(MFISUPRANGE, RID_SUPRANGE),
	RID(CFISUPRANGE, RID_SUPRANGE),
	RID(CHANNELLIST, RID_HEXDUMP),
	RID(REGULATORYDOMAINS, RID_STRING),
	RID(TEMPTYPE, RID_WORD),
	RID(CIS, RID_HEXDUMP),
	RID(STAID, RID_COMPID),
	RID(STASUPRANGE, RID_SUPRANGE),
	RID(MFIACTRANGES, RID_SUPRANGE),
	RID(CFIACTRANGES2, RID_SUPRANGE),
	RID(PRODUCTNAME, RID_STRING),
	RID(PORTSTATUS, RID_WORD),
	RID(CURRENTSSID, RID_STRING),
	RID(CURRENTBSSID, RID_HWADDR),
	RID(COMMSQUALITY, RID_HEXDUMP),
	RID(CURRENTTXRATE, RID_WORD),
	RID(CURRENTBEACONINTERVAL, RID_WORD),
	RID(CURRENTSCALETHRESHOLDS, RID_HEXDUMP),
	RID(PROTOCOLRSPTIME, RID_WORD),
	RID(SHORTRETRYLIMIT, RID_WORD),
	RID(LONGRETRYLIMIT, RID_WORD),
	RID(MAXTRANSMITLIFETIME, RID_WORD),
	RID(MAXRECEIVELIFETIME, RID_WORD),
	RID(CFPOLLABLE, RID_WORD),
	RID(AUTHENTICATIONALGORITHMS, RID_HEXDUMP),
	RID(PRIVACYOPTIONIMPLEMENTED, RID_WORD),
	RID(DBMCOMMSQUALITY, RID_HEXDUMP),
	RID(CURRENTTXRATE1, RID_WORD),
	RID(CURRENTTXRATE2, RID_WORD),
	RID(CURRENTTXRATE3, RID_WORD),
	RID(CURRENTTXRATE4, RID_WORD),
	RID(CURRENTTXRATE5, RID_WORD),
	RID(CURRENTTXRATE6, RID_WORD),
	RID(OWNMACADDR, RID_HWADDR),
	RID(SCANRESULTSTABLE, RID_HEXDUMP),
	RID(HOSTSCANRESULTS, RID_HEXDUMP),
	RID(AUTHENTICATIONUSED, RID_HEXDUMP),
	RID(PHYTYPE, RID_WORD),
	RID(CURRENTCHANNEL, RID_WORD),
	RID(CURRENTPOWERSTATE, RID_WORD),
	RID(CCAMODE, RID_WORD),
	RID(SUPPORTEDDATARATES, RID_HEXDUMP),

	RID(BUILDSEQ, RID_HEXDUMP),
	RID(FWID, RID_STRING)
};

#define PROC_LIMIT (PAGE_SIZE - 80)

static int prism2_rids_proc_read(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	int i, j, len, strlen, total = 0;
	unsigned char buf[256];
	struct hfa384x_comp_ident *compid;
	struct hfa384x_sup_range *range;

	for (i = 0; i < sizeof(rid_table) / sizeof(rid_table[0]); i++) {
		if (total + (p - page) <= off) {
			total += p - page;
			p = page;
		}
		if (total + (p - page) > off + count)
			break;
		if ((p - page) > PROC_LIMIT)
			break;

		len = hfa384x_get_rid(local->dev, rid_table[i].rid, buf,
				      sizeof(buf), 0);
		if (len < 0)
			continue;

		p += sprintf(p, "%04X=%s=", rid_table[i].rid,
			     rid_table[i].name);

		switch (rid_table[i].type) {
		case RID_HEXDUMP:
			for (j = 0; j < len; j++)
				p += sprintf(p, "<%02x>", buf[j]);
			p += sprintf(p, "\n");
			break;

		case RID_WORD:
			if (len != 2) {
				p += sprintf(p, "<INVALID RID_WORD LEN %d>\n",
					     len);
			} else {
				u16 val = __le16_to_cpu(*(u16 *)buf);
				p += sprintf(p, "%d\n", val);
			}
			break;

		case RID_HWADDR:
			if (len != 6) {
				p += sprintf(p,
					     "<INVALID RID_HWADDR LEN %d>\n",
					     len);
			} else {
				p += sprintf(p, MACSTR "\n", MAC2STR(buf));
			}
			break;

		case RID_STRING:
			strlen = __le16_to_cpu(*(u16 *)buf);
			if (strlen > len)
				strlen = len;
			for (j = 2; j < strlen + 2; j++) {
				if (buf[j] >= 32 && buf[j] < 127)
					p += sprintf(p, "%c", buf[j]);
				else
					p += sprintf(p, "<%02x>", buf[j]);
			}
			p += sprintf(p, "\n");
			break;

		case RID_COMPID:
			if (len != sizeof(*compid)) {
				p += sprintf(p, "<INVALID RID_COMPID LEN "
					     "%d>\n", len);
				break;
			}
			compid = (struct hfa384x_comp_ident *) buf;
			p += sprintf(p, "0x%02x v%d.%d.%d\n",
				     __le16_to_cpu(compid->id),
				     __le16_to_cpu(compid->major),
				     __le16_to_cpu(compid->minor),
				     __le16_to_cpu(compid->variant));
			break;

		case RID_SUPRANGE:
			if (len != sizeof(*range)) {
				p += sprintf(p, "<INVALID RID_SUPRANGE LEN "
					     "%d>\n", len);
				break;
			}
			range = (struct hfa384x_sup_range *) buf;
			p += sprintf(p, "%d 0x%02x %d %d-%d\n",
				     __le16_to_cpu(range->role),
				     __le16_to_cpu(range->id),
				     __le16_to_cpu(range->variant),
				     __le16_to_cpu(range->bottom),
				     __le16_to_cpu(range->top));
			break;

		default:
			p += sprintf(p, "<UNKNOWN TYPE %d>\n",
				     rid_table[i].type);
			break;
		}
	}

	total += (p - page);
	if (total >= off + count)
		*eof = 1;

	if (total < off) {
		*eof = 1;
		return 0;
	}

	len = total - off;
	if (len > (p - page))
		len = p - page;
	*start = p - len;
	if (len > count)
		len = count;

	return len;
}


static int prism2_unknown_rids_proc_read(char *page, char **start, off_t off,
					 int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	u16 rid;
	int pos, rid_entries, len, j, total = 0;
	unsigned char buf[256];

	pos = 0;
	rid_entries = sizeof(rid_table) / sizeof(rid_table[0]);

	for (rid = 0xfc00; rid <= 0xfdff; rid++) {
		if (pos < rid_entries) {
			if (rid_table[pos].rid == rid) {
				pos++;
				continue;
			}
			while (pos < rid_entries && rid_table[pos].rid < rid)
				pos++;
		}

		if (total + (p - page) <= off) {
			total += p - page;
			p = page;
		}
		if (total + (p - page) > off + count)
			break;
		if ((p - page) > PROC_LIMIT)
			break;

		len = hfa384x_get_rid(local->dev, rid, buf, sizeof(buf), 0);

		if (len < 0)
			continue;

		p += sprintf(p, "%04X=", rid);
		for (j = 0; j < len; j++)
			p += sprintf(p, "<%02x>", buf[j]);
		p += sprintf(p, "\n");
	}

	total += (p - page);
	if (total >= off + count)
		*eof = 1;

	if (total < off) {
		*eof = 1;
		return 0;
	}

	len = total - off;
	if (len > (p - page))
		len = p - page;
	*start = p - len;
	if (len > count)
		len = count;

	return len;

}


static int prism2_wds_proc_read(char *page, char **start, off_t off,
				int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	prism2_wds_info_t *wds;
	unsigned long flags;

	if (off > PROC_LIMIT) {
		*eof = 1;
		return 0;
	}

	spin_lock_irqsave(&local->wdslock, flags);
	wds = local->wds;
	while (wds != NULL) {
		p += sprintf(p, "%s\t" MACSTR "\n",
			     wds->dev.name, MAC2STR(wds->remote_addr));
		if ((p - page) > PROC_LIMIT) {
			printk(KERN_DEBUG "%s: wds proc did not fit\n",
			       local->dev->name);
			break;
		}
		wds = wds->next;
	}
	spin_unlock_irqrestore(&local->wdslock, flags);

	if ((p - page) <= off) {
		*eof = 1;
		return 0;
	}

	*start = page + off;

	return (p - page - off);
}


void prism2_init_proc(local_info_t *local)
{
	local->proc = NULL;

	if (proc_net != NULL) {
		if (prism2_proc == NULL)
			prism2_proc = proc_mkdir("prism2", proc_net);
		if (prism2_proc != NULL)
			local->proc = proc_mkdir(local->dev->name,
						 prism2_proc);
	}

	if (local->proc == NULL) {
		printk(KERN_INFO "/proc/net/prism2 creation failed\n");
		return;
	}

	create_proc_read_entry("debug", 0, local->proc,
			       prism2_debug_proc_read, local);
	create_proc_read_entry("stats", 0, local->proc,
			       prism2_stats_proc_read, local);
	create_proc_read_entry("registers", 0, local->proc,
			       prism2_registers_proc_read, local);
	create_proc_read_entry("rids", 0, local->proc,
			       prism2_rids_proc_read, local);
	create_proc_read_entry("unknown_rids", 0, local->proc,
			       prism2_unknown_rids_proc_read, local);
	create_proc_read_entry("wds", 0, local->proc,
			       prism2_wds_proc_read, local);
}


void prism2_remove_proc(local_info_t *local)
{
	if (local->proc != NULL) {
		remove_proc_entry("wds", local->proc);
		remove_proc_entry("unknown_rids", local->proc);
		remove_proc_entry("rids", local->proc);
		remove_proc_entry("registers", local->proc);
		remove_proc_entry("stats", local->proc);
		remove_proc_entry("debug", local->proc);
		if (local->dev != NULL && local->dev->name != NULL &&
		    prism2_proc != NULL)
			remove_proc_entry(local->dev->name, prism2_proc);
	}
}


void prism2_cleanup_proc(void)
{
	if (prism2_proc != NULL) {
		prism2_proc = NULL;
		remove_proc_entry("prism2", proc_net);
	}
}
