/*
 * Host AP (software wireless LAN access point) user space daemon for
 * Host AP kernel driver
 * Copyright (c) 2002, SSH Communications Security Corp
 * Jouni Malinen <jkm@ssh.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <netinet/in.h>
#include <string.h>
#include <sys/ioctl.h>

#include <sys/socket.h>
#include <features.h>    /* for the glibc version number */
#if (__GLIBC__ == 2 && __GLIBC_MINOR__ >= 1) || __GLIBC__ > 2
#include <net/if.h>
#include <netpacket/packet.h>
#include <net/ethernet.h>     /* the L2 protocols */
#include <net/if_arp.h>
#else
#include <asm/types.h>
#include <linux/if_packet.h>
#include <linux/if_ether.h>   /* The L2 protocols */
#include <linux/if_arp.h>
#endif

#include <endian.h>
#if __BYTE_ORDER == __LITTLE_ENDIAN
#define le_to_host16(n) (n)
#define host_to_le16(n) (n)
#else
#include <byteswap.h>
#define le_to_host16(n) bswap_16(n)
#define host_to_le16(n) bswap_16(n)
#endif

typedef unsigned int u32;
typedef unsigned short u16;
typedef unsigned char u8;

#include "../driver/modules/prism2_wlan.h"



/* FIX: make own type or union for mgmt frames.. maybe mgmt frame with union
 * for different subtypes ?? */


struct ieee80211_hdr {
	u16 frame_control;
	u16 duration_id;
	u8 addr1[6];
	u8 addr2[6];
	u8 addr3[6];
	u16 seq_ctrl;
	/* followed by 'u8 addr4[6];' if ToDS and FromDS is set in data frame
	 */
} __attribute__ ((packed));

#define IEEE80211_HDRLEN (sizeof(struct ieee80211_hdr))


struct ieee80211_mgmt {
	u16 frame_control;
	u16 duration;
	u8 da[6];
	u8 sa[6];
	u8 bssid[6];
	u16 seq_ctrl;
	union {
		struct {
			u16 auth_alg;
			u16 auth_transaction;
			u16 status_code;
			/* possibly followed by Challenge text */
			u8 variable[0];
		} auth;
		struct {
			u16 reason_code;
		} deauth;
		struct {
			u16 capab_info;
			u16 listen_interval;
			/* followed by SSID and Supported rates */
			u8 variable[0];
		} assoc_req;
		struct {
			u16 capab_info;
			u16 status_code;
			u16 aid;
			/* followed by Supported rates */
			u8 variable[0];
		} assoc_resp, reassoc_resp;
		struct {
			u16 capab_info;
			u16 listen_interval;
			u8 current_ap[6];
			/* followed by SSID and Supported rates */
			u8 variable[0];
		} reassoc_req;
		struct {
			u16 reason_code;
		} disassoc;
		struct {
			u8 timestamp[8];
			u16 beacon_int;
			u16 capab_info;
			/* followed by some of SSID, Supported rates,
			 * FH Params, DS Params, CF Params, IBSS Params, TIM */
			u8 variable[0];
		} beacon;
	} u;
} __attribute__ ((packed));


#define IEEE80211_FC(type, stype) host_to_le16((type << 2) | (stype << 4))



typedef struct hostapd_data {
	int sock; /* raw packet socket for driver access */
	u8 own_addr[6];
} hostapd;


static int allowed_address(hostapd *hapd, u8 *addr)
{
	/* TODO: add support for filtering unwanted client addresses */
	return 1;
}


static void handle_auth(hostapd *hapd, struct ieee80211_mgmt *mgmt, size_t len)
{
	u16 auth_alg, auth_transaction, status_code;
	u16 resp = WLAN_STATUS_SUCCESS;

	if (len < IEEE80211_HDRLEN + sizeof(mgmt->u.auth)) {
		printf("handle_auth - too short payload (len=%d)\n", len);
		return;
	}

	auth_alg = le_to_host16(mgmt->u.auth.auth_alg);
	auth_transaction = le_to_host16(mgmt->u.auth.auth_transaction);
	status_code = le_to_host16(mgmt->u.auth.status_code);

	printf("authentication: STA=" MACSTR " auth_alg=%d "
	       "auth_transaction=%d status_code=%d\n",
	       MAC2STR(mgmt->sa), auth_alg, auth_transaction, status_code);

	if (auth_alg != WLAN_AUTH_OPEN) {
		printf("Unknown authentication algorithm (%d)\n", auth_alg);
		resp = WLAN_STATUS_NOT_SUPPORTED_AUTH_ALG;
		goto fail;
	}

	if (auth_transaction != 1) {
		printf("Unknown authentication transaction number (%d)\n",
		       auth_transaction);
		resp = WLAN_STATUS_UNKNOWN_AUTH_TRANSACTION;
		goto fail;
	}

	if (!allowed_address(hapd, mgmt->sa)) {
		printf("Station " MACSTR " not allowed to authenticate.\n",
		       MAC2STR(mgmt->sa));
		resp = WLAN_STATUS_UNSPECIFIED_FAILURE;
		goto fail;
	}

	/* FIX: update STA table */

 fail:

	/* use the queued buffer for transmission because it is large enough
	 * and not needed anymore */
	mgmt->frame_control = IEEE80211_FC(WLAN_FC_TYPE_MGMT,
					   WLAN_FC_STYPE_AUTH);
	memcpy(mgmt->da, mgmt->sa, ETH_ALEN);
	memcpy(mgmt->sa, hapd->own_addr, ETH_ALEN);
	/* Addr3 = BSSID - already set */

	mgmt->u.auth.auth_alg = host_to_le16(WLAN_AUTH_OPEN);
	mgmt->u.auth.auth_transaction = host_to_le16(2);
	mgmt->u.auth.status_code = host_to_le16(resp);

	if (send(hapd->sock, mgmt, IEEE80211_HDRLEN + sizeof(mgmt->u.auth), 0)
	    < 0)
		perror("handle_auth: send");
}


static void handle_assoc(hostapd *hapd, struct ieee80211_mgmt *mgmt,
			 size_t len, int reassoc)
{
	u16 capab_info, listen_interval;
	u16 resp = WLAN_STATUS_SUCCESS;
	int send_deauth = 0, send_len;

	if (len < IEEE80211_HDRLEN + (reassoc ? sizeof(mgmt->u.reassoc_req) :
				      sizeof(mgmt->u.assoc_req))) {
		printf("handle_assoc(reassoc=%d) - too short payload (len=%d)"
		       "\n", reassoc, len);
		return;
	}

	if (reassoc) {
		capab_info = le_to_host16(mgmt->u.reassoc_req.capab_info);
		listen_interval = le_to_host16(
			mgmt->u.reassoc_req.listen_interval);
		printf("association request: STA=" MACSTR " capab_info=0x%02x "
		       "listen_interval=%d current_ap=" MACSTR "\n",
		       MAC2STR(mgmt->sa), capab_info, listen_interval,
		       MAC2STR(mgmt->u.reassoc_req.current_ap));
	} else {
		capab_info = le_to_host16(mgmt->u.assoc_req.capab_info);
		listen_interval = le_to_host16(
			mgmt->u.assoc_req.listen_interval);
		printf("association request: STA=" MACSTR " capab_info=0x%02x "
		       "listen_interval=%d\n",
		       MAC2STR(mgmt->sa), capab_info, listen_interval);
	}

	/* FIX: parse supp_rates and ssid; check whether ssid is ours */

	/* FIX: get STA data */
	/* FIX: allocate AID (old or new) */

	if (0 /* FIX: not yet authenticated */) {
		printf("STA " MACSTR " trying to associate before "
		       "authentication\n", MAC2STR(mgmt->sa));
		send_deauth = 1;
		resp = WLAN_STATUS_UNSPECIFIED_FAILURE;
		goto fail;
	}

	if (0 /* FIX: no more space for new AIDs */) {
		printf("no room for more AIDs\n");
		resp = WLAN_STATUS_AP_UNABLE_TO_HANDLE_NEW_STA;
		goto fail;
	}

	/* FIX: set STA flag assocated; add to kernel driver */


 fail:

	/* use the queued buffer for transmission because it is large enough
	 * and not needed anymore */
	mgmt->frame_control =
		IEEE80211_FC(WLAN_FC_TYPE_MGMT,
			     (send_deauth ? WLAN_FC_STYPE_DEAUTH :
			      (reassoc ? WLAN_FC_STYPE_REASSOC_RESP :
			       WLAN_FC_STYPE_ASSOC_RESP)));
	memcpy(mgmt->da, mgmt->sa, ETH_ALEN);
	memcpy(mgmt->sa, hapd->own_addr, ETH_ALEN);
	/* Addr3 = BSSID - already set */

	send_len = IEEE80211_HDRLEN;
	if (send_deauth) {
		send_len += sizeof(mgmt->u.deauth);
		mgmt->u.deauth.reason_code = host_to_le16(resp);
	} else {
		char *p;
		send_len += sizeof(mgmt->u.assoc_resp);
		mgmt->u.assoc_resp.capab_info =
			host_to_le16(WLAN_CAPABILITY_ESS);
		mgmt->u.assoc_resp.status_code = host_to_le16(resp);
		mgmt->u.assoc_resp.aid = host_to_le16(0 /* FIX */
						      | BIT(14) | BIT(15));
		/* Supported rates */
		p = (char *) mgmt->u.assoc_resp.variable;
		*p++ = WLAN_EID_SUPP_RATES;
		*p++ = 4; /* len */
		*p++ = 0x82; /* 1 Mbps, base set */
		*p++ = 0x84; /* 2 Mbps, base set */
		*p++ = 0x0b; /* 5.5 Mbps */
		*p++ = 0x16; /* 11 Mbps */
	}

	if (send(hapd->sock, mgmt, send_len, 0) < 0)
		perror("handle_assoc: send");
}


static void handle_mgmt(hostapd *hapd, char *buf, size_t len, u16 stype)
{
	struct ieee80211_mgmt *mgmt = (struct ieee80211_mgmt *) buf;

	if (stype == WLAN_FC_STYPE_BEACON) {
		printf("mgmt::beacon\n");
		/* FIX: implement handle_beacon() */
		return;
	}

	if (memcmp(mgmt->bssid, hapd->own_addr, ETH_ALEN) != 0) {
		printf("MGMT: BSSID=" MACSTR " not our address\n",
		       MAC2STR(mgmt->bssid));
		return;
	}


	if (stype == WLAN_FC_STYPE_PROBE_REQ) {
		printf("mgmt::probe_req\n");
		return;
	}

	if (memcmp(mgmt->da, hapd->own_addr, ETH_ALEN) != 0) {
		printf("MGMT: DA=" MACSTR " not our address\n",
		       MAC2STR(mgmt->da));
		return;
	}

	switch (stype) {
	case WLAN_FC_STYPE_AUTH:
		printf("mgmt::auth\n");
		handle_auth(hapd, mgmt, len);
		break;
	case WLAN_FC_STYPE_ASSOC_REQ:
		printf("mgmt::assoc_req\n");
		handle_assoc(hapd, mgmt, len, 0);
		break;
	case WLAN_FC_STYPE_REASSOC_REQ:
		printf("mgmt::reassoc_req\n");
		handle_assoc(hapd, mgmt, len, 1);
		break;
	default:
		printf("unknown mgmt frame subtype %d\n", stype);
		break;
	}
}


static void handle_frame(hostapd *hapd, char *buf, size_t len)
{
	struct ieee80211_hdr *hdr;
	u16 fc, extra_len, type, stype;
	unsigned char *extra = NULL;
	size_t data_len = len;
	int ver;

	if (len < 24) {
		printf("handle_frame: too short (%d)\n", len);
		return;
	}

	hdr = (struct ieee80211_hdr *) buf;
	fc = le_to_host16(hdr->frame_control);

	ver = fc & WLAN_FC_PVER;

	/* protocol version 3 is reserved for indicating extra data after the
	 * payload */
	if (ver == 3) {
		u16 *elen;
		elen = (u16 *) (buf + len - 1);
		extra_len = le_to_host16(*elen);
		printf("extra data in frame (elen=%d)\n", extra_len);
		if (extra_len + 2 > len) {
			printf("  extra data overflow\n");
			return;
		}
		len -= extra_len + 2;
		extra = buf + len;
	} else if (ver != 0) {
		printf("unknown protocol version %d\n", ver);
		return;
	}

	type = WLAN_FC_GET_TYPE(fc);
	stype = WLAN_FC_GET_STYPE(fc);

	switch (type) {
	case WLAN_FC_TYPE_MGMT:
		printf("MGMT\n");
		handle_mgmt(hapd, buf, data_len, stype);
		break;
	case WLAN_FC_TYPE_CTRL:
		printf("CTRL\n");
		break;
	case WLAN_FC_TYPE_DATA:
		printf("DATA\n");
		break;
	default:
		printf("unknown frame type %d\n", type);
		break;
	}
}


int main(int argc, char *argv[])
{
	hostapd hapd;
	int i, len;
	unsigned char buf[3000];
        struct ifreq ifr;
	struct sockaddr_ll addr;

	if (argc != 2) {
		fprintf(stderr, "usage: hostapd <wlan#>\n");
		exit(1);
	}

	memset(&hapd, 0, sizeof(hapd));

	hapd.sock = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_ALL));
	if (hapd.sock < 0) {
		perror("socket");
		exit(1);
	}

        memset(&ifr, 0, sizeof(ifr));
        snprintf(ifr.ifr_name, sizeof(ifr.ifr_name), "%sap", argv[1]);
        if (ioctl(hapd.sock, SIOCGIFINDEX, &ifr) != 0) {
		perror("ioctl(SIOCGIFINDEX)");
                close(hapd.sock);
                exit(1);
        }

	memset(&addr, 0, sizeof(addr));
	addr.sll_family = AF_PACKET;
	addr.sll_ifindex = ifr.ifr_ifindex;
	printf("Opening raw packet socket for ifindex %d\n", addr.sll_ifindex);

	if (bind(hapd.sock, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
		perror("bind");
		close(hapd.sock);
		exit(1);
	}

        memset(&ifr, 0, sizeof(ifr));
        snprintf(ifr.ifr_name, sizeof(ifr.ifr_name), "%s", argv[1]);
        if (ioctl(hapd.sock, SIOCGIFHWADDR, &ifr) != 0) {
		perror("ioctl(SIOCGIFHWADDR)");
                close(hapd.sock);
                exit(1);
        }

	if (ifr.ifr_hwaddr.sa_family != ARPHRD_ETHER) {
		printf("Invalid HW-addr family 0x%04x\n",
		       ifr.ifr_hwaddr.sa_family);
		close(hapd.sock);
		exit(1);
	}
	memcpy(hapd.own_addr, ifr.ifr_hwaddr.sa_data, ETH_ALEN);

	printf("Using interface %sap with hwaddr " MACSTR "\n",
	       argv[1], MAC2STR(hapd.own_addr));


	for (;;) {
		len = recv(hapd.sock, buf, sizeof(buf), 0);
		if (len < 0) {
			perror("recvfrom");
			exit(1);
		}
		printf("Received %d bytes:", len);
		for (i = 0; i < len; i++)
			printf(" %02x", buf[i]);
		printf("\n");

		handle_frame(&hapd, buf, len);
	}

	return 0;
}
