/* Last Change: 20010429 - doe */
/*
 *  $Id: P54-06,v 1.16 1998/12/10 00:01:28 route Exp $
 *  Trusted path ACL implementation for OpenBSD 2.4
 *
 *  Copyright (c) 1998 route|daemon9 and Mike D. Schiffman
 *  All rights reserved.
 *  Originally published in Phrack Magazine (http://www.phrack.com).
 *
 *  Thanks to nirva for helping me choose an ADT.
 *  See <sys/kern_tpe.h> for more info.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <sys/kern_tpe.h>

void
tpe_init()
{
	memset(tpe_acl, TPE_INITIALIZER, sizeof(uid_t) * TPE_ACL_SIZE);
	tpe_acl_candidates = 0;
	tpe_ld_check = 1;
#if (AUTO_ADD_ROOT)
	tpe_acl[0] = 0;
#endif
}


void
tpe_show()
{
	int             i;

	printf("%d trusted users: ", tpe_acl_candidates);
	for (i = 0; i < tpe_acl_candidates; i++) {
		printf("%d ", tpe_acl[i]);
	}
	printf("\n");
}


int
tpe_add(uid_t candidate)
{
	if (tpe_acl_candidates == TPE_ACL_SIZE) {
		/*
	         *  Full list.
	         */
		return (NACK);
	}
	/*
         *  Don't add duplicates.
         */
	if ((tpe_search(candidate, 0, tpe_acl_candidates)) == NACK) {
		/*
	         *      Add to the end of the list, then sort.
	         */
		tpe_acl_candidates++;
		tpe_acl[tpe_acl_candidates] = candidate;
		tpe_sort(0, tpe_acl_candidates);

		printf("tpe: UID %d added to trust list\n", candidate);
	} else {
		printf("tpe: duplicate UID %d not added\n", candidate);
	}
	return (ACK);
}


int
tpe_remove(uid_t candidate)
{
	int             n;

	if (tpe_acl_candidates == 0) {
		/*
	         *  Empty list.
	         */
		return (NACK);
	}
	if ((n = tpe_search(candidate, 0, tpe_acl_candidates)) != NACK) {
		/*
	         *  Remove the candidate (mark the slot as unused), resort the list.
	         */
		tpe_acl[n] = TPE_INITIALIZER;
		tpe_acl_candidates--;
		tpe_sort(0, tpe_acl_candidates);

		printf("tpe: UID %d removed from trust list\n", candidate);
		return (ACK);
	}
	/*
         *  Not found.
         */
	return (NACK);
}


int
tpe_verify(uid_t candidate)
{
	if ((tpe_search(candidate, 0, tpe_acl_candidates)) != NACK) {
		return (ACK);
	} else {
		return (NACK);
	}
}


void
tpe_sort(int low, int high)
{
	int             i, j, n;

	/*
         *  Standard insertion sort.
         */
	for (i = low + 1; i <= high; i++) {
		COMPSWAP(tpe_acl[low], tpe_acl[i]);
	}

	for (i = low + 2; i <= high; i++) {
		j = i;
		n = tpe_acl[i];
		while (LESS(n, tpe_acl[j - 1])) {
			tpe_acl[j] = tpe_acl[j - 1];
			j--;
		}
		tpe_acl[j] = n;
	}
}

int
tpe_search(uid_t candidate, int low, int high)
{
	int             n;

	/*
         *  Standard binary search.  XXX - should be iterative.
         */
	n = (low + high) / 2;

	if (low > high) {
		return (NACK);
	}
	if (candidate == tpe_acl[n]) {
		return (n);
	}
	if (low == high) {
		return (NACK);
	}
	if (LESS(candidate, tpe_acl[n])) {
		return (tpe_search(candidate, low, n - 1));
	} else {
		return (tpe_search(candidate, n + 1, high));
	}
}
